/*
 * C interface to gpg
 *
 * error reporting: function return value
 * error messages: print to stderr
 * (rationale: most error messages come from gpg anyway, and I'd
 * like to be able to reuse this module in other programs)
 *
 * todo:
 * [ ] remove netbuild dependencies
 * - concat
 * - pathnames.h
 * - malloc_or_else
 *
 * [ ] should pass all operating parameters in a single structure
 * - working directory
 * - verbosity
 * - stream to be used for error reporting
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdarg.h>
#include <string.h>
#include <sys/wait.h>
#include <unistd.h>
#include "concat.h"
#include "gpglib.h"
#include "misc.h"
#include "pathnames.h"
#include "spawn.h"

static int verbose = 0;

void
gpg_set_verbosity (int verbosity)
{
    verbose = verbosity;
}

/* 
 * wrapper around spawn_in_dir() which handles verbose flag
 */

static int
do_command (char *dir, char **command, int fd1, int fd2, int fd3)
{
    int status;
    char *foo;

    if (verbose > 0) {
	int i;

	fprintf (stderr, "+");
	for (i = 0; command[i]; ++i)
	    fprintf (stderr, " %s", command[i]);
	fprintf (stderr, "\n");
    }
    status = spawn_in_dir (dir, command, fd1, fd2, fd3);
    if (verbose > 0)
	fprintf (stderr, "=> exit status %d\n", status);
    return status;
}

int
gpg_import_keys (char *dir, char *new_keyring)
{
    char *child_argv[20];
    int i = 0;
    char *gpgpath = PATH_GPG;
    int result;
    char **command;
    char *foo;

    if (new_keyring == NULL)
	return -1;

#if 0
    if ((gpgpath = getenv ("GPGPATH")) == NULL)
	return -1;
#endif

    if (*new_keyring != '/') {
	char cwd[1024];

	getcwd (cwd, sizeof (cwd));
	new_keyring = concat (cwd, "/", new_keyring, NULL);
    }
    command = make_argv (gpgpath, "--homedir", ".", "--import",
			 new_keyring, NULL);
    result = do_command (dir, command,
			 verbose > 0 ? 1 : -1,
			 verbose > 0 ? 2 : -1,
			 -1);
    free_argv (command);
    return result;
}

int
gpg_verify (char *dir, char *file, char *signature, char *status)
{
    char *child_argv[20];
    int i = 0;
    char *gpgpath = PATH_GPG;
    char **command;
    int result;
    int fd3;

    if (file == NULL || signature == NULL || status == NULL)
	return -1;

#if 0
    if ((gpgpath = getenv ("GPGPATH")) == NULL)
	return -1;
#endif

    status = concat (dir, "/", status, NULL);
    if ((fd3 = open (status, O_RDWR|O_CREAT|O_TRUNC, 0600)) < 0) {
	fprintf (stderr, "gpg_verify: open (%s): %s\n", status,
		 strerror (errno));
	return -1;
    }
    free (status);

    command = make_argv (gpgpath, "--homedir", ".", "--status-fd", "3",
			 "--verify", signature, file, NULL);

    result = do_command (dir, command, 
			 (verbose > 0) ? 1 : -1, 
			 (verbose > 0) ? 2 : -1, 
			 fd3);
    free_argv (command);
    close (fd3);

    return result;
}

#define MIN_GPG_VERSION "1.2.2"

int
gpg_check_version ()
{
    char *gpgpath = PATH_GPG;
    int fd1;
    char tempname[20];
    FILE *fp;
    char buf[1024];
    char **command;
    int result;

    strcpy (tempname, "/tmp/gpglib.XXXXXX");
    if ((fd1 = mkstemp (tempname)) < 0) {
	fprintf (stderr, "mkstemp(%s): %s\n", tempname, strerror (errno));
	return -1;
    }

    command = make_argv (gpgpath, "--version", NULL);
    result = do_command (".", command, fd1, 2, 3);
    free_argv (command);
    close (fd1);

    if (result != 0) {
	fprintf (stderr,
		 "gpg_check_version: %s --version exited with status %d\n",
		 gpgpath, result);
	return -1;
    }

    if ((fp = fopen (tempname, "r")) == NULL) {
	fprintf (stderr, "fopen (%s): %s\n", tempname, strerror (errno));
	return -1;
    }
    if (fscanf (fp, "gpg (GnuPG) %20s", tempname) != 1) {
	fprintf (stderr,
		 "gpg_check_version: unexpected version string format\n");
	fclose (fp);
	return -1;
    }
    fclose (fp);

    if (version_compare (MIN_GPG_VERSION, tempname) > 0) {
	fprintf (stderr, "%s is version %s, at least %s required\n",
		 gpgpath, tempname, MIN_GPG_VERSION);
	return -1;
    }
    return 0;
}

int
gpg_check_status (char *dir, char *status_file)
{
    char **command;
    int result;
    int fd1;

    fd1 = open ("/dev/null", O_WRONLY, 0);
    command = make_argv ("grep", "TRUST_UNDEFINED", status_file, NULL);
    result = do_command (dir, command, fd1, fd1, 3);

    free_argv (command);
    close (fd1);

    return (result == 0) ? -1 : 0;
}

#ifdef TEST
main ()
{
    gpg_check_version ();
}
#endif
