/*
 * functions callable from constraint expressions
 *
 * error reporting: special return value for expression errors
 * error messages: none
 *
 * dependencies: custom, symtab
 *
 * todo:
 */

#include <stdio.h>
#include <unistd.h>
#include "custom.h"
#include "symtab.h"

extern struct string_list *dir_list;
extern struct pattern *lib_pattern_list;

static VAL *
fn_version_compare (VAL *args)
{
    static VAL result;
    char *a1, *a2;

    if (args->next == NULL || args->next->next != NULL) 
	result.valtype = ERR;
    else if (args->valtype != STR || args->next->valtype != STR)
	result.valtype = ERR;
    else {
        result.valtype = INT;
    	result.intval = version_compare (args->strval, args->next->strval);
    }
    return &result;
}


/*
 * have_file (filename, mode)
 * 
 * mode can contain any of 'r', 'w', 'x' or be the empty string.
 * if not specified, mode is treated as the empty string.
 */

static VAL *
fn_have_file (VAL *args)
{
    static VAL result;
    char *a2;
    int mode = 0;

    if (args->valtype != STR)
	goto err;

    if (args->next == NULL) 
	a2 = "";
    else if (args->next->valtype == STR && args->next->next == NULL)
	a2 = args->next->strval;
    else
	goto err;

    while (*a2) {
	switch (*a2) {
	    case 'r':
		mode |= R_OK;
		break;
	    case 'w':
		mode |= W_OK;
		break;
	    case 'x':
		mode |= X_OK;
		break;
	    default:
		goto err;
	}
        ++a2;
    }
    if (mode == 0)
	mode = F_OK;
    result.valtype = INT;
    result.intval = (access (args->strval, mode) == 0);
    return &result;

err:
    result.valtype = ERR;
    return &result;
}

static VAL *
fn_have_library (VAL *args)
{
    static VAL result;
    struct string_list *dir;
    struct pattern *pat;
    int i;

    if (args->valtype != STR || args->next != NULL)
	goto err;

    result.valtype = INT;
    result.intval = 0;
    for (dir = dir_list; dir; dir=dir->next) {
        for (pat = lib_pattern_list; pat; pat=pat->next) {
            char huge[1024];    /* XXX */

            sprintf (huge, pat->pat, args->strval);
#if 0
	    fprintf (stderr, "looking for %s in %s\n", huge, dir->this);
#endif
            if (find_in_dir (dir->this, huge)) {
                result.intval = 1;
		goto done;
            }
        }
    }
done:
    return &result;

err:
    result.valtype = ERR;
    return &result;
}

static VAL *
fn_error (VAL *args)
{
    static VAL result;

    result.valtype = INT;
    result.intval = 0;
    if (args == NULL || args->valtype == ERR)
	result.intval = 1;
    return &result;
}

static VAL *
fn_print (VAL *args)
{
    static VAL result;

    fprintf (stderr, "\n");
    symtab_printf_val (stderr, args);
    fprintf (stderr, "\n");

    result.valtype = INT;
    result.intval = 1;
    return &result;
}

static VAL *
fn_string_in_list (VAL *args)
{
    static VAL result;
    char *str0;

    result.valtype = INT;

    if (args == NULL || args->valtype != STR) {
	result.valtype = ERR;
	return &result;
    }
    str0 = args->strval;

    args = args->next;
    while (args) {
	if (args->valtype != STR) {
	    result.valtype = ERR;
	    return &result;
	}
	if (strcmp (str0, args->strval) == 0) {
	    result.intval = 1;
	    return &result;
	}
	args = args->next;
    }
    result.intval = 0;
    return &result;
}

expr_functions_init ()
{
    symtab_bind_function ("version_compare", fn_version_compare);
    symtab_bind_function ("have_file", fn_have_file);
    symtab_bind_function ("have_library", fn_have_library);
    symtab_bind_function ("error", fn_error);
    symtab_bind_function ("print", fn_print);
    symtab_bind_function ("string_in_list", fn_string_in_list);
}
