%{
/*
 * evaluator for metadata constraint expressions
 *
 * error reporting: special function return value for expression errors
 * error messages: none
 *
 * dependencies: symtab
 *
 * todo:
 */

#include "conf.h"

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_STRING_H
#include <string.h>
#else
char *strdup ();
#endif
#include "symtab.h"
#include "expr.h"

static char yytext[1000];

static VAL errval = { ERR };
static VAL oneval = { INT, 1 };
static VAL zeroval = { INT, 0 };
static VAL result ;
static VAL lookup_id ();
static char *concat ();
static int yyerror ();
static char *tempstr;
extern VAL val_proto;
%}

%start expression

%token CONST ID STRING 

%right '?' ':'
%left OR
%left AND
%left '|'
%left '^'
%left '&'
%left EQ NEQ
%left '<' '>' LEQ GEQ
%left RSHIFT LSHIFT
%left '+' '-'
%left '*' '/' '%'
%right UMINUS '!' '~'
%union {
    VAL val;
}
%type <val> expression expr id arglist STRING CONST ID

%%
expression: expr
		{
		    result = $1;
		}
	;

id: ID
		{
		    extern char yytext[];

		    /*
		     * this is just a way to save yytext before
		     * it's gone.  I'm too lazy to set up a separate
		     * yacc type to hold just an identifier.
		     */
		    $$ = val_proto;
		    $$.strval = strdup (yytext);
		}
	;


arglist: expr
		{
		    $$ = *(symtab_copyval (&($1)));
		}
	| arglist ',' expr
		{
		    $$ = *(symtab_append (&($1), symtab_copyval (&($3))));
		}
	;

expr: 
	id
		{
		    struct symbol *ptr;

		    ptr = symtab_get_symbol ($1.strval);
		    if (ptr == NULL)
			$$ = errval;
		    else if (ptr->value.valtype == FUNC)
			$$ = errval;
		    else if (ptr->value.valtype == PSEUDO) {
			VAL bar;

			(*ptr->value.pseudo)('g', $1.strval, &bar,
					     ptr->value.pseudo_arg);
			$$ = bar;
		    }
		    else
			$$ = ptr->value;
		}
	| id '(' arglist ')'
		{
		    struct symbol *ptr;

		    ptr = symtab_get_symbol ($1.strval);

		    if (ptr == NULL)
			$$ = errval;
		    else if (ptr->value.valtype != FUNC)
			$$ = errval;
		    else
		        $$ = *(*(ptr->value.funval))(&($3));
		}
	| id '(' ')'
		{
		    struct symbol *ptr;

		    ptr = symtab_get_symbol ($1.strval);

		    if (ptr == NULL)
			$$ = errval;
		    else if (ptr->value.valtype != FUNC)
			$$ = errval;
		    else
		        $$ = *(*(ptr->value.funval))(NULL);
		}
	| CONST
		{
		    extern char yytext[];

		    $$ = val_proto;
		    $$.valtype = INT;
		    $$.intval = atoi (yytext);
		}
	| STRING
		{
		    $$ = val_proto;
		    $$.valtype = STR;
		    $$.strval = strdup (yytext); /* XXX core leakage */
		}
	| '(' expr ')'
		{
		    $$ = $2;
		}
	| '-' expr	%prec UMINUS
		{
		    if ($2.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = - $2.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| '!' expr
		{
		    if ($2.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = ! $2.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| '~' expr
		{
		    if ($2.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = ~ $2.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr '*' expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval * $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr '/' expr
		{
		    if ($1.valtype == INT && $3.valtype == INT && $3.intval != 0) {
			$$.valtype = INT;
			$$ = val_proto;
			$$.intval = $1.intval / $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr '%' expr
		{
		    if ($1.valtype == INT && $3.valtype == INT && $3.intval != 0) {
			$$.valtype = INT;
			$$ = val_proto;
			$$.intval = $1.intval % $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr '+' expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$.valtype = INT;
			$$ = val_proto;
			$$.intval = $1.intval + $3.intval;
		    }
		    else if ($1.valtype == STR && $3.valtype == STR) {
			$$ = val_proto;
			$$.valtype = STR;
			$$.strval = concat ($1.strval, $3.strval);
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr '-' expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval - $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr RSHIFT expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval >> $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr LSHIFT expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$.valtype = INT;
			$$ = val_proto;
			$$.intval = $1.intval << $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr '<' expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval < $3.intval;
		    }
		    else if ($1.valtype == STR && $3.valtype == STR) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = strcmp ($1.strval, $3.strval) < 0;
		    }
		    else {
			$$ = zeroval;
		    }
		}
	| expr '>' expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval > $3.intval;
		    }
		    else if ($1.valtype == STR && $3.valtype == STR) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = strcmp ($1.strval, $3.strval) > 0;
		    }
		    else {
			$$ = zeroval;
		    }
		}
	| expr LEQ expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval <= $3.intval;
		    }
		    else if ($1.valtype == STR && $3.valtype == STR) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = strcmp ($1.strval, $3.strval) <= 0;
		    }
		    else {
			$$ = zeroval;
		    }
		}
	| expr GEQ expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval >= $3.intval;
		    }
		    else if ($1.valtype == STR && $3.valtype == STR) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = strcmp ($1.strval, $3.strval) >= 0;
		    }
		    else {
			$$ = zeroval;
		    }
		}
	| expr EQ expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval == $3.intval;
		    }
		    else if ($1.valtype == STR && $3.valtype == STR) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = strcmp ($1.strval, $3.strval) == 0;
		    }
		    else {
			$$ = zeroval;
		    }
		}
	| expr NEQ expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval != $3.intval;
		    }
		    else if ($1.valtype == STR && $3.valtype == STR) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = strcmp ($1.strval, $3.strval) != 0;
		    }
		    else {
			$$ = oneval;
		    }
		}
	| expr '&' expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval & $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr '^' expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval ^ $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr '|' expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval | $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr AND expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$.valtype = INT;
			$$.intval = $1.intval && $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr OR expr
		{
		    if ($1.valtype == INT && $3.valtype == INT) {
			$$ = val_proto;
			$$.valtype = INT;
			$$.intval = $1.intval || $3.intval;
		    }
		    else {
			$$ = errval;
		    }
		}
	| expr '?' expr ':' expr
		{
		    if ($1.valtype != INT)
			$$ = errval;
		    $$ = $1.intval ? $3 : $5;
		}
	;

%%

/* eschew ctype because of Posix locale brain-damage */

#define ISDIGIT(c) ((c) >= '0' && (c) <= '9')
#define ISODIGIT(c) ((c) >= '0' && (c) <= '7')
#define ISXDIGIT(c) (ISDIGIT(c) || ((c) >= 'a' && (c) <= 'f') || ((c) >= 'A' && (c) <= 'F'))
#define ISUPPER(c) ((c) >= 'A' && (c) <= 'Z')
#define ISLOWER(c) ((c) >= 'a' && (c) <= 'z')
#define ISALPHA(c) (ISUPPER(c) || ISLOWER(c))
#define ISALNUM(c) (ISALPHA(c) || ISDIGIT(c))

static char *ptr;
static int error = 0;

struct {
    char *str;
    int sym;
} twos[] = {
    { ">>", RSHIFT },
    { "<<", LSHIFT },
    { "<=", LEQ },
    { ">=", GEQ },
    { "==", EQ },
    { "!=", NEQ },
    { "&&", AND },
    { "||", OR },
};

/*
 * lookup an identifier.
 * if the identifier is not defined, treat as if it were zero
 */

static struct val
lookup_id (str)
char *str;
{
    struct val val;

    if (symtab_get_value (str, &val) < 0)
	return zeroval;
    return val;
}

static char *ones = "()-!~*/%+<>&|^?:,";

static int
unhex (c)
char c;
{
    if (c >= 'A' && c <= 'F')
	return c - 'A' + 10;
    if (c >= 'a' && c <= 'f')
	return c - 'a' + 10;
    if (c >= '0' && c <= '9')
	return c - '0';
}

static char *
concat (a, b)
char *a, *b;
{
    int len = 1;
    char *result;

    if (a)
	len += strlen (a);
    if (b)
	len += strlen (b);
    result = malloc (len);
    *result = '\0';
    if (a)
	strcpy (result, a);
    if (b)
	strcat (result, b);
    return result;
}

int
yylex ()
{
    int i;

    while (*ptr == ' ' || *ptr == '\t' || *ptr == '\r' || *ptr == '\n')
	++ptr;

    /* two-character symbols */
    for (i = 0; i < sizeof(twos)/sizeof(*twos); ++i) {
	if ((ptr[0] == twos[i].str[0]) && (ptr[1] == twos[i].str[1])) {
	    yytext[0] = ptr[0];
	    yytext[1] = ptr[1];
	    yytext[2] = '\0';
	    ptr += 2;
	    return twos[i].sym;
	}
    }
    /* one-character symbols */
    for (i = 0; ones[i]; ++i) {
	if (ptr[0] == ones[i]) {
	    yytext[0] = ones[i];
	    yytext[1] = '\0';
	    ++ptr;
	    return ones[i];
	}
    }
    /* integer constants (hex, octal, decimal) */
    if (ISDIGIT (*ptr)) {
	char *dst = yytext;

	if (*ptr == '0' && (ptr[1] == 'x' || ptr[1] == 'X')) {
	    *dst++ = *ptr++;
	    *dst++ = *ptr++;
	    while (dst < (yytext + sizeof(yytext) - 1) && ISXDIGIT (*ptr))
		*dst++ = *ptr++;
	    *dst++ = '\0';
	}
	else if (*ptr == '0') {
	    *dst++ = *ptr++;
            while (dst < (yytext + sizeof(yytext) - 1) && ISODIGIT (*ptr))
                *dst++ = *ptr++;
            *dst++ = '\0';
	}
	else {
	    while (dst < (yytext + sizeof(yytext) - 1) && ISDIGIT (*ptr))
	        *dst++ = *ptr++;
	    *dst++ = '\0';
	}
	return CONST;
    }
    /* identifiers */
    else if (ISALPHA (*ptr)) {
	char *dst = yytext;

        while (dst < (yytext + sizeof(yytext) - 1) && 
               ISALNUM (*ptr) || *ptr == '.' || *ptr == '_' || *ptr == '$') 
            *dst++ = *ptr++;
        *dst++ = '\0';
	return ID;
    }
    /* quoted strings */
    else if (*ptr == '"') {
	char *dst = yytext;

	++ptr;
        while (dst < (yytext + sizeof(yytext) - 2)) {
	    int octal = 0;

	    if (*ptr == '\0')
		return 0;
	    else if (*ptr == '"') {
		++ptr;
		*dst++ = '\0';
		return STRING;
	    }
	    if (*ptr == '\\') {
		++ptr;
		switch (*ptr) {
		case 'a': *dst++ = '\007'; ptr++; break;
		case 'b': *dst++ = '\b'; ptr++; break;
		case 'e': *dst++ = '\033'; ptr++; break;
		case 'f': *dst++ = '\f'; ptr++; break;
		case 'n': *dst++ = '\n'; ptr++; break;
		case 'r': *dst++ = '\r'; ptr++; break;
		case 't': *dst++ = '\t'; ptr++; break;
		case 'v': *dst++ = '\v'; ptr++; break;
		case '\\': *dst++ = '\\'; ptr++; break;
		case 'x':
		    ++ptr;
		    octal = 0;
		    if (ISXDIGIT (*ptr))
			octal = unhex (*ptr++);
		    if (ISXDIGIT (*ptr))
			octal = octal * 16 + unhex (*ptr++);
		    *dst++ = octal;
		    break;
		default:
		    if (ISDIGIT (*ptr)) {
			octal = *ptr++ - '0'; 
			if (ISDIGIT (*ptr))
			    octal = octal * 8 + *ptr++ - '0';
			if (ISDIGIT (*ptr))
			    octal = octal * 8 + *ptr++ - '0';
			*dst++ = octal;
		    }
		    else
			*dst++ = *ptr++;
		}
	    }
	    else
		*dst++ = *ptr++;
	}
    }
    /* anything else */
    else {
	yytext[0] = *ptr++;
	yytext[1] = '\0';
	return 0;
    }
}

static int
yyerror ()
{
    error = 1;
}

VAL *
expr_evaluate (char *str)
{
    ptr = str;
    error = 0;
    yyparse (); 
    if (error)
	return &errval;
    return &result;
}
