/*
 * routines to allow site and platform-specific
 * customization of nb behavior
 *
 * error reporting: function return values
 * error messages: print to stderr
 *
 * dependencies: netbuild
 *
 * todo:
 */

#include "conf.h"

#include <stdio.h>
#ifdef STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif
#ifdef HAVE_LIBGEN_H
#include <libgen.h>
#endif
#include <unistd.h>
#include "concat.h"
#include "custom.h"
#include "field.h"
#include "global.h"
#include "misc.h"
#include "netfind.h"
#include "pathnames.h"
#include "symtab.h"

#ifdef TEST
#include "arch.h"

int verbosity = 1;

#else
extern int verbosity;
#endif

void
append (struct list **list, struct list *new)
{
    struct list *ptr;

    if (*list == NULL)
	*list = new;
    else {
	for (ptr = *list; ptr->next != NULL; ptr=ptr->next);
	ptr->next = new;
    }
    new->next = NULL;
}

/*
 * append a string to a list of strings
 */

void
append_string (struct string_list **list, char *str)
{
    struct string_list *new =
	(struct string_list *) malloc (sizeof (struct string_list));
    struct string_list *ptr;

    new->this = strdup (str);
    append ((struct list **) list, (void *) new);
}

static void
add_option (char *opt, int nargs, int bits)
{
    extern struct options_list *options_list; /* netbuild.c */

    struct option *new =
	(struct option *) malloc (sizeof (struct option));
    
#if 0
    fprintf (stderr, "add_option (%s nargs=%d bits=%x)\n", opt, nargs, bits);
#endif
    new->pattern = strdup (opt);
    new->nargs = nargs;
    new->flags = bits;
    append ((struct list **) &options_list, (struct list *) new);
}

static void
add_library (char *libname)
{
    extern struct string_list *dir_list; /* netbuild.c */

#if 0
    fprintf (stderr, "add_library (%s)\n", libname);
#endif
    append_string (&dir_list, libname);
}

static void
add_library_pattern (char *pattern, int flags)
{
    extern struct pattern_list *lib_pattern_list; /* netbuild.c */

    struct pattern *new =
	(struct pattern *) malloc (sizeof (struct pattern));

#if 0
    fprintf (stderr, "add_library_pattern (%s, flags=%x)\n", pattern, flags);
#endif
    new->pat = pattern;
    new->flags = flags;
    append ((struct list **) &lib_pattern_list, (struct list *) new);
}

static int
split (char *str, char **args, int nargs, char c)
{
    char *cptr;
    int n = 0;

    while (*str) {
	if (n+1 >= nargs)
	    return -1;

	if (cptr = strchr (str, c)) {
	    *cptr = '\0';
	    args[n++] = str;
	    str = cptr + 1;
	}
	else {
	    args[n++] = str;
	    return n;
	}
    }
}

struct {
    char *name;
    int bits;
} flags[] = {
    { "libarg", LIBARG },
    { "static", STATIC },
    { "dynamic", DYNAMIC },
    { "localdir", LOCALDIR },
    { "verbose", VERBOSE },
};

static int
lookup_flag (char *name)
{
    int i;

    for (i = 0; i < sizeof (flags) / sizeof (*flags); ++i)
	if (strcasecmp (name, flags[i].name) == 0)
	    return flags[i].bits;
    return 0;
}

static void
check_number (char *filename, int lineno, char *s)
{
    char *t = s;

    while (*t) {
	if (!isdigit (*t)) {
	    fprintf (stderr, "%s, line %d: '%s' should be numeric\n",
		     filename, lineno, s);
	    exit (1);
	}
	++t;
    }
    return;
}

static void
process_option (char *filename, int lineno, char *buf)
{
    char *args[500];
    int i, n;
    int bits;

    n = split (buf, args, sizeof(args)/sizeof(*args), ':');
    switch (n) {
    case 0:
	break;
    case 1:
	add_option (args[0], 1, 0);
	break;
    case 2:
	check_number (filename, lineno, args[1]);
	add_option (args[0], atoi (args[1]), 0);
	break;
    case 3:
	check_number (filename, lineno, args[1]);
	bits = 0;
	for (i = 2; i < n; ++i)
	    bits |= lookup_flag (args[i]);
	add_option (args[0], atoi (args[1]), bits);
	break;
    }
}

static void
process_options (char *filename, int lineno, char *opts)
{
    char buf[1024];		/* XXX fixed-length array, may overflow */
    char *dst;

    while (*opts) {
	while (isspace (*opts))
	    ++opts;
	if (*opts == '\0')
	    return;
	if (*opts == '{') {
	    ++opts;
	    dst = buf;
	    while (*opts && *opts != '}')
		*dst++ = *opts++;
	    *dst = '\0';
	    process_option (filename, lineno, buf);
	    ++opts;
	}
    }
}


void 
process_library_path (char *buf)
{
    char *args[500];
    int i, n;

    n = split (buf, args, sizeof (args) / sizeof (*args), ':');
    for (i = 0; i < n; ++i)
	add_library (args[i]);
}


static void
process_library_pattern (char *filename, int lineno, char *buf)
{
    char *args[500];
    int n;
    int bits;

    n = split (buf, args, sizeof(args)/sizeof(*args), ':');
    switch (n) {
    case 2:
	add_library_pattern (args[0], lookup_flag (args[1]));
	break;
    }
}

static void
process_library_patterns (char *filename, int lineno, char *opts)
{
    char buf[1024];		/* XXX fixed-length array, may overflow */
    char *dst;

    while (*opts) {
	while (isspace (*opts))
	    ++opts;
	if (*opts == '\0')
	    return;
	if (*opts == '{') {
	    ++opts;
	    dst = buf;
	    while (*opts && *opts != '}')
		*dst++ = *opts++;
	    *dst = '\0';
	    process_library_pattern (filename, lineno, buf);
	    ++opts;
	}
	else {
	    /* XXX malformed library patterns */
	    return;
	}
    }
}

/*
 * find the config file to use.  check files in this order:
 * $HOME/NetBuild/$NB_ARCH/$HOSTNAME
 * $HOME/NetBuild/$NB_ARCH/default
 * $PATH_CONFIG_DIR/$HOSTNAME
 * $PATH_CONFIG_DIR/default
 */

char *
find_config_file (char *hostname, char *arch)
{
    char *filename;

    filename = concat (getenv ("HOME"), "/NetBuild/", arch, "/", hostname,
		       NULL);
    if (access (filename, R_OK) == 0)
	return filename;
    free (filename);

    filename = concat (getenv ("HOME"), "/NetBuild/", arch, "/default", NULL);
    if (access (filename, R_OK) == 0)
	return filename;
    free (filename);

    /*
     * for the normal case where the user who is running nb is the
     * same as the user who compiled it, these two tests are redundant.
     * but they should produce the same result.
     */

    filename = concat (PATH_CONFIG_DIR, "/", hostname, NULL);
    if (access (filename, R_OK) == 0)
	return filename;
    free (filename);

    filename = concat (PATH_CONFIG_DIR, "/default", NULL);
    if (access (filename, R_OK) == 0)
	return filename;
    free (filename);

    return NULL;
}

static int
count_newlines (char *str)
{
    int nl = 0;
    char *ptr;

    while (ptr = strchr (str, '\n')) {
	++nl;
	str = ptr + 1;
    }
    return nl;
}

/*
 * strip leading and trailing space characters
 */

static char *
strip (char *buf)
{
    char *ptr;

    while (isspace (*buf))
        ++buf;
    ptr = buf + strlen (buf) - 1;
    while (ptr > buf && isspace (*ptr)) {
        *ptr-- = '\0';
    }
    return buf;
}

static int
read_config_internal (char *filename, char *hostname, char *arch,
		      char *compiler)
{
    FILE *fp;
    char buf[1024];
    int clen;
    int lineno;

    clen = strlen (compiler);

    if ((fp = fopen (filename, "r")) == NULL) {
	perror (concat ("nb: ", filename, NULL));
	return;
    }
    lineno = 1;
    while (fgets (buf, sizeof (buf), fp) != NULL) {
	if (buf[0] == '[' && buf[clen+1] == ']' &&
	    strncasecmp (buf+1, compiler, clen) == 0 &&
	    isblankline (buf+clen+2)) {
	    while (get_header_field (buf, sizeof (buf), fp)) {
		if (*buf == '[') {
		    fclose (fp);
		    return 0;
		}
		else if (match_field_name ("options", buf))
		    process_options (filename, lineno, field_body (buf));
		else if (match_field_name ("library-path", buf))
		    process_library_path (field_body (buf));
		else if (match_field_name ("library-patterns", buf))
		    process_library_patterns (filename, lineno, field_body (buf));
#if 0
		else if (match_field_name ("run-time-environment", buf)) {
		    /* XXX might should be an expression to evaluate 
		       rather than just a string constant */
		    symtab_set_value_str ("source.run-time-environment", 
                                          field_body (buf));
		}
#endif
		else if (strchr (field_name (buf), '.')) {
#if 0
		    fprintf (stderr, "symtab_set_value_str (\"%s\", \"%s\")\n", field_name (buf), strip (field_body (buf)));
#endif
		    symtab_set_value_str (field_name (buf), strip (field_body (buf)));
		}
		else if (isblankline (buf))
		    ;
		else {
		    fprintf (stderr, "line %d: unrecognized config file setting\n", lineno);
		    fprintf (stderr, "%s", buf);
		}
		lineno += count_newlines (buf);
	    }
	}
	++lineno;
    }
    fclose (fp);

    return -1;
}

int
read_config (char *hostname, char *arch, char *compiler)
{
    char *filename = find_config_file (hostname, arch);
    int x;

    if (filename == NULL) {
	fprintf (stderr, "nb: no config file found\n");
	return -1;
    }

    if (verbosity > 0)
	fprintf (stderr, "[using config file %s]\n", filename);

    if (read_config_internal (filename, hostname, arch, compiler) == 0)
	return 0;
    /*
     * if compiler is an absolute path name, and we didn't find
     * a stanza for this compiler, try again with just the base
     * name of the compiler
     */
    if (*compiler == '/')
	return read_config_internal (filename, hostname, arch,
				     basename (compiler));
    return -1;
}


#ifdef TEST

main (int argc, char **argv)
{
    char hostname[1024];
    char *myarch;
    char *compiler;

    gethostname (hostname, sizeof (hostname));
    myarch = get_netbuild_arch ();
    compiler = argv[1];

    fprintf (stderr, "reading from %s\n", find_config_file (hostname, myarch));
    read_config (hostname, myarch, compiler);
}
#endif

