/*
 * Trace analysis pipe (create a sound based on a trace file)
 *		J-Y Peterschmitt - 02/93
 *
 * tr_sendmix.c
 * 
 * trace_file : stdin > tr_sendmix nb length scale > stdout : sound
 *
 * nb		: number of processors
 * length	: number of sound samples generated for each event,
 *		  or each period of scale microseconds (execution time)
 * scale	: when there is no new event happening at the current 
 *		  time, we skip scale microseconds (execution time).
 *		  In fact, we play a sound of length samples.
 *
 * ==> scale microseconds of execution time (time described by the
 * timestamps in the trace file) will be associated with length sound
 * samples (a sound lasting length / SMP_RATE seconds). When nothing
 * at all is happening, we play a "blank" sound (low frequency, low
 * intensity) of length samples.
 *
 * The program maps each of the nb processors to a unique frequency
 * (the highest proc number has the highest frequency).
 * Then, sequentially analyzing the input trace file, the program
 * plays the frequency associated with a processor, whenever this
 * processor has sent one or several messages, and that all these
 * messages have not yet been received. 
 * At a given time, we play simultaneously the frequencies associated
 * with all the currently sending processors.
 * We play a "sharper" tone when a processor starts sending a message.
 * The sound is louder when more processors are sending
 *
 */

#include <stdio.h>
#include <math.h>
#include <memory.h>
#include <ulaw2linear.h>
#include <audio_errno.h>
#include <audio_hdr.h>

#define PERROR(str_error) fprintf(stderr, "%s\n", str_error)

#define PG_TRACESTART	1      	/* subset of PICL			*/
#define PG_SEND		4      	/* event types used			*/
#define PG_RECV		6      	/* by ParaGraph				*/
#define PG_RECV_BLOCKING 7
#define PG_RECV_WAKING	8
#define PG_COMPSTATS	11
#define PG_TRACEEXIT	19
#define PG_BLOCKBEGIN	20
#define PG_BLOCKEND	21
#define PG_TRACEMARK	22

#define MAX_LINES 10		/* number of lines in the trace		*/
				/* buffer				*/
#define LINE_SIZE 100		/* max number of chars / line		*/
#define MAX_INFO 80		/* length of the info string		*/
#define OUT_BUFSIZE 8000	/* size of the sound buffer		*/
#define SMP_RATE 8000		/* number of samples / sec.		*/
#define PI 3.141593
#define F0 300			/* Min freq of the generated		*/
				/* sounds	      			*/
#define FN 3520			/* Max freq				*/
#define MAX_PROC 128		/* Max number of processors		*/
#define DFLT_NBPROC 16
#define DFLT_LENGTH 200		/* default length of a sound		*/
				/* event       				*/
#define MIN_LENGTH 50
#define DFLT_UNIT 5		/* length of a sound event in		*/
				/* microseconds		      		*/
#define MIN_UNIT 1
#define BLANK_FREQ 150		/* characteristics of the		*/
#define BLANK_MAG 0.04		/* blank sound				*/
#define INTENSITY 150.0		/* "intensity" of the generated		*/
				/* sounds	       			*/

typedef struct
{
    char str[LINE_SIZE];
} Trace;

Trace trace[MAX_LINES];		/* trace buffer				*/

short tsin[SMP_RATE];		/* samples of a 1 HZ sine		*/

int overflow_proc = DFLT_NBPROC;/* current value of the overflow limit	*/
int *freq_lut;			/* frequency lookup table 		*/
float *mag_lut;			/* magnitude lookup table		*/
int *cur_pos;			/* current position of a given processor*/
				/*in the sine wave array (current sample)*/
int *nb_waiting;		/* number of messages sent by a given	*/
				/* processor, but not received yet	*/

float *val_tab;			/* array of size "length", where the	*/
				/* sound is computed			*/

unsigned char out_buf[OUT_BUFSIZE], *p_out;	/* sound buffer		*/
int fd_out, out_bufsize;	/* number of samples in the sound buffer*/
int nb_proc, length, unit;     	/* command line parameters' value	*/

int current_time = -1;		/* curent time (timestamp) in		*/
				/* microseconds. This time is		*/
				/*increased as we analyze the trace file*/
int nb_sending = 0;		/* number of sending processors		*/
int emp = -1;			/* number of a proc whose sound needs to*/
				/* be emphasised at a given time	*/


/* Initialization function						*/

void tr_init()
{
    int		i;
    double	inc, angle;
    
    /* compute the sine table used to generate the sound 		*/
    
    for(angle = -PI, inc = 2*PI / (SMP_RATE-1), i = 0;
	i < SMP_RATE;
	i++, angle += inc)
    {
	tsin[i] = audio_d2s(sin(angle));
    }
    
    /* make sure the pointer to the current sample points to		*/
    /* the beginning of the out buffer					*/
    p_out = out_buf;
    out_bufsize = 0;
}

/* Open the stdout stream to start the pipe				*/

void pipe_open()
{
    fd_out = fileno(stdout);
}

/* Create a valid header for the generated sound	       		*/

void pipe_header()
{
    Audio_hdr au_desc;
    char au_info[MAX_INFO];
    
    au_desc.sample_rate = SMP_RATE;
    au_desc.samples_per_unit = 1;
    au_desc.bytes_per_unit = 1;
    au_desc.channels = 1;
    au_desc.encoding = AUDIO_ENCODING_ULAW;
    au_desc.data_size = AUDIO_UNKNOWN_SIZE;    
    sprintf(au_info, "Trace sound file generated by tr_sendmix");
    
    /* send the header to stdout	      				*/
    audio_write_filehdr(fd_out, &au_desc, au_info, 
			(unsigned) strlen(au_info));
}

/* Close the pipe      							*/

void pipe_close()
{
    (void) close(fd_out);
}

/* Read as many lines as possible, to fill the trace buffer		*/

int getlines()
{
    int i;
    
    for(i = 0; i < MAX_LINES; i++)
    {
	if(fgets(trace[i].str, LINE_SIZE, stdin) == NULL)
	{
	    break;     			/* end of file reached	       	*/
	}
    }	
    return(i);
}

/* Empty the sound buffer, if there is not enough room to store		*/
/* the samples corresponding to the next event			       	*/

void flush_outbuf()
{
    if((out_bufsize + length) >= OUT_BUFSIZE)
    {
	write(fd_out, (char *) out_buf, out_bufsize);
	p_out = out_buf;
	out_bufsize = 0;
    }
}

/* Create the frequency and magnitude lookup tables		       	*/

void create_lut(nb)
int nb;
{
    int i, freq;
    float lambda;
    
    /* There are two ways to compute the mapping between		*/
    /* the proc number and the frequency of the sound			*/
    /* - a linear distribution between the min freq allowed		*/
    /*		and the max frequency		 			*/
    /* - a logarithmic distribution between the min and the		*/
    /*		max freq      						*/
    /* In both cases, the magnitude of the sound is chosen		*/
    /* so that the product "freq * mag" remains constant,		*/
    /* in order to deal with sounds of the same "intensity"		*/
    
    /* linear distribution						*/
    /* f(k) = f(0) + k * (f(N) - f(0)) / N		   		*/
    /* where N = MAX_PROC_NO (max number of processors)			*/
    
    /* logarithmic distribution				       		*/
    /* f(k) = f(0) * exp( k/N * log(f(N) / f(0)))	       		*/
    /* where N = MAX_PROC_NO (max number of processors)			*/
    
    lambda = (float) log((double) (FN / F0)) / nb;
    for(i = 0; i < nb; i++)
    {
	freq = F0 * exp((double) (i * lambda));
	*(freq_lut + i) = freq;
	*(mag_lut + i) = INTENSITY / freq;
    }
}

/* Create a blank sound when nothing interesting is happening		*/

void create_blank()
{
    int i;
    static int ind = 0;
    
    for(i = 0; 
	i < length; 
	i++, ind = (ind + BLANK_FREQ) % SMP_RATE)
    {
	*p_out = audio_s2u((short) (BLANK_MAG * tsin[ind]));
	p_out++;
	out_bufsize++;
    }
}

/* Create the next LENGTH sound samples			      		*/

void create_au()
{
    int i, j, pos, freq, *p_pos, *p_waiting, *p_freq;
    float *p_mag, *p_value, mag, tot_mag, lambda;

    memset((char *) val_tab, 0, length * sizeof(float));
    tot_mag = 0;
    for(i = 0, p_pos = cur_pos, p_waiting = nb_waiting, p_freq = freq_lut,
	p_mag = mag_lut;
	i < nb_proc;
	i++, p_pos++, p_waiting++, p_freq++, p_mag++)
    {
	if(*p_waiting > 0)	/* all the messages sent by the current */
	{			/* proc have not yet been received	*/
	    pos = *p_pos;
	    mag = *p_mag;
	    if(i == emp)	/* make the sound associated with this	*/
	    {			/* proc "sharper" than the others	*/
		tot_mag += mag * nb_sending;
		mag *= mag * nb_sending;
		emp = -1;
	    }
	    else
	    {
		tot_mag += mag;
		mag *= mag;
	    }
	    
	    freq = *p_freq;
	    for(j = 0, p_value = val_tab;
		j < length;
		j++, p_value++)
	    {
		*p_value += mag * tsin[pos];
		pos = (pos + freq) % SMP_RATE;
	    }
	    *p_pos = pos;
	}
    }
    
    /* determine the "normalization" factor of the sound		*/
    lambda = 1 / (tot_mag * (1 + 7 * (nb_proc - nb_sending) / nb_proc));
    
    for(j = 0, p_value = val_tab;
	j < length;
	j++, p_value++)
    {
	*p_out = audio_s2u((short) (*p_value * lambda));
	p_out++;
	out_bufsize++;
    }
    
}

/* Case of a send event	      						*/

void send2au(str)
char *str;
{
    int proc;
    
    if(sscanf(str, "%*d %*d %*d %d", &proc) != 1)
    {
	PERROR("send2au : unable to get proc number");
    }
    else if((proc + 1) > nb_proc)
    {
	/* PERROR("send2au : too many processors");			*/
	if((proc + 1) > overflow_proc)
	{
	    PERROR("send2au : too many processors");
	    overflow_proc = proc + 1;
	    fprintf(stderr, "\t\t%d processors\n", overflow_proc);
	}
    }
    else if(*(nb_waiting + proc) > 0)
    {
	/* This processor has already sent a message			*/
	/* (but the message hasn't been received yet)			*/
	emp = proc;
	*(nb_waiting + proc) += 1;
    }
    else
    {
	/* This processor was not sending messages			*/
	*(cur_pos + proc) = 0;
	emp = proc;
	nb_sending++;
	*(nb_waiting + proc) = 1;
    }
}

/* Case of a receive event						*/

void recv2au(str)
char *str;
{
    int proc;
    
    /* We get the proc number of the SENDING processor, not		*/
    /* of the receiving one						*/
    if(sscanf(str, "%*d %*d %*d %*d %d", &proc) != 1)
    {
	PERROR("recv2au : unable to get proc number");
    }
    else if((proc + 1) > nb_proc)
    {
	/* PERROR("recv2au : too many processors");			*/
	if((proc + 1) > overflow_proc)
	{
	    PERROR("recv2au : too many processors");
	    overflow_proc = proc + 1;
	    fprintf(stderr, "\t\t%d processors\n", overflow_proc);
	}
    }
    else if(*(nb_waiting + proc) <= 0)
    {
	PERROR("recv2au : receiving an unsent message!");
    }
    else
    {
	*(nb_waiting + proc) -= 1;
	if(*(nb_waiting + proc) == 0)
	{
	    /* All the messages sent by proc have been received		*/
	    nb_sending--;	
	}
    }
}

/* Analyse the content of the trace buffer				*/

void trace2au(nb)
int nb;
{
    int i, type, t_sec, t_musec;
    int trace_time;
    
    for(i = 0; i < nb; i++)
    {
	/* get the time of the new event				*/
	if(sscanf(trace[i].str, "%d %d %d", &type, 
		  &t_sec, &t_musec) !=  3)
	{
	    PERROR("Error analysing trace line. Line skipped");
	    continue;
	}
	trace_time = (int) t_sec * 1e6 + t_musec;
	
	if(current_time < 0)
	    current_time = trace_time;	/* time initialization		*/
	
	
	if(current_time < trace_time)
	{
	    /* We need to catch up with the trace time			*/
	    do
	    {
		flush_outbuf();
		if(nb_sending > 0)
		    create_au();
		else
		    create_blank();
		current_time += unit;
	    } while(current_time < trace_time);
	}
	
	flush_outbuf();
	switch(type)
	{
	  case PG_TRACESTART :
	      break;
	    case PG_SEND :
		send2au(trace[i].str);
	      create_au();
	      break;
	    case PG_RECV :
		recv2au(trace[i].str);
	      if(nb_sending > 0)
		  create_au();
	      else
		  create_blank();
	      break;
	    case PG_RECV_BLOCKING :
		break;
	    case PG_RECV_WAKING :
		recv2au(trace[i].str);
	      if(nb_sending > 0)
		  create_au();
	      else
		  create_blank();
	      break;
	    case PG_COMPSTATS :
		break;
	    case PG_TRACEEXIT :
		break;
	    case PG_BLOCKBEGIN :
		break;
	    case PG_BLOCKEND :
		break;
	    case PG_TRACEMARK :
		break;
	      default :
		  ;
	  }
    }
}

/* on-line "manual" 							*/

void usage()
{
    PERROR("\nWrong parameters. Please specify 3 integers");
    PERROR("\tand check the range of the supplied parameters");
    PERROR("Usage : tr_sendmix nb length scale");
    PERROR("\tnb : number of processors");
    PERROR("\tlength : number of sound samples generated for each event");
    PERROR("\tscale : time scale");
}

/* main program								*/

main(argc, argv)
int argc;
char *argv[];
{
    int i, nb_lines;
    
    if(argc == 4)
    {
	if((sscanf(argv[1], "%d", &nb_proc) != 1) ||
	   (nb_proc < 1) || (nb_proc > MAX_PROC) ||
	   (sscanf(argv[2], "%d", &length) != 1) || 
	   (length < MIN_LENGTH) ||
	   (length > OUT_BUFSIZE) ||
	   (sscanf(argv[3], "%d", &unit) != 1) ||
	   (unit < MIN_UNIT))
	{
	    usage();
	    exit();
	}
    }
    else
    {
	PERROR("Using default parameters...");
	nb_proc = DFLT_NBPROC;
	length = DFLT_LENGTH;
	unit = DFLT_UNIT;
    }
    overflow_proc = nb_proc;
    
    /* allocate memory for misc tables					*/
    
    freq_lut = (int *) malloc(nb_proc * sizeof(int));
    if(freq_lut == NULL)
    {
	PERROR("Error allocating memory for freq_lut");
	exit();
    }
    
    mag_lut = (float *) malloc(nb_proc * sizeof(float));
    if(mag_lut == NULL)
    {
	PERROR("Error allocating memory for mag_lut");
	exit();
    }
    
    cur_pos = (int *) malloc(nb_proc * sizeof(int));
    if(cur_pos == NULL)
    {
	PERROR("Error allocating memory for cur_pos");
	exit();
    }

    nb_waiting = (int *) malloc(nb_proc * sizeof(int));
    if(nb_waiting == NULL)
    {
	PERROR("Error allocating memory for nb_waiting");
	exit();
    }
    for(i = 0; i < nb_proc; i++)
	*(nb_waiting + i) = 0;
    
    val_tab = (float *) malloc(length * sizeof(float));
    if(val_tab == NULL)
    {
	PERROR("Error allocating memory for val_tab");
	exit();
    }
    
    create_lut(nb_proc);
    tr_init();
    pipe_open();
    pipe_header();
    
    while((nb_lines = getlines()) > 0)
    {
	trace2au(nb_lines);
    }

    /* force the flushing of the output buffer a last time		*/
    length = OUT_BUFSIZE;
    flush_outbuf();
    
    fflush(stdout);
    pipe_close();
}
