function [p, stats] = colamdtree (S, knobs)
% COLAMDTREE Column approximate minimum degree permutation (COLAMD), followed
%    by a column elimination tree post-ordering (COLETREE).
%    P = COLAMDTREE(S) returns the column approximate minimum degree permutation
%    vector for the sparse matrix S.  For a non-symmetric matrix S, S (:,P)
%    tends to have sparser LU factors than S.  The Cholesky factorization of
%    S (:,P)' * S (:,P) also tends to be sparser than that of S'*S.  COLAMD
%    tends to be faster than COLMMD and tends to return a better ordering.
%
%    See also COLAMD, SYMAMD, SYMAMDTREE, COLMMD, SYMMMD, SYMRCM, COLPERM.
%
%    Usage:  P = colamdtree (S)
%            P = colamdtree (S, knobs)
%            [P, stats] = colamdtree (S)
%            [P, stats] = colamdtree (S, knobs)
%
%    stats (1) and stats (2) are the number of dense rows and columns
%    ignored in COLAMD, and stats (3) is the number of garbage collections
%    performed.  knobs (1) and knobs (2) control how COLAMD ignores dense
%    rows and columns (similar to spparms ('wh_frac')).
%
%    Authors:  Stefan I. Larimore and Timothy A. Davis, University of Florida,
%    (davis@cise.ufl.edu); in collaboration with John Gilbert, Xerox PARC, and
%    Esmond Ng, Oak Ridge National Laboratory.  This work was supported by the
%    National Science Foundation, under grants DMS-9504974 and DMS-9803599.
%    COLAMD and SYMAMD are available at http://www.cise.ufl.edu/~davis/colamd.

%    Tested under Matlab 5.2.0.3084, using Solaris 2.6.  August 3, 1998.

%-------------------------------------------------------------------------------
% Perform the colamd ordering:
%-------------------------------------------------------------------------------

if (nargout <= 1 & nargin == 1)
    p = colamd (S) ;
elseif (nargout <= 1 & nargin == 2)
    p = colamd (S, knobs) ;
elseif (nargout == 2 & nargin == 1)
    [p, stats] = colamd (S) ;
elseif (nargout == 2 & nargin == 2)
    [p, stats] = colamd (S, knobs) ;
else
    help colamdtree ;
    error ('colamdtree:  wrong number of input and/or output arguments') ;
end

%-------------------------------------------------------------------------------
% column elimination tree post-ordering:
%-------------------------------------------------------------------------------

[ignore, q] = sparsfun ('coletree', S (:,p)) ;
p = p (q) ;

