/* ========================================================================== */
/* === colamd mexFunction =================================================== */
/* ========================================================================== */

/*
    Usage:

	P = colamd (A) ;

	P = colamd (A, knobs) ;

	[ P, stats ] = colamd (A) ;

	[ P, stats ] = colamd (A, knobs) ;

    Returns a permutation vector P such that the LU factorization of A (:,P)
    tends to be sparser than that of A.  The Cholesky factorization of
    (A (:,P))'*(A (:,P)) will also tend to be sparser than that of A'*A.
    This routine provides the same functionality as COLMMD, but is much faster
    and returns a better permutation vector.  Note that the COLMMD m-file in
    Matlab 5.2 also performs a column elimination tree post-ordering.  This
    mexFunction does not do this post-ordering.  This mexFunction is a
    replacement for the p = sparsfun ('colmmd', A) operation.

    The knobs and stats vectors are optional:

	knobs (1)	rows with which are knobs (1) percent or more full are
			removed prior to ordering.

	knobs (2)	columns which are knobs (2)percent ore more full are
			removed prior to ordering, and placed last in the
			column permutation.

	stats (1)	the number of dense (or empty) rows ignored

	stats (2)	the number of dense (or empty) columms.  These
			are ordered last, in their natural order.

	stats (3)	the number of garbage collections performed.

    Calls/uses the following Matlab 5.2 routines/definitions/include files:

	mex.h			matrix.h		mexFunction
	mexCallMATLAB		mexErrMsgTxt		mexPrintf
	mxArray			mxCalloc		mxCreateDoubleMatrix
	mxCreateString		mxDestroyArray		mxFree
	mxGetIr			mxGetJc			mxGetM
	mxGetN			mxGetNumberOfDimensions	mxGetPr
	mxGetScalar		mxIsSparse		mxREAL

    Calls/uses the following ANSI C library routines/definitions/include files:

	string.h		stdlib.h		memcpy

    Calls/uses the following routines/defintions in colamd.c and colamd.h:

	colamd.h		colamd_set_defaults	colamd_recommended
	colamd			COLAMD_KNOBS		COLAMD_STATS
	COLAMD_DENSE_ROW	COLAMD_DENSE_COL	COLAMD_DEFRAG_COUNT

    Authors:

	The authors of the code itself are Stefan I. Larimore and Timothy A.
	Davis (davis@cise.ufl.edu), University of Florida.  The algorithm was
	developed in collaboration with John Gilbert, Xerox PARC, and Esmond
	Ng, Oak Ridge National Laboratory.

    Date:

	August 3, 1998.  Version 1.0.  Written under Matlab 5.2.0.3084, and
	tested in Solaris 2.6.

    Acknowledgements:

	This work was supported by the National Science Foundation, under
	grants DMS-9504974 and DMS-9803599.

    Notice:

	Copyright (c) 1998 by the University of Florida.  All Rights Reserved.

	THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY
	EXPRESSED OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.

	Permission is hereby granted to use or copy this program for any
	purpose, provided the above notices are retained on all copies.
	User documentation of any code that uses this code must cite the
	Authors, the Copyright, and "Used by permission."  If this code is
	accessible from within Matlab, then typing "help colamd" or "colamd"
	(with no arguments) must cite the Authors.  Permission to modify the
	code and to distribute modified code is granted, provided the above
	notices are retained, and a notice that the code was modified is
	included with the above copyright notice.  You must also retain the
	Availability information below, of the original version.

	This software is provided free of charge.

    Availability:

	This file is located at

		http://www.cise.ufl.edu/~davis/colamd/colamdmex.c

	The colamd.h and colamd.c files are required, located in the same
	directory.  All codes are purely ANSI C compliant.
*/

/* ========================================================================== */
/* === Include files and prototypes ========================================= */
/* ========================================================================== */

#include "colamd.h"
#include "mex.h"
#include "matrix.h"
#include <string.h>
#include <stdlib.h>

static void colamd_help (void) ;

/* ========================================================================== */
/* === colamd mexFunction =================================================== */
/* ========================================================================== */

void mexFunction
(
    /* === Parameters ======================================================= */

    int nlhs,			/* number of left-hand sides */
    mxArray *plhs [],		/* left-hand side matrices */
    int nrhs,			/* number of right--hand sides */
    const mxArray *prhs []	/* right-hand side matrices */
)
{
    /* === Local variables ================================================== */

    int *A ;			/* colamd's copy of the matrix, and workspace */
    int *p ;			/* colamd's copy of the column pointers */
    int Alen ;			/* size of A */
    int n_col ;			/* number of columns of A */
    int n_row ;			/* number of rows of A */
    int nnz ;			/* number of entries in A */
    int full ;			/* TRUE if input matrix full, FALSE if sparse */
    double knobs [COLAMD_KNOBS] ; /* colamd user-controllable parameters */
    double *out_perm ;		/* output permutation vector */
    double *out_stats ;		/* output stats vector */
    double *in_knobs ;		/* input knobs vector */
    int i ;			/* loop counter */
    mxArray *Ainput ;		/* input matrix handle */
    mxArray *string ;		/* name of matlab function mexCallMATLAB */
    mxArray *parameter ;	/* return matrix from mexCallMATLAB */
    int spumoni ;		/* matlab variable for verbosity */

    /* === Get spumoni ====================================================== */

    string = mxCreateString ("spumoni") ;
    mexCallMATLAB (1, &parameter, 1, &string, "spparms") ;
    spumoni = (int) mxGetScalar (parameter) ;
    mxDestroyArray (parameter) ;
    mxDestroyArray (string) ;

    /* === Check inputs ===================================================== */

    if (nrhs < 1 || nrhs > 2 || nlhs < 0 || nlhs > 2)
    {
	colamd_help () ;
	mexErrMsgTxt (
	"colamd: incorrect number of input and/or output arguments") ;
    }

    /* === Get knobs from matlab and/or user================================= */

    colamd_set_defaults (knobs) ;

    /* check for user passed knobs */
    if (nrhs == 2)
    {
	if (mxGetM (prhs [1]) * mxGetN (prhs [1]) < 2)
	{
	    colamd_help () ;
	    mexErrMsgTxt ("colamd: knobs argument must be of size 2 or more") ;
	}

	/* get pointer to knobs */
	in_knobs = mxGetPr (prhs [1]) ;

	knobs [COLAMD_DENSE_ROW] = in_knobs [COLAMD_DENSE_ROW] ;
	knobs [COLAMD_DENSE_COL] = in_knobs [COLAMD_DENSE_COL] ;
    }
    else
    {
	/* use spparms ('wh_frac') for the dense row and column setting */
	string = mxCreateString ("wh_frac") ;
	mexCallMATLAB (1, &parameter, 1, &string, "spparms") ;
	knobs [COLAMD_DENSE_ROW] = mxGetScalar (parameter) ;
	knobs [COLAMD_DENSE_COL] = knobs [COLAMD_DENSE_ROW] ;
	mxDestroyArray (parameter) ;
	mxDestroyArray (string) ;
    }

    /* print help info, and knob settings if spumoni is set */
    if (spumoni > 1)
    {
	colamd_help () ;
    }
    if (spumoni > 0)
    {
	mexPrintf ("colamd:  dense row fraction: %f\n",
	    knobs [COLAMD_DENSE_ROW]) ;
	mexPrintf ("colamd:  dense col fraction: %f\n",
	    knobs [COLAMD_DENSE_COL]) ;
    }

    /* === If A is full, convert to a sparse matrix ========================= */

    Ainput = (mxArray *) prhs [0] ;
    if (mxGetNumberOfDimensions (Ainput) != 2)
    {
	colamd_help () ;
	mexErrMsgTxt ("colamd: input matrix must be 2-dimensional") ;
    }
    full = !mxIsSparse (Ainput) ;
    if (full)
    {
	mexCallMATLAB (1, &Ainput, 1, (mxArray **) prhs, "sparse") ;
    }

    /* === Allocate workspace for colamd ==================================== */

    /* get size of matrix */
    n_row = mxGetM (Ainput) ;
    n_col = mxGetN (Ainput) ;

    /* get column pointer vector so we can find nnz */
    p = (int *) mxCalloc (n_col+1, sizeof (int)) ;
    (void) memcpy (p, mxGetJc (Ainput), (n_col+1)*sizeof (int)) ;
    nnz = p [n_col] ;
    Alen = colamd_recommended (nnz, n_row, n_col) ;

    /* === Copy input matrix into workspace ================================= */
    A = (int *) mxCalloc (Alen, sizeof (int)) ;
    (void) memcpy (A, mxGetIr (Ainput), nnz*sizeof (int)) ;

    if (full)
    {
	mxDestroyArray (Ainput) ;
    }

    /* === Order the columns (destroys A) =================================== */

    if (!colamd (n_row, n_col, Alen, A, p, knobs))
    {
	colamd_help () ;
	mexErrMsgTxt
	("colamd: internal error!  Please report this to the authors.\n") ;
    }

    /* === Return the permutation vector ==================================== */

    plhs [0] = mxCreateDoubleMatrix (1, n_col, mxREAL) ;
    out_perm = mxGetPr (plhs [0]) ;
    for (i = 0 ; i < n_col ; i++)
    {
	/* colamd is 0-based, but Matlab expects this to be 1-based */
	out_perm [i] = p [i] + 1 ;
    }
    mxFree (p) ;

    /* === Return the stats vector ========================================== */

    /* print stats if spumoni > 0 */
    if (spumoni > 0)
    {
	mexPrintf ("colamd:  dense rows ignored:            %d\n",
		A [COLAMD_DENSE_ROW]) ;
	mexPrintf ("colamd:  dense columns ignored:         %d\n",
		A [COLAMD_DENSE_COL]) ;
	mexPrintf ("colamd:  number of garbage collections: %d\n",
		A [COLAMD_DEFRAG_COUNT]) ;
	mexPrintf ("colamd:  columns jumbled? (1=yes):      %d\n",
		A [COLAMD_JUMBLED_COLS]) ;
    }

    if (nlhs == 2)
    {
	plhs [1] = mxCreateDoubleMatrix (1, COLAMD_STATS, mxREAL) ;
	out_stats = mxGetPr (plhs [1]) ;
	for (i = 0 ; i < COLAMD_STATS ; i++)
	{
	    out_stats [i] = A [i] ;
	}
    }

    mxFree (A) ;
}


/* ========================================================================== */
/* === colamd_help ========================================================== */
/* ========================================================================== */

/* NOTICE:  You do not have permission to delete the Authors' names from this */
/* function.  Typing "help colamd", or just "colamd" must cite the Authors. */

static void colamd_help (void)
{
    mexPrintf ("\n") ;
    mexPrintf (" COLAMD Column approximate minimum degree permutation.\n") ;
    mexPrintf ("    P = COLAMD (S) returns the column approximate") ;
    mexPrintf (" minimum degree permutation\n    vector for the sparse") ;
    mexPrintf (" matrix S.  For a non-symmetric matrix S, S (:,P)\n") ;
    mexPrintf ("    tends to have sparser LU factors than S.  The") ;
    mexPrintf (" Cholesky factorization of\n    S (:,P)' * S (:,P) also") ;
    mexPrintf (" tends to be sparser than that of S'*S.  COLAMD\n    tends") ;
    mexPrintf (" to be faster than COLMMD and tends to return a better") ;
    mexPrintf (" ordering.\n\n    See also COLAMDTREE, SYMAMD, SYMAMDTREE,") ;
    mexPrintf (" COLMMD, SYMMMD, SYMRCM, COLPERM.\n\n") ;
    mexPrintf ("    Usage:  P = colamd (S)\n") ;
    mexPrintf ("            P = colamd (S, knobs)\n") ;
    mexPrintf ("            [P, stats] = colamd (S)\n") ;
    mexPrintf ("            [P, stats] = colamd (S, knobs)\n\n") ;
    mexPrintf ("    stats (1) and stats (2) are the number of") ;
    mexPrintf (" dense rows and columns\n    ignored in COLAMD, and") ;
    mexPrintf (" stats (3) is the number of garbage collections\n") ;
    mexPrintf ("    performed.  knobs (1) and knobs (2) control how COLAMD") ;
    mexPrintf (" ignores dense rows\n    and columns (similar to spparms") ;
    mexPrintf (" ('wh_frac')).\n\n") ;
    mexPrintf ("    Authors:  Stefan I. Larimore and Timothy A. Davis,") ;
    mexPrintf (" University of Florida,\n") ;
    mexPrintf ("    (davis@cise.ufl.edu); in collaboration with John Gilbert") ;
    mexPrintf (", Xerox PARC, and\n    Esmond Ng, Oak Ridge National") ;
    mexPrintf (" Laboratory.  This work was supported by the\n    National") ;
    mexPrintf (" Science Foundation, under grants DMS-9504974 and") ;
    mexPrintf (" DMS-9803599.\n    COLAMD and SYMAMD are available") ;
    mexPrintf (" at http://www.cise.ufl.edu/~davis/colamd.\n\n") ;
}

