C================================================================C
C                                                                C
C     This file is part of the                                   C
C     Distributed Iterative Systems Solvers library              C
C     (c) 1994 Victor Eijkhout, eijkhout@cs.utk.edu              C
C                                                                C
C     Current version: 0.9                                       C
C     This file last generated 94/11/04                          C
C                                                                C
C================================================================C
C================================================================
C     
C     Matrix manipulation routines for a distributed memory computer
C     
C     For different matrix types, the vectors and matrices have
C     different structures. The interpretation of the data in a vector
C     array is determined by the `vec_inf' pointer array.
C     Here are the common elements in that array.
C     @(n_vectyp): a type identifier for the data format
C     vec_inf(2): the number of owned variables
C     vec_inf(3): the number of local variables
C     vec_inf(4): the number of global variables
C     (these first three are identical for uniprocessor formats)
C     vec_inf(5): the location of vector structure information
C     vec_inf(6): the location of vector embedding information
C     Further definition of embedding and structure is dependent
C     on the specific data structures. See the appropriate files.
C
C================================================================
C================================================================
C
C     Matrix times vector product
C
C     if `trans' the matrix is used by columns.
C     if `update' the operation is y <- y+Ax.
C     if `do_diag' is false, the matrix is assumed to be symmetric,
C     and the multiplication is by a strict triangle.
C
C     Guide to matrix parts:
C     0 (all): involve all of your local variables
C     2 (brd): involve only bordering variables
C     1 (own): involve only your owned variables
C----------------------------------------------------------------
      subroutine mvp(y,x,vec_inf,
     >      matrix,mat_ptr,mat_idx, trans,update,do_diag)
      
C     Arguments
C---- 
      integer mat_ptr(*),mat_idx(*),vec_inf(*),
     >     trans
      double precision matrix(*), x(*),y(*)
      logical update,do_diag

C     Dump quantities
C----
      integer dmp_trace_val
      common /dmp_trace/dmp_trace_val

C     Number of nonzeros in the matrix
C----
      integer nnzero
      common /mat_nnzero/nnzero

      if (dmp_trace_val.ge.2) then
         if (trans.eq.1) then
            if (update) then
               call pd0('Transpose (update) mvp$')
            else
               call pd0('Transpose mvp$')
            endif
         else
            if (update) then
               call pd0('Regular (update) mvp$')
            else
               call pd0('Regular mvp$')
            endif
         endif
      endif

C     Perform actual product
C---- 
      if (10*(vec_inf(1)/10).eq.10) then
C     matrix vector multiply by diagonals: pass block of diagonals
C     and their offsets
C----
         call mvp_grd(y,
     >        matrix(1),matrix(1+vec_inf(2)), x,
     >        vec_inf(vec_inf(5)+2),vec_inf(vec_inf(5)+2+1),
     >        vec_inf(vec_inf(5)),
     >        mat_ptr, trans,update,do_diag)
      else if (10*(vec_inf(1)/10).eq.20) then
C     Diagonal storage matrix vector product
C----
         call mvp_dia(y, matrix,mat_ptr,mat_idx, x,vec_inf,
     >        vec_inf(2),
     >        trans,update,do_diag)
      else if (10*(vec_inf(1)/10).eq.30) then
C     Row compressed storage matrix vector product
C----
         call mvp_crs(y, matrix,mat_ptr,mat_idx, x,vec_inf,
     >        trans,update,do_diag)
      else
         call strange_matrix_fmt(vec_inf,'MVP$')
      endif

      call addflp(2*nnzero)

      return
      end
C----------------------------------------------------------------
      subroutine matrix_norm_estimate(a_norm,matrix,mat_ptr,mat_idx,
     >     comm_context,vec_inf)

C     Arguments
C----
      double precision a_norm(1),matrix(*)
      integer mat_ptr(*),mat_idx(*),vec_inf(*),comm_context(*)

      if (10*(vec_inf(1)/10).eq.10) then
         call matrix_grid_norm_est(a_norm(1),matrix,
     >        vec_inf(vec_inf(5)+2),vec_inf(vec_inf(5)+2+1))
      else if (10*(vec_inf(1)/10).eq.30) then
         call matrix_cprs_norm_est(a_norm(1),matrix,
     >        mat_ptr(2*vec_inf(2)+2),
     >        mat_ptr,mat_idx,vec_inf)
      else if (10*(vec_inf(1)/10).eq.20) then
         call matrix_diag_norm_est(a_norm(1),matrix,
     >        mat_ptr,vec_inf)
      else
         call strange_matrix_fmt(vec_inf,'Norm estimate$')
      endif
      call dpgmax(a_norm(1),1,comm_context,'Matrix norm; gmax$')

      return
      end
C----------------------------------------------------------------
      subroutine matrix_grid_norm_est(a_norm,matrix,
     >     ipts,jpts)

C     Arguments
C----
      integer ipts,jpts
      double precision a_norm,matrix(ipts,jpts)

C     Local
C----
      integer ip,jp
      double precision sum

      sum = 0.d0
      do 10 ip=1,ipts
         do 20 jp=1,jpts
            sum = sum+matrix(ip,jp)
 20      continue
 10   continue
      a_norm = sum/(ipts*jpts)

      return
      end
C----------------------------------------------------------------
      subroutine matrix_cprs_norm_est(a_norm,matrix,dloc,
     >     mat_ptr,mat_idx,vec_inf)

C     Arguments
C----
      integer mat_ptr(*),mat_idx(*),dloc(*),vec_inf(*)
      double precision a_norm,matrix(*)

C     Local
C----
      integer var,rfirst,rowlen
      double precision sum

      sum = 0.d0
      do 10 var=1,vec_inf(2)
         call crs_get_pointer(rfirst,rowlen,var,mat_ptr)
         sum = sum+matrix(rfirst+dloc(var)-1)
 10   continue
      a_norm = sum/vec_inf(2)

      return
      end
C----------------------------------------------------------------
      subroutine compute_nnzero(mat_ptr,vec_inf)

C     Arguments
C----
      integer mat_ptr(*),vec_inf(*)

C     Number of nonzeros in the matrix
C----
      integer nnzero
      common /mat_nnzero/nnzero

C     Functions
C----
      integer compute_nnzero_diag,compute_nnzero_grid,
     >     compute_nnzero_cprs
      logical trace_setup

      if (10*(vec_inf(1)/10).eq.10) then
         nnzero = compute_nnzero_grid(mat_ptr,vec_inf)
      else if (10*(vec_inf(1)/10).eq.30) then
         nnzero = compute_nnzero_cprs(mat_ptr,vec_inf)
      else if (10*(vec_inf(1)/10).eq.20) then
         nnzero = compute_nnzero_diag(mat_ptr,vec_inf)
      else
         call strange_matrix_fmt(vec_inf,'Compute nnzero$')
      endif
      nnzero = 2*nnzero-vec_inf(2)
      if (trace_setup()) call pd1i('Flops in mvp:$',nnzero)
      call force_range(nnzero,1,0,'Compute nnzero$')

      return
      end
C----------------------------------------------------------------
      function get_nnzero()

C     Arguments
C----
      integer get_nnzero

C     Number of nonzeros in the matrix
C----
      integer nnzero
      common /mat_nnzero/nnzero

      get_nnzero = nnzero

      return
      end
C----------------------------------------------------------------
      subroutine process_matrix(mat,
     >     mat_ptr,leng_mat_ptr, mat_idx, mat_con,leng_mat_con,
     >     comm_context, vec_inf,leng_vec_inf)

C     Arguments
C----
      double precision mat(*)
      integer vec_inf(*),leng_vec_inf,
     >     mat_ptr(*),leng_mat_ptr, mat_idx(*),
     >     mat_con(*),leng_mat_con,comm_context(*)

C     Processor grid information
C---- 
      integer 
     >     pgrid_dimension,pgrid_size(4),proc_ijk(4),
     >     buffer_size,buffer_pointers(99),
     >     neighbr_buffer_size
      common /prcgrd/
     >     pgrid_dimension,pgrid_size,proc_ijk,
     >     buffer_size,buffer_pointers,
     >     neighbr_buffer_size

C     Functions
C----
      logical trace_matrices,trace_setup,trace_progress
      external int_var2proc
      integer int_var2proc

C     Local
C----
      logical trace

      trace = (trace_progress().or.trace_setup())
      if (trace) call pd0('Preprocessing matrix$')

C     For future reference (multi_colour ordering)
C     determine the global number of variables
C----
      if (.not.10*(vec_inf(1)/10).eq.10)
     >     call exch_global_variables(vec_inf,comm_context)

C     For grid and diagonal storage, determining traffic
C     patterns is relatively easy
C----
      if (10*(vec_inf(1)/10).eq.10) then
         call process_grid_matrix(mat_con,leng_mat_con,
     >        mat_ptr,vec_inf,leng_vec_inf)
         goto 99
      else if (10*(vec_inf(1)/10).eq.20) then
         call diag_traffic_patterns(mat_con,leng_mat_con,comm_context,
     >        mat_ptr,mat_idx,vec_inf,leng_vec_inf)
         goto 99
      endif

C     For compressed row/col storage:
C     Figure out what the border variables are; put them among the locals,
C     and copy them to a separate place ( vec_inf(brdvrs) )
C     This will also put the relative location of the bordvrs
C     in mat_ptr(1*vec_inf(2)+2, .... )
C----
      call force_range(1*vec_inf(2)+2+vec_inf(2),
     >     1,leng_mat_ptr,
     >     'No space for matblc pointers$')
      call border_vars_from_matrix_slice(vec_inf,leng_vec_inf,
     >     mat_ptr,mat_idx)

C     Now remap the index array to local numbering instead of global
C     (this is wasteful I think: the traffic patterns need to unmap
C     it again temporarily)
C----
      call remap_local_matrix(mat_ptr,mat_idx,mat,vec_inf)

C     Determine who you are going to send to, and who
C     will be sending to you
C
C----
      if (trace) call pt0('>> Constructing all traffic$')
      call cprs_set_traffic_patterns(mat_con,leng_mat_con,
     >     comm_context,vec_inf,leng_vec_inf, mat_ptr,mat_idx)
      if (trace) call pt0('<< Traffic patterns constructed$')

C     Find the diagonal in the matrix
C----
      call force_range(3*vec_inf(2)+2+vec_inf(2),
     >     1,leng_mat_ptr,
     >     'No space for 2*vec_inf(2)+2 pointers$')
      call extract_diagonal_loc(
     >     mat_ptr(2*vec_inf(2)+2),mat_ptr(3*vec_inf(2)+2),
     >     vec_inf,mat_ptr,mat_idx)

      if (trace_matrices()) then
C     Make a full dump of matrix and connectivity
         call crs_trace_dump(mat,vec_inf,mat_ptr,mat_idx,mat_con,0)
      else if (trace_setup()) then
C     Dump only in/out connectivity
         call crs_trace_dump(mat,vec_inf,mat_ptr,mat_idx,mat_con,1)
         call crs_trace_dump(mat,vec_inf,mat_ptr,mat_idx,mat_con,3)
         call crs_trace_dump(mat,vec_inf,mat_ptr,mat_idx,mat_con,4)
      endif

 99   continue
C     Count how many nonzeros there are
C     (does this serve any other purpose than flop counting?
C----
      call compute_nnzero(mat_ptr,vec_inf)

C     Create buffer space for communication
      call compute_buffer_needed(vec_inf,mat_con)
      call vector_gen_x_alloc(vec_inf,mat_con)

      return
      end
C----------------------------------------------------------------
C     Problem format. About sparsity structure and such.
C----------------------------------------------------------------
      subroutine default_problem_format(buffer,nitems)

C     Arguments
C----
      integer buffer(*),nitems

C     I/O channels
C     initialized in comp/v
C----
      integer 
     >     inchan,outchn,errchn,dmpchn,blkchn,solchn,logchn,
     >     tmp_channel,host_channel
      common /io_channels/
     >     inchan,outchn,errchn,dmpchn,blkchn,solchn,logchn,
     >     tmp_channel,host_channel

C     I/O channel status
C----
      logical
     >     dmp_channel_open,sol_channel_open,log_channel_open,
     >     tmp_channel_open,err_channel_open
      common /io_channel_status/
     >     dmp_channel_open,sol_channel_open,log_channel_open,
     >     tmp_channel_open,err_channel_open

C     Functions
C----
      logical tracer_proc

      if (tracer_proc().and.log_channel_open) then
      write(logchn,*) 
     >     '-------------------------------------------------------'
      write(logchn,*) 
     >     '[1] Problem format: 1-20=grid, 21/2/3=compressed'
      write(logchn,*) '[2] If grid: space dimension;'
      write(logchn,*) '    Compressed format: 2=row 4=col 5=symmetric'
      write(logchn,*) '[4] Sparsity: (0=def) symmetric (1) unsymmetric'
      write(logchn,*) '[3] Storage: (0=def) unsymmetric 1=symmetric'
      write(logchn,*) '[5] Variables partitioning:'
      write(logchn,*) '    1=consecutive, 2=external function'
      write(logchn,*) '[6] Rhs: (1=def) unit (2) external'
      write(logchn,*) '[7] Mat: 1=internally generated 2=from file'
      endif

      nitems = 7
C     fmt : 2D
      buffer(1) = 10
C     compressed format
      buffer(2) = 2
C     symmetric storage
      buffer(3) = 0
C     symmetric sparsity
      buffer(4) = 1
C     partitioning
      buffer(5) = 3

      return
      end
C----------------------------------------------------------------
      subroutine set_demo_matrix_file(cbuffer)

C     Arguments
C----
      character*3 cbuffer

C     Name of the external matrix file
C----
      character*50 matnam
      common /inmatx/matnam
      integer matnam_len
      common /inmatxi/matnam_len

C     I/O channels
C     initialized in comp/v
C----
      integer 
     >     inchan,outchn,errchn,dmpchn,blkchn,solchn,logchn,
     >     tmp_channel,host_channel
      common /io_channels/
     >     inchan,outchn,errchn,dmpchn,blkchn,solchn,logchn,
     >     tmp_channel,host_channel

C     I/O channel status
C----
      logical
     >     dmp_channel_open,sol_channel_open,log_channel_open,
     >     tmp_channel_open,err_channel_open
      common /io_channel_status/
     >     dmp_channel_open,sol_channel_open,log_channel_open,
     >     tmp_channel_open,err_channel_open

      matnam = '   .mat'
      matnam(1:3) = cbuffer(1:3)
      matnam_len = 7

      if (log_channel_open) then
         write(logchn,*) 'Name of matrix file:'
         write(logchn,*) '---- <',cbuffer,'>'
      endif

      return
      end
C----------------------------------------------------------------
      subroutine set_compress_format(vec_inf,spec,symm)

C     Arguments
C----
      integer vec_inf(*),spec,symm

      if (symm.eq.1) then
         vec_inf(1) = 30+5
      else
         vec_inf(1) = 30+spec
      endif

      return
      end
C----------------------------------------------------------------
      subroutine set_diag_format(vec_inf,symm)

C     Arguments
C----
      integer symm,vec_inf(*)

      vec_inf(1) = 20+symm

      return
      end
C----------------------------------------------------------------
      subroutine set_grid_format(vec_inf,dim,symm)

C     Arguments
C----
      integer dim,symm,vec_inf(*)

      vec_inf(1) = 10+2*dim+symm

      return
      end
C----------------------------------------------------------------
      subroutine dump_problem_format(vec_inf)

C     Arguments
C----
      integer vec_inf(*)

C     I/O channels
C     initialized in comp/v
C----
      integer 
     >     inchan,outchn,errchn,dmpchn,blkchn,solchn,logchn,
     >     tmp_channel,host_channel
      common /io_channels/
     >     inchan,outchn,errchn,dmpchn,blkchn,solchn,logchn,
     >     tmp_channel,host_channel

C     I/O channel status
C----
      logical
     >     dmp_channel_open,sol_channel_open,log_channel_open,
     >     tmp_channel_open,err_channel_open
      common /io_channel_status/
     >     dmp_channel_open,sol_channel_open,log_channel_open,
     >     tmp_channel_open,err_channel_open

      if (log_channel_open) then
         write(logchn,*) 'Format:',vec_inf(1)
         if (10*(vec_inf(1)/10).eq.10) then
            write(logchn,*) 'Grid dimension:',(vec_inf(1)-10)/2
         else if (10*(vec_inf(1)/10).eq.30) then
            write(logchn,*) 'Compress fmt:',vec_inf(1)-30
         endif
         if (mod(vec_inf(1),2).eq.1)
     >        write(logchn,*) '- symmetric storage'
      endif

      if (dmp_channel_open) then
         if (10*(vec_inf(1)/10).eq.10) call pd0('Format: g$')
         if (10*(vec_inf(1)/10).eq.20) call pd0('Format: d$')
         if (10*(vec_inf(1)/10).eq.30) then
            call pd0('Format: i$')
            call pd0('(harbo compression)$')
         endif
      endif

      return
      end
C----------------------------------------------------------------
      subroutine strange_matrix_fmt(vec_inf,txt)

C     Arguments
C----
      character*(*) txt
      integer vec_inf(*)

      call pe1i('Unknown matrix format$',vec_inf(1))
      call stop_connections(txt)

      return
      end
C----------------------------------------------------------------
      function problem_dimension(vec_inf)

C     Argument
C----
      integer problem_dimension,vec_inf(*)

C     If this is a dimensional problem, there is a real dimension
C     otherwise, there is just a linear string of variables
C----
      if (10*(vec_inf(1)/10).eq.10) then
         problem_dimension = (vec_inf(1)-10)/2
      else
         problem_dimension = 1
      endif
      call force_range(problem_dimension,1,-1,
     >   'Problem dimension$')

      return
      end
C----------------------------------------------------------------
      subroutine set_global_variables(vars,vec_inf)

C     Arguments
C----
      integer vars,vec_inf(*)

      call force_range(vars,1,0,'Tot #vars (set)$')
      vec_inf(4) = vars

      return
      end
C----------------------------------------------------------------
      subroutine exch_global_variables(vec_inf,comm_context)

C     Arguments
C----
      integer vec_inf(*),comm_context(*)

C     Local
C----
      integer totvrs(1)

      totvrs(1) = vec_inf(2)
      call ingsum(totvrs,1,comm_context)
      call inbcst(totvrs,1,comm_context,'Global #vars$')
      call set_global_variables(totvrs(1),vec_inf)

      return
      end
C----------------------------------------------------------------
      subroutine init_multicolour(vec_inf,leng_vec_inf,
     >    mat_con,comm_context, mat_ptr,mat_idx,
     >     clr_buff1,clr_buff2, trace)

C     Arguments
C----
      integer vec_inf(*),leng_vec_inf,mat_con(*),comm_context(*),
     >     mat_ptr(*),mat_idx(*)
      double precision clr_buff1(*),clr_buff2(*)
      logical trace

C     Functions
C----
      logical trace_matrices

      if (10*(vec_inf(1)/10).eq.30) then
         call init_crs_multicolour(vec_inf,leng_vec_inf,
     >        mat_con,comm_context,mat_ptr,mat_idx,
     >        clr_buff1,clr_buff2, trace)
      else if (10*(vec_inf(1)/10).eq.10) then
         call init_grd_multicolour(vec_inf,leng_vec_inf,
     >        mat_con,comm_context,mat_ptr,
     >        clr_buff1,clr_buff2, trace)
      else if (10*(vec_inf(1)/10).eq.20) then
         call init_dia_multicolour(vec_inf,leng_vec_inf,
     >        mat_con,comm_context,mat_ptr,mat_idx,
     >        clr_buff1,clr_buff2, trace)
      else
         call strange_matrix_fmt(vec_inf,'Init multicolour$')
      endif

      if (trace_matrices()) call trace_multicolour(vec_inf)

      return
      end
C----------------------------------------------------------------
      subroutine colour_loc(iloc,ilen, clr,vec_inf)

C     Arguments
C----
      integer iloc,ilen, clr,vec_inf(*)

      iloc = vec_inf(vec_inf(9)+2*clr-1)
      ilen = vec_inf(vec_inf(9)+2*clr)

      return
      end
C----------------------------------------------------------------
      function n_colours(vec_inf)

C     Arguments
C----
      integer vec_inf(*), n_colours

      n_colours = vec_inf(vec_inf(8))

      return
      end
C----------------------------------------------------------------
      subroutine trace_multicolour(vec_inf)

C     Arguments
C----
      integer vec_inf(*)

C     I/O channels
C     initialized in comp/v
C----
      integer 
     >     inchan,outchn,errchn,dmpchn,blkchn,solchn,logchn,
     >     tmp_channel,host_channel
      common /io_channels/
     >     inchan,outchn,errchn,dmpchn,blkchn,solchn,logchn,
     >     tmp_channel,host_channel

C     Local
C----
      integer clr,iloc,ilen,idum

      write(dmpchn,*) 'Colours:'
      do 10 clr=1,vec_inf(vec_inf(8))
         call colour_loc(iloc,ilen, clr,vec_inf)
         write(dmpchn,*) clr,':',(vec_inf(iloc+idum-1), idum=1,ilen)
 10   continue

      return
      end
C----------------------------------------------------------------
C     Extract the matrix diagonal into a separate array.
C     For the purpose of incomplete factorization pivots;
C     this array has the *local* length, no border.
C----------------------------------------------------------------
      subroutine extract_matrix_diagonal(xtr,dloc,
     >     vec_inf,matrix,mat_ptr,mat_idx)

C     Arguments
C----
      integer dloc(*),vec_inf(*),mat_ptr(*),mat_idx(*)
      double precision xtr(*),matrix(*)

C     Function
C----
      logical trace_setup

C     Local
C----
      integer dia,size

      if (10*(vec_inf(1)/10).eq.10) then
         size = vec_inf(3)
         call vvcopy(xtr,matrix,size)
      else if (10*(vec_inf(1)/10).eq.30) then
         call extract_cprs_matrix_diagonal(xtr,dloc, matrix,mat_ptr,
     >        vec_inf)
      else if (10*(vec_inf(1)/10).eq.20) then
         size = vec_inf(2)
         do 10 dia=1,mat_ptr(1)
            if (mat_ptr(1+dia).eq.0) goto 11
 10      continue
         call stop_connections('Extract dia; matrix no main$')
 11      continue
         if (trace_setup()) call pd1i('Extract main diag=dia$',dia)
         call vvcopy(xtr,matrix(1+(dia-1)*size),size)
      else
         call strange_matrix_fmt(vec_inf,'extract diagonal$')
      endif

      return
      end
C----------------------------------------------------------------
C     Dump matrix
C----------------------------------------------------------------
      subroutine dump_matrix(matrix,mat_ptr,mat_idx,vec_inf)
      
C     Arguments
C---- 
      integer mat_ptr(*),mat_idx(*),vec_inf(*)
      double precision matrix(*)
      
C     Functions
C----
      logical tracer_proc

      if (10*(vec_inf(1)/10).eq.10) then
         if (tracer_proc()) call pt0('Dumping ort matrix$')
         call dump_matx_grid(matrix,mat_ptr,vec_inf)
      else if (10*(vec_inf(1)/10).eq.20) then
         if (tracer_proc()) call pt0('Dumping diag matrix$')
         call dump_matx_diag(matrix,mat_ptr,vec_inf)
      else if (10*(vec_inf(1)/10).eq.30) then
         if (tracer_proc()) call pt0('Dumping crs matrix$')
         call dump_matx_cprs(matrix,mat_ptr,mat_idx,vec_inf)
      else
         call strange_matrix_fmt(vec_inf,'Dump Matrix$')
      endif
      
      return
      end
C----------------------------------------------------------------
C     Primitive dump vector routine; 
C     everything goes to the dump channel
C----------------------------------------------------------------
      subroutine dump_vector(x,vec_inf,txt,itxt)

C     Arguments
C----
      integer vec_inf(*),itxt
      double precision x(*)
      character*(*) txt

C     Local
C----
      integer bord

      if (10*(vec_inf(1)/10).eq.10) then
         bord = vec_inf(vec_inf(5))
         call dump_vector_g(x,
     >        1-bord,vec_inf(vec_inf(5)+2)+bord,
     >        1-bord,vec_inf(vec_inf(5)+2+1)+bord,
     >        vec_inf,
     >        1,vec_inf(vec_inf(5)+2),1,vec_inf(vec_inf(5)+2+1),
     >        txt,itxt)
      else if (10*(vec_inf(1)/10).eq.30) then
         call dump_vector_i(x, vec_inf, txt,itxt)
      else if (10*(vec_inf(1)/10).eq.20) then
         call dump_vector_d(x, vec_inf, txt,itxt)
      else
         call strange_matrix_fmt(vec_inf,'Dump vector$')
      endif

      return
      end
C----------------------------------------------------------------
      subroutine dump_vector_o(x,vec_inf,txt,itxt)

C     Arguments
C----
      integer vec_inf(*),itxt
      double precision x(*)
      character*(*) txt

      if (10*(vec_inf(1)/10).eq.10) then
         call dump_vector_g(x,
     >        1,vec_inf(vec_inf(5)+2),
     >        1,vec_inf(vec_inf(5)+2+1),
     >        vec_inf,
     >        1,vec_inf(vec_inf(5)+2),1,vec_inf(vec_inf(5)+2+1),
     >        txt,itxt)
      else if (10*(vec_inf(1)/10).eq.30) then
         call dump_vector_i(x, vec_inf, txt,itxt)
      else if (10*(vec_inf(1)/10).eq.20) then
         call dump_vector_od(x, vec_inf, txt,itxt)
      else
         call strange_matrix_fmt(vec_inf,'Dump vector$')
      endif

      return
      end
C----------------------------------------------------------------
      subroutine dump_border(x,vec_inf,txt,itxt)

C     Arguments
C----
      integer vec_inf(*),itxt
      double precision x(*)
      character*(*) txt

C     Local
C----
      integer bord

      if (10*(vec_inf(1)/10).eq.10) then
         bord = vec_inf(vec_inf(5))
         call dump_border_g(x,
     >        1-bord,vec_inf(vec_inf(5)+2)+bord,
     >        1-bord,vec_inf(vec_inf(5)+2+1)+bord,
     >        vec_inf,
     >        1,vec_inf(vec_inf(5)+2),1,vec_inf(vec_inf(5)+2+1),
     >        txt,itxt)
      else if (10*(vec_inf(1)/10).eq.30) then
         call dump_border_i(x, vec_inf, txt,itxt)
      endif

      return
      end
C----------------------------------------------------------------
C     Restore a global vector to local storage
C----------------------------------------------------------------
      subroutine load_global_vector(x,vec_inf,vector_name,itmp)

C     Arguments
C----
      integer vec_inf(*),itmp(*)
      double precision x(*)
      character*3 vector_name

C     I/O channels
C     initialized in comp/v
C----
      integer 
     >     inchan,outchn,errchn,dmpchn,blkchn,solchn,logchn,
     >     tmp_channel,host_channel
      common /io_channels/
     >     inchan,outchn,errchn,dmpchn,blkchn,solchn,logchn,
     >     tmp_channel,host_channel

C     Local
C----
      character*50 vector_file

      vector_file = '   .dat '
      vector_file(1:3) = vector_name(1:3)
      call setup_in_file(
     >     'pvm3/examples/cg/local.data/',28,
     >     vector_file,7,'Reading vector from file$')

      if (10*(vec_inf(1)/10).eq.10) then
c$$$         bord = vec_inf(vec_inf(5))
c$$$         call rst2vc(x,
c$$$     >        1-bord,vec_inf(vec_inf(5)+2)+bord,
c$$$     >        1-bord,vec_inf(vec_inf(5)+2+1)+bord,
c$$$     >        1,vec_inf(vec_inf(5)+2),1,vec_inf(vec_inf(5)+2+1), txt,itxt)
      else if (10*(vec_inf(1)/10).eq.30) then
         call load_global_vector_i(x, vec_inf, itmp)
      endif

      close(inchan)

      return
      end
