package AT.Ac.univie.imp.loeffler.pde.threeD.fd;
import  AT.Ac.univie.imp.loeffler.util.*;
import  AT.Ac.univie.imp.loeffler.parallel.*;

/**
 * An implementation of the damped Jacobi method of relaxing (thereby smoothing) the solution of a linear elliptic PDE.
 *
 * @author Gerald Loeffler (Gerald.Loeffler@univie.ac.at)
 */
public final class DampedJacobi implements Smoother {
     /**
      * in theory the ideal weighting factor, i.e. the relative contribution of a new iteration value at a grid element.
      */
     public static final double IDEAL_WEIGHTING_FACTOR  = 2.0/3.0;

     /**
      * construct from PDE and weighting factor.
      *
      * @param pde             a representation of the PDE
      * @param weightingFactor the relative contribution of a new iteration value at a grid element (0 < weightingFactor
      *                        <= 1)
      */
     public DampedJacobi(PDE pde, double weightingFactor) {
          Contract.pre(pde != null,"pde not null-object");
          Contract.pre(weightingFactor > 0 && weightingFactor <= 1,"0 < weightingFactor <= 1");
          
          this.pde             = pde;
          this.weightingFactor = weightingFactor;
          
          oneMinusWF = 1.0 - weightingFactor;
     }

     /**
      * implements method from Smoother.
      *
      * @see Smoother#smooth
      */
     public BoundaryGrid smooth(ConstBoundaryGrid u, ConstNoBoundaryGrid f) {
          Contract.pre(u != null && f != null,"all objects not null-objects");
          
          size = u.size();
          
          Contract.pre(f.size() == size,"size of u equal size of f");
          
          totalRange     = new IntRange1D(1,size - 2);
          BoundaryGrid g = (BoundaryGrid) u.newInstance(size,0);
          new DampedJacobiSmoothParallelizer(this,g,u,f);
          return g;
     }
     
     /**
      * implements of the parallel part of smooth().
      */
     void smoothProper(BoundaryGrid g, ConstBoundaryGrid u, ConstNoBoundaryGrid f, int myNum, int totalNum) {
          IntRange1D range = Parallelizer.partition(totalRange,myNum,totalNum);
          for (int x = range.from(); x <= range.to(); x++) {
               for (int y = 1; y <= (size - 2); y++) {
                    for (int z = 1; z <= (size - 2); z++) {
                         // calculate new value
                         g.set(x,y,z,oneMinusWF*u.get(x,y,z) + weightingFactor*pde.evaluate(u,f,x,y,z));
                    }
               }
          }
     }

     private PDE        pde;
     private double     weightingFactor;
     private double     oneMinusWF;
     private int        size;
     private IntRange1D totalRange;
}
