      SUBROUTINE SQRED (MDEC, N, A, G, F, S, WORK, JOB)
      INTEGER MDEC, N, JOB
      DOUBLE PRECISION A(MDEC,N), G((N*(N+1))/2), F((N*(N+1))/2)
      DOUBLE PRECISION S(MDEC,2*N), WORK(2*N)
*
*     TRANSFORM THE HAMILTONIAN MATRIX
*
*                        T
*     (*)          H = (A   G)
*                      (F  -A)
*
*     INTO A SQUARE REDUCED HAMILTONIAN MATRIX
*
*                          T
*     (**)         H' = (A'  G')
*                       (F' -A')
*
*     BY A SYMPLECTIC-ORTHOGONAL SIMILARITY TRANSFORMATION
*
*     (***)        S = ( S1 S2)
*                      (-S2 S1).
*
*             T                                          2
*     H = SH'S.  S IS BOTH SYMPLECTIC AND ORTHOGONAL.  H' IS BLOCK
*     UPPER TRIANGULAR WITH N BY N BLOCKS.
*
*     ******************************************************************
*     *     ON OUTPUT:                                2                *
*     *     THE SQUARE ROOTS OF THE EIGENVALUES OF (A'  + F'G')        *
*     *     ARE THE EIGENVALUES OF H. (EIGENVALUES OF H APPEAR         *
*     *     IN +/- PAIRS SO BOTH COMPLEX SQUARE ROOTS ARE REQUIRED.)   *
*     *                                                                *
*     *        2                                                       *
*     *     (A' + F'G') IS A REAL UPPER HESSENBERG MATRIX, SO ITS      *
*     *     EIGENVALUES MAY BE COMPUTED WITH EISPACK SUBROUTINE HQR.   *
*     *                                                                *
*     ******************************************************************
*
*     MDEC:(INPUT) INTEGER SCALAR
*        MDEC CONTAINS THE FIRST DIMENSION OF MATRICES A AND S JUST AS
*        DECLARED IN THE CALLING PROGRAM.
*
*     N:(INPUT) INTEGER SCALAR
*        N CONTAINS THE ORDER OF THE MATRIX A.  F AND G ARE N BY N
*        SYMMETRIC MATRICES.
*
*     A:(INPUT/OUTPUT) DP ARRAY OF DIMENSION AT LEAST (MDEC,N)
*        ON INPUT A CONTAINS THE MATRIX A IN (*).
*        ON OUTPUT A HAS BEEN REPLACED BY THE MATRIX A' IN (**).
*
*     G:(INPUT/OUTPUT) DP ARRAY OF LENGTH AT LEAST (N*(N+1))/2
*        ON INPUT G CONTAINS THE SYMMETRIC MATRIX G IN (*).
*        ON OUTPUT G HAS BEEN REPLACED BY THE MATRIX G' IN (**).
*        G IS STORED IN SYMMETRIC STORAGE MODE.
*
*     F:(INPUT/OUTPUT) DP ARRAY OF LENGTH AT LEAST (N*(N+1))/2
*        ON INPUT F CONTAINS THE SYMMETRIC MATRIX F IN (*).
*        ON OUTPUT F HAS BEEN REPLACED BY THE MATRIX F' IN (**).
*        F IS STORED IN SYMMETRIC MODE.
*
*     S:(OUTPUT) DP ARRAY OF DIMENSION AT LEAST (MDEC,2*N)
*        IF JOB IS NONZERO, THEN S IS RETURNED WITH THE FIRST N ROWS OF
*        THE MATRIX S IN (***).  IF JOB IS ZERO, THEN S IS UNREFERENCED.
*
*     WORK:(WORKSPACE) DP ARRAY OF LENGTH AT LEAST 2*N
*
*     JOB:(INPUT) INTEGER SCALAR
*        IF JOB IS NONZERO, THEN THE FIRST N ROWS OF THE SYMPLECTIC-
*        ORTHOGONAL SIMILARITY TRANSFORMATION (***) ARE ACCUMULATED AND
*        RETURNED IN S.  IF JOB IS ZERO, THEN S IS UNREFERENCED.
*
*
*     RALPH BYERS, DEPARTMENT OF MATHEMATICS, UNIVERSITY OF KANSAS,
*                  LAWRENCE, KANSAS  66045, BYERS@NA-NET.STANFORD.EDU
*
*     SQRED IMPLEMENTS VAN LOAN'S ``SQUARE REDUCED'' ALGORITHM [1]
*     FOR CALCULATING ALL EIGENVALUES OF A HAMILTONIAN MATRIX. THIS 
*     IMPLEMENTATION APPEARS IN [2].  
*
*     USERS MAY CONTACT RALPH BYERS AT THE ABOVE ADDRESS WITH COMMENTS
*     OR BUG REPORTS.  HOWEVER, HE MAKES NO PROMISES TO FIX BUGS OR
*     TO SUPPORT THIS SOFTWARE IN ANY WAY.
*
*
*     REFERENCES:
*      [1] CHARLES VAN LOAN, ``A SYMPLECTIC METHOD FOR APPROXIMATING
*          ALL EIGENVALUES OF A HAMILTONIAN MATRIX'', LINEAR ALGEBRA
*          AND ITS APPLICATIONS, VOLUME 61, 1984, 233-251.
*
*      [2] RALPH BYERS, ``HAMILTONIAN AND SYMPLECTIC ALGORITHMS FOR 
*          THE ALGEBRAIC RICCATI EQUATION'', PHD DISSERTATION, CENTER
*          FOR APPLIED MATHEMATICS, CORNELL UNIVERSITY, ITHACA, NEW
*          YORK, 1983
*
*      REQUIRED SUBROUTINES: GENREF, VECREF, SYMREF 
*      BLAS:                 DDOT, DCOPY, DROTG, DROTG
************************************************************************

*     ========== LOCAL VARIABLES ==========
      INTEGER I, I1, IK, J, JP1JP1, JP1K, JP11, J1, K, KJ
      DOUBLE PRECISION COSINE, SINE, SUM, SUM1, SUM2, TEMP, V, X, Y
      DOUBLE PRECISION T11, T12, T21, T22

*     ========== NAMED CONSTANTS ==========
      DOUBLE PRECISION ZERO, ONE, TWO
      PARAMETER (ZERO = 0.0D0, ONE = 1.0D0, TWO = 2.0D0)

*     ========== BLAS ==========
      DOUBLE PRECISION DDOT
      EXTERNAL DCOPY, DROT, DROTG

*     ========== REQUIRED SUBROUTINES ==========
      EXTERNAL GENREF, SYMREF, VECREF

************************************************************************

*        ========== TRANSFORM TO SQUARE REDUCED FORM ==========
         J1 = 1
         DO 110 J = 1, N-1
              JP11 = J1 + J
*                                                T
*             ========== WORK := JTH COLUMN OF FA - AF ==========
              I1 = JP11
              DO 50 I = J+1, N
                   SUM1 = DDOT(I-1, F(I1), 1, A(J,1), MDEC)
                   IK = I1 + I - 1
                   DO 30 K = I, N
                        SUM1 = SUM1 + F(IK)*A(J,K)
                        IK = IK + K
30                 CONTINUE

                   SUM2 = DDOT(J-1, A(I,1), MDEC, F(J1), 1)
                   KJ = JP11 - 1
                   DO 40 K = J, N
                        SUM2 = SUM2 + A(I,K)*F(KJ)
                        KJ = KJ + K
40                 CONTINUE

                   WORK(I) = SUM1 - SUM2
                   I1 = I1 + I
50            CONTINUE
*             ========== SYMPLECTIC REFLECTION TO ZERO        2
*                        COMPONENTS (N+J+1,J) THRU (2N,J) OF H =========
              CALL GENREF (J+1, J+2, N, WORK, 1, V)
              CALL VECREF (J+1, J+2, N, WORK, 1, V, A, 1, MDEC, N)
              CALL VECREF (J+1, J+2, N, WORK, 1, V, A, MDEC, 1, N)
              CALL SYMREF (J+1, J+2, N, WORK, 1, V, G, N, 1, WORK(N+1))
              CALL SYMREF (J+1, J+2, N, WORK, 1, V, F, N, 1, WORK(N+1))

              IF (JOB .NE. 0) THEN
*                  ========== SAVE REFLECTION ==========
                   CALL DCOPY (N-J, WORK(J+1), 1, S(J+1,J), 1)
                   S(J+1,J) = V
              ENDIF
*                                                               2
*             ====== (X,Y) := ((J+1,J),(N+J+1,J)) COMPONENT OF H ======
              SUM = DDOT(J, G(JP11), 1, F(J1), 1)
              JP1K = JP11 + J
              KJ = JP11 + J - 1
              DO 60 K = J+1, N
                   SUM = SUM + G(JP1K)*F(KJ)
                   JP1K = JP1K + K
                   KJ   = KJ   + K
60            CONTINUE
              X = DDOT (N, A(1,J+1), 1, A(J,1), MDEC) + SUM
              Y = WORK(J+1)
*             ========== SYMPLECTIC ROTATION TO       2
*                        ZERO COMPONENT (N+J+1,J) OF H ==========
              CALL DROTG (X, Y, COSINE, SINE)
              CALL DROT  (J, A(1,J+1), 1, F(JP11), 1, COSINE, SINE)
              CALL DROT  (J, A(J+1,1), MDEC, G(JP11), 1, COSINE, SINE)

              JP1JP1 = JP11 + J
              T11 =   COSINE*A(J+1,J+1) + SINE*F(JP1JP1)
              T12 =   COSINE*G(JP1JP1)  - SINE*A(J+1,J+1)
              T21 =   COSINE*F(JP1JP1)  - SINE*A(J+1,J+1)
              T22 = -(COSINE*A(J+1,J+1) + SINE*G(JP1JP1))
              A(J+1,J+1) = T11*COSINE + T12*SINE
              G(JP1JP1)  = T12*COSINE - T11*SINE
              F(JP1JP1)  = T21*COSINE + T22*SINE

              JP1K = JP1JP1 + J + 1
              DO 70 K = J+2, N
                   TEMP     = COSINE*A(K,J+1) + SINE*F(JP1K)
                   F(JP1K)  = COSINE*F(JP1K)  - SINE*A(K,J+1)
                   A(K,J+1) = TEMP

                   TEMP     = COSINE*A(J+1,K) + SINE*G(JP1K)
                   G(JP1K)  = COSINE*G(JP1K)  - SINE*A(J+1,K)
                   A(J+1,K) = TEMP

                   JP1K = JP1K + K
70            CONTINUE
              IF (JOB .NE. 0) THEN
*                  ========== SAVE ROTATION ==========
                   S(J,J)   = COSINE
                   S(J,N+J) = SINE
              ENDIF
*                                               2T
*             ========== WORK := JTH COLUMN OF A  + GF ==========
              I1 = JP11
              DO 100 I = J+1, N
                   SUM = DDOT(J, G(I1), 1, F(J1), 1)

                   IK = I1 + J
                   KJ = JP11 + J - 1
                   DO 80 K = J+1, I-1
                        SUM = SUM + G(IK)*F(KJ)
                        IK = IK + 1
                        KJ = KJ + K
80                 CONTINUE

                   DO 90 K = I, N
                        SUM = SUM + G(IK)*F(KJ)
                        IK = IK + K
                        KJ = KJ + K
90                 CONTINUE

                   WORK(I) = DDOT(N, A(1,I), 1, A(J, 1), MDEC) + SUM
                   I1 = I1 + I
100           CONTINUE
*             ========== SYMPLECTIC REFLECTION TO ZERO     2
*                        COMPONENTS (J+2,J) THRU (N,J) OF H ==========
              CALL GENREF (J+1, J+2, N, WORK, 1, V)
              CALL VECREF (J+1, J+2, N, WORK, 1, V, A, 1, MDEC, N)
              CALL VECREF (J+1, J+2, N, WORK, 1, V, A, MDEC, 1, N)
              CALL SYMREF (J+1, J+2, N, WORK, 1, V, G, N, 1, WORK(N+1))
              CALL SYMREF (J+1, J+2, N, WORK, 1, V, F, N, 1, WORK(N+1))

              IF (JOB .NE. 0) THEN
*                  ========== SAVE REFLECTION ==========
                   CALL DCOPY (N-J, WORK(J+1), 1, S(J+1,N+J), 1)
                   S(J+1,N+J) = V
              ENDIF

              J1 = JP11
110      CONTINUE

         IF (JOB .NE. 0) THEN
*             ========== ACCUMULATE TRANSFORMATIONS ==========
              DO 130 J = N-1, 1, -1
*                  ========== INITIALIZE J+1ST COLUMN OF S ==========
                   DO 120 I = 1, N
                        S(I,J+1)   = ZERO
                        S(I,N+J+1) = ZERO
120                CONTINUE
                   S(J+1,J+1) = ONE

*                  ========== SECOND REFLECTION ==========
                   CALL VECREF (J+1, J+2, N, S(1,N+J), 1, S(J+1,N+J)
     $                        , S(1,J+1), 1, MDEC, N-J)
                   CALL VECREF (J+1, J+2, N, S(1,N+J), 1, S(J+1,N+J)
     $                        , S(1,N+J+1), 1, MDEC, N-J)
*                  ========== ROTATION ==========
                   CALL DROT (N-J, S(J+1,J+1), MDEC, S(J+1,N+J+1), MDEC,
     $                        S(J,J), S(J,N+J))
*                  ========== FIRST REFLECTION ==========
                   CALL VECREF (J+1, J+2, N, S(1,J), 1, S(J+1,J),
     $                          S(1,J+1), 1, MDEC, N-J)
                   CALL VECREF (J+1, J+2, N, S(1,J), 1, S(J+1,J),
     $                          S(1,N+J+1), 1, MDEC, N-J)

130           CONTINUE
*             ========== FIRST COLUMN IS FIRST COLUMN OF IDENTITY=======
              DO 140 I = 1, N
                   S(I,1) = ZERO
                   S(I,N+1) = ZERO
140           CONTINUE
              S(1,1) = ONE
         ENDIF

      END
      SUBROUTINE VECREF (L, M, N, U, INCU, S, VEC, INCOMP, INCVEC, NVEC)
      INTEGER L, M, N, INCU, INCOMP, INCVEC, NVEC
      DOUBLE PRECISION U(1), S, VEC(1)
C
C   APPLY A HOUSEHOLDER REFLECTION TO A SET OF VECTORS
C
C   THE REFLECTION IS GIVEN BY
C                   T
C      H = I - S*V*V
C
C   WHERE S IS A SCALAR AND V IS A VECTOR OF LENGTH AT LEAST MAX(L,N)
C      V(L) = 1
C      V(I) = 0     IF I.NE.L .AND. ((I.LT.M) .OR. (N.LT.I))
C      V(I) = U(I)  IF M.LE.I .AND. I.LE.N
C
C   H MAY BE CONSTRUCTED BY SUBROUTINE GENREF.
C
C
C   L, M, N:(INPUT) INTEGER SCALARS
C      L, M, AND N ALONG WITH S AND V DEFINE THE REFLECTION AS
C      DESCRIBED ABOVE.  L, M, AND N MUST BE GREATER THAN ZERO AND
C      SATISFY (M.LE.N) .AND. ((L.LT.M) .OR. (L.GT.N))
C
C   U:(INPUT) DP ARRAY OF LENGTH AT LEAST MAX(L,N)
C      THE NONZERO COMPONENTS OF V OTHER THAN V(L) ARE
C      GIVEN BY THE CORRESPONDING COMPONENTS OF U.
C      (IE V(I) = U(I) FOR (M.LE.I) .AND. (I.LE.N))
C
C   INCU:(INPUT)  INTEGER SCALAR
C      INCU IS THE INCREMENT BETWEEN SUCCESSIVE COMPONENTS OF U.
C      INCU MUST BE GREATER THAN ZERO.
C
C   S:(INPUT) DP SCALAR
C      S ALONG WITH V, L, M, AND N DEFINE THE REFLECTION AS DESCRIBED
C      ABOVE.
C
C   VEC:(INPUT/OUTPUT) DP ARRAY OF LENGTH AT LEAST NVEC*MAX(L,N)
C      ON INPUT VEC CONTAINS THE VECTORS TO BE MULTIPLIED BY H.
C      ON OUTPUT EACH VECTOR W IS OVERWRITTEN BY H*W.
C
C   INCOMP:(INPUT) INTEGER SCALAR
C      INCOMP GIVES THE INCREMENT BETWEEN SUCCESSIVE COMPONENTS
C      OF VECTORS STORED IN VEC.  INCOMP MUST BE GREATER THAN ZERO.
C
C   INCVEC:(INPUT)  INTEGER SCALAR
C      INCVEC GIVES THE INCREMENT BETWEEN SUCCESSIVE VECTORS IN VEC.
C      INCVEC MUST BE GREATER THAN ZERO.
C
C   NVEC:(INPUT)  INTEGER SCALAR
C      NVEC GIVES THE NUMBER OF VECTORS CONTAINED IN VEC.
C
C
C
C
C     RALPH BYERS   JANUARY 1982
C     RALPH BYERS, DEPARTMENT OF MATHEMATICS, UNIVERSITY OF KANSAS,
C                  LAWRENCE, KANSAS  66045, BYERS@NA-NET.STANFORD.EDU
C
C
C     USERS MAY CONTACT RALPH BYERS AT THE ABOVE ADDRESS WITH COMMENTS
C     OR BUG REPORTS.  HOWEVER, HE MAKES NO PROMISES TO FIX BUGS OR
C     TO SUPPORT THIS SOFTWARE IN ANY WAY.
C
C
C     REFERENCES:
C      [1] GENE GOLUB AND CHARLES VAN LOAN, ``MATRIX COMPUTATIONS'',
C          THE JOHNS HOPKINS UNIVERSITY PRESS, BALTIMORE, 1983
C***********************************************************************
C
C   LOCAL VARIABLES
C
      INTEGER I, IFLAG, J, JSUBU, LSUBU, LSUBV, MSUBU, MSUBV, NSUBV
      DOUBLE PRECISION SCALE, SUM, ZERO
      INTEGER INCREM, INDEX, MAP
      DATA ZERO/0.0D0/
C
C   SUBSCRIPT MAP
C
      MAP(INDEX,INCREM) = 1 + (INDEX-1)*INCREM
C
C   CHECK INPUT
C
      IFLAG = 0
      IF ((L.LE.0) .OR. (M.LE.0) .OR. (INCU.LE.0)
     $     .OR. (INCOMP.LE.0) .OR. (INCVEC.LE.0))  IFLAG = 3
      IF ((N.LT.M) .OR. ((M.LE.L) .AND. (L.LE.N))) IFLAG = 2
      IF ((S.EQ.ZERO) .OR. (NVEC.LE.0))            IFLAG = 1
      IF (IFLAG.NE.0)  GO TO 40
C
C   APPLY H
C
      LSUBU = MAP(L,INCU)
      MSUBU = MAP(M,INCU)
      LSUBV = MAP(L,INCOMP)
      MSUBV = MAP(M,INCOMP)
      NSUBV = MAP(N,INCOMP)
C
      DO 30 I = 1, NVEC
         SUM = VEC(LSUBV)
         JSUBU = MSUBU
         DO 10 J = MSUBV, NSUBV, INCOMP
              SUM = SUM + U(JSUBU)*VEC(J)
              JSUBU = JSUBU + INCU
10       CONTINUE
         SCALE = S*SUM
C
         VEC(LSUBV) = VEC(LSUBV) - SCALE
         JSUBU = MSUBU
         DO 20 J = MSUBV, NSUBV, INCOMP
              VEC(J) = VEC(J) - SCALE*U(JSUBU)
              JSUBU = JSUBU + INCU
20       CONTINUE
C
         LSUBV = LSUBV + INCVEC
         MSUBV = MSUBV + INCVEC
         NSUBV = NSUBV + INCVEC
30    CONTINUE
C
40    CONTINUE
      RETURN
      END
      SUBROUTINE SYMREF (L, M, N, U, INCU, S, G, NDIMG, MATRIX, WORK)
      INTEGER L, M, N, INCU, NDIMG, MATRIX
      DOUBLE PRECISION U(1), S, G(1), WORK(1)
C
C   SIMILARITY TRANSFORMATION OF A SYMMETRIC MATRIX BY THE REFLECTION
C                      T
C           H = I-S*V*V
C
C   WHERE S IS A SCALAR AND V IS A VECTOR OF LENGTH AT LEAST MAX(L,N)
C      V(L) = 1
C      V(I) = 0     IF I.NE.L .AND. ((I.LT.M) .OR. (I.GT.N))
C      V(I) = U(I)  IF L.LE.I .AND. I.LE.N
C
C   L, M, N,:(INPUT)  INTEGER SCALARS
C      L, M, AND N ALONG WITH S AND V DEFINE THE REFLECTION AS
C      DESCRIBED ABOVE.  L, M, AND N MUST BE GREATER THAN ZERO AND
C      SATISFY (M.LE.N) .AND. ((L.LT.M) .OR. (N.LT.L))
C
C   U:(INPUT)  DP ARRAY OF LENGTH AT LEAST MAX(L,N)
C      THE NONZERO COMPONENTS OF V ARE GIVEN BY THE CORRESPONDING
C      COMPONENTS OF U (EXCEPT FOR V(L) WHICH IS DEFINED TO BE ONE).
C      IE V(I) = U(I) FOR (L.LE.I) .AND. (I.LE.N)
C
C   INCU:(INPUT)  INTEGER SCALAR
C      INCU GIVES THE INCREMENT BETWEEN SUCCESSIVE COMPONENTS OF U.
C      INCU MUST BE GREATER THAN ZERO.
C
C   S:(INPUT)  DP SCALAR
C      S ALONG WITH V , L, M, AND N DEFINE THE REFLECTION AS
C      DESCRIBED ABOVE.
C
C   G:(INPUT/OUTPUT)  DP ARRAY OF LENGTH AT LEAST (NDIMG*(NDIMG+1))/2
C      ON INPUT G CONTAINS A SYMMETRIC MATRIX IN SYMMETRIC STORAGE
C      MODE.  ON OUTPUT A TRAILING PRINCIPAL SUB-MATRIX OF G IS REPLACED
C      BY A SIMILARITY TRANSFORMATION OF ITSELF BY THE REFLECTION
C      DESCRIBED ABOVE.
C
C   NDIMG:(INPUT)  INTEGER SCALAR
C      NDIMG GIVES THE ORDER OF THE SYMMETRIC MATRIX GIVEN BY G.
C      NDIMG MUST BE AT GREATER THAN OF EQUAL TO MAX(L,N).
C
C   MATRIX:(INPUT)  INTEGER SCALAR
C      MATRIX GIVES THE FIRST ROW OF THE TRAILING PRINCIPAL SUB-MATRIX
C      OF G TO BE TRANSFORMED.  MATRIX MUST SATISFY THE CONDITION
C      (0.LT.MATRIX) .AND. (MATRIX.LE.MIN0(L,M)).
C
C   WORK:(WORKSPACE)  DP ARRAY OF LENGTH AT LEAST M-N+2
C
C   REFERENCES: WILKINSON,'THE ALGEBRAIC EIGENVALUE PROBLEM', ...
C
C     RALPH BYERS  JANUARY 1982
C     RALPH BYERS, DEPARTMENT OF MATHEMATICS, UNIVERSITY OF KANSAS,
C                  LAWRENCE, KANSAS  66045, BYERS@NA-NET.STANFORD.EDU
C
C
C     USERS MAY CONTACT RALPH BYERS AT THE ABOVE ADDRESS WITH COMMENTS
C     OR BUG REPORTS.  HOWEVER, HE MAKES NO PROMISES TO FIX BUGS OR
C     TO SUPPORT THIS SOFTWARE IN ANY WAY.
C
C
C     REFERENCES:
C      [1] GENE GOLUB AND CHARLES VAN LOAN, ``MATRIX COMPUTATIONS'',
C          THE JOHNS HOPKINS UNIVERSITY PRESS, BALTIMORE, 1983
C
C***********************************************************************
C
C   LOCAL VARIABLES
C
      INTEGER I, IFLAG, IJSUBG, ILSUBG, IMSUBG, ISUBU, ISUBWK, J, JSUBU
      INTEGER JSUBWK, K, KJSUBG, KSUBU, LJSUBG, LLSUBG, MJSUBG, MSUBU
      INTEGER MAPG
      INTEGER NMMP2
      DOUBLE PRECISION SCALE, SUM, TWO, ZERO
C
C   FORTRAN SUPPLIED FUNCTIONS
C
      INTEGER MAX0, MIN0
C
C   CONSTANTS
C
      DATA TWO/2.0D0/, ZERO/0.0D0/
C
C   SYMMETRIC STORAGE MAP
C
      MAPG(I,J) = (MAX0(I,J)*(MAX0(I,J)-1))/2 + MIN0(I,J)
C
C***********************************************************************
C
C   CHECK INPUT
C
      IFLAG = 0
      IF ((L.LE.0) .OR. (M.LE.0) .OR. (INCU.LE.0) .OR. (MATRIX.LE.0))
     $                                                         IFLAG = 3
      IF ((N.LT.M) .OR. ((M.LE.L) .AND. (L.LE.N))
     $    .OR. (MATRIX.GT.MIN0(L,M)) .OR. (NDIMG.LT.MAX0(L,N)))IFLAG = 2
      IF (S.EQ.ZERO) IFLAG = 1
      IF (IFLAG.NE.0) GO TO 120
C
C        T         T     T                        T      T
C   H*G*H = G - V*W - W*V         WHERE W = S*(G*V - S*(V *G*V)*V).
C                                       T       T
C   ROWS AND COLUMNS IN WHICH ONE OF V*W AND W*V  IS ZERO.
C
      MSUBU = 1 + (M-1)*INCU
C
      LJSUBG = MAPG(L,MATRIX)
      MJSUBG = MAPG(M,MATRIX)
      DO 30 J = MATRIX, NDIMG
         IF (((M.LE.J) .AND. (J.LE.N)) .OR. (J.EQ.L)) GO TO 25
              SUM = G(LJSUBG)
              KJSUBG = MJSUBG
              KSUBU = MSUBU
              DO 10 K = M, N
                   SUM = SUM + U(KSUBU)*G(KJSUBG)
                   KSUBU = KSUBU + INCU
                   IF (K.GE.J) KJSUBG = KJSUBG + K
                   IF (K.LT.J) KJSUBG = KJSUBG + 1
10            CONTINUE
              SCALE = S*SUM
C
              G(LJSUBG) = G(LJSUBG) - SCALE
              KSUBU = MSUBU
              KJSUBG = MJSUBG
              DO 20 K = M, N
                   G(KJSUBG) = G(KJSUBG) - U(KSUBU)*SCALE
                   KSUBU = KSUBU + INCU
                   IF (K.GE.J) KJSUBG = KJSUBG + K
                   IF (K.LT.J) KJSUBG = KJSUBG + 1
20            CONTINUE
C
25       CONTINUE
         IF (J.LT.L) LJSUBG = LJSUBG + 1
         IF (J.GE.L) LJSUBG = LJSUBG + J
         IF (J.LT.M) MJSUBG = MJSUBG + 1
         IF (J.GE.M) MJSUBG = MJSUBG + J
30    CONTINUE
C                                     T       T
C   ROWS AND COLUMNS IN WHICH BOTH V*W AND W*V ARE NONZERO
C
C           WORK = G*V
C
      LLSUBG = MAPG(L,L)
      SUM = G(LLSUBG)
      LJSUBG = MAPG(L,M)
      JSUBU = MSUBU
      DO 40 J = M, N
         SUM = SUM + G(LJSUBG)*U(JSUBU)
         JSUBU = JSUBU + INCU
         IF (J.LT.L) LJSUBG = LJSUBG + 1
         IF (J.GE.L) LJSUBG = LJSUBG + J
40    CONTINUE
      WORK(1) = SUM
C
      ISUBWK = 2
      ILSUBG = MAPG(M,L)
      IMSUBG = MAPG(M,M)
      DO 60 I = M, N
         SUM = G(ILSUBG)
         JSUBU = MSUBU
         IJSUBG = IMSUBG
         DO 50 J = M, N
              SUM = SUM + G(IJSUBG)*U(JSUBU)
              JSUBU = JSUBU + INCU
              IF (J.LT.I) IJSUBG = IJSUBG + 1
              IF (J.GE.I) IJSUBG = IJSUBG + J
50       CONTINUE
         WORK(ISUBWK) = SUM
C
         IF (I.LT.L) ILSUBG = ILSUBG + 1
         IF (I.GE.L) ILSUBG = ILSUBG + I
         IF (I.LT.M) IMSUBG = IMSUBG + 1
         IF (I.GE.M) IMSUBG = IMSUBG + I
         ISUBWK = ISUBWK + 1
60    CONTINUE
C                      T
C           SCALE = S*V *G*V/2
C
      ISUBU = MSUBU
      SUM = WORK(1)
      NMMP2 = N - M + 2
      DO 70 I = 2, NMMP2
         SUM = SUM + U(ISUBU)*WORK(I)
         ISUBU = ISUBU + INCU
70    CONTINUE
      SCALE = S*SUM/TWO
C
C           OVERWRITE WORK BY W
C
      WORK(1) = S*(WORK(1) - SCALE)
      ISUBU = MSUBU
      NMMP2 = N - M + 2
      DO 80 I = 2, NMMP2
         WORK(I) = S*(WORK(I) - SCALE*U(ISUBU))
         ISUBU = ISUBU + INCU
80    CONTINUE
C                           T     T
C                G = G - V*W - W*V
C
      LLSUBG = MAPG(L,L)
      G(LLSUBG) = G(LLSUBG) - WORK(1) - WORK(1)
C
      LJSUBG = MAPG(L,M)
      JSUBWK = 2
      JSUBU = MSUBU
      DO 90 J = M, N
         G(LJSUBG) = G(LJSUBG) - WORK(JSUBWK) - WORK(1)*U(JSUBU)
         JSUBU = JSUBU + INCU
         JSUBWK = JSUBWK + 1
         IF (J.LT.L) LJSUBG = LJSUBG + 1
         IF (J.GE.L) LJSUBG = LJSUBG + J
90    CONTINUE
C
      IMSUBG = MAPG(M,M)
      ISUBU = MSUBU
      ISUBWK = 2
      DO 110 I = M, N
         IJSUBG = IMSUBG
         JSUBU = MSUBU
         JSUBWK = 2
         DO 100 J = M, I
              G(IJSUBG) = G(IJSUBG) - U(ISUBU)*WORK(JSUBWK) -
     $                                            WORK(ISUBWK)*U(JSUBU)
              JSUBU = JSUBU + INCU
              JSUBWK = JSUBWK + 1
              IJSUBG = IJSUBG + 1
100      CONTINUE
C
         IMSUBG = IMSUBG + I
         ISUBU = ISUBU + INCU
         ISUBWK = ISUBWK + 1
110   CONTINUE
C
120   CONTINUE
      RETURN
      END
      SUBROUTINE GENREF (L, M, N, U, INCU, S)
      INTEGER L, M, N, INCU
      DOUBLE PRECISION U(1), S
C
C   CONSTRUCT A HOUSEHOLDER REFLECTION
C                          T
C             H = I - S*V*V
C
C   WHERE S IS A SCALAR AND V IS A VECTOR OF LENGTH AT LEAST MAX(L,N).
C      V(L) := 1
C      V(I) := 0     IF  (I.NE.L) .AND. ((I.LT.M) .OR. (N.LT.I))
C      V(I) FOR (M.LE.I .AND. I.LE.N) IS CHOSEN SO THE VECTOR
C         W := H*U   SATISFIES
C                             2      2        2        2            2
C         W(L) = +/- SQRT(U(L) + U(M) + U(M+1) + U(M+2) + ... + U(N) )
C         W(I) = 0     IF M.LE.I .AND. I.LE.N
C         W(I) = U(I)  IF I.NE.L .AND. ((I.LT.M) .OR. (N.LT.I))
C
C   L, M, N:(INPUT)  INTEGER SCALARS
C      ALONG WITH S AND V L, M, N DEFINE A REFLECTION AS
C      DESCRIBED ABOVE.  L, M, AND N MUST BE GREATER THAN ZERO AND
C      SATISFY (M.LE.N) .AND. ((L.LT.M) .OR. (N.LT.L)).
C
C   U:(INPUT/OUTPUT)  DP ARRAY OF LENGTH AT LEAST MAX(L,N)
C      ON INPUT U IS USED TO DETERMINE V AS DESCRIBED ABOVE.
C      ON OUTPUT U(L) IS OVERWRITTEN BY W(L) DESCRIBED ABOVE,
C      AND U(I) IS OVERWRITTEN BY V(I) FOR (M.LE.I) .AND. (I.LE.N).
C
C   INCU:(INPUT)  INTEGER SCALAR
C      INCU GIVES THE INCREMENT BETWEEN SUCCESSIVE COMPONENTS OF U.
C      INCU MUST BE GREATER THAN ZERO.
C
C   S:(OUTPUT)  DP SCALAR
C      ALONG WITH V, L, M, AND N, S DEFINES THE REFLECTION H  AS
C      DESCRIBED ABOVE.
C
C
C     RALPH BYERS,  JANUARY 1982
C     RALPH BYERS, DEPARTMENT OF MATHEMATICS, UNIVERSITY OF KANSAS,
C                  LAWRENCE, KANSAS  66045, BYERS@NA-NET.STANFORD.EDU
C
C
C     USERS MAY CONTACT RALPH BYERS AT THE ABOVE ADDRESS WITH COMMENTS
C     OR BUG REPORTS.  HOWEVER, HE MAKES NO PROMISES TO FIX BUGS OR
C     TO SUPPORT THIS SOFTWARE IN ANY WAY.
C
C
C     REFERENCES:
C      [1] GENE GOLUB AND CHARLES VAN LOAN, ``MATRIX COMPUTATIONS'',
C          THE JOHNS HOPKINS UNIVERSITY PRESS, BALTIMORE, 1983
C***********************************************************************
C
C   LOCAL VARIABLES
C
      INTEGER I, IFLAG, K, LSUB, MAP, MSUB, NSUB
      DOUBLE PRECISION ONE, SUM, SUP, ULMWL, WL, ZERO
C
C   FORTRAN SUPPLIED FUNCTIONS
C
      DOUBLE PRECISION DABS, DMAX1, DSIGN, DSQRT
C
C   CONSTANTS
C
      DATA ONE/1.0D0/, ZERO/0.0D0/
C
C   SUBSCRIPT MAP FOR U
C
      MAP(K) = 1 + (K-1)*INCU
C
C   CHECK INPUT
C
      S = ZERO
C
      IFLAG = 0
      IF ((L.LE.0) .OR. (M.LE.0) .OR. (INCU.LE.0)) IFLAG = 3
      IF ((N.LT.M) .OR. ((M.LE.L) .AND. (L.LE.N))) IFLAG = 2
      IF (IFLAG .NE. 0) GO TO 40
C
      LSUB = MAP(L)
      MSUB = MAP(M)
      NSUB = MAP(N)
C
      SUP = DABS(U(LSUB))
      DO 10 I = MSUB, NSUB, INCU
         SUP = DMAX1(SUP, DABS(U(I)))
10    CONTINUE
      IF (SUP .EQ. ZERO) IFLAG = 1
      IF (IFLAG .NE. 0) GO TO 40
C
C   CONSTRUCT V AND S
C
      SUM = (U(LSUB)/SUP)**2
      DO 20 I = MSUB, NSUB, INCU
         SUM = SUM + (U(I)/SUP)**2
20    CONTINUE
      WL = -DSIGN(SUP*DSQRT(SUM), U(LSUB))
      ULMWL = U(LSUB) - WL
C
      S = ONE - U(LSUB)/WL
      U(LSUB) = WL
      DO 30 I = MSUB, NSUB, INCU
         U(I) = U(I) / ULMWL
30    CONTINUE
C
40    CONTINUE
C
      RETURN
      END
