/*
 * Algebraic manipulator program.
 *
 * Mathomatic Copyright (c) 1996 George Gesslein II.
 *
 * Currently, this program only supports binary and unary operators.
 * Unary operators are implemented as a binary operation with a dummy operand.
 *
 * In the storage format, each level of parenthesis is indicated
 * by a level number (origin 1).  The deeper the level, the
 * higher the level number.
 *
 * The storage format for expressions is a fixed size array of elements
 * "token_type", which may be a CONSTANT, VARIABLE, or OPERATOR.
 * The array always alternates between operand (CONSTANT or VARIABLE)
 * and OPERATOR.  There is a separate integer for each array which
 * contains the current length of the expression stored in the array.
 * This length is always odd.
 *
 * Only one POWER operator is allowed per level in the storage format,
 * and no other operators may be on that level.  Same with the FACTORIAL
 * operator.
 *
 * Any number of TIMES and DIVIDE operators may be on the same
 * level, because they are simple multiplicative class operators.
 * The same for PLUS and MINUS.
 */

#include "am.h"
#include <signal.h>
#if	LINUX
#include <fenv.h>
#endif

#define	VERSION		"9.6"

void		fphandler();
void		inthandler();

int		n_equations;
int		cur_equation;

int		n_lhs[N_EQUATIONS];
int		n_rhs[N_EQUATIONS];

token_type	*lhs[N_EQUATIONS];
token_type	*rhs[N_EQUATIONS];

int		n_tlhs;
int		n_trhs;

token_type	tlhs[N_TOKENS];
token_type	trhs[N_TOKENS];

token_type	*scratch;

/* Set the following to "true" if you want upper and lower case variables. */
int		case_sensitive_flag = false;

int		debug_level;
int		domain_check;
int		color_flag;
int		partial_flag;	/* true for partial unfactoring */
int		symb_flag;	/* true for "simplify symbolic" */
int		high_prec;	/* true while saving equations */
int		input_column;
int		sign_flag;	/* true when all "sign" variables are to compare equal */
int		usage_flag;
int		coption;
char		*prog_name;
char		*tmp_file = "mathxxx.tmp";
double		epsilon;

sign_array_type	sign_array;

FILE		*gfp;

jmp_buf		jmp_save;

int		start;
int		eargc;
char		**eargv;

int
main(argc, argv)
int	argc;
char	**argv;
{
	int		i;
	char		*cp;
	double		numerator, denominator;

	prog_name = argv[0];
	init_gvars();
	gfp = stdout;
	setbuf(stdout, NULL);		/* make standard output unbuffered */
	eargc = argc;
	eargv = argv;
	epsilon = 0.00000000000005;	/* the correct value for doubles */
	start = 1;
	if (argc > 1
	    && (strcmp(argv[1], "-c") == 0 || strcmp(argv[1], "-C") == 0)) {
		start++;
		coption = true;
	}
#ifdef	TUTOR
	printf("Mathomatic Version %s tutorial.\n", VERSION);
#else
	printf("Mathomatic Version %s (full-size version).\n", VERSION);
#endif
	print_copyright();
	init_mem();
#if	!TUTOR
	if (argc >= 1) {
		i = strlen(argv[0]);
		for (; i > 0; i--) {
			if (argv[0][i] == '/' || argv[0][i] == '\\') {
				i = 0;
				break;
			}
			if (argv[0][i] == '.')
				break;
		}
		if (i <= 0) {
			i = strlen(argv[0]);
		}
		if (i && (argv[0][i-1] == 'C' || argv[0][i-1] == 'c')) {
			if (!coption) {
#if	UNIX
				printf("ANSI color mode enabled.  To disable, rename this program to 'am',\n");
				printf("or use the '-c' option.\n");
#else
				printf("\nANSI color mode enabled.  If the 'ansi.sys' driver is not loaded\n");
				printf("or if a color display is not installed, rename this program to 'am.exe'.\n");
#endif
			}
			color_flag = true;
		}
	}
#endif
	if (coption) {
		if (color_flag) {
			printf("ANSI color mode toggled off by the '-c' option.\n");
			color_flag = false;
		} else {
			printf("ANSI color mode enabled.\n");
			color_flag = true;
		}
	}
	default_color();
	printf("\n");
#if	LINUX
	feenableexcept(FE_INVALID | FE_DIVBYZERO | FE_OVERFLOW);
#endif
#if	false
	ieee_handler("set", "common", fphandler);
#else
	if (signal(SIGFPE, fphandler) == SIG_ERR) {
		printf("Unable to trap floating point errors!\n");
		exit(1);
	}
#endif
	signal(SIGINT, inthandler);
	if ((i = setjmp(jmp_save)) != 0) {
		clean_up();
		if (i != 3) {
			if (i == 13)
				printf("Operation abruptly aborted.\n");
			else
				printf("Operation aborted.\n");
		}
	} else {
		if (!f_to_fraction(0.5, &numerator, &denominator)
		    || !f_to_fraction(1.0/3.0, &numerator, &denominator)) {
			printf("Cannot convert floating point values to fractions.\n");
			printf("Your Floating Point Unit may be crippled.\n");
		}
#if	!TUTOR
		for (i = start; i < argc; i++) {
			if (!read_in(argv[i])) {
				reedit(argv[i]);
				break;
			} else if (strcmp(argv[i], tmp_file) == 0) {
				unlink(tmp_file);
			}
		}
#endif
	}
#if	TUTOR
	if (start < argc) {
		if (strcmp(argv[start], "-") == 0)
			start++;
		ex_limited();
	} else {
		tutor_proc();
	}
#else
	for (;;) {
		input_column = printf("%d%s", cur_equation + 1, PROMPT);
		if ((cp = fgets((char *) &tlhs[0], N_TOKENS * sizeof(token_type), stdin)) == NULL)
			break;
		process(cp);
	}
#endif
	printf("\n");
	quit("");
}

print_copyright()
{
	printf("Copyright (C) 1996 George Gesslein II.\n");
}

#if	TUTOR
int
ex_limited()
{
	int	i;
	char	*cp;

	printf("Leaving tutorial and entering Mathomatic.\n\n");
	printf("Mathomatic Version %s (emailware version).\n", VERSION);
	print_copyright();
	if (color_flag) {
		printf("\nANSI color mode enabled.\n");
	}
	default_color();
	printf("\n");
	clear("all");
	if ((i = setjmp(jmp_save)) != 0) {
		clean_up();
		if (i != 3) {
			if (i == 13)
				printf("Operation abruptly aborted.\n");
			else
				printf("Operation aborted.\n");
		}
	} else {
		for (i = start; i < eargc; i++) {
			if (!read_in(eargv[i])) {
				reedit(eargv[i]);
				break;
			} else if (strcmp(eargv[i], tmp_file) == 0) {
				unlink(tmp_file);
			}
		}
	}
	for (;;) {
		input_column = printf("%d%s", cur_equation + 1, PROMPT);
		if ((cp = fgets((char *) &tlhs[0], N_TOKENS * sizeof(token_type), stdin)) == NULL)
			break;
		process(cp);
	}
	printf("\n");
	quit("");
}
#endif

/*
 * Allocate the equation spaces.
 */
int
init_mem()
{
	int	i;
	char	*junk;

	if ((junk = malloc(4096)) == NULL
	    || (scratch = (token_type *) malloc((N_TOKENS + 10) * sizeof(token_type))) == NULL) {
		printf("Not enough memory.\n");
		exit(1);
	}
	for (i = 0; i < N_EQUATIONS; i++) {
#if	!HALLOC
		lhs[i] = (token_type *) malloc(N_TOKENS * sizeof(token_type));
#else
		lhs[i] = (token_type *) halloc((long) N_TOKENS, sizeof(token_type));
#endif
		if (lhs[i] == NULL)
			break;
#if	!HALLOC
		rhs[i] = (token_type *) malloc(N_TOKENS * sizeof(token_type));
#else
		rhs[i] = (token_type *) halloc((long) N_TOKENS, sizeof(token_type));
#endif
		if (rhs[i] == NULL) {
#if	!HALLOC
			free((char *) lhs[i]);
#else
			hfree((char *) lhs[i]);
#endif
			break;
		}
	}
	free(junk);
	n_equations = i;
	clear("all");
#if	TUTOR
	if (n_equations != N_EQUATIONS) {
		printf("Not enough memory.\n");
		exit(1);
	}
#else
	if (n_equations != N_EQUATIONS && n_equations < 5) {
		printf("\nDue to the small amount of available memory,\n");
		printf("you only have room for %d equations.\n", n_equations);
	} else {
		printf("%d equation spaces allocated (total size is %ld KBytes).\n",
		    n_equations, ((long) N_TOKENS * sizeof(token_type) * 2L * n_equations) / 1000L);
	}
#endif
}

#if	MSC
void
fphandler(sig, num)
int	sig, num;
{
	_fpreset();
	switch (num) {
	case FPE_INVALID:
		printf("Floating point value invalid!\n");
		break;
	case FPE_ZERODIVIDE:
		printf("Divide by zero error!\n");
		break;
	case FPE_OVERFLOW:
		printf("Floating point overflow!\n");
		break;
	case FPE_UNDERFLOW:
		printf("Floating point underflow!\n");
		break;
	case FPE_INEXACT:
		printf("Floating point result inexact!\n");
		break;
	default:
		printf("Unknown floating point error!  Code = 0x%x.\n", num);
		break;
	}
	longjmp(jmp_save, 2);
}
#else
void
fphandler(sig, num)
int	sig, num;
{
#if	LINUX
	printf("Floating point exception!\n");
	feenableexcept(FE_INVALID | FE_DIVBYZERO | FE_OVERFLOW);
	signal(SIGFPE, fphandler);
#else
	printf("Floating point exception!\n");
	signal(SIGFPE, fphandler);
#endif
	longjmp(jmp_save, 2);
}
#endif

#if	MATHERR
int
matherr(x)
struct exception	*x;
{
	switch (x->type) {
	case DOMAIN:
		if (domain_check) {
			domain_check = false;
			return 1;
		}
		printf("Domain error in constant!\n");
		break;
	case SING:
		printf("Singularity error in constant!\n");
		break;
	case OVERFLOW:
		printf("Overflow error in constant!\n");
		break;
	case UNDERFLOW:
		printf("Underflow error in constant!\n");
		break;
	case TLOSS:
		printf("Total loss of significance in constant!\n");
		break;
	case PLOSS:
		printf("Partial loss of significance in constant!\n");
		break;
	default:
		printf("Unknown error in constant!\n");
		break;
	}
	longjmp(jmp_save, 2);
}

int
check_err()
{
}
#else
int
check_err()
{
	switch (errno) {
	case EDOM:
		errno = 0;
		if (domain_check) {
			domain_check = false;
		} else {
			printf("Domain error in constant!\n");
			longjmp(jmp_save, 2);
		}
		break;
	case ERANGE:
		errno = 0;
		printf("Overflow error in constant!\n");
		longjmp(jmp_save, 2);
		break;
	}
}
#endif

void
inthandler(sig, num)
int	sig, num;
{
	printf("\n");
	quit("");
}
