/*
 * Algebraic manipulator commands.
 *
 * Copyright (c) 1996 George Gesslein II.
 */

#include "am.h"
#include "externs.h"
#if	!UNIX
#include <process.h>
#endif

extern	int	euclid_count;

int
ngcd(d1, d2)
double	d1, d2;
{
	int	i;
	int	gcd_iterations;
	double	d3, d4;
	double	d5, d6;

	if (fabs(d1) >= 1e14 || fabs(d2) >= 1e14) {
		printf("Each constant for this command must be 14 digits or less.\n");
		return false;
	}
	d3 = gcd(d1, d2);
	if (d3 == 0.0) {
		printf("No GCD found.\n");
		return false;
	}
	gcd_iterations = euclid_count;
	if (fabs(d1 / d3) < (1.0 - epsilon)
	    || fabs(d2 / d3) < (1.0 - epsilon)) {
		printf("The computed GCD is too large!\n");
		return false;
	}
	modf(fabs(d1 / d3) + 0.5, &d4);
	d4 = fabs(d1 / d4);
	if (fabs(modf(fabs(d1 / d4), &d5) - 0.5) < (0.5 - epsilon)
	    || fabs(modf(fabs(d2 / d4), &d6) - 0.5) < (0.5 - epsilon)) {
		printf("The computed GCD is too inaccurate!\n");
		return false;
	}
	modf(fabs(d1 / d4) + 0.5, &d5);
	modf(fabs(d2 / d4) + 0.5, &d6);
	if (gcd(d5, d6) != 1.0) {
		printf("The computed GCD is too small!\n");
		return false;
	}
	if (d3 != d4) {
		printf("Corrected ");
	}
	printf("GCD = %.14lg (number of Euclidean algorithm iterations = %d)\n", d4, gcd_iterations);
	printf("Least Common Multiple (LCM) = %.14lg\n", fabs((d1 * d2) / d4));
	return true;
}

int
pause_cmd(cp)
char	*cp;
{
	char	*cp1;

	printf("Press the Enter key to continue or type \"quit\" to exit program...");
	if ((cp1 = fgets((char *) &tlhs[0], N_TOKENS * sizeof(token_type), stdin)) == NULL) {
		quit("");
	}
	str_tolower(cp1);
	if (strncmp(cp1, "quit", 4) == 0)
		quit("");
	return true;
}

int
copy(cp)
char	*cp;
{
	int	i, j, k;
	int	i1;
	char	exists[N_EQUATIONS];

	if (!get_range(&cp, &i, &j)) {
		return false;
	}
	if (extra_garbage(cp))
		return false;
	for (i1 = 0; i1 < N_EQUATIONS; i1++) {
		exists[i1] = false;
	}
	for (i1 = i; i1 <= j; i1++) {
		if (n_lhs[i1] > 0) {
			exists[i1] = true;
		}
	}
	for (i1 = i; i1 <= j; i1++) {
		if (exists[i1]) {
			k = next_espace();
			blt(&lhs[k][0], &lhs[i1][0], n_lhs[i1] * sizeof(token_type));
			n_lhs[k] = n_lhs[i1];
			blt(&rhs[k][0], &rhs[i1][0], n_rhs[i1] * sizeof(token_type));
			n_rhs[k] = n_rhs[i1];
			list_sub(k);
		}
	}
	return true;
}

int
complex_sub(cp, imag_flag)
char	*cp;
int	imag_flag;
{
	int	i, j, k;
	int	beg;
	int	found_imag;
	int	has_imag, has_real;

	if ((i = get_default_en(cp)) < 0)
		return false;
	if (n_lhs[i] != 1) {
		printf("Please solve this equation first.\n");
		return false;
	}
	j = next_espace();
	uf_simp(&rhs[i][0], &n_rhs[i]);
	factorv(&rhs[i][0], &n_rhs[i], (long) IMAGINARY);
	partial_flag = false;
	uf_simp(&rhs[i][0], &n_rhs[i]);
	partial_flag = true;
	n_rhs[j] = 1;
	rhs[j][0].level = 1;
	rhs[j][0].kind = CONSTANT;
	rhs[j][0].token.constant = 0.0;
	has_imag = false;
	has_real = false;
	k = 0;
	for (beg = k; beg < n_rhs[i]; beg = k, k++) {
		found_imag = false;
		for (;; k++) {
			if (k >= n_rhs[i])
				break;
			if (rhs[i][k].level == 1 && rhs[i][k].kind == OPERATOR
			    && (rhs[i][k].token.operatr == PLUS || rhs[i][k].token.operatr == MINUS)) {
				break;
			}
			if (rhs[i][k].kind == VARIABLE && rhs[i][k].token.variable == IMAGINARY) {
				found_imag = true;
			}
		}
		if (found_imag)
			has_imag = true;
		else
			has_real = true;
		if (found_imag == imag_flag) {
			if (beg == 0) {
				blt(&rhs[j][0], &rhs[i][0], k * sizeof(token_type));
				n_rhs[j] = k;
			} else {
				blt(&rhs[j][n_rhs[j]], &rhs[i][beg], (k - beg) * sizeof(token_type));
				n_rhs[j] += (k - beg);
			}
		}
	}
	if (!imag_flag && !has_real) {
		printf("No real parts found.\n");
		return false;
	}
	if (!has_imag) {
		printf("No imaginary parts found.  Equation is not complex.\n");
		return false;
	}
	if (!has_real) {
		printf("No real parts found.  Equation is not complex.\n");
		return false;
	}
	blt(&lhs[j][0], &lhs[i][0], n_lhs[i] * sizeof(token_type));
	n_lhs[j] = n_lhs[i];
	printf("(");
	list_proc(&lhs[j][0], n_lhs[j]);
	if (imag_flag) {
		printf(") is now equal to the imaginary part.\n");
	} else {
		printf(") is now equal to the real part.\n");
	}
	simp_divide(&rhs[j][0], &n_rhs[j], 0L);
	cur_equation = j;
	list_sub(cur_equation);
	return true;
}

int
real(cp)
char	*cp;
{
	return complex_sub(cp, false);
}

int
imaginary(cp)
char	*cp;
{
	return complex_sub(cp, true);
}

int
tally(cp)
char	*cp;
{
	int	i;

	if (extra_garbage(cp))
		return false;
	trhs[0].kind = CONSTANT;
	trhs[0].level = 1;
	trhs[0].token.constant = 0.0;
	n_trhs = 1;
	for (;;) {
		printf("Running total = ");
		list_proc(&trhs[0], n_trhs);
		printf("\n");
oops:
		input_column = printf("Enter value: ");
		if (!get_expr(&tlhs[0], &n_tlhs)) {
			goto oops;
		}
		if (n_tlhs == 0)
			break;
		if ((n_trhs + 1 + n_tlhs) > N_TOKENS) {
			error_huge();
		}
		for (i = 0; i < n_tlhs; i++) {
			tlhs[i].level++;
		}
		for (i = 0; i < n_trhs; i++) {
			trhs[i].level++;
		}
		trhs[n_trhs].kind = OPERATOR;
		trhs[n_trhs].level = 1;
		trhs[n_trhs].token.operatr = PLUS;
		n_trhs++;
		blt(&trhs[n_trhs], &tlhs[0], n_tlhs * sizeof(token_type));
		n_trhs += n_tlhs;
		subst_constants(&trhs[0], &n_trhs);
		simp_side(&trhs[0], &n_trhs);
		uf_simp(&trhs[0], &n_trhs);
		factorv(&trhs[0], &n_trhs, (long) IMAGINARY);
		simp_side(&trhs[0], &n_trhs);
		uf_simp(&trhs[0], &n_trhs);
	}
	return true;
}

int
calculate(cp)
char	*cp;
{
	int	i, j, k;
	int	level;
	long	last_v;
	long	v;
	long	counter;
	long	counter_max;
	sign_array_type	sa_mark;
	sign_array_type	sa_value;
	jmp_buf	save_save;

	if ((i = get_default_en(cp)) < 0)
		return false;
	n_trhs = n_rhs[i];
	blt(&trhs[0], &rhs[i][0], n_trhs * sizeof(token_type));
	last_v = -1;
	for (;;) {
		v = -1;
		for (j = 0; j < n_rhs[i]; j += 2) {
			if (rhs[i][j].kind == VARIABLE && rhs[i][j].token.variable > IMAGINARY) {
				if (rhs[i][j].token.variable > last_v
				    && (v == -1 || rhs[i][j].token.variable < v))
					v = rhs[i][j].token.variable;
			}
		}
		if (v == -1)
			break;
		last_v = v;
		if ((v & VAR_MASK) == SIGN) {
			continue;
		}
oops:
		input_column = printf("Enter ");
		input_column += list_var(v, true);
		input_column += printf(": ");
		if (!get_expr(&tlhs[0], &n_tlhs)) {
			goto oops;
		}
		if (n_tlhs == 0)
			continue;
		for (j = 0; j < n_tlhs; j++)
			if (tlhs[j].kind == VARIABLE)
				tlhs[j].token.variable = -tlhs[j].token.variable;
		for (j = n_trhs - 1; j >= 0; j--) {
			if (trhs[j].kind == VARIABLE && trhs[j].token.variable == v) {
				level = trhs[j].level;
				if (n_trhs + n_tlhs - 1 > N_TOKENS) {
					error_huge();
				}
				blt(&trhs[j+n_tlhs], &trhs[j+1], (n_trhs - (j + 1)) * sizeof(token_type));
				n_trhs += n_tlhs - 1;
				blt(&trhs[j], &tlhs[0], n_tlhs * sizeof(token_type));
				for (k = j; k < j + n_tlhs; k++)
					trhs[k].level += level;
			}
		}
	}
	for (j = 0; j < n_trhs; j += 2)
		if (trhs[j].kind == VARIABLE && trhs[j].token.variable < 0)
			trhs[j].token.variable = -trhs[j].token.variable;
	blt(save_save, jmp_save, sizeof(jmp_save));
	if (setjmp(jmp_save) != 0) {
		clean_up();
		blt(jmp_save, save_save, sizeof(jmp_save));
		return false;
	}
	subst_constants(&trhs[0], &n_trhs);
	simp_side(&trhs[0], &n_trhs);
	blt(jmp_save, save_save, sizeof(jmp_save));
	for (j = 0; j < ARR_CNT(sa_mark); j++)
		sa_mark[j] = false;
	for (j = 0; j < n_trhs; j += 2) {
		if (trhs[j].kind == VARIABLE
		    && (trhs[j].token.variable & VAR_MASK) == SIGN) {
			sa_mark[(trhs[j].token.variable >> VAR_SHIFT) & SUBSCRIPT_MASK] = true;
		}
	}
	for (j = 0, k = 0; j < ARR_CNT(sa_mark); j++) {
		if (sa_mark[j]) {
			k++;
		}
	}
	counter_max = (1L << k) - 1;
	blt(save_save, jmp_save, sizeof(jmp_save));
	counter = 0;
	if (setjmp(jmp_save) != 0) {
		clean_up();
		counter++;
	}
	for (; counter <= counter_max; counter++) {
		blt(&tlhs[0], &trhs[0], n_trhs * sizeof(token_type));
		n_tlhs = n_trhs;
		for (j = 0, k = 0; j < ARR_CNT(sa_mark); j++) {
			if (sa_mark[j]) {
				sa_value[j] = (((1L << k) & counter) != 0);
				k++;
			}
		}
		for (j = 0; j < n_tlhs; j += 2) {
			if (tlhs[j].kind == VARIABLE
			    && (tlhs[j].token.variable & VAR_MASK) == SIGN) {
				if (sa_value[(tlhs[j].token.variable >> VAR_SHIFT) & SUBSCRIPT_MASK]) {
					tlhs[j].kind = CONSTANT;
					tlhs[j].token.constant = -1.0;
				} else {
					tlhs[j].kind = CONSTANT;
					tlhs[j].token.constant = 1.0;
				}
			}
		}
		for (j = 0, k = false; j < ARR_CNT(sa_mark); j++) {
			if (sa_mark[j]) {
				if (k)
					printf(", ");
				else {
					printf("Solution #%ld with ", counter + 1);
				}
				list_var((long) SIGN + (((long) j) << VAR_SHIFT), true);
				printf(" = ");
				if (sa_value[j]) {
					printf("-1");
				} else {
					printf("1");
				}
				k = true;
			}
		}
		if (k)
			printf(":\n");
		simp_side(&tlhs[0], &n_tlhs);
		uf_simp(&tlhs[0], &n_tlhs);
		factorv(&tlhs[0], &n_tlhs, (long) IMAGINARY);
		simp_side(&tlhs[0], &n_tlhs);
		uf_simp(&tlhs[0], &n_tlhs);
		printf(" ");
		list_proc(&lhs[i][0], n_lhs[i]);
		printf(" = ");
		list_proc(&tlhs[0], n_tlhs);
		printf("\n");
	}
	blt(jmp_save, save_save, sizeof(jmp_save));
	return true;
}

int
clear(cp)
char	*cp;
{
	int	i, j;

	if (is_all(cp)) {
		cur_equation = 0;
		for (i = 0; i < n_equations; i++) {
			n_lhs[i] = 0;
		}
	} else {
		if (!get_range(&cp, &i, &j)) {
			return false;
		}
		if (extra_garbage(cp))
			return false;
		for (; i <= j; i++) {
			n_lhs[i] = 0;
		}
	}
	return true;
}

int
compare_rhs(i, j, diff_signp)
int	i, j;
int	*diff_signp;
{
	int	rv;

	sign_flag = true;
	rv = se_compare(&rhs[i][0], n_rhs[i], &rhs[i][0], n_rhs[i], diff_signp);
	sign_flag = false;
	if (!rv || *diff_signp) {
		printf("Error in compare function or too many terms to compare!\n");
		longjmp(jmp_save, 2);
	}
	sign_flag = true;
	rv = se_compare(&rhs[i][0], n_rhs[i], &rhs[j][0], n_rhs[j], diff_signp);
	sign_flag = false;
	return rv;
}

int
compare(cp)
char	*cp;
{
	int		i, j, k;
	token_type	want;
	int		diff_sign;
	int		already_solved;

	i = atoi(cp) - 1;
	cp = skip_num(cp);
	if (i < 0 || i >= n_equations) {
		printf("Invalid equation number.\n");
		usage_flag = true;
		return false;
	}
	if (*cp == '\0') {
		j = cur_equation;
	} else {
		if (strncmp(cp, "with", 4) == 0) {
			cp = skip_space(cp + 4);
			j = atoi(cp) - 1;
			cp = skip_num(cp);
			if (j < 0 || j >= n_equations) {
				printf("Invalid equation number.\n");
				usage_flag = true;
				return false;
			}
			if (extra_garbage(cp))
				return false;
		} else {
			printf("Syntax error.\n");
			usage_flag = true;
			return false;
		}
	}
	if (i == j) {
		printf("Ridiculous command.\n");
		return false;
	}
	if (n_lhs[i] <= 0 || n_lhs[j] <= 0) {
		printf("Undefined equation.\n");
		return false;
	}
	printf("Comparing equation #%d with #%d...\n", i + 1, j + 1);
	already_solved = (solved_equation(i) && solved_equation(j));
	if (already_solved) {
		simp_loop(&rhs[i][0], &n_rhs[i]);
		simp_loop(&rhs[j][0], &n_rhs[j]);
		if (compare_rhs(i, j, &diff_sign)) {
			goto times_neg1;
		}
		printf("Simplifying both equations...\n");
		simpa_side(&rhs[i][0], &n_rhs[i], true, false);
		list_sub(i);
		simpa_side(&rhs[j][0], &n_rhs[j], true, false);
		list_sub(j);
		if (compare_rhs(i, j, &diff_sign)) {
			goto times_neg1;
		}
		uf_simp(&rhs[i][0], &n_rhs[i]);
		uf_simp(&rhs[j][0], &n_rhs[j]);
		if (compare_rhs(i, j, &diff_sign)) {
			goto times_neg1;
		}
	}
	printf("Solving both equations for zero and unfactoring...\n");
	want.level = 1;
	want.kind = CONSTANT;
	want.token.constant = 0.0;
	if (!solve_sub(&want, 1, &lhs[i][0], &n_lhs[i], &rhs[i][0], &n_rhs[i])
	    || !solve_sub(&want, 1, &lhs[j][0], &n_lhs[j], &rhs[j][0], &n_rhs[j])) {
		printf("Can't solve for zero!\n");
		return false;
	}
	uf_simp(&rhs[i][0], &n_rhs[i]);
	uf_simp(&rhs[j][0], &n_rhs[j]);
	if (compare_rhs(i, j, &diff_sign)) {
		printf("Equations are identical.\n");
		return true;
	}
	printf("Simplifying both equations...\n");
	simpa_side(&rhs[i][0], &n_rhs[i], true, false);
	simpa_side(&rhs[j][0], &n_rhs[j], true, false);
	if (compare_rhs(i, j, &diff_sign)) {
		printf("Equations are identical.\n");
		return true;
	}
	if (!solve_sub(&want, 1, &lhs[i][0], &n_lhs[i], &rhs[i][0], &n_rhs[i])
	    || !solve_sub(&want, 1, &lhs[j][0], &n_lhs[j], &rhs[j][0], &n_rhs[j])) {
		printf("Can't solve for zero!\n");
		return false;
	}
	uf_simp(&rhs[i][0], &n_rhs[i]);
	uf_simp(&rhs[j][0], &n_rhs[j]);
	if (compare_rhs(i, j, &diff_sign)) {
		printf("Equations are identical.\n");
		return true;
	}
	printf("Equations may differ.\n");
	return false;
times_neg1:
	if (!diff_sign && lhs[i][0].token.variable == lhs[j][0].token.variable) {
		printf("Equations are identical.\n");
		return true;
	}
	printf("Variable (");
	list_proc(&lhs[i][0], n_lhs[i]);
	printf(") in the first equation is\nequal to (");
	if (diff_sign) {
		printf("-");
	}
	list_proc(&lhs[j][0], n_lhs[j]);
	printf(") in the second equation.\n");
	return(!diff_sign);
}

int
debug(cp)
char	*cp;
{
	debug_level = atoi(cp);
	if (debug_level)
		printf("Debug level set to %d.\n", debug_level);
	else
		printf("Debug output turned off.\n");
	return true;
}

int
derivative(cp)
char	*cp;
{
	long	v;
	int	i, j;
	long	l;
	int	found;
	int	er;

	if (n_lhs[cur_equation] <= 0) {
		printf("Undefined equation.\n");
		return false;
	}
	v = 0;
	if (*cp == '\0') {
		if (!prompt_var(&v)) {
			return false;
		}
	} else {
		if (strncmp(cp, "all", 3) == 0) {
			v = MATCH_ANY;
			cp += 3;
		} else {
			cp = parse_var2(&v, cp);
			if (cp == NULL) {
				return false;
			}
			if (*cp) {
				printf("Only one variable or \"all\" may be specified.\n");
				printf("\"all\" differentiates with respect to all normal variables.\n");
				usage_flag = true;
				return false;
			}
		}
	}
	i = next_espace();
	found = false;
	er = !solved_equation(cur_equation);
	if (!er) {
		er = ((lhs[cur_equation][0].token.variable & VAR_MASK) <= SIGN);
	}
	for (j = 0; j < n_rhs[cur_equation]; j++) {
		if (rhs[cur_equation][j].kind == VARIABLE) {
			if (rhs[cur_equation][j].token.variable == v)
				found = true;
		}
	}
	if (er) {
		printf("Warning: This equation is not solved for a normal variable.\n");
	}
	if (v && v != MATCH_ANY && !found) {
		printf("Variable not found in RHS.  Result would be zero.\n");
		return false;
	}
	if (!differentiate(&rhs[cur_equation][0], n_rhs[cur_equation], &rhs[i][0], &n_rhs[i], v, false)) {
		printf("Differentiation failed.\n");
		return false;
	}
	simpd_side(&rhs[i][0], &n_rhs[i]);
	blt(&lhs[i][0], &lhs[cur_equation][0], n_lhs[cur_equation] * sizeof(token_type));
	n_lhs[i] = n_lhs[cur_equation];
	if (!er) {
		l = lhs[i][0].token.variable;
		l = (unsigned long) l + (unsigned long) PRIME_INCREMENT;
		if (l <= 0) {
			printf("Warning: Too many primes in variable name.  Variable unchanged.\n");
		} else {
			lhs[i][0].token.variable = l;
		}
	}
	cur_equation = i;
	list_sub(cur_equation);
	return true;
}

int
integrate(cp)
char	*cp;
{
	long	v;
	long	l;
	int	i, j, k;
	int	i1, j1;
	int	level;
	int	er;
	int	iterations;
	int	first_size;
	token_type	*ep;
	int	trap_flag;

	iterations = 100;	/* must be even */
	trap_flag = false;
	if (n_lhs[cur_equation] <= 0) {
		printf("Undefined equation.\n");
		return false;
	}
	if (*cp == '\0') {
		if (!prompt_var(&v))
			return false;
	} else {
		cp = parse_var2(&v, cp);
		if (cp == NULL) {
			return false;
		}
		if (strncmp(cp, "trap", 4) == 0) {
			trap_flag = true;
		} else if (*cp) {
			printf("This program will only integrate with respect to one variable.\n");
			usage_flag = true;
			return false;
		}
	}
	if ((v & VAR_MASK) <= SIGN) {
		printf("This program will only integrate with respect to a normal variable.\n");
		return false;
	}
	i = next_espace();
	er = !solved_equation(cur_equation);
	if (!er) {
		er = ((lhs[cur_equation][0].token.variable & VAR_MASK) <= SIGN);
	}
	if (er) {
		printf("Please solve this equation for a normal variable and try again.\n");
		return false;
	}
oops:
	input_column = printf("Enter lower limit: ");
	if (!get_expr(&tlhs[0], &n_tlhs)) {
		goto oops;
	}
	if (n_tlhs == 0) {
		printf("Command aborted.\n");
		return false;
	}
	subst_constants(&tlhs[0], &n_tlhs);
oops2:
	input_column = printf("Enter upper limit: ");
	if (!get_expr(&trhs[0], &n_trhs)) {
		goto oops2;
	}
	if (n_trhs == 0) {
		printf("Command aborted.\n");
		return false;
	}
	subst_constants(&trhs[0], &n_trhs);
	if ((n_tlhs + 1 + n_trhs + 2) > N_TOKENS) {
		error_huge();
	}
	printf("Approximating the definite integral using ");
	if (trap_flag) {
		printf("the trapezoidal method...\n");
	} else {
		printf("Simpson's rule...\n");
	}
	for (j = 0; j < n_trhs; j++)
		trhs[j].level += 2;
	trhs[n_trhs].level = 2;
	trhs[n_trhs].kind = OPERATOR;
	trhs[n_trhs].token.operatr = MINUS;
	n_trhs++;
	j = n_trhs;
	blt(&trhs[n_trhs], &tlhs[0], n_tlhs * sizeof(token_type));
	n_trhs += n_tlhs;
	for (; j < n_trhs; j++)
		trhs[j].level += 2;
	trhs[n_trhs].level = 1;
	trhs[n_trhs].kind = OPERATOR;
	trhs[n_trhs].token.operatr = DIVIDE;
	n_trhs++;
	trhs[n_trhs].level = 1;
	trhs[n_trhs].kind = CONSTANT;
	trhs[n_trhs].token.constant = iterations;
	n_trhs++;
	simp_loop(&trhs[0], &n_trhs);
	rhs[i][0].level = 1;
	rhs[i][0].kind = CONSTANT;
	rhs[i][0].token.constant = 0.0;
	n_rhs[i] = 1;
	for (j = 0; j <= iterations; j++) {
		if (j > 0 && color_flag
		    && ((j % (iterations / 10)) == 0 || j == iterations)) {
			printf("%ld%s", (j * 100L) / iterations, "% complete.");
			if (j == iterations)
				printf("\n");
			else
				printf("\r");
		}
		if ((n_rhs[i] + 1 + n_rhs[cur_equation]) > N_TOKENS)
			error_huge();
		for (k = 0; k < n_rhs[i]; k++)
			rhs[i][k].level++;
		ep = &rhs[i][n_rhs[i]];
		ep->level = 1;
		ep->kind = OPERATOR;
		ep->token.operatr = PLUS;
		n_rhs[i]++;
		i1 = n_rhs[i];
		blt(&rhs[i][i1], &rhs[cur_equation][0], n_rhs[cur_equation] * sizeof(token_type));
		n_rhs[i] += n_rhs[cur_equation];
		subst_constants(&rhs[i][i1], &n_rhs[cur_equation]);
		for (k = i1; k < n_rhs[i]; k++)
			rhs[i][k].level += 2;
		for (k = i1; k < n_rhs[i]; k += 2) {
			if (rhs[i][k].kind == VARIABLE
			    && rhs[i][k].token.variable == v) {
				level = rhs[i][k].level;
				j1 = n_tlhs + 2 + n_trhs;
				if ((n_rhs[i] + j1) > N_TOKENS)
					error_huge();
				blt(&rhs[i][k+1+j1], &rhs[i][k+1], (n_rhs[i] - (k + 1)) * sizeof(token_type));
				n_rhs[i] += j1;
				j1 = k;
				blt(&rhs[i][k], &tlhs[0], n_tlhs * sizeof(token_type));
				k += n_tlhs;
				for (; j1 < k; j1++)
					rhs[i][j1].level += level + 1;
				ep = &rhs[i][k];
				ep->level = level + 1;
				ep->kind = OPERATOR;
				ep->token.operatr = PLUS;
				ep++;
				ep->level = level + 2;
				ep->kind = CONSTANT;
				ep->token.constant = j;
				ep++;
				ep->level = level + 2;
				ep->kind = OPERATOR;
				ep->token.operatr = TIMES;
				k += 3;
				j1 = k;
				blt(&rhs[i][k], &trhs[0], n_trhs * sizeof(token_type));
				k += n_trhs;
				for (; j1 < k; j1++)
					rhs[i][j1].level += level + 2;
				k--;
			}
		}
		if (j > 0 && j < iterations) {
			if ((n_rhs[i] + 2) > N_TOKENS)
				error_huge();
			ep = &rhs[i][n_rhs[i]];
			ep->level = 2;
			ep->kind = OPERATOR;
			ep->token.operatr = TIMES;
			ep++;
			ep->level = 2;
			ep->kind = CONSTANT;
			if (trap_flag) {
				ep->token.constant = 2.0;
			} else {
				if ((j & 1) == 1) {
					ep->token.constant = 4.0;
				} else {
					ep->token.constant = 2.0;
				}
			}
			n_rhs[i] += 2;
		}
		elim_loop(&rhs[i][0], &n_rhs[i]);
		ufactor(&rhs[i][0], &n_rhs[i]);
		simp_divide(&rhs[i][0], &n_rhs[i], 0L);
		if (j > 0) {
			if (j == 1) {
				first_size = n_rhs[i];
				if (first_size < 4)
					first_size = 4;
			} else {
				if ((n_rhs[i] / 8) >= first_size) {
					printf("Integration failed.\n");
					return false;
				}
			}
		}
	}
	if ((n_rhs[i] + 3 + n_trhs) > N_TOKENS)
		error_huge();
	for (k = 0; k < n_rhs[i]; k++)
		rhs[i][k].level++;
	ep = &rhs[i][n_rhs[i]];
	ep->level = 1;
	ep->kind = OPERATOR;
	ep->token.operatr = DIVIDE;
	ep++;
	ep->level = 1;
	ep->kind = CONSTANT;
	if (trap_flag) {
		ep->token.constant = 2.0;
	} else {
		ep->token.constant = 3.0;
	}
	ep++;
	ep->level = 1;
	ep->kind = OPERATOR;
	ep->token.operatr = TIMES;
	n_rhs[i] += 3;
	k = n_rhs[i];
	blt(&rhs[i][k], &trhs[0], n_trhs * sizeof(token_type));
	n_rhs[i] += n_trhs;
	for (; k < n_rhs[i]; k++)
		rhs[i][k].level++;
	elim_loop(&rhs[i][0], &n_rhs[i]);
	ufactor(&rhs[i][0], &n_rhs[i]);
	simp_divide(&rhs[i][0], &n_rhs[i], 0L);
	if (n_rhs[i] == 1) {
		printf("Integration successful.  Result = ");
		list_proc(&rhs[i][0], n_rhs[i]);
		printf("\n");
		return true;
	}
	blt(&lhs[i][0], &lhs[cur_equation][0], n_lhs[cur_equation] * sizeof(token_type));
	n_lhs[i] = n_lhs[cur_equation];
	l = lhs[i][0].token.variable;
	l -= PRIME_INCREMENT;
	if (l > 0) {
		lhs[i][0].token.variable = l;
	}
	printf("Integration successful.\n");
	cur_equation = i;
	list_sub(cur_equation);
	return true;
}

int
polydivide(cp)
char	*cp;
{
	long	v, v_tmp;
	int	i, j, k;
	int	nl, nr;
	int	div_flag;
	double	d1, d2;
	double	d3, d4;

	if (*cp == '\0') {
		v = 0;
	} else {
		cp = parse_var2(&v, cp);
		if (cp == NULL) {
			return false;
		}
		if (extra_garbage(cp))
			return false;
	}
	i = next_espace();
oops:
	input_column = printf("Enter dividend: ");
	if (!get_expr(&rhs[i][0], &nr)) {
		goto oops;
	}
	if (nr == 0) {
		printf("Command aborted.\n");
		return false;
	}
oops2:
	input_column = printf("Enter divisor: ");
	if (!get_expr(&lhs[i][0], &nl)) {
		goto oops2;
	}
	if (nl == 0) {
		printf("Command aborted.\n");
		return false;
	}
	simp_loop(&lhs[i][0], &nl);
	simp_loop(&rhs[i][0], &nr);
	uf_simp(&lhs[i][0], &nl);
	uf_simp(&rhs[i][0], &nr);
	if (nl == 1 && lhs[i][0].kind == CONSTANT
	    && nr == 1 && rhs[i][0].kind == CONSTANT) {
		d1 = rhs[i][0].token.constant;
		d2 = lhs[i][0].token.constant;
		d4 = modf(d1 / d2, &d3);
		printf("Result of numerical division: %.14lg\n", d1 / d2);
		printf("Quotient: %.14lg, Remainder: %.14lg\n", d3, d4 * d2);
		return ngcd(d1, d2);
	}
	div_flag = 2;
	partial_flag = false;
	v_tmp = v;
	if (poly_div(&rhs[i][0], nr, &lhs[i][0], nl, &v_tmp, &div_flag)) {
		simp_divide(&tlhs[0], &n_tlhs, 0L);
		simp_divide(&trhs[0], &n_trhs, 0L);
		printf("Polynomial division successful using variable (");
		list_var(v_tmp, true);
		printf(").\nThe quotient is:\n");
		list_proc(&tlhs[0], n_tlhs);
		printf("\n\nThe remainder is:\n");
		list_proc(&trhs[0], n_trhs);
		printf("\n");
	} else {
		printf("Polynomial division failed.\n");
	}
	printf("\n");
	j = poly_gcd(&rhs[i][0], nr, &lhs[i][0], nl, v, false, false);
	if (!j) {
		j = poly_gcd(&lhs[i][0], nl, &rhs[i][0], nr, v, false, false);
	}
	if (j) {
		simp_divide(&trhs[0], &n_trhs, 0L);
		printf("Found polynomial Greatest Common Divisor (iterations = %d):\n", j);
		list_proc(&trhs[0], n_trhs);
		printf("\n");
	} else {
		printf("No polynomial GCD found.\n");
	}
	partial_flag = true;
	return true;
}

int
taylor(cp)
char	*cp;
{
	long	v;
	int	i, j, k;
	int	i1, j1;
	int	level;
	int	found;
	int	er;
	int	n;
	int	order;
	token_type	*ep;
	double	d;
	char	*cp1;

	if (n_lhs[cur_equation] <= 0) {
		printf("Undefined equation.\n");
		return false;
	}
	if (*cp == '\0') {
		if (!prompt_var(&v))
			return false;
	} else {
		cp = parse_var2(&v, cp);
		if (cp == NULL) {
			return false;
		}
		if (extra_garbage(cp))
			return false;
	}
	i = next_espace();
	found = false;
	er = !solved_equation(cur_equation);
	if (!er) {
		er = ((lhs[cur_equation][0].token.variable & VAR_MASK) <= SIGN);
	}
	for (j = 0; j < n_rhs[cur_equation]; j++) {
		if (rhs[cur_equation][j].kind == VARIABLE) {
			if (rhs[cur_equation][j].token.variable == v)
				found = true;
		}
	}
	if (!found) {
		printf("Variable not found in RHS.\n");
		return false;
	}
	if (er) {
		printf("Warning: This equation is not solved for a normal variable.\n");
	}
	printf("Taylor approximation of current equation about ");
	list_var(v, true);
	printf(" = point.\n");
	if (!differentiate(&rhs[cur_equation][0], n_rhs[cur_equation], &trhs[0], &n_trhs, v, false)) {
		printf("Differentiation failed.\n");
		return false;
	}
oops:
	input_column = printf("Enter point: ");
	if (!get_expr(&tlhs[0], &n_tlhs)) {
		goto oops;
	}
	if (n_tlhs == 0) {
		printf("Command aborted.\n");
		return false;
	}
oops2:
	printf("Enter order (number of derivatives to take): ");
	if ((cp1 = fgets((char *) &scratch[0], N_TOKENS * sizeof(token_type), stdin)) == NULL)
		return false;
	if (*cp1 == '\0' || *cp1 == '\n') {
		printf("Derivatives will be taken until they reach zero.\n");
		order = 32000;
	} else {
		if (!isascii(*cp1) || !isdigit(*cp1))
			goto oops2;
		order = atoi(cp1);
		if (order < 0) {
			goto oops2;
		}
	}
	n = 0;
	i1 = 0;
	blt(&rhs[i][0], &rhs[cur_equation][0], n_rhs[cur_equation] * sizeof(token_type));
	n_rhs[i] = n_rhs[cur_equation];
loop_again:
	for (k = i1; k < n_rhs[i]; k += 2) {
		if (rhs[i][k].kind == VARIABLE && rhs[i][k].token.variable == v) {
			level = rhs[i][k].level;
			if ((n_rhs[i] + n_tlhs - 1) > N_TOKENS)
				error_huge();
			blt(&rhs[i][k+n_tlhs], &rhs[i][k+1], (n_rhs[i] - (k + 1)) * sizeof(token_type));
			n_rhs[i] += n_tlhs - 1;
			j1 = k;
			blt(&rhs[i][k], &tlhs[0], n_tlhs * sizeof(token_type));
			k += n_tlhs;
			for (; j1 < k; j1++)
				rhs[i][j1].level += level;
			k--;
		}
	}
	if ((n_rhs[i] + n_tlhs + 9) > N_TOKENS)
		error_huge();
	for (k = i1; k < n_rhs[i]; k++)
		rhs[i][k].level++;
	k = n_rhs[i];
	ep = &rhs[i][k];
	ep->level = 1;
	ep->kind = OPERATOR;
	ep->token.operatr = TIMES;
	ep++;
	ep->level = 3;
	ep->kind = VARIABLE;
	ep->token.variable = v;
	ep++;
	ep->level = 3;
	ep->kind = OPERATOR;
	ep->token.operatr = MINUS;
	k += 3;
	j1 = k;
	blt(&rhs[i][k], &tlhs[0], n_tlhs * sizeof(token_type));
	k += n_tlhs;
	for (; j1 < k; j1++)
		rhs[i][j1].level += 3;
	ep = &rhs[i][k];
	ep->level = 2;
	ep->kind = OPERATOR;
	ep->token.operatr = POWER;
	ep++;
	ep->level = 2;
	ep->kind = CONSTANT;
	ep->token.constant = n;
	ep++;
	ep->level = 1;
	ep->kind = OPERATOR;
	ep->token.operatr = DIVIDE;
	ep++;
	for (d = 1.0, j = 2; j <= n; j++)
		d *= j;
	ep->level = 1;
	ep->kind = CONSTANT;
	ep->token.constant = d;
	k += 4;
	n_rhs[i] = k;
	for (; i1 < k; i1++)
		rhs[i][i1].level++;
	if (n < order) {
		if (n > 0) {
			if (!differentiate(&trhs[0], n_trhs, &trhs[0], &n_trhs, v, false)) {
				printf("Differentiation failed.\n");
				return false;
			}
		}
		simpd_side(&trhs[0], &n_trhs);
		if (n_trhs != 1 || trhs[0].kind != CONSTANT
		    || trhs[0].token.constant != 0.0) {
			if ((k + 1 + n_trhs) > N_TOKENS)
				error_huge();
			rhs[i][k].level = 1;
			rhs[i][k].kind = OPERATOR;
			rhs[i][k].token.operatr = PLUS;
			k++;
			i1 = k;
			blt(&rhs[i][k], &trhs[0], n_trhs * sizeof(token_type));
			n_rhs[i] = k + n_trhs;
			n++;
			goto loop_again;
		}
	}
	elim_loop(&rhs[i][0], &n_rhs[i]);
	blt(&lhs[i][0], &lhs[cur_equation][0], n_lhs[cur_equation] * sizeof(token_type));
	n_lhs[i] = n_lhs[cur_equation];
	cur_equation = i;
	list_sub(cur_equation);
	return true;
}

int
eliminate(cp)
char	*cp;
{
	long	v;
	int	i, j, k;
	int	n;
	int	using_flag;
	char	used[N_EQUATIONS];

	using_flag = false;
	for (i = 0; i < ARR_CNT(used); i++)
		used[i] = false;
	if (n_lhs[cur_equation] <= 0) {
		printf("Undefined equation.\n");
		return false;
	}
next_var:
	if (*cp == '\0') {
		if (!prompt_var(&v))
			return false;
	} else {
		cp = parse_var2(&v, cp);
		if (cp == NULL) {
			return false;
		}
	}
	if (!found_variable(cur_equation, v)) {
		printf("Variable (");
		list_var(v, true);
		printf(") not found in current equation.\n");
		usage_flag = true;
		return false;
	}
	using_flag = (strncmp(cp, "using", 5) == 0);
	if (using_flag) {
		cp = skip_space(cp + 5);
		i = atoi(cp) - 1;
		cp = skip_num(cp);
		if (i < 0 || i >= n_equations) {
			printf("Invalid equation number.\n");
			usage_flag = true;
			return false;
		}
		if (extra_garbage(cp))
			return false;
		if (i == cur_equation) {
			printf("Ridiculous command.\n");
			return false;
		}
	} else {
		i = cur_equation;
		for (n = 1;; n++) {
			if (n >= n_equations) {
				printf("Replacement equation not found.\n");
				return false;
			}
			if (i <= 0)
				i = n_equations - 1;
			else
				i--;
			if (used[i])
				continue;
			if (found_variable(i, v))
				break;
		}
	}
	if (!elim_sub(i, v, using_flag))
		return false;
	simp_sub(cur_equation);
	used[i] = true;
	if (*cp != '\0') {
		goto next_var;
	}
	list_sub(cur_equation);
	return true;
}

/*
 * Return true if equation number "i" is solved for a variable.
 */
int
solved_equation(i)
int	i;
{
	int	k;

	if (n_lhs[i] != 1 || lhs[i][0].kind != VARIABLE)
		return false;
	for (k = 0; k < n_rhs[i]; k += 2) {
		if (rhs[i][k].kind == VARIABLE
		    && rhs[i][k].token.variable == lhs[i][0].token.variable) {
			return false;
		}
	}
	return true;
}

/*
 * Return true if variable "v" exists in equation number "i".
 */
int
found_variable(i, v)
int	i;
long	v;
{
	int	j;

	if (n_lhs[i] <= 0)
		return false;
	for (j = 0; j < n_lhs[i]; j += 2) {
		if (lhs[i][j].kind == VARIABLE && lhs[i][j].token.variable == v) {
			return true;
		}
	}
	for (j = 0; j < n_rhs[i]; j += 2) {
		if (rhs[i][j].kind == VARIABLE && rhs[i][j].token.variable == v) {
			return true;
		}
	}
	return false;
}

int
elim_sub(i, v, using_flag)
int	i, using_flag;
long	v;
{
	int	j, k;
	int	len;
	int	level;
	token_type	want;

	if (n_lhs[i] <= 0) {
		printf("Equation #%d is undefined.\n", i + 1);
		return false;
	}
	if (!using_flag) {
		printf("Solving equation #%d for (", i + 1);
		list_var(v, true);
		printf(")...\n");
	}
	want.level = 1;
	want.kind = VARIABLE;
	want.token.variable = v;
	if (!solve_sub(&want, 1, &lhs[i][0], &n_lhs[i], &rhs[i][0], &n_rhs[i])) {
		printf("Can't solve equation #%d.\n", i + 1);
		return false;
	}
	len = n_rhs[i];
	for (j = n_rhs[cur_equation] - 1; j >= 0; j--) {
		if (rhs[cur_equation][j].kind == VARIABLE && rhs[cur_equation][j].token.variable == v) {
			level = rhs[cur_equation][j].level;
			if (n_rhs[cur_equation] + len - 1 > N_TOKENS) {
				error_huge();
			}
			blt(&rhs[cur_equation][j+len], &rhs[cur_equation][j+1], (n_rhs[cur_equation] - (j + 1)) * sizeof(token_type));
			n_rhs[cur_equation] += len - 1;
			blt(&rhs[cur_equation][j], &rhs[i][0], len * sizeof(token_type));
			for (k = j; k < j + len; k++)
				rhs[cur_equation][k].level += level;
		}
	}
	for (j = n_lhs[cur_equation] - 1; j >= 0; j--) {
		if (lhs[cur_equation][j].kind == VARIABLE && lhs[cur_equation][j].token.variable == v) {
			level = lhs[cur_equation][j].level;
			if (n_lhs[cur_equation] + len - 1 > N_TOKENS) {
				error_huge();
			}
			blt(&lhs[cur_equation][j+len], &lhs[cur_equation][j+1], (n_lhs[cur_equation] - (j + 1)) * sizeof(token_type));
			n_lhs[cur_equation] += len - 1;
			blt(&lhs[cur_equation][j], &rhs[i][0], len * sizeof(token_type));
			for (k = j; k < j + len; k++)
				lhs[cur_equation][k].level += level;
		}
	}
	return true;
}

/*
 * Display equations in fraction format.
 * Return true if any equations were displayed.
 */
int
group(cp)
char	*cp;
{
	int	i, j;
	jmp_buf	save_save;
	int	displayed;
	int	factor_flag;

	displayed = false;
	factor_flag = false;
	if (strncmp(cp, "factor", 4) == 0) {
		factor_flag = true;
		cp = skip_param(cp);
	}
	if (!get_range(&cp, &i, &j)) {
		return false;
	}
	if (extra_garbage(cp))
		return false;
	blt(save_save, jmp_save, sizeof(jmp_save));
	if (setjmp(jmp_save) != 0) {
		domain_check = false;
		i++;
		printf("Skipping equation #%d.\n", i);
	}
	for (; i <= j; i++) {
		if (n_lhs[i] > 0) {
			group_sub(i);
			if (factor_flag) {
				fi_sub(i);
			}
			flist_sub(i);
			displayed = true;
		}
	}
	blt(jmp_save, save_save, sizeof(jmp_save));
	return displayed;
}

int
list(cp)
char	*cp;
{
	int	i, j;

	if (!get_range(&cp, &i, &j)) {
		return false;
	}
	if (extra_garbage(cp))
		return false;
	for (; i <= j; i++) {
		list_sub(i);
	}
	return true;
}

int
replace(cp)
char	*cp;
{
	int	i, j, k;
	int	n;
	int	level;
	long	last_v;
	long	v, v1;
	char	*cp_start;
	char	temp_flag;

	cp_start = cp;
	i = cur_equation;
	if (n_lhs[i] <= 0) {
		printf("Undefined equation.\n");
		return false;
	}
	if (strncmp(cp, "temp", 4) == 0) {
		temp_flag = true;
		cp = skip_param(cp);
	} else {
		temp_flag = false;
	}
	v1 = 0L;
	if (*cp) {
		cp = parse_var2(&v1, cp);
		if (cp == NULL) {
			return false;
		}
		if (!found_variable(i, v1)) {
			printf("Variable (");
			list_var(v1, true);
			printf(") not found in current equation.\n");
			return false;
		}
	}
	n_tlhs = n_lhs[i];
	blt(&tlhs[0], &lhs[i][0], n_tlhs * sizeof(token_type));
	n_trhs = n_rhs[i];
	blt(&trhs[0], &rhs[i][0], n_trhs * sizeof(token_type));
	last_v = -1;
	for (;;) {
		v = -1;
		for (j = 0; j < n_lhs[i]; j++) {
			if (lhs[i][j].kind == VARIABLE) {
				if (lhs[i][j].token.variable > last_v
				    && (v == -1 || lhs[i][j].token.variable < v))
					v = lhs[i][j].token.variable;
			}
		}
		for (j = 0; j < n_rhs[i]; j++) {
			if (rhs[i][j].kind == VARIABLE) {
				if (rhs[i][j].token.variable > last_v
				    && (v == -1 || rhs[i][j].token.variable < v))
					v = rhs[i][j].token.variable;
			}
		}
		if (v == -1) {
			break;
		}
		last_v = v;
		if (v1) {
			if (v != v1)
				continue;
			if (*cp) {
				if (strncmp(cp, "with", 4) != 0) {
					usage_flag = true;
					return false;
				}
				cp = skip_space(cp + 4);
				input_column += (cp - cp_start);
				if ((cp = parse_section(&scratch[0], &n, cp)) == NULL
				    || n <= 0) {
					usage_flag = true;
					return false;
				}
				if (extra_garbage(cp))
					return false;
				goto do_this;
			}
		}
oops:
		input_column = printf("Enter ");
		input_column += list_var(v, true);
		input_column += printf(": ");
		if (!get_expr(&scratch[0], &n)) {
			goto oops;
		}
		if (n == 0)
			continue;
do_this:
		for (j = 0; j < n; j++) {
			if (scratch[j].kind == VARIABLE) {
				scratch[j].token.variable = -scratch[j].token.variable;
			}
		}
		for (j = n_tlhs - 1; j >= 0; j--) {
			if (tlhs[j].kind == VARIABLE && tlhs[j].token.variable == v) {
				level = tlhs[j].level;
				if (n_tlhs + n - 1 > N_TOKENS) {
					error_huge();
				}
				blt(&tlhs[j+n], &tlhs[j+1], (n_tlhs - (j + 1)) * sizeof(token_type));
				n_tlhs += n - 1;
				blt(&tlhs[j], &scratch[0], n * sizeof(token_type));
				for (k = j; k < j + n; k++)
					tlhs[k].level += level;
			}
		}
		for (j = n_trhs - 1; j >= 0; j--) {
			if (trhs[j].kind == VARIABLE && trhs[j].token.variable == v) {
				level = trhs[j].level;
				if (n_trhs + n - 1 > N_TOKENS) {
					error_huge();
				}
				blt(&trhs[j+n], &trhs[j+1], (n_trhs - (j + 1)) * sizeof(token_type));
				n_trhs += n - 1;
				blt(&trhs[j], &scratch[0], n * sizeof(token_type));
				for (k = j; k < j + n; k++)
					trhs[k].level += level;
			}
		}
	}
	for (j = 0; j < n_tlhs; j++)
		if (tlhs[j].kind == VARIABLE && tlhs[j].token.variable < 0)
			tlhs[j].token.variable = -tlhs[j].token.variable;
	for (j = 0; j < n_trhs; j++)
		if (trhs[j].kind == VARIABLE && trhs[j].token.variable < 0)
			trhs[j].token.variable = -trhs[j].token.variable;
	simp_side(&tlhs[0], &n_tlhs);
	simp_side(&trhs[0], &n_trhs);
	if (temp_flag) {
		printf(" ");
		list_proc(&tlhs[0], n_tlhs);
		printf(" = ");
		list_proc(&trhs[0], n_trhs);
		printf("\n");
	} else {
		n_lhs[i] = n_tlhs;
		blt(&lhs[i][0], &tlhs[0], n_tlhs * sizeof(token_type));
		n_rhs[i] = n_trhs;
		blt(&rhs[i][0], &trhs[0], n_trhs * sizeof(token_type));
		list_sub(i);
	}
	return true;
}

int
sensitivity(cp)
char	*cp;
{
	long	va[20];
	char	fa[ARR_CNT(va)];
	int	vc;
	int	i, j, k;
	int	n;
	int	er;
	int	level;
	token_type	*ep;
	token_type	want;

	if (n_lhs[cur_equation] <= 0) {
		printf("Undefined equation.\n");
		return false;
	}
	if (*cp == '\0') {
		if (!prompt_var(&va[0]))
			return false;
		vc = 1;
	} else {
		for (vc = 0; (cp != NULL) && *cp; vc++) {
			if (vc >= ARR_CNT(va)) {
				printf("Too many variables specified!\n");
				return false;
			}
			cp = parse_var2(&va[vc], cp);
			if (cp == NULL) {
				return false;
			}
		}
	}
	for (i = 0; i < vc; i++) {
		if ((va[i] & VAR_MASK) <= SIGN || (va[i] & PERCENT_CHANGE)) {
			printf("One of the variables specified is not a normal variable!\n");
			return false;
		}
		fa[i] = false;
	}
	i = next_espace();
	er = !solved_equation(cur_equation);
	if (!er) {
		er = ((lhs[cur_equation][0].token.variable & VAR_MASK) <= SIGN
		    || (lhs[cur_equation][0].token.variable & PERCENT_CHANGE));
	}
	if (er) {
		printf("Please solve this equation for a normal variable and try again.\n");
		return false;
	}
	n = n_rhs[cur_equation];
	blt(&rhs[i][0], &rhs[cur_equation][0], n * sizeof(token_type));
	n_rhs[i] = n;
	blt(&lhs[i][0], &rhs[cur_equation][0], n * sizeof(token_type));
	for (j = n_rhs[i] - 1; j >= 0; j--) {
		if (rhs[i][j].kind == VARIABLE) {
			for (k = 0; k < vc; k++) {
				if (rhs[i][j].token.variable == va[k])
					break;
			}
			if (k >= vc)
				continue;
			fa[k] = true;
			if (n_rhs[i] + 6 > N_TOKENS) {
				error_huge();
			}
			blt(&rhs[i][j+7], &rhs[i][j+1], (n_rhs[i] - (j + 1)) * sizeof(token_type));
			n_rhs[i] += 6;
			level = ++rhs[i][j].level;
			ep = &rhs[i][j+1];
			ep->level = level;
			ep->kind = OPERATOR;
			ep->token.operatr = TIMES;
			ep++;
			ep->level = level + 1;
			ep->kind = CONSTANT;
			ep->token.constant = 1.0;
			ep++;
			ep->level = level + 1;
			ep->kind = OPERATOR;
			ep->token.operatr = PLUS;
			ep++;
			ep->level = level + 2;
			ep->kind = VARIABLE;
			ep->token.variable = va[k] | PERCENT_CHANGE;
			ep++;
			ep->level = level + 2;
			ep->kind = OPERATOR;
			ep->token.operatr = DIVIDE;
			ep++;
			ep->level = level + 2;
			ep->kind = CONSTANT;
			ep->token.constant = 100.0;
		}
	}
	er = false;
	for (j = 0; j < vc; j++) {
		if (!fa[j]) {
			printf("Variable (");
			list_var(va[j], true);
			printf(") not found in RHS.\n");
			er = true;
		}
	}
	if (er) {
		return false;
	}
	for (j = 0; j < n; j++)
		lhs[i][j].level++;
	if (n + 6 > N_TOKENS) {
		error_huge();
	}
	ep = &lhs[i][n];
	ep->level = 1;
	ep->kind = OPERATOR;
	ep->token.operatr = TIMES;
	ep++;
	ep->level = 2;
	ep->kind = CONSTANT;
	ep->token.constant = 1.0;
	ep++;
	ep->level = 2;
	ep->kind = OPERATOR;
	ep->token.operatr = PLUS;
	ep++;
	ep->level = 3;
	ep->kind = VARIABLE;
	want.level = 1;
	want.kind = VARIABLE;
	want.token.variable = lhs[cur_equation][0].token.variable | PERCENT_CHANGE;
	ep->token.variable = want.token.variable;
	ep++;
	ep->level = 3;
	ep->kind = OPERATOR;
	ep->token.operatr = DIVIDE;
	ep++;
	ep->level = 3;
	ep->kind = CONSTANT;
	ep->token.constant = 100.0;
	n += 6;
	if (!solve_sub(&want, 1, &lhs[i][0], &n, &rhs[i][0], &n_rhs[i])) {
		printf("Solve failed!\n");
		return false;
	}
	n_lhs[i] = n;
	cur_equation = i;
	list_sub(cur_equation);
	return true;
}

int
simplify(cp)
char	*cp;
{
	int	i, j;
	int	poly_flag;
	int	quick_flag;
	int	symb;

	poly_flag = true;
	quick_flag = false;
	symb = false;
check_again:
	if (strncmp(cp, "smart", 4) == 0) {
		poly_flag = false;
		cp = skip_param(cp);
		goto check_again;
	}
	if (strncmp(cp, "symbolic", 4) == 0) {
		symb = true;
		cp = skip_param(cp);
		goto check_again;
	}
	if (strncmp(cp, "quick", 4) == 0) {
		quick_flag = true;
		cp = skip_param(cp);
		goto check_again;
	}
	if (!get_range(&cp, &i, &j)) {
		return false;
	}
	if (extra_garbage(cp))
		return false;
	if (symb)
		symb_flag = true;
	for (; i <= j; i++) {
		if (n_lhs[i] > 0) {
			simpa_sub(i, poly_flag, quick_flag);
			list_sub(i);
		}
	}
	symb_flag = false;
	return true;
}

int
factor(cp)
char	*cp;
{
	int	i, j;
	int	i1;
	long	v;

	v = 0;
	if (!get_range(&cp, &i, &j)) {
		return false;
	}
	do {
		if (*cp) {
			if ((cp = parse_var2(&v, cp)) == NULL) {
				return false;
			}
		}
		for (i1 = i; i1 <= j; i1++) {
			if (n_lhs[i1] > 0) {
				simpv_side(&lhs[i1][0], &n_lhs[i1], v);
				simpv_side(&rhs[i1][0], &n_rhs[i1], v);
			}
		}
	} while (*cp);
	for (i1 = i; i1 <= j; i1++) {
		list_sub(i1);
	}
	return true;
}

int
unfactor(cp)
char	*cp;
{
	int	i, j;

	if (!get_range(&cp, &i, &j)) {
		return false;
	}
	if (extra_garbage(cp))
		return false;
	for (; i <= j; i++) {
		if (n_lhs[i] > 0) {
			unfa_sub(i);
			list_sub(i);
		}
	}
	return true;
}

int
quit(cp)
char	*cp;
{
	reset_attr();
	unlink(tmp_file);
	printf("Thank you for using Mathomatic!\n");
	printf("Please visit \"http://www.mathomatic.com\".\n");
	exit(0);
}

int
read_in(cp)
char	*cp;
{
	int	i;
	int	rv;
	FILE	*fp;
	jmp_buf	save_save;

	if (*cp == '\0') {
		printf("This command reads in the specified file as if you typed it in.\n");
		usage_flag = true;
		return false;
	}
	fp = fopen(cp, "r");
	if (fp == NULL) {
		printf("Can't open file '%s'.\n", cp);
		return false;
	}
	blt(save_save, jmp_save, sizeof(jmp_save));
	if ((rv = setjmp(jmp_save)) != 0) {
		clean_up();
		printf("Read operation aborted.\n");
		goto end_read;
	}
	while (cp = fgets((char *) &tlhs[0], N_TOKENS * sizeof(token_type), fp)) {
		i = strlen(cp);
		if (i == 0 || cp[i-1] != '\n') {
			printf("Incomplete line encountered.\n");
			longjmp(jmp_save, 3);
		}
		input_column = printf("%d%s", cur_equation + 1, PROMPT);
		printf("%s", cp);
		if (!process(cp)) {
			longjmp(jmp_save, 3);
		}
	}
end_read:
	blt(jmp_save, save_save, sizeof(jmp_save));
	if (ferror(fp)) {
		printf("I/O error reading file.\n");
		rv = 1;
	}
	fclose(fp);
	usage_flag = false;
	return(!rv);
}

int
edit(cp)
char	*cp;
{
	int	i, j;
	FILE	*fp;
	int	rv;

	if (*cp == '\0') {
		fp = fopen(tmp_file, "w");
		if (fp == NULL) {
			printf("Can't create temporary file '%s'.\n", tmp_file);
			return false;
		}
		fprintf(fp, "; After you are done editing,\n");
		fprintf(fp, "; save and exit to load into Mathomatic.\n");
		gfp = fp;
		high_prec = true;
		list("all");
		high_prec = false;
		gfp = stdout;
		if (fflush(fp) || ferror(fp)) {
			rv = false;
		} else {
			rv = true;
		}
		if (fclose(fp))
			rv = false;
		if (!rv) {
			printf("Error writing temporary file.\n");
			return false;
		}
		rv = edit_sub(tmp_file);
		unlink(tmp_file);
		return rv;
	} else {
		if (access(cp, 6)) {
			printf("You can only edit existing/writable files or the current equations.\n");
			usage_flag = true;
			return false;
		}
		return edit_sub(cp);
	}
}

int
reedit(cp)
char	*cp;
{
	if (access(cp, 06))
		return false;
	printf("Prepare to run the editor.\n");
	pause_cmd("");
	return edit_sub(cp);
}

#if	UNIX
int
edit_sub(cp)
char	*cp;
{
	int	i, j;
	char	cl[500];
	char	*cp1;

edit_again:
	cp1 = getenv("EDITOR");
	if (cp1 == NULL) {
		printf("EDITOR environment variable not set.\n");
		return false;
	}
	strcpy(cl, cp1);
	strcat(cl, " ");
	strcat(cl, cp);
	if (system(cl)) {
		printf("Error executing editor.  Check EDITOR environment variable.\n");
		return false;
	}
	default_color();
	clear("all");
	if (!read_in(cp)) {
		printf("Prepare to run the editor.\n");
		pause_cmd("");
		goto edit_again;
	}
	unlink(tmp_file);
	return true;
}

#else

int
edit_sub(cp)
char	*cp;
{
#define	EDIT_BAT	"amedit.bat"

	int	i, j;
	char	*comspec;

	comspec = getenv("COMSPEC");
	if (comspec == NULL) {
		printf("Environment variable COMSPEC not defined!\n");
		return false;
	}
	for (j = n_equations - 1; j >= 0; j--) {
#if	HALLOC
		hfree((char *) rhs[j]);
		hfree((char *) lhs[j]);
#else
		free((char *) rhs[j]);
		free((char *) lhs[j]);
#endif
	}
	printf("Running \"%s\"...\n", EDIT_BAT);
	if (coption) {
		execlp(comspec, comspec, "/c", EDIT_BAT, cp, prog_name, "-c", NULL);
	} else {
		execlp(comspec, comspec, "/c", EDIT_BAT, cp, prog_name, NULL);
	}
	printf("Cannot execute command interpreter.  errno = %d\n", errno);
	exit(1);
}
#endif

#if	!UNIX
int
graph(cp)
char	*cp;
{
#define	GRAPH_PROG	"graph.exe"

	int	i, j;
	FILE	*fp;
	int	rv;
	char	str[100];
	char	*options;

	if ((i = get_default_en(cp)) < 0)
		return false;
	simp_i(&rhs[i][0], &n_rhs[i]);
	group_sub(i);
	fp = fopen(tmp_file, "w");
	if (fp == NULL) {
		printf("Can't create temporary file '%s'.\n", tmp_file);
		return false;
	}
	gfp = fp;
	high_prec = true;
	rv = list("all");
	for (j = cur_equation + 1; j < n_equations; j++) {
		if (n_lhs[j] > 0) {
			fprintf(gfp, "#%d\n", cur_equation + 1);
			break;
		}
	}
	high_prec = false;
	gfp = stdout;
	if (fflush(fp) || ferror(fp)) {
		rv = false;
	}
	if (fclose(fp))
		rv = false;
	if (!rv) {
		printf("Error writing temporary file.\n");
		return false;
	}
#if	TUTOR
	sprintf(str, "%du", i + 1);
#else
	sprintf(str, "%d", i + 1);
#endif
	for (j = n_equations - 1; j >= 0; j--) {
#if	HALLOC
		hfree((char *) rhs[j]);
		hfree((char *) lhs[j]);
#else
		free((char *) rhs[j]);
		free((char *) lhs[j]);
#endif
	}
	options = "";
	if (coption) {
		options = "-c";
	}
	execlp(GRAPH_PROG, GRAPH_PROG, str, prog_name, options, NULL);
	printf("ERROR: Cannot execute '%s', errno = %d.\n\n", GRAPH_PROG, errno);
	execlp(prog_name, prog_name, options, tmp_file, NULL);
	exit(1);
}
#endif

int
save(cp)
char	*cp;
{
	FILE	*fp;
	char	*cp1;
	int	rv;

	if (*cp == '\0') {
		printf("This command saves all equations in a file when followed by a filename.\n");
		usage_flag = true;
		return false;
	}
	if (access(cp, 0) == 0) {
		do {
			printf("'%s' exists.  Overwrite (Y/N)? ", cp);
			if ((cp1 = fgets((char *) &trhs[0], N_TOKENS * sizeof(token_type), stdin)) == NULL) {
				return false;
			}
			str_tolower(cp1);
			if (*cp1 == 'n') {
				printf("Command aborted.\n");
				return true;
			}
		} while (*cp1 != 'y');
	}
	fp = fopen(cp, "w");
	if (fp == NULL) {
		printf("Can't create file '%s'.\n", cp);
		return false;
	}
	gfp = fp;
	high_prec = true;
	rv = list("all");
	high_prec = false;
	gfp = stdout;
	if (fflush(fp) || ferror(fp)) {
		rv = false;
	}
	if (fclose(fp))
		rv = false;
	if (rv) {
		printf("All equations saved in file: '%s'.\n", cp);
	} else {
		printf("Error encountered while saving equations.\n");
	}
	return rv;
}

int
print(cp)
char	*cp;
{
#define	PRINT_DEVICE	"prn"

	int	i;
	char	*filename;
	FILE	*fp;
	int	rv;
	int	append_flag;

	append_flag = false;
	filename = PRINT_DEVICE;
	for (i = strlen(cp) - 1; i >= 0; i--) {
		if (cp[i] == '>') {
			filename = skip_space(&cp[i+1]);
			if (i && cp[i-1] == '>') {
				i--;
				append_flag = true;
			}
			cp[i] = '\0';
			break;
		}
	}
	if (*cp == '\0' || filename[0] == '\0') {
		usage_flag = true;
		return false;
	}
	if (append_flag) {
		fp = fopen(filename, "a");
	} else {
		fp = fopen(filename, "w");
	}
	if (fp == NULL) {
		printf("Can't open '%s' for writing.\n", filename);
		return false;
	}
	gfp = fp;
	rv = group(cp);
#if	UNIX
	printf("Text to print is in file: '%s'.\n", filename);
#else
	if (rv && strcmp(filename, PRINT_DEVICE) == 0) {
		fprintf(fp, "\014");
	}
#endif
	gfp = stdout;
	if (fflush(fp) || ferror(fp)) {
		printf("Printer error.\n");
		rv = false;
	}
	if (fclose(fp)) {
		printf("Printer error.\n");
		rv = false;
	}
	return rv;
}

int
remark(cp)
char	*cp;
{
	return true;
}

/*
 * Get default equation number from a command parameter string.
 * The equation number must be the only parameter.
 * Return -1 on error.
 */
int
get_default_en(cp)
char	*cp;
{
	int	i;

	if (*cp == '\0')
		i = cur_equation;
	else {
		i = atoi(cp) - 1;
		cp = skip_num(cp);
		if (i < 0 || i >= n_equations) {
			printf("Invalid equation number.\n");
			usage_flag = true;
			return -1;
		}
		if (extra_garbage(cp))
			return -1;
	}
	if (n_lhs[i] <= 0) {
		printf("Undefined equation.\n");
		return -1;
	}
	return i;
}

/*
 * Get an expression from the user.
 * Return true if successful.
 */
int
get_expr(equation, np)
token_type	*equation;
int		*np;
{
	char	buf[2000];
	char	*cp1;

	if ((cp1 = fgets(&buf[0], sizeof(buf) - 1, stdin)) == NULL) {
		printf("EOF encountered!\n");
		longjmp(jmp_save, 2);
	}
	if (!case_sensitive_flag) {
		str_tolower(cp1);
	}
	set_error_level(cp1);
	cp1 = parse_section(equation, np, cp1);
	return(cp1 != NULL);
}

/*
 * Prompt for a variable from the user.
 * Return true if successful.
 */
int
prompt_var(vp)
long	*vp;
{
	char	buf[80];
	char	*cp1;

	printf("Enter variable: ");
	if ((cp1 = fgets(&buf[0], sizeof(buf) - 1, stdin)) == NULL) {
		printf("EOF encountered!\n");
		longjmp(jmp_save, 2);
	}
	if (!case_sensitive_flag) {
		str_tolower(cp1);
	}
	cp1 = parse_var2(vp, cp1);
	if (cp1 == NULL)
		return false;
	if (*cp1) {
		printf("Only one variable may be specified.\n");
		return false;
	}
	return true;
}
