/*
 * Algebraic manipulator solve and display routines.
 *
 * Copyright (c) 1996 George Gesslein II.
 */

#include "am.h"
#include "externs.h"

int		carray[] = {
	32, 33, 31, 35, 34, 36, 37
};

reset_attr()
{
	if (color_flag) {
		printf("\033[0m");
	}
}

set_color(color)
int	color;
{
	if (color_flag) {
		printf("\033[%dm", carray[color%ARR_CNT(carray)]);
	}
}

default_color()
{
	reset_attr();
	set_color(0);
}

/*
 * Initialize the global variables.
 */
init_gvars()
{
	domain_check = false;
	high_prec = false;
	partial_flag = true;
	symb_flag = false;
	sign_flag = false;
}

/*
 * Clean up when processing is unexpectedly
 * interrupted or terminated.
 */
clean_up()
{
	int	i;

	init_gvars();
	if (gfp != stdout)
		fclose(gfp);
	gfp = stdout;
	for (i = 0; i < n_equations; i++) {
		if (n_rhs[i] <= 0)
			n_lhs[i] = 0;
	}
}

set_sign_array()
{
	int	i, j;

	for (i = 0; i < ARR_CNT(sign_array); i++)
		sign_array[i] = false;
	for (i = 0; i < n_equations; i++) {
		if (n_lhs[i] > 0) {
			for (j = 0; j < n_lhs[i]; j += 2) {
				if (lhs[i][j].kind == VARIABLE
				    && (lhs[i][j].token.variable & VAR_MASK) == SIGN) {
					sign_array[(lhs[i][j].token.variable >> VAR_SHIFT) & SUBSCRIPT_MASK] = true;
				}
			}
			for (j = 0; j < n_rhs[i]; j += 2) {
				if (rhs[i][j].kind == VARIABLE
				    && (rhs[i][j].token.variable & VAR_MASK) == SIGN) {
					sign_array[(rhs[i][j].token.variable >> VAR_SHIFT) & SUBSCRIPT_MASK] = true;
				}
			}
		}
	}
}

/*
 * Return next unused sign variable in "*vp".
 * Mark it used.
 */
int
next_sign(vp)
long	*vp;
{
	int	i;

	for (i = 2;; i++) {
		if (i >= ARR_CNT(sign_array)) {
			*vp = SIGN;
			printf("Out of unique sign variables.\n");
			return false;
		}
		if (!sign_array[i]) {
			*vp = SIGN + (((long) i) << VAR_SHIFT);
			sign_array[i] = true;
			break;
		}
	}
	return true;
}

/*
 * Return true if all real numbers raise to the "power"
 * result in positive, real numbers.
 */
int
always_positive(power)
double		power;
{
	return(fmod(power, 2.0) == 0.0);
}

/*
 * Give the operators in an expression the correct priority.
 */
prior_sub(equation, n)
token_type	*equation;
int		n;
{
	int	i;

	for (i = 1; i < n; i += 2) {
		if (equation[i].token.operatr == FACTORIAL) {
			binary_parenthesize(equation, n, i);
		}
	}
	for (i = 1; i < n; i += 2) {
		if (equation[i].token.operatr == POWER) {
			binary_parenthesize(equation, n, i);
		}
	}
	for (i = 1; i < n; i += 2) {
		if (equation[i].token.operatr == TIMES
		    || equation[i].token.operatr == DIVIDE) {
			binary_parenthesize(equation, n, i);
		}
	}
}

int
solve(want, have)
int	want, have;
{
	int		rv;

	if (n_lhs[want] > 0) {
		rv = solve_sub(&lhs[want][0], n_lhs[want],
		    &lhs[have][0], &n_lhs[have],
		    &rhs[have][0], &n_rhs[have]);
	} else {
		rv = solve_sub(&rhs[want][0], n_rhs[want],
		    &rhs[have][0], &n_rhs[have],
		    &lhs[have][0], &n_lhs[have]);
	}
	n_lhs[want] = 0;
	if (!rv)
		printf("Solve failed.\n");
	return rv;
}

/*
 * Mathomatic solve routine.
 */
int
solve_sub(wantp, wantn, leftp, leftnp, rightp, rightnp)
token_type	*wantp;
int		wantn;
token_type	*leftp;
int		*leftnp;
token_type	*rightp;
int		*rightnp;
{
	int		i, j, k;
	int		found, found_count;
	int		worked;
	int		diff_sign;
	int		op, op_kind;
	token_type	*p1, *b1;
	token_type	*ep;
	token_type	want0;
	long		v;
	int		need_flip;
	int		uf_flag;
	int		qtries;
	int		zflag;
	int		zsolve;
	int		inc_count;
	int		zero_solved;
	double		d1, d2;

	qtries = 0;
	zero_solved = false;
	uf_flag = false;
	inc_count = 0;
	n_tlhs = *leftnp;
	blt(&tlhs[0], leftp, n_tlhs * sizeof(*leftp));
	n_trhs = *rightnp;
	blt(&trhs[0], rightp, n_trhs * sizeof(*rightp));
	if (wantn != 1) {
		printf("This program will only solve for a single variable or for zero.\n");
		return false;
	}
	if (n_tlhs <= 0 || n_trhs <= 0) {
		printf("Undefined equation.\n");
		return false;
	}
	if (wantp->kind == VARIABLE) {
		v = wantp->token.variable;
		found = false;
		ep = &trhs[n_trhs];
		for (p1 = &trhs[0]; p1 < ep; p1 += 2) {
			if (p1->kind == VARIABLE && p1->token.variable == v) {
				found = true;
				break;
			}
		}
		ep = &tlhs[n_tlhs];
		for (p1 = &tlhs[0]; p1 < ep; p1 += 2) {
			if (p1->kind == VARIABLE && p1->token.variable == v) {
				found = true;
				break;
			}
		}
		if (!found) {
			printf("Variable not found.\n");
			return false;
		}
		zsolve = false;
	} else {
		v = 0;
		if (wantp->kind != CONSTANT || wantp->token.constant != 0.0) {
			printf("This program will only solve for a single variable or for zero.\n");
			return false;
		}
		zsolve = true;
	}
	uf_power(&tlhs[0], &n_tlhs);
	uf_power(&trhs[0], &n_trhs);
simp_again:
	list_tdebug(2);
	simps_side(&tlhs[0], &n_tlhs, zsolve);
	if (uf_flag) {
		simp_loop(&trhs[0], &n_trhs);
		uf_simp(&trhs[0], &n_trhs);
		factorv(&trhs[0], &n_trhs, v);
	} else {
		simps_side(&trhs[0], &n_trhs, zsolve);
	}
	list_tdebug(1);
no_simp:
	op = 0;
	ep = &trhs[n_trhs];
	for (b1 = p1 = &trhs[0]; p1 < ep; p1++) {
		if (p1->kind == VARIABLE) {
			if (v == p1->token.variable || (zsolve && op == DIVIDE)) {
				if (op == 0) {
					for (p1++;; p1++) {
						if (p1 >= ep) {
							op = PLUS;
							break;
						}
						if (p1->level == 1 && p1->kind == OPERATOR) {
							switch (p1->token.operatr) {
							case TIMES:
							case DIVIDE:
								op = TIMES;
								break;
							default:
								op = PLUS;
								break;
							}
							break;
						}
					}
				}

				if (!zsolve && (op == TIMES || op == DIVIDE)) {
					b1 = &trhs[0];
					op = PLUS;
					for (p1 = b1; p1 < ep; p1++)
						p1->level++;
				}

				if (!g_of_f(op, b1, &trhs[0], &n_trhs, &tlhs[0], &n_tlhs))
					return false;
				goto simp_again;
			}
		} else if (p1->level == 1 && p1->kind == OPERATOR) {
			op = p1->token.operatr;
			b1 = p1 + 1;
		}
	}
	if (uf_flag) {
		simps_side(&trhs[0], &n_trhs, zsolve);
	}
left_again:
	worked = 1;
	uf_flag = false;
see_work:
	if (se_compare(wantp, wantn, &tlhs[0], n_tlhs, &diff_sign)) {
		if (!diff_sign) {
			if (!zsolve) {
				ep = &trhs[n_trhs];
				for (p1 = &trhs[0]; p1 < ep; p1 += 2) {
					if (p1->kind == VARIABLE && p1->token.variable == v)
						goto fool2;
				}
			} else {
zero_simp:
				list_tdebug(2);
				uf_power(&trhs[0], &n_trhs);
				do {
					simp_ssub(&trhs[0], &n_trhs, 0L, 0.0, false, 2, true);
				} while (uf_power(&trhs[0], &n_trhs));
				list_tdebug(1);
				ep = &trhs[n_trhs];
				op = 0;
				for (p1 = &trhs[1]; p1 < ep; p1 += 2) {
					if (p1->level == 1) {
						op = p1->token.operatr;
						if (op == DIVIDE) {
							goto no_simp;
						}
						if (op != TIMES) {
							break;
						}
					}
				}
				switch (op) {
				case TIMES:
					for (p1 = &trhs[0]; p1 < ep; p1 += 2) {
						if (p1->kind == CONSTANT
						    && p1->level == 1
						    && p1->token.constant != 1.0) {
							p1->token.constant = 1.0;
							goto zero_simp;
						}
					}
					for (b1 = &trhs[1]; b1 < ep; b1 += 2) {
						if (b1->level == 2
						    && b1->token.operatr == POWER
						    && (b1 + 1)->level == 2
						    && (b1 + 1)->kind == CONSTANT
						    && (b1 + 1)->token.constant > 0.0) {
							p1 = b1 + 2;
							blt(b1, p1, (char *) ep - (char *) p1);
							n_trhs -= 2;
							goto zero_simp;
						}
					}
					break;
				case POWER:
					if ((p1 + 1)->level == 1
					    && (p1 + 1)->kind == CONSTANT
					    && (p1 + 1)->token.constant > 0.0) {
						n_trhs -= 2;
						goto zero_simp;
					}
					break;
				}
			}
			blt(leftp, &tlhs[0], n_tlhs * sizeof(*leftp));
			*leftnp = n_tlhs;
			blt(rightp, &trhs[0], n_trhs * sizeof(*rightp));
			*rightnp = n_trhs;
			return true;
		}
	}
fool2:
	found_count = 0;
	need_flip = 0;
	found = 0;
	op = 0;
	ep = &tlhs[n_tlhs];
	for (b1 = p1 = &tlhs[0];; p1++) {
		if (p1 >= ep || (p1->level == 1 && p1->kind == OPERATOR)) {
			if (!found) {
				if ((p1 < ep || found_count || zsolve || n_tlhs > 1 || tlhs[0].kind != CONSTANT)
				    && (p1 - b1 != 1 || b1->kind != CONSTANT
				    || b1->token.constant != 1.0
				    || p1 >= ep || p1->token.operatr != DIVIDE)) {
					if (op == 0) {
						for (;; p1++) {
							if (p1 >= ep) {
								op = PLUS;
								break;
							}
							if (p1->level == 1 && p1->kind == OPERATOR) {
								switch (p1->token.operatr) {
								case POWER:
								case TIMES:
								case DIVIDE:
									op = TIMES;
									break;
								case PLUS:
								case MINUS:
									op = PLUS;
									break;
								}
								break;
							}
						}
					}
					if (zsolve) {
						if (p1 < ep) {
							switch (op) {
							case PLUS:
							case MINUS:
							case DIVIDE:
								break;
							default:
								goto fin1;
							}
						} else {
							if (op != DIVIDE) {
								b1 = &tlhs[0];
								op = PLUS;
								for (p1 = b1; p1 < ep; p1++)
									p1->level++;
							}
						}
					}
					if (!g_of_f(op, b1, &tlhs[0], &n_tlhs, &trhs[0], &n_trhs))
						return false;
					list_tdebug(2);
					if (uf_flag) {
						simp_loop(&tlhs[0], &n_tlhs);
					} else {
						simps_side(&tlhs[0], &n_tlhs, zsolve);
					}
					simps_side(&trhs[0], &n_trhs, zsolve);
					list_tdebug(1);
/*					worked = 1; */
					goto see_work;
				}
			} else if (op == DIVIDE) {
				need_flip += found;
			}
			if (p1 >= ep) {
				if (found_count == 0) {
					for (i = 0; i < n_trhs; i++) {
						if (trhs[i].kind == VARIABLE
						    && trhs[i].token.variable == v)
							return false;
					}
					subst_constants(&tlhs[0], &n_tlhs);
					subst_constants(&trhs[0], &n_trhs);
					simps_side(&tlhs[0], &n_tlhs, zsolve);
					simps_side(&trhs[0], &n_trhs, zsolve);
					uf_simp(&tlhs[0], &n_tlhs);
					uf_simp(&trhs[0], &n_trhs);
					if (se_compare(&tlhs[0], n_tlhs, &trhs[0], n_trhs, &diff_sign) && !diff_sign) {
						printf("This equation is an identity; that is,\n");
						printf("the left equation side is identical to the right equation side.\n");
						return false;
					}
					found = false;
					for (i = 0; i < n_tlhs; i += 2) {
						if (tlhs[i].kind == VARIABLE
						    && tlhs[i].token.variable > IMAGINARY) {
							found = true;
							break;
						}
					}
					for (i = 0; i < n_trhs; i += 2) {
						if (trhs[i].kind == VARIABLE
						    && trhs[i].token.variable > IMAGINARY) {
							found = true;
							break;
						}
					}
					if (found) {
						printf("This equation is independent of (");
						list_proc(&wantp[0], wantn);
						printf(").\n");
						return false;
					}
no_val:
					printf("There are no possible values for (");
					list_proc(&wantp[0], wantn);
					printf(").\n");
					return false;
				}
				zflag = (n_trhs == 1 && trhs[0].kind == CONSTANT
				    && trhs[0].token.constant == 0.0);
				if (need_flip >= found_count) {
					if (zflag)
						goto no_val;
					printf("Taking the reciprocal of both sides of the equation...\n");
					if (!flip(&tlhs[0], &n_tlhs, &trhs[0], &n_trhs))
						return false;
					list_tdebug(2);
					simps_side(&tlhs[0], &n_tlhs, zsolve);
					simps_side(&trhs[0], &n_trhs, zsolve);
					list_tdebug(1);
					goto left_again;
				}
				if (worked && !uf_flag) {
					worked--;
					printf("Unfactoring...\n");
					partial_flag = false;
					uf_simp(&tlhs[0], &n_tlhs);
					partial_flag = true;
					factorv(&tlhs[0], &n_tlhs, v);
					list_tdebug(1);
					uf_flag = true;
					goto see_work;
				}
				if (uf_flag) {
					simps_side(&tlhs[0], &n_tlhs, zsolve);
					uf_flag = false;
					goto see_work;
				}
split_solve:
				op = 0;
				b1 = &tlhs[0];
				for (i = 1; i < n_tlhs; i += 2) {
					if (tlhs[i].level == 1) {
						op_kind = tlhs[i].token.operatr;
						if (op_kind == TIMES || op_kind == DIVIDE) {
							if (op == 0) {
								op = TIMES;
							}
						} else {
							op = op_kind;
							break;
						}
						if (zflag && inc_count) {
							if (op_kind == DIVIDE
							    || (tlhs[i+1].level == 1 && tlhs[i+1].kind != CONSTANT)) {
								op = op_kind;
								b1 = &tlhs[i+1];
								if (op_kind == DIVIDE)
									break;
							}
						} else {
							if (op_kind == DIVIDE) {
								for (j = i + 2; j < n_tlhs && tlhs[j].level > 1; j += 2) {
									if (tlhs[j].level == 2) {
										op_kind = tlhs[j].token.operatr;
										if (op_kind == PLUS || op_kind == MINUS) {
											op = DIVIDE;
											b1 = &tlhs[i+1];
										}
										break;
									}
								}
							}
						}
					}
				}
				if ((zflag && inc_count && op == TIMES
				    && b1->level == 1 && b1->kind != CONSTANT)
				    || op == DIVIDE) {
					if (op == TIMES) {
						if (inc_count)
							inc_count--;
						printf("Removing possible solution: \"");
						qtries = 0;	/* might be quadratic after removing solution */
						k = b1 - &tlhs[0];
						for (j = k + 1; j < n_tlhs; j += 2)
							if (tlhs[j].level <= 1)
								break;
						list_proc(b1, j - k);
						printf(" = 0\".\n");
					} else
						printf("Juggling...\n");
					if (!g_of_f(op, b1, &tlhs[0], &n_tlhs, &trhs[0], &n_trhs))
						return false;
					uf_flag = true;
					goto simp_again;
				}
				b1 = NULL;
				for (i = 0; i < n_tlhs; i++) {
					if (tlhs[i].kind == OPERATOR
					    && tlhs[i].token.operatr == POWER
					    && tlhs[i+1].level == tlhs[i].level
					    && tlhs[i+1].kind == CONSTANT
					    && tlhs[i+1].token.constant < 1.0
					    && tlhs[i+1].token.constant > -1.0) {
						d1 = 1.0 / tlhs[i+1].token.constant;
						d1 = modf(d1, &d2);
						if (d1 != 0.0)
							continue;
						for (j = i - 1; (j >= 0) && tlhs[j].level >= tlhs[i].level; j--) {
							if (tlhs[j].kind == VARIABLE && tlhs[j].token.variable == v) {
								if (b1) {
									if (fabs(b1->token.constant) > fabs(tlhs[i+1].token.constant)) {
										b1 = &tlhs[i+1];
									}
								} else
									b1 = &tlhs[i+1];
								break;
							}
						}
					}
				}
				if (b1 && zero_solved) {
					printf("Raising both sides to the power of %g and unfactoring...\n", 1.0 / b1->token.constant);
					inc_count++;
					zero_solved = false;
					qtries = 0;
					if (!increase(b1->token.constant, v))
						return false;
					uf_flag = true;
					goto simp_again;
				}
				if (qtries) {
					if (inc_count || !zflag)
						return false;
					inc_count++;
					goto split_solve;
				}
				printf("Solving for zero...\n");
				*leftnp = n_tlhs;
				blt(leftp, &tlhs[0], n_tlhs * sizeof(*leftp));
				*rightnp = n_trhs;
				blt(rightp, &trhs[0], n_trhs * sizeof(*rightp));
				want0.level = 1;
				want0.kind = CONSTANT;
				want0.token.constant = 0.0;
				if (!solve_sub(&want0, 1, leftp, leftnp, rightp, rightnp))
					return false;
				if (zero_solved)
					qtries++;
				else
					zero_solved = true;
				if (quad_solve(v)) {
					goto left_again;
				} else
					goto simp_again;
			} else {
fin1:
				found = 0;
				op = p1->token.operatr;
				b1 = p1 + 1;
			}
		} else if (p1->kind == VARIABLE) {
			if (v == p1->token.variable) {
				found_count++;
				found++;
			}
		}
	}
}

/*
 * Isolate the expression containing "v" raised to the power of "d",
 * then raise both sides of the equation to the power of 1/"d".
 * Return true if successful.
 */
int
increase(d, v)
double	d;
long	v;
{
	int		foundp;
	int		op;
	token_type	*b1, *p1;
	token_type	*ep;
	int		zflag;
	int		j, k;

	partial_flag = false;
	ufactor(&tlhs[0], &n_tlhs);
	partial_flag = true;
	simp_ssub(&tlhs[0], &n_tlhs, v, d, true, false, false);
	simp_ssub(&tlhs[0], &n_tlhs, 0L, 0.0, true, false, true);
lonely:
	foundp = false;
	ep = &tlhs[n_tlhs];
	for (p1 = &tlhs[1];; p1 += 2) {
		if (p1 >= ep) {
			return false;
		}
		if (p1->level == 1) {
			break;
		}
		if (p1->token.operatr == POWER
		    && (p1 + 1)->level == p1->level
		    && (p1 + 1)->kind == CONSTANT
		    && (p1 + 1)->token.constant == d) {
			for (b1 = p1 - 1;;) {
				if (b1->level < p1->level)
					break;
				if (b1->kind == VARIABLE
				    && b1->token.variable == v) {
					foundp = true;
					break;
				}
				if (b1 == &tlhs[0])
					break;
				b1--;
			}
		}
	}
	zflag = (n_trhs == 1 && trhs[0].kind == CONSTANT
	    && trhs[0].token.constant == 0.0);
	if (p1->token.operatr == POWER && (p1 + 1)->level == 1
	    && (p1 + 1)->kind == CONSTANT && (p1 + 1)->token.constant == d) {
		if (!g_of_f(POWER, p1 + 1, &tlhs[0], &n_tlhs, &trhs[0], &n_trhs))
			return false;
		return true;
	} else {
		if (foundp) {
			b1 = p1 + 1;
			op = p1->token.operatr;
		} else {
			b1 = &tlhs[0];
			if (p1 - b1 == 1 && p1->token.operatr == DIVIDE
			    && b1->kind == CONSTANT && b1->token.constant == 1.0) {
				if (!flip(&tlhs[0], &n_tlhs, &trhs[0], &n_trhs))
					return false;
				list_tdebug(2);
				simp_loop(&tlhs[0], &n_tlhs);
				simp_loop(&trhs[0], &n_trhs);
				list_tdebug(1);
				goto lonely;
			}
			switch (p1->token.operatr) {
			case TIMES:
			case DIVIDE:
				op = TIMES;
				break;
			case PLUS:
			case MINUS:
				op = PLUS;
				break;
			default:
				return false;
			}
		}
		if (op == TIMES && zflag) {
			foundp = false;
			k = b1 - &tlhs[0];
			for (j = k + 1;; j += 2) {
				if (tlhs[j-1].kind == VARIABLE
				    && tlhs[j-1].token.variable == v)
					foundp = true;
				if (j >= n_tlhs || tlhs[j].level <= 1)
					break;
			}
			if (foundp) {
				printf("Removing possible solution: \"");
				list_proc(b1, j - k);
				printf(" = 0\".\n");
			}
		}
		if (!g_of_f(op, b1, &tlhs[0], &n_tlhs, &trhs[0], &n_trhs))
			return false;
		list_tdebug(2);
		simp_loop(&tlhs[0], &n_tlhs);
		simp_loop(&trhs[0], &n_trhs);
		list_tdebug(1);
		goto lonely;
	}
}

/*
 * Quadratic and biquadratic solve routine.
 * Solves any equation of the form "0 = ax^(2n)+bx^n+c" for "x",
 * where "x" is an expression containing passed variable "v",
 * and "n" is a constant.
 */
int
quad_solve(v)
long	v;
{
	int		i, j, k;
	token_type	*p1, *ep;
	token_type	*p2;
	token_type	*x1p, *x2p;
	token_type	*a1p, *a2p, *a2ep;
	token_type	*b1p, *b2p, *b2ep;
	token_type	*x1tp, *a1tp;
	token_type	x1_storage[200];
	int		nx1;
	int		op, op2, opx1, opx2;
	int		found;
	int		diff_sign;
	int		len;
	int		alen, blen;
	int		aloc;
	double		high_power;

	list_tdebug(2);
	uf_simp(&trhs[0], &n_trhs);
	factorv(&trhs[0], &n_trhs, v);
	list_tdebug(1);

	found = false;
	op = 0;
	high_power = 0.0;
	ep = &trhs[n_trhs];
	for (x1tp = p1 = &trhs[0];; p1++) {
		if (p1 >= ep || (p1->level == 1 && p1->kind == OPERATOR)) {
			if (p1 < ep) {
				switch (p1->token.operatr) {
				case PLUS:
				case MINUS:
					break;
				default:
					return false;
				}
			}
			if (op == TIMES || op == DIVIDE) {
				found = false;
				op2 = 0;
				for (a1tp = p2 = x1tp;; p2++) {
					if (p2 >= p1)
						break;
					if (p2->level == 2) {
						if (p2->kind == OPERATOR) {
							x1tp = p2 + 1;
							op2 = p2->token.operatr;
							found = false;
						}
					} else {
						if (p2->kind == OPERATOR) {
							if (p2->level == 3 && p2->token.operatr == POWER) {
								if (found && (op2 == TIMES || op2 == 0)
								    && (p2 + 1)->level == 3
								    && (p2 + 1)->kind == CONSTANT
								    && (p2 + 1)->token.constant > high_power) {
									high_power = (p2 + 1)->token.constant;
									x1p = x1tp;
									a1p = a1tp;
									a2p = p2 + 2;
									a2ep = p1;
								}
							}
						} else if (p2->kind == VARIABLE && p2->token.variable == v) {
							found = true;
						}
					}
				}
			} else if (op == POWER) {
				if (found && (p1 - 1)->level == 2
				    && (p1 - 1)->kind == CONSTANT
				    && (p1 - 1)->token.constant > high_power) {
					high_power = (p1 - 1)->token.constant;
					a1p = x1p = x1tp;
					a2p = p1;
					a2ep = a2p;
				}
			}
			if (p1 >= ep) {
				if (high_power == 0.0)
					return false;
				else
					break;
			}
		}
		if (p1->level == 1) {
			if (p1->kind == OPERATOR) {
				op = 0;
				x1tp = p1 + 1;
				found = false;
			}
		} else {
			if (p1->kind == OPERATOR) {
				if (p1->level == 2) {
					op = p1->token.operatr;
				}
			} else if (op == 0) {
				if (p1->kind == VARIABLE && p1->token.variable == v)
					found = true;
			}
		}
	}
	if (a1p > &trhs[0] && (a1p - 1)->token.operatr == MINUS)
		opx1 = MINUS;
	else
		opx1 = PLUS;
	if (high_power == 2.0) {
		nx1 = (a2p - x1p) - 2;
		if (nx1 > ARR_CNT(x1_storage))
			return false;
		blt(x1_storage, x1p, nx1 * sizeof(token_type));
	} else {
		nx1 = (a2p - x1p);
		if (nx1 > ARR_CNT(x1_storage))
			return false;
		blt(x1_storage, x1p, nx1 * sizeof(token_type));
		x1_storage[nx1-1].token.constant /= 2.0;
	}
	opx2 = 0;
	op = 0;
	for (x2p = p1 = &trhs[0];; p1++) {
		if (p1 >= ep || (p1->level == 1 && p1->kind == OPERATOR)) {
			if (se_compare(x1_storage, nx1, x2p, p1 - x2p, &diff_sign)) {
				b1p = x2p;
				b2p = p1;
				b2ep = b2p;
				break;
			}
			if (op == TIMES || op == DIVIDE) {
				op2 = 0;
				for (b1p = p2 = x2p;; p2++) {
					if (p2 >= p1 || (p2->level == 2 && p2->kind == OPERATOR)) {
						if (op2 == 0 || op2 == TIMES) {
							if (se_compare(x1_storage, nx1, x2p, p2 - x2p, &diff_sign)) {
								b2p = p2;
								b2ep = p1;
								goto big_bbreak;
							}
						}
						if (p2 >= p1)
							break;
					}
					if (p2->level == 2) {
						if (p2->kind == OPERATOR) {
							x2p = p2 + 1;
							op2 = p2->token.operatr;
						}
					}
				}
			}
			if (p1 >= ep)
				return false;
		}
		if (p1->level == 1) {
			if (p1->kind == OPERATOR) {
				op = 0;
				opx2 = p1->token.operatr;
				x2p = p1 + 1;
			}
		} else {
			if (p1->kind == OPERATOR) {
				if (p1->level == 2) {
					op = p1->token.operatr;
				}
			}
		}
	}
big_bbreak:
	switch (opx2) {
	case 0:
		opx2 = PLUS;
	case PLUS:
		if (diff_sign)
			opx2 = MINUS;
		break;
	case MINUS:
		if (diff_sign)
			opx2 = PLUS;
		break;
	default:
		return false;
	}
	blt(&scratch[0], b1p, (int) ((char *) x2p - (char *) b1p));
	len = x2p - b1p;
	scratch[len].level = 7;
	scratch[len].kind = CONSTANT;
	if (opx2 == MINUS)
		scratch[len].token.constant = -1.0;
	else
		scratch[len].token.constant = 1.0;
	len++;
	blt(&scratch[len], b2p, (int) ((char *) b2ep - (char *) b2p));
	len += (b2ep - b2p);
	blen = len;
	j = min_level(&scratch[0], len);
	j = 7 - j;
	for (i = 0; i < len; i++)
		scratch[i].level += j;
	scratch[len].level = 6;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = POWER;
	len++;
	scratch[len].level = 6;
	scratch[len].kind = CONSTANT;
	scratch[len].token.constant = 2.0;
	len++;
	scratch[len].level = 5;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = MINUS;
	len++;
	scratch[len].level = 6;
	scratch[len].kind = CONSTANT;
	scratch[len].token.constant = 4.0;
	len++;
	scratch[len].level = 6;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = TIMES;
	len++;
	aloc = len;
	blt(&scratch[len], a1p, (int) ((char *) x1p - (char *) a1p));
	len += (x1p - a1p);
	scratch[len].level = 7;
	scratch[len].kind = CONSTANT;
	if (opx1 == MINUS)
		scratch[len].token.constant = -1.0;
	else
		scratch[len].token.constant = 1.0;
	len++;
	blt(&scratch[len], a2p, (int) ((char *) a2ep - (char *) a2p));
	len += (a2ep - a2p);
	alen = len - aloc;
	j = min_level(&scratch[aloc], len - aloc);
	j = 7 - j;
	for (i = aloc; i < len; i++)
		scratch[i].level += j;
	scratch[len].level = 6;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = TIMES;
	len++;
	k = len;
	scratch[len].level = 1;
	scratch[len].kind = CONSTANT;
	scratch[len].token.constant = 0.0;
	len++;
	for (p2 = p1 = &trhs[0];; p1++) {
		if (p1 >= ep || (p1->level == 1 && p1->kind == OPERATOR)) {
			if ((p2 <= x1p && p1 > x1p) || (p2 <= x2p && p1 > x2p))
				goto skip_this;
			if (p2 == &trhs[0]) {
				scratch[len].level = 1;
				scratch[len].kind = OPERATOR;
				scratch[len].token.operatr = PLUS;
				len++;
			}
			blt(&scratch[len], p2, (int) ((char *) p1 - (char *) p2));
			len += (p1 - p2);
skip_this:
			if (p1 >= ep)
				break;
			else
				p2 = p1;
		}
	}
	for (i = k; i < len; i++)
		scratch[i].level += 6;
	scratch[len].level = 4;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = POWER;
	len++;
	scratch[len].level = 4;
	scratch[len].kind = CONSTANT;
	scratch[len].token.constant = 0.5;
	len++;
	scratch[len].level = 3;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = TIMES;
	len++;
	scratch[len].level = 3;
	scratch[len].kind = VARIABLE;
	next_sign(&scratch[len].token.variable);
	len++;
	scratch[len].level = 2;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = MINUS;
	len++;
	if (len + blen + 3 + alen > N_TOKENS) {
		error_huge();
	}
	blt(&scratch[len], &scratch[0], blen * sizeof(token_type));
	len += blen;
	scratch[len].level = 1;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = DIVIDE;
	len++;
	scratch[len].level = 2;
	scratch[len].kind = CONSTANT;
	scratch[len].token.constant = 2.0;
	len++;
	scratch[len].level = 2;
	scratch[len].kind = OPERATOR;
	scratch[len].token.operatr = TIMES;
	len++;
	blt(&scratch[len], &scratch[aloc], alen * sizeof(token_type));
	len += alen;
	for (i = 0; i < len; i++) {
		if (scratch[i].kind == VARIABLE && scratch[i].token.variable == v)
			return false;
	}
	if (debug_level > 0) {
		printf("Plugging equation into the quadratic formula:\n");
	}
	blt(&tlhs[0], x1_storage, nx1 * sizeof(token_type));
	n_tlhs = nx1;
	simp_loop(&tlhs[0], &n_tlhs);
	blt(&trhs[0], &scratch[0], len * sizeof(token_type));
	n_trhs = len;
	simp_loop(&trhs[0], &n_trhs);
	list_tdebug(2);
	uf_simp_no_repeat(&trhs[0], &n_trhs);
	list_tdebug(2);
	uf_simp(&trhs[0], &n_trhs);
	simps_side(&trhs[0], &n_trhs, false);
	list_tdebug(1);
	if (high_power == 2.0) {
		printf("Equation was quadratic.\n");
	} else if (high_power == 4.0) {
		printf("Equation was biquadratic.\n");
	} else {
		printf("Equation was quasi-quadratic.\n");
	}
	return true;
}

/*
 * Apply the inverse of the operation "op" followed by expression "operandp",
 * which is somewhere in "side1p", to both sides of an equation,
 * which is "side1p" and "side2p".
 *
 * Return true unless something is wrong.
 */
int
g_of_f(op, operandp, side1p, side1np, side2p, side2np)
int		op;
token_type	*operandp;
token_type	*side1p;
int		*side1np;	/* pointer to the length of "side1p" */
token_type	*side2p;
int		*side2np;	/* pointer to the length of "side2p" */
{
	int		i;
	token_type	*p1, *p2, *ep;
	int		oldn;
	int		level;
	double		numerator, denominator;

	oldn = *side1np;
	ep = &side1p[oldn];
	switch (op) {
	case PLUS:
	case MINUS:
	case TIMES:
	case DIVIDE:
	case POWER:
		break;
	case FACTORIAL:
		i = oldn - 2;
		if (i >= 0 && side1p[i].level == 1 && side1p[i].token.operatr == op) {
			i = *side2np - 2;
			if (i >= 0 && side2p[i].level == 1 && side2p[i].token.operatr == op) {
				*side1np -= 2;
				*side2np -= 2;
				return true;
			}
		}
		return false;
	default:
		return false;
	}
	if (operandp > side1p) {
		level = (operandp - 1)->level;
	} else
		level = 1;
	if (level == 1) {
		for (p1 = operandp + 1; p1 < ep; p1 += 2) {
			if (p1->level == 1) {
				if (p1->token.operatr == POWER || p1->token.operatr == FACTORIAL) {
					op = PLUS;
				} else {
					break;
				}
			}
		}
	} else {
		printf("Bug in call to g_of_f()!\n");
		return false;
	}
	if (debug_level > 0) {
		switch (op) {
		case PLUS:
			printf("Subtracting");
			break;
		case MINUS:
			printf("Adding");
			break;
		case TIMES:
			printf("Dividing both sides of the equation by");
			break;
		case DIVIDE:
			printf("Multiplying both sides of the equation by");
			break;
		case POWER:
			printf("Raising both sides of the equation to the power of");
			break;
		}
		printf(" \"");
		if (op == POWER)
			printf("1/(");
		list_proc(operandp, p1 - operandp);
		switch (op) {
		case PLUS:
			printf("\" from both sides of the equation:\n");
			break;
		case MINUS:
			printf("\" to both sides of the equation:\n");
			break;
		case POWER:
			printf(")");
		default:
			printf("\":\n");
			break;
		}
	}
	if (*side1np + (p1 - operandp) + 3 > N_TOKENS
	    || *side2np + (p1 - operandp) + 5 > N_TOKENS) {
		error_huge();
	}
	for (p2 = side1p; p2 < ep; p2++)
		p2->level++;
	ep = &side2p[*side2np];
	for (p2 = side2p; p2 < ep; p2++)
		p2->level++;
	p2 = &side1p[oldn];
	switch (op) {
	case POWER:
		p2->level = 1;
		p2->kind = OPERATOR;
		p2->token.operatr = POWER;
		p2++;
		p2->level = 2;
		p2->kind = CONSTANT;
		p2->token.constant = 1.0;
		p2++;
		p2->level = 2;
		p2->kind = OPERATOR;
		p2->token.operatr = DIVIDE;
		p2++;
		blt(p2, operandp, (int) ((char *) p1 - (char *) operandp));
		*side1np += 3 + (p1 - operandp);
		ep = &side1p[*side1np];
		for (; p2 < ep; p2++)
			p2->level++;
		break;
	case TIMES:
		p2->level = 1;
		p2->kind = OPERATOR;
		p2->token.operatr = DIVIDE;
		p2++;
		blt(p2, operandp, (int) ((char *) p1 - (char *) operandp));
		*side1np += 1 + (p1 - operandp);
		break;
	case DIVIDE:
		p2->level = 1;
		p2->kind = OPERATOR;
		p2->token.operatr = TIMES;
		p2++;
		blt(p2, operandp, (int) ((char *) p1 - (char *) operandp));
		*side1np += 1 + (p1 - operandp);
		break;
	case PLUS:
		p2->level = 1;
		p2->kind = OPERATOR;
		p2->token.operatr = MINUS;
		p2++;
		blt(p2, operandp, (int) ((char *) p1 - (char *) operandp));
		*side1np += 1 + (p1 - operandp);
		break;
	case MINUS:
		p2->level = 1;
		p2->kind = OPERATOR;
		p2->token.operatr = PLUS;
		p2++;
		blt(p2, operandp, (int) ((char *) p1 - (char *) operandp));
		*side1np += 1 + (p1 - operandp);
		break;
	}
	blt(&side2p[*side2np], &side1p[oldn], (int) (*side1np - oldn) * sizeof(*side1p));
	*side2np += *side1np - oldn;
	if (op == POWER && p1 - operandp == 1 && operandp->kind == CONSTANT) {
		f_to_fraction(operandp->token.constant, &numerator, &denominator);
		if (always_positive(numerator)) {
			ep = &side2p[*side2np];
			for (p2 = side2p; p2 < ep; p2++)
				p2->level++;
			p2->level = 1;
			p2->kind = OPERATOR;
			p2->token.operatr = TIMES;
			p2++;
			p2->level = 1;
			p2->kind = VARIABLE;
			next_sign(&p2->token.variable);
			*side2np += 2;
		}
	}
	if (op == POWER) {
		*side1np = (operandp - 1) - side1p;
	}
	return true;
}

/*
 * Take the reciprocal of both equation sides.
 */
int
flip(side1p, side1np, side2p, side2np)
token_type	*side1p;
int		*side1np;
token_type	*side2p;
int		*side2np;
{
	token_type	*p1, *ep;

	if (*side1np + 2 > N_TOKENS || *side2np + 2 > N_TOKENS) {
		error_huge();
	}
	ep = &side1p[*side1np];
	for (p1 = side1p; p1 < ep; p1++)
		p1->level++;
	ep = &side2p[*side2np];
	for (p1 = side2p; p1 < ep; p1++)
		p1->level++;
	blt(side1p + 2, side1p, *side1np * sizeof(*side1p));
	*side1np += 2;
	blt(side2p + 2, side2p, *side2np * sizeof(*side2p));
	*side2np += 2;
	side1p->level = 1;
	side1p->kind = CONSTANT;
	side1p->token.constant = 1.0;
	side1p++;
	side1p->level = 1;
	side1p->kind = OPERATOR;
	side1p->token.operatr = DIVIDE;

	side2p->level = 1;
	side2p->kind = CONSTANT;
	side2p->token.constant = 1.0;
	side2p++;
	side2p->level = 1;
	side2p->kind = OPERATOR;
	side2p->token.operatr = DIVIDE;
	return true;
}

/*
 * Display the equation in equation number "n" in single line format.
 */
int
list_sub(n)
int	n;
{
	if (n_lhs[n] <= 0)
		return false;
	fprintf(gfp, "#%d: ", n + 1);
	list_proc(&lhs[n][0], n_lhs[n]);
	fprintf(gfp, " = ");
	list_proc(&rhs[n][0], n_rhs[n]);
	fprintf(gfp, "\n");
	return true;
}

list_tdebug(level)
int	level;
{
	if (debug_level >= level) {
		list_debug(level, &tlhs[0], n_tlhs, &trhs[0], n_trhs);
	}
}

list_debug(level, p1, n1, p2, n2)
int		level;
token_type	*p1;
int		n1;
token_type	*p2;
int		n2;
{
	if (debug_level >= level) {
		printf("debug %d: ", level);
		list_proc(p1, n1);
		printf(" = ");
		list_proc(p2, n2);
		printf("\n");
	}
}

side_debug(level, p1, n1)
int		level;
token_type	*p1;
int		n1;
{
	if (debug_level >= level) {
		printf("debug %d: ", level);
		list_proc(p1, n1);
		printf("\n");
	}
}

/*
 * Output a variable name in ASCII if "out_flag" is true.
 * Return length of variable (number of ASCII characters).
 */
int
list_var(v, out_flag)
long	v;
int	out_flag;
{
	int		i, j;
	long		l1;
	static char	var_str[80];

	var_str[0] = '\0';
	switch ((int) (v & VAR_MASK)) {
	case 0:
		strcat(var_str, "NULL");
		break;
	case SIGN:
		strcat(var_str, "sign");
		break;
	case IMAGINARY:
		strcat(var_str, "i#");
		break;
	case V_E:
		strcat(var_str, "e#");
		break;
	case V_PI:
		strcat(var_str, "pi");
		break;
	case LIMIT0:
		strcat(var_str, "limit_of_0");
		break;
	case MATCH_ANY:
		strcat(var_str, "all");
		break;
	case SPECIAL:
		strcat(var_str, "Answer");
		break;
	default:
		j = (v >> 7) & 0x7f;
		if (j) {
			var_str[0] = j;
			var_str[1] = '_';
			var_str[2] = (v & 0x7f);
			var_str[3] = '\0';
		} else {
			var_str[0] = (v & 0x7f);
			var_str[1] = '\0';
		}
		break;
	}
	j = (v >> VAR_SHIFT) & SUBSCRIPT_MASK;
	if (j) {
		j--;
		sprintf(&var_str[strlen(var_str)], "%d", j);
	}
	l1 = v;
	i = 0;
	while ((l1 -= PRIME_INCREMENT) >= 0) {
		strcat(var_str, "'");
		if (++i >= 15)
			break;
	}
	if (v & PERCENT_CHANGE) {
		strcat(var_str, "_percent_change");
	}
	if (out_flag) {
		fprintf(gfp, "%s", var_str);
	}
	return(strlen(var_str));
}

/*
 * Display an expression in single line format.
 */
list_proc(equation, n)
token_type	*equation;
int		n;
{
	int	i,j,k,l;
	int	min1;
	int	cur_level;

	cur_level = min1 = min_level(equation, n);
	for (i = 0; i < n; i++) {
		j = cur_level - equation[i].level;
		k = abs(j);
		for (l = 1; l <= k; l++) {
			if (j > 0) {
				cur_level--;
				putc(')', gfp);
				if (gfp == stdout) {
					set_color(cur_level-min1);
				}
			} else {
				cur_level++;
				if (gfp == stdout) {
					set_color(cur_level-min1);
				}
				putc('(', gfp);
			}
		}
		switch (equation[i].kind) {
		case CONSTANT:
			if (high_prec) {
				fprintf(gfp, "%.20lg", equation[i].token.constant);
			} else {
				fprintf(gfp, "%.14lg", equation[i].token.constant);
			}
			break;
		case VARIABLE:
			list_var(equation[i].token.variable, true);
			break;
		case OPERATOR:
			switch (equation[i].token.operatr) {
			case PLUS:
				putc('+', gfp);
				break;
			case MINUS:
				putc('-', gfp);
				break;
			case TIMES:
				putc('*', gfp);
				break;
			case DIVIDE:
				putc('/', gfp);
				break;
			case POWER:
				putc('^', gfp);
				break;
			case FACTORIAL:
				putc('!', gfp);
				i++;
				break;
			}
			break;
		}
	}
	j = cur_level - min1;
	for (; j > 0;) {
		putc(')', gfp);
		j--;
		if (gfp == stdout) {
			set_color(j);
		}
	}
}

int	cur_line;
int	cur_pos;
int	max_line;
int	min_line;

/*
 * Display the equation at equation number "n" in fraction format.
 */
int
flist_sub(n)
int	n;
{
	int	i;
	int	sind;
	char	buf[30];
	int	len;
	int	len2, len3;
	int	pos;
	int	high, low;
	int	max2_line, min2_line;

	if (n_lhs[n] <= 0)
		return false;
	sprintf(&buf[0], "#%d: ", n + 1);
	cur_line = 0;
	cur_pos = 0;
	max_line = 0;
	min_line = 0;
	sind = n_rhs[n];
	high = 0;
	low = 0;
	len = strlen(buf);
	len += flist_recurse(&lhs[n][0], n_lhs[n], false, 0, 0, 1, false, &high, &low);
	if (high > max_line)
		max_line = high;
	if (low < min_line)
		min_line = low;
	len += 3;
make_smaller:
	len2 = flist_recurse(&rhs[n][0], sind, false, 0, 0, 1, false, &high, &low);
	if ((len + len2) >= SCREEN_COLUMNS && sind > 0) {
		for (sind--; sind > 0; sind--) {
			if (rhs[n][sind].level == 1 && rhs[n][sind].kind == OPERATOR
			    && rhs[n][sind].token.operatr != DIVIDE
			    && rhs[n][sind].token.operatr != FACTORIAL) {
				break;
			}
		}
		goto make_smaller;
	}
	if (high > max_line)
		max_line = high;
	if (low < min_line)
		min_line = low;
	len3 = flist_recurse(&rhs[n][sind], n_rhs[n] - sind, false, 0, 0, 1, false, &max2_line, &min2_line);
	fprintf(gfp, "\n");
	if (max(len + len2, len3) >= SCREEN_COLUMNS) {
		printf("Equation number %d is too wide to convert to fraction format.\nUsing single line format.\n", n + 1);
		list_sub(n);
		return true;
	}
	for (i = max_line; i >= min_line; i--) {
		cur_line = i;
		cur_pos = 0;
		if (i == 0) {
			fprintf(gfp, "%s", buf);
			cur_pos = strlen(buf);
		}
		pos = strlen(buf);
		pos += flist_recurse(&lhs[n][0], n_lhs[n], true, 0, pos, 1, false, &high, &low);
		if (i == 0) {
			fprintf(gfp, "%s", " = ");
			cur_pos += 3;
		}
		pos += 3;
		flist_recurse(&rhs[n][0], sind, true, 0, pos, 1, false, &high, &low);
		fprintf(gfp, "\n");
	}
	if (sind < n_rhs[n]) {
		fprintf(gfp, "\n");
		for (i = max2_line; i >= min2_line; i--) {
			cur_line = i;
			cur_pos = 0;
			flist_recurse(&rhs[n][sind], n_rhs[n] - sind, true, 0, 0, 1, false, &high, &low);
			fprintf(gfp, "\n");
		}
	}
	return true;
}

int
flist_recurse(equation, n, out_flag, line, pos, cur_level, top_flag, highp, lowp)
token_type	*equation;
int		n;
int		out_flag;
int		line;
int		pos;
int		cur_level;
int		top_flag;
int		*highp, *lowp;
{
	int	i,j,k,l;
	int	l1, l2;
	int	ii;
	int	stop_at;
	int	div_loc;
	int	len_div;
	int	level;
	int	start_level;
	int	cflag;
	int	oflag;
	int	len, len1, len2;
	int	high, low;
	char	buf[50];

	len = 0;
	start_level = cur_level;
	*highp = line;
	*lowp = line;
	if (n <= 0) {
		return 0;
	}
	oflag = (out_flag && line == cur_line);
	cflag = (color_flag && gfp == stdout && oflag);
	if (cflag) {
		set_color(cur_level-1);
	}
	if (oflag) {
		for (; cur_pos < pos; cur_pos++) {
			putc(' ', gfp);
		}
	}
	ii = 0;
check_again:
	stop_at = n;
	div_loc = -1;
	for (i = ii; i < n; i++) {
		if (equation[i].kind == OPERATOR && equation[i].token.operatr == DIVIDE) {
			level = equation[i].level;
			for (j = i - 2; j > 0; j -= 2) {
				if (equation[j].level < level)
					break;
			}
			j++;
			if (div_loc < 0) {
				div_loc = i;
				stop_at = j;
			} else {
				if (j < stop_at) {
					div_loc = i;
					stop_at = j;
				} else if (j == stop_at) {
					if (level < equation[div_loc].level)
						div_loc = i;
				}
			}
		}
	}
	for (i = ii; i < n; i++) {
		if (i == stop_at) {
			j = cur_level - equation[div_loc].level;
			k = abs(j) - 1;
		} else {
			j = cur_level - equation[i].level;
			k = abs(j);
		}
		for (l = 1; l <= k; l++) {
			if (j > 0) {
				cur_level--;
				if (oflag)
					putc(')', gfp);
				len++;
				if (cflag) {
					set_color(cur_level-1);
				}
			} else {
				cur_level++;
				if (cflag) {
					set_color(cur_level-1);
				}
				if (oflag)
					putc('(', gfp);
				len++;
			}
		}
		if (i == stop_at) {
			level = equation[div_loc].level;
			len1 = flist_recurse(&equation[stop_at], div_loc - stop_at, false, line + 1, pos + len, level, true, &high, &low);
			l1 = (2 * (line + 1)) - low;
			for (j = div_loc + 2; j < n; j += 2) {
				if (equation[j].level <= level)
					break;
			}
			len2 = flist_recurse(&equation[div_loc+1], j - (div_loc + 1), false, line - 1, pos + len, level, false, &high, &low);
			l2 = (2 * (line - 1)) - high;
			ii = j;
			len_div = max(len1, len2);
			j = 0;
			if (len1 < len_div) {
				j = (len_div - len1) / 2;
			}
			flist_recurse(&equation[stop_at], div_loc - stop_at, out_flag, l1, pos + len + j, level, true, &high, &low);
			if (high > *highp)
				*highp = high;
			if (low < *lowp)
				*lowp = low;
			if (oflag) {
				if (cflag) {
					set_color(level-1);
				}
				for (j = 0; j < len_div; j++) {
					putc('-', gfp);
				}
				if (cflag) {
					set_color(cur_level-1);
				}
			}
			j = 0;
			if (len2 < len_div) {
				j = (len_div - len2) / 2;
			}
			flist_recurse(&equation[div_loc+1], ii - (div_loc + 1), out_flag, l2, pos + len + j, level, false, &high, &low);
			if (high > *highp)
				*highp = high;
			if (low < *lowp)
				*lowp = low;
			len += len_div;
			goto check_again;
		}
		switch (equation[i].kind) {
		case CONSTANT:
			sprintf(buf, "%.14lg", equation[i].token.constant);
			if (oflag)
				fprintf(gfp, "%s", buf);
			len += strlen(buf);
			break;
		case VARIABLE:
			len += list_var(equation[i].token.variable, oflag);
			break;
		case OPERATOR:
			switch (equation[i].token.operatr) {
			case PLUS:
				strcpy(buf, " + ");
				break;
			case MINUS:
				strcpy(buf, " - ");
				break;
			case TIMES:
				strcpy(buf, "*");
				break;
			case POWER:
				strcpy(buf, "^");
				break;
			case FACTORIAL:
				strcpy(buf, "!");
				i++;
				break;
			default:
				printf("Unknown operator in fraction list routine!\n");
				longjmp(jmp_save, 2);
				break;
			}
			if (oflag)
				fprintf(gfp, "%s", buf);
			len += strlen(buf);
			break;
		}
	}
	j = cur_level - start_level;
	for (; j > 0;) {
		cur_level--;
		if (oflag)
			putc(')', gfp);
		len++;
		j--;
		if (cflag) {
			set_color(cur_level-1);
		}
	}
	if (cflag) {
		set_color(0);
	}
	if (oflag)
		cur_pos += len;
	return len;
}

/*
 * Return the number of the next empty equation space.
 */
int
next_espace()
{
	int	i,n;

	for (n = cur_equation, i = 1; i <= n_equations; n = (n + 1) % n_equations, i++) {
		if (n_lhs[n] == 0)
			break;
	}
	if (i > n_equations) {
		printf("Out of free equation spaces.\n");
		printf("Use the 'clear' command on unnecessary equations and try again.\n");
		longjmp(jmp_save, 3);
	}
	return n;
}

/*
 * Return the minimum level encountered in a sub-expression.
 */
int
min_level(equation, n)
token_type	*equation;
int		n;
{
	int		min1;
	token_type	*p1, *ep;

	if (n <= 1) {
		if (n <= 0) {
			printf("Programming error in call to min_level()!\n");
			longjmp(jmp_save, 2);
		}
		return equation[0].level;
	}
	min1 = equation[1].level;
	ep = &equation[n];
	for (p1 = &equation[3]; p1 < ep; p1 += 2) {
		if (p1->level < min1)
			min1 = p1->level;
	}
	return min1;
}

error_huge()
{
#if	TUTOR
	printf("Equation too big for emailware version of Mathomatic!\n");
#else
	printf("Equation too big!\n");
#endif
	longjmp(jmp_save, 2);
}
