C#######################################################################
C PSTSWM Version 1.0 (8/1/93)                                          #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
      SUBROUTINE SIMPLIC(DTA, ALP, DALP, WTS, WTACSJ, ANNP1, A2NNP1, 
     &                   TRIGS, MOUNT, TOPOSC, DIV, ZETA, PHI, UCOS, 
     &                   VCOS, WS1, WS2, WS3, DIVSC, ZETASC, PHISC)
C                                                                              
C This is the main computational procedure for one semi-implicit time-   
C step. Timestepping is calculated for the prognostic variables
C geopotential, divergence and vorticity using leapfrog timestepping
C with a spectral transform algorithm. The old timelevel is LNM1, The
C derivative is evaluated at timelevel LN, and the computed values go
C into timelevel LNP1. 
C
C The nonlinear products and forcing terms (for test case 4) are 
C evaluated at the gridpoints by routine NONLIM. A real Fourier 
C transform (RFTLON) is then used for each latitude and vertical layer.
C Contributions to the new timelevel spectral coefficients are computed 
C by the routine FTDPIV. Divergence and geopotential coefficients are
C then updated in routine DZUPIM, and an optional linear diffusion
C operator is applied in spectral space in routine DIFFUSE. Finally, 
C routines DZPUV and RFTLON compute the grid U,V wind fields from 
C divergence and vorticity spectral coefficients and inverse transforms
C the prognostic variables to gridpoint space. An Asselin filter can be
C also be used to prevent modal splitting.
C                                                                              
C called by: COMP1
C calls: DIFFUSE, DPUPIM, DZPUV, RFTLON, FTDPIV, NONLIM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C constants & timesteps
      INCLUDE 'consts.i'
C domain decomposition information
      INCLUDE 'physical.i'
      INCLUDE 'spectral.i'
C time dependent fields
      INCLUDE 'tdvars.i'
C transform arrays
      INCLUDE 'trnsfm.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C timestep
      REAL DTA
C associated Legendre polynomials
      REAL ALP(NFSPEC_S,NLLATH_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S,NLLATH_S)
C Gaussian weights
      REAL WTS(NLAT)
C 1.0/(A*COS(LAT)**2)
      REAL WTACSJ(NLAT)
C A/(N*(N+1))
      REAL ANNP1(0:KK)
C (N*(N+1))/A**2
      REAL A2NNP1(0:KK)
C trigonometric function values used by RFTLON 
      COMPLEX TRIGS(NTRIGS)
C mountain height
      REAL MOUNT(NLLON_P,NLLAT_P)
C spectral coefficients of mountains
      COMPLEX TOPOSC(MXLSPEC_S)
C
C     Input/Output
C
C divergence field
      REAL DIV(MXLLON_P,NLVER_P,NLLAT_P,6)
C vorticity field
      REAL ZETA(MXLLON_P,NLVER_P,NLLAT_P,6)
C geopotential field
      REAL PHI(MXLLON_P,NLVER_P,NLLAT_P,6)
C eastward wind field (scaled by COS(THETA))
      REAL UCOS(MXLLON_P,NLVER_P,NLLAT_P)
C northward wind field (scaled by COS(THETA))
      REAL VCOS(MXLLON_P,NLVER_P,NLLAT_P)
C
C     Work Space
C
C work array 1
C (big enough for REAL (MXLLON_P,NLVER_P,NLLAT_P,8) 
C             and COMPLEX (MXLFC_S,NLVER_S,NLLAT_S,8)
C             and COMPLEX (4,MXLSPEC_S,NLVER_S))
      REAL WS1(1)
C work array 2
C (big enough for REAL (MXLLON_P,NLVER_P,NLLAT_P,8,BUFSWS2) 
C             and REAL (NLLON_P,NLVER_P,3) or
C             and COMPLEX (14,MXLFC_S,NLVER_S,NLLATH_S)
C             and REAL (0:KK,3))
      REAL WS2(1)
C work array 3
C (big enough for COMPLEX (MXLSPEC_S,NLVER_S,3,BUFSWS3))
      REAL WS3(1)
C
C     Output
C
C computed divergence at new timelevel
      COMPLEX DIVSC(MXLSPEC_S,NLVER_S)
C computed vorticity new timelevel
      COMPLEX ZETASC(MXLSPEC_S,NLVER_S)                        
C computed geopotential new timestep
      COMPLEX PHISC(MXLSPEC_S,NLVER_S)                        
C
C---- Local Variables --------------------------------------------------
C
C latitude, vertical layer, and longitude indices
      INTEGER J, JV, I
C Asselin filter coefficient
      REAL FAC
C
C---- Executable Statements --------------------------------------------
C
C     Evaluate non-linear product and forcing terms and old timelevel of
C     vorticity, divergence, geopotential.
C     (results in WS1)
C      CALL TRACEEVENT('entry', 1, 1, NSTEP)
        CALL NONLIM(DTA, WTACSJ, MOUNT, DIV, ZETA, PHI, UCOS, VCOS,
     &              WS2, WS1)
C      CALL TRACEEVENT('exit', 1, 1, NSTEP)
C
C     Fourier transform non-linear terms in place. (results in WS1)
C      CALL TRACEEVENT('entry', 2, 1, NSTEP)
        CALL RFTLON(-1, 8, TRIGS, WS2, WS1)
C      CALL TRACEEVENT('exit', 2, 1, NSTEP)
C
C     Divergence and geopotential timestepping (spectral code)
C     and forward Legendre transform and update of vorticity spectral 
C     coefficients. (data in WS1)
C      CALL TRACEEVENT('entry', 3, 1, NSTEP)
        CALL FTDPIV (DTA, ALP, DALP, WTS, WTACSJ, A2NNP1,
     &               WS1, WS2, WS3, DIVSC, ZETASC, PHISC)
C      CALL TRACEEVENT('exit', 3, 1, NSTEP)
C
C     Update divergence and mass continuity spectral coefficients
C     for semi-implicit timestepping. (See H. Ritchie's paper.)
C      CALL TRACEEVENT('entry', 4, 1, NSTEP)
        CALL DPUPIM(DTA, A2NNP1, DIVSC, PHISC)
C      CALL TRACEEVENT('exit', 4, 1, NSTEP)
C
C     a little linear diffusion (time lagged as in CCM)
C      CALL TRACEEVENT('entry', 5, 1, NSTEP)
        IF (HDC .NE. 0.0) THEN 
          CALL DIFFUSE(DTA, A2NNP1, TOPOSC, WS2, DIVSC, ZETASC, PHISC)
        ENDIF
C
C       Compute first half of Asselin filter.
        IF (AFC .NE. 0.0) THEN
          FAC = 1.0 - 2.0*AFC
          DO J=1,NLLAT_P 
            DO JV=1,NLVER_P
              DO I=1,NLLON_P
                ZETA(I,JV,J,LN) = FAC*ZETA(I,JV,J,LN) 
     &                          + AFC*ZETA(I,JV,J,LNM1)
                DIV(I,JV,J,LN)  = FAC*DIV(I,JV,J,LN) 
     &                          + AFC*DIV(I,JV,J,LNM1)
                PHI(I,JV,J,LN)  = FAC*PHI(I,JV,J,LN) 
     &                          + AFC*PHI(I,JV,J,LNM1) 
              ENDDO
            ENDDO
          ENDDO
        ENDIF
C      CALL TRACEEVENT('exit', 5, 1, NSTEP)
C
C     Compute velocity fields and transform divergence, geopotential,
C     and vorticity back to gridpoint space:
C      1) inverse Legendre transform and associated computations
C      2) inverse FFT
C      CALL TRACEEVENT('entry', 6, 1, NSTEP)
        CALL DZPUV(LNP1, ALP, DALP, ANNP1, DIVSC, ZETASC, PHISC, WS1, 
     &             WS3, DIV, ZETA, PHI, UCOS, VCOS)
C      CALL TRACEEVENT('exit', 6, 1, NSTEP)
C
C     (If (LNP1 .EQ. 1), then the order is DIV, ZETA, PHI, UCOS, VCOS.
C      Otherwise, the order is UCOS, VCOS, DIV, ZETA, PHI.)
C      CALL TRACEEVENT('entry', 7, 1, NSTEP)
        IF (LNP1 .EQ. 1) THEN
          CALL RFTLON(+1, 5, TRIGS, WS2, DIV)
        ELSE
          CALL RFTLON(+1, 5, TRIGS, WS2, UCOS)
        ENDIF
C      CALL TRACEEVENT('exit', 7, 1, NSTEP)
C
C     Compute second half of Asselin filter.
C      CALL TRACEEVENT('entry', 8, 1, NSTEP)
        IF (AFC .NE. 0.0) THEN
          DO J=1,NLLAT_P 
            DO JV=1,NLVER_P
              DO I=1,NLLON_P
                ZETA(I,JV,J,LN) = ZETA(I,JV,J,LN) 
     &                          + AFC*ZETA(I,JV,J,LNP1)
                DIV(I,JV,J,LN)  = DIV(I,JV,J,LN) 
     &                          + AFC*DIV(I,JV,J,LNP1)
                PHI(I,JV,J,LN)  = PHI(I,JV,J,LN) 
     &                          + AFC*PHI(I,JV,J,LNP1) 
              ENDDO
            ENDDO
          ENDDO
        ENDIF
C      CALL TRACEEVENT('exit', 8, 1, NSTEP)
C
      RETURN
      END
