C#######################################################################
C PSTSWM Version 1.0 (8/1/93)                                          #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
      SUBROUTINE SHIFTCAST(COMMOPT, BUFFERS, PROTOPT, FORCETYPE, ME,
     &                     LEFT, RIGHT, MYINDEX, BASE, STEP, MAXSTEP,
     &                     LTH, LOCVEC, EXTVEC, CURBUF)
C
C This subroutine calculates one step of a ring-shift algorithm for
C broadcasting a distributed vector.
C
C Communication options (COMMOPT) for SHIFT include:
C  IF (COMMOPT .EQ. 0) simple shift: (calc) ... send/recv/(calc)
C  IF (COMMOPT .EQ. 1) ordered shift: 
C    (calc) ... [send/recv]|[recv/send]/(calc)
C  IF (COMMOPT .EQ. 2) send early:
C    send/(calc) recv/send/(calc) ... recv/(calc)
C Communication protocol options (PROTOPT) for SHIFT include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5) .AND. (BUFFERS .GT. 3)
C    recv-ahead
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: ILTCAST
C calls: SR1, SR2, SR3, SRBEGIN, SREND
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C communication algorithm option
      INTEGER COMMOPT
C number of communication buffers (to use in recv-ahead algorithms)
      INTEGER BUFFERS
C communication protocol option
      INTEGER PROTOPT
C forcetype message type offset
      INTEGER FORCETYPE
C my processor id
      INTEGER ME
C processor ids for neighbors in logical ring
      INTEGER LEFT, RIGHT
C index of "me" in logical ring
      INTEGER MYINDEX
C message type offset to use in interprocessor communication
      INTEGER BASE
C current step of the algorithm (an increasing sequence)
      INTEGER STEP
C maximum (final) step in algorithm
      INTEGER MAXSTEP
C length of segments of the data vector
      INTEGER LTH
C local segment of the data vector
      REAL LOCVEC(LTH)
C
C     Input/Output
C
C external segments of the data vector
      REAL EXTVEC(LTH,BUFFERS)
C pointer to beginning of segment to be used next
      INTEGER CURBUF
C
C------ Local Variables ------------------------------------------------
C
C loop index
      INTEGER I
C buffer pointers
      INTEGER NXTBUF, PREVBUF
C vector length (in bytes)
      INTEGER MSGLTH
C order of send/recv in ordered sendrecv algorithm
      INTEGER ORDER
C                                                                              
C---- Executable Statements --------------------------------------------
C
C     Calculate message length.
      MSGLTH = RBYTES*LTH
C
C     Calculate order for ordered algorithms.
      ORDER = MOD(MYINDEX,2)
C
      IF (BUFFERS .EQ. 2) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       double buffer algorithms
C       (require at least two buffers to support nonblocking send 
C        variants - 1 buffer being sent, 1 buffer "active")
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C       Choose communication algorithm.
        IF ((COMMOPT .EQ. 0) .OR. (COMMOPT .EQ. 1)) THEN
C***********************************************************************
C         (0)   simple shift: (calc) ... send/recv/(calc)
C         (1) odd/even shift: (calc) ... [send/recv]|[recv/send]/(calc)
C***********************************************************************
C
          IF (STEP .EQ. 2) THEN
C
C           Send current data on and request new data.
            CALL SRBEGIN(COMMOPT, PROTOPT, FORCETYPE, ORDER, ME,
     &                   STEP+BASE, LEFT, MSGLTH, LOCVEC, RIGHT,
     &                   MSGLTH, EXTVEC(1,1))
            CURBUF = 1
C
          ELSEIF (STEP .GT. 2) THEN
C
            PREVBUF = CURBUF
            CURBUF  = MOD(CURBUF,2) + 1
C
C           Make sure that previous send buffer is available for 
C           reuse as a receive buffer.
            CALL SREND(COMMOPT, PROTOPT, FORCETYPE, ORDER, ME,
     &                 STEP-1+BASE, LEFT, RIGHT, MSGLTH, 
     &                 EXTVEC(1,PREVBUF))
C
C           Send current data on and request new data.
            CALL SRBEGIN(COMMOPT, PROTOPT, FORCETYPE, ORDER, ME,
     &                   STEP+BASE, LEFT, MSGLTH, EXTVEC(1,PREVBUF), 
     &                   RIGHT, MSGLTH, EXTVEC(1,CURBUF))
C
          ENDIF
C
          IF ((STEP .GE. 2) .AND. (STEP .EQ. MAXSTEP)) THEN
C
C             Make sure that the final sendrecv is complete.
              CALL SREND(COMMOPT, PROTOPT, FORCETYPE, ORDER, ME,
     &                   STEP+BASE, LEFT, RIGHT, MSGLTH,
     &                   EXTVEC(1,CURBUF))
C
          ENDIF
C
        ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C         send-ahead shift: send/(calc) recv/send/(calc) ... recv/(calc)
C***********************************************************************
C
          IF (STEP .EQ. 1) THEN
C
            IF (STEP .LT. MAXSTEP) THEN
C             Send current data on and request new data.
              CALL SRBEGIN(COMMOPT, PROTOPT, FORCETYPE, ORDER, ME,
     &                     STEP+1+BASE, LEFT, MSGLTH, LOCVEC, RIGHT,
     &                     MSGLTH, EXTVEC(1,1))
              CURBUF = 0
            ENDIF
C
          ELSE
C
            CURBUF = MOD(CURBUF,2) + 1
            NXTBUF = MOD(CURBUF,2) + 1
C
C           Make sure that previous send and receive buffers are
C           available for reuse.
            CALL SREND(COMMOPT, PROTOPT, FORCETYPE, ORDER, ME,
     &                 STEP+BASE, LEFT, RIGHT, MSGLTH, EXTVEC(1,CURBUF))
C
C           Send current data on and request new data.
            IF (STEP .LT. MAXSTEP) THEN
              CALL SRBEGIN(COMMOPT, PROTOPT, FORCETYPE, ORDER, ME,
     &                     STEP+1+BASE, LEFT, MSGLTH, EXTVEC(1,CURBUF), 
     &                     RIGHT, MSGLTH, EXTVEC(1,NXTBUF))
            ENDIF
C
          ENDIF
C
        ENDIF
C
      ELSEIF (BUFFERS .GT. 3) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       multiple (>3) buffer algorithms
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C       If first time, then post "buffers-3" receive requests
C       (eventually, buffers-2 outstanding requests, one buffer active, 
C       and one buffer being sent).
        IF (STEP .EQ. 1) THEN
C
C         Post all but first receive request (disabling safe forcetype
C         protocol - only needs to be used in the "first" request).
          DO I=2,BUFFERS-3
            CALL SR1(PROTOPT, FORCETYPE, .FALSE., ME, I+1+BASE, RIGHT, 
     &               MSGLTH, EXTVEC(1,I))
          ENDDO
C
C         Post first receive request (with safe forcetype protocol).
          CALL SR1(PROTOPT, FORCETYPE, .TRUE., ME, 2+BASE, RIGHT, 
     &             MSGLTH, EXTVEC(1,1))
C
        ENDIF
C
C       Choose communication algorithm.
        IF ((COMMOPT .EQ. 0) .OR. (COMMOPT .EQ. 1)) THEN
C***********************************************************************
C         (0) simple shift (with recv-ahead): 
C              (calc) ... send/recvend/(calc)
C         (1) odd/even shift (with recv-ahead):
C              (calc) ... [send/recvend]|[recvend/send]/(calc)
C***********************************************************************
C
          IF (STEP .EQ. 2) THEN
C
C           Send current data on (using safeforce protocol).
            CALL SR2(COMMOPT, PROTOPT, FORCETYPE, .TRUE., ORDER, ME, 
     &               2+BASE, LEFT, MSGLTH, LOCVEC,
     &               RIGHT, MSGLTH, EXTVEC(1,1))
            CURBUF = 1
C
          ELSEIF (STEP .GT. 2) THEN
C
C           Make sure that previous send buffer is available for 
C           reuse as a receive buffer.
            CALL SR3(COMMOPT, PROTOPT, FORCETYPE, ME, STEP-1+BASE, 
     &               RIGHT, MSGLTH, EXTVEC(1,CURBUF))
C
C           Send current data on.
            NXTBUF = MOD(CURBUF,BUFFERS) + 1
            IF (STEP .GE. BUFFERS-1) THEN
C             Using safeforce protocol.
              CALL SR2(COMMOPT, PROTOPT, FORCETYPE, .TRUE., ORDER, ME,
     &                 STEP+BASE, LEFT, MSGLTH, EXTVEC(1,CURBUF),
     &                 RIGHT, MSGLTH, EXTVEC(1,NXTBUF))
            ELSE
C             Not using safeforce protocol.
              CALL SR2(COMMOPT, PROTOPT, FORCETYPE, .FALSE., ORDER, ME,
     &                 STEP+BASE, LEFT, MSGLTH, EXTVEC(1,CURBUF),
     &                 RIGHT, MSGLTH, EXTVEC(1,NXTBUF))
            ENDIF
            CURBUF = NXTBUF
C
          ENDIF
C
C         Make sure that the final sendrecv is complete.
          IF ((STEP .GE. 2) .AND. (STEP .EQ. MAXSTEP)) THEN
            CALL SR3(COMMOPT, PROTOPT, FORCETYPE, ME, STEP+BASE, 
     &               RIGHT, MSGLTH, EXTVEC(1,CURBUF))
          ENDIF
C
        ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C         send-ahead shift (with recv-ahead):
C          send/(calc) recvend/send/(calc) ... recvend/(calc)
C***********************************************************************
C
          IF (STEP .EQ. 1) THEN
C
            IF (STEP .LT. MAXSTEP) THEN
C
C             Send current data on.
              CALL SR2(COMMOPT, PROTOPT, FORCETYPE, .TRUE., ORDER, ME, 
     &                 2+BASE, LEFT, MSGLTH, LOCVEC, RIGHT, MSGLTH,
     &                 EXTVEC(1,1)) 
C
C             Initialize buffer pointers.
              CURBUF = 0
              NXTBUF = 1
C
            ENDIF
C
          ELSE
C
            CURBUF = MOD(CURBUF,BUFFERS) + 1
C
C           Make sure that previous send and receive buffers are
C           available for reuse.
            CALL SR3(COMMOPT, PROTOPT, FORCETYPE, ME, STEP+BASE, 
     &               RIGHT, MSGLTH, EXTVEC(1,CURBUF))
C
C           Send current data on.
            IF (STEP .LT. MAXSTEP) THEN
              NXTBUF = MOD(CURBUF,BUFFERS) + 1
              IF (STEP .GE. BUFFERS-2) THEN
C               Using safeforce protocol.
                CALL SR2(COMMOPT, PROTOPT, FORCETYPE, .TRUE., ORDER, ME,
     &                   STEP+1+BASE, LEFT, MSGLTH, EXTVEC(1,CURBUF), 
     &                   RIGHT, MSGLTH, EXTVEC(1,NXTBUF))
              ELSE
C               Not using safeforce protocol.
                CALL SR2(COMMOPT, PROTOPT, FORCETYPE, .FALSE., ORDER,
     &                   ME, STEP+1+BASE, LEFT, MSGLTH,
     &                   EXTVEC(1,CURBUF), RIGHT, MSGLTH,
     &                   EXTVEC(1,NXTBUF)) 
              ENDIF
            ENDIF
C
          ENDIF
C
        ENDIF
C
C***********************************************************************
C       Post next recv-ahead receive request.
C       (Note 1: This is unsafe unless (BUFFERS .GT. 3). If 
C        (BUFFERS .EQ. 3), then next recv-ahead will receive into a
C        buffer that may not be finished sending yet (in nonblocking
C        send variant).) 
C       (Note 2: If (BUFFERS .EQ. MAXSTEP+1), then first BUFFERS-2
C        requests are all that are needed by shift algorithm, and the
C        following test is never satisfied (no more requests made).
C        If (BUFFERS .EQ. MAXSTEP), then this test succeeds once, but
C        only need MAXSTEP-1 buffers, so do not "wrap around" (i.e. no
C        reuse of old buffers). If (BUFFERS .EQ. MAXSTEP-1), then this
C        test succeeds twice, but only need MAXSTEP-1 buffers, so no
C        reuse of old buffers.) 
C***********************************************************************
C
        IF (STEP-2+BUFFERS .LE. MAXSTEP) THEN
          NXTBUF = MOD(STEP-4+BUFFERS,BUFFERS) + 1
          CALL SR1(PROTOPT, FORCETYPE, .TRUE., ME, 
     &             (STEP-2+BUFFERS)+BASE, RIGHT, MSGLTH, 
     &             EXTVEC(1,NXTBUF))
        ENDIF
C
      ELSE
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       illegal number of buffers specified
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
        WRITE(0,100) ME, BUFFERS
  100   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE SHIFT ',/,
     &          ' ILLEGAL COMMUNICATION BUFFER SIZE SPECIFIED',/,
     &          ' PROCID = ',I4,' BUFFERS = ',I4)
        STOP
C
      ENDIF
C
      RETURN
      END

