C#######################################################################
C PSTSWM Version 1.0 (8/1/93)                                          #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C sendrecv.f                                                           C
C                                                                      C
C The following routines implement the following low level             C
C interprocessor communication commands used in PSTSWM:                C
C                                                                      C
C SENDRECV - used in algorithms where sends and receives come in       C
C            logical pairs, for example, when shifting data around a   C
C            ring.                                                     C
C SRBEGIN  - used when initiating a SENDRECV                           C
C SREND    - used when completing a SENDRECV initiated by              C
C            SRBEGIN                                                   C
C SR1      - first of three routines which implement SENDRECV          C
C SR2      - second of three routines which implement SENDRECV         C
C SR3      - third of three routines which implement SENDRECV          C
C                                                                      C
C These routines all have the following communication options:         C
C  if (COMMOPT .EQ. 0) simple                                          C
C  if (COMMOPT .EQ. 1) ordered                                         C
C They also have the following communication protocol options:         C
C  if (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send                 C
C  if (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive              C
C  if (PROTOPT .EQ. 4 .OR. 5)         forcetype                        C
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous                C
C                                                                      C
C SRBEGIN/SREND and SR1/SR2/SR3 represent different ways of            C
C partitioning the basic SENDRECV routine                              C
C                                                                      C
C Use of SRBEGIN/SREND and SR1/SR2/SR3 allows receives used in         C
C SENDRECV to be posted ahead of time, and sends and/or receives       C
C to be completed just before they are needed.                         C
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SENDRECV(COMMOPT, PROTOPT, FORCETYPE, ORDER, ME, BASE,
     &                    SNDDEST, SNDLTH, SNDMSG, RCVSRC, RCVLTH,
     &                    RCVMSG)
C
C This subroutine sends the message in the SNDMSG buffer to SNDDEST,
C and receives a message into RCVMSG. 
C
C Communication options (COMMOPT) for SENDRECV include:
C  if (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  if (COMMOPT .EQ. 1) ordered SENDRECV: [send/recv]|[recv/send]
C Communication protocol options (PROTOPT) for SENDRECV include:
C  if (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  if (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  if (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  if (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: RING1
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C Communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C forcetype message type offset
      INTEGER FORCETYPE
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message type offset 
C (should be > NPROCS to guarantee correctness)
      INTEGER BASE
C message destination
      INTEGER SNDDEST
C length of message to be sent
      INTEGER SNDLTH
C message to be sent
      CHARACTER*1 SNDMSG(SNDLTH)
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      CHARACTER*1 RCVMSG(RCVLTH)
C
C---- Local Variables --------------------------------------------------
C
C buffer for protocol messages
      CHARACTER*1 BUFTMP
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple SENDRECV: send/recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           simple SENDRECV
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
          ELSE
C           Do not block for send, increasing odds that receive will
C           be posted before the message arrives.
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL SENDEND0(BASE+ME)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C    
C         IF (PROTOPT .EQ. 2) THEN
C           Post receive before send, increasing odds that receive will 
C           be posted before message arrives.
            CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECVEND0(BASE+RCVSRC)
C         ELSE
C           This case is "included" for completeness. There is no
C           advantage to using sendbegin/end here, so code is commented
C           out. There is an advantage to this case in SRBEGIN, for
C           example. 
C           CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
C           CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
C           CALL RECVEND0(BASE+RCVSRC)
C           CALL SENDEND0(BASE+ME)
C         ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C    
C         IF (PROTOPT .EQ. 4) THEN
C           Post receive before send to allow use of forcetypes.
            CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
            CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
C         ELSE
C           This case is "included" for completeness. There is no
C           advantage to using sendbegin/end here, so code is commented
C           out. There is an advantage to this case in SRBEGIN, for
C           example. 
C           CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
C           CALL SEND0(BUFTMP, 1, ME, RCVSRC)
C           CALL RECV0(BUFTMP, 1, SNDDEST)
C           CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
C           CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
C           CALL SENDEND0(BASE+ME+FORCETYPE)
C         ENDIF
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SENDRECV:',/,
     &             ' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered SENDRECV:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C
C           ordered SENDRECV
            IF (ORDER .EQ. 1) THEN
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            ELSE
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
          ELSE
C
C           Do not block for initial send, increasing odds that receive 
C           will be posted before message arrives.
            IF (ORDER .EQ. 1) THEN
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SENDEND0(BASE+ME)
            ELSE
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
C         IF (PROTOPT .EQ. 2) THEN
C
C           Post receive before initial send, increasing odds that
C           receive will be posted before message arrives.
            IF (ORDER .EQ. 1) THEN
              CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECVEND0(BASE+RCVSRC)
            ELSE
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
C         ELSE
C
C           This case is "included" for completeness. There is no
C           advantage to using sendbegin/end here, so code is commented
C           out. There is an advantage to this case in SRBEGIN, for
C           example. 
C           IF (ORDER .EQ. 1) THEN
C             CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
C             CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
C             CALL RECVEND0(BASE+RCVSRC)
C             CALL SENDEND0(BASE+ME)
C           ELSE
C             CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
C             CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
C           ENDIF
C
C         ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
C         IF (PROTOPT .EQ. 4) THEN
C
C           Post receive before send to allow use of forcetypes.
            IF (ORDER .EQ. 1) THEN
              CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
              CALL RECV0(BUFTMP, 1, SNDDEST)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
              CALL SEND0(BUFTMP, 1, ME, RCVSRC)
              CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
            ELSE
              CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
              CALL SEND0(BUFTMP, 1, ME, RCVSRC)
              CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
              CALL RECV0(BUFTMP, 1, SNDDEST)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
            ENDIF
C
C         ELSE
C
C           This case is "included" for completeness. There is no
C           advantage to using sendbegin/end here, so code is commented
C           out. There is an advantage to this case in SRBEGIN, for
C           example. 
C           IF (ORDER .EQ. 1) THEN
C             CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
C             CALL RECV0(BUFTMP, 1, SNDDEST)
C             CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, 
C    &                        SNDDEST)
C             CALL SEND0(BUFTMP, 1, ME, RCVSRC)
C             CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
C             CALL SENDEND0(BASE+ME+FORCETYPE)
C           ELSE
C             CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
C             CALL SEND0(BUFTMP, 1, ME, RCVSRC)
C             CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
C             CALL RECV0(BUFTMP, 1, SNDDEST)
C             CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
C           ENDIF
C
C         ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         synchronous ordered SENDRECV 
          IF (ORDER .EQ. 1) THEN
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
          ELSE
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined SENDRECV option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SENDRECV:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SRBEGIN(COMMOPT, PROTOPT, FORCETYPE, ORDER, ME, BASE, 
     &                   SNDDEST, SNDLTH, SNDMSG, RCVSRC, RCVLTH, 
     &                   RCVMSG)
C
C This subroutine begins a SENDRECV operation that will be completed by
C a SREND. It initiates the sending of the message in the SNDMSG 
C buffer to SNDDEST, and the receiving of a message into RCVMSG. 
C Everything is completed except the sendend in the nonblocking
C send option, and the recv or recvend in the delayed-recv option.
C
C Communication options (COMMOPT) for SRBEGIN/END include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: [send/recv]|[recv/send]
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SRBEGIN/END include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: SHIFT, SHIFTSUM
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C forcetype message type offset
      INTEGER FORCETYPE
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message type offset 
C (should be > NPROCS to guarantee correctness)
      INTEGER BASE
C message destination
      INTEGER SNDDEST
C length of message to be sent
      INTEGER SNDLTH
C message to be sent
      CHARACTER*1 SNDMSG(SNDLTH)
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      CHARACTER*1 RCVMSG(RCVLTH)
C
C---- Local Variables --------------------------------------------------
C
C buffer for protocol messages
      CHARACTER*1 BUFTMP
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple SENDRECV: send/recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           simple SENDRECV
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
          ELSE
C           Do not block for send, increasing odds that receive will
C           be posted before message arrives
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C    
          IF (PROTOPT .EQ. 2) THEN
C           Post receive before send, increasing odds that receive will
C           be posted before message arrives.
            CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECVEND0(BASE+RCVSRC)
          ELSE
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECVEND0(BASE+RCVSRC)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C    
          IF (PROTOPT .EQ. 4) THEN
C           Post receive before send to allow use of forcetypes.
            CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
            CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
          ELSE
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
            CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
          ENDIF
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SRBEGIN:',/,
     &             ' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered SENDRECV:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C
C           ordered SENDRECV
            IF (ORDER .EQ. 1) THEN
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            ELSE
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
          ELSE
C
C           Do not block for initial send, increasing odds that receive 
C           will be posted before message arrives.
            IF (ORDER .EQ. 1) THEN
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            ELSE
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
          IF (PROTOPT .EQ. 2) THEN
C
C           Post receive before initial send, increasing odds that receive 
C           will be posted before message arrives.
            IF (ORDER .EQ. 1) THEN
              CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECVEND0(BASE+RCVSRC)
            ELSE
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
          ELSE
C
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            IF (ORDER .EQ. 1) THEN
              CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECVEND0(BASE+RCVSRC)
            ELSE
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C
C           Post receive before initial send, increasing odds that receive 
C           will be posted before message arrives.
            IF (ORDER .EQ. 1) THEN
              CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
              CALL RECV0(BUFTMP, 1, SNDDEST)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
              CALL SEND0(BUFTMP, 1, ME, RCVSRC)
              CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
            ELSE
              CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
              CALL SEND0(BUFTMP, 1, ME, RCVSRC)
              CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
              CALL RECV0(BUFTMP, 1, SNDDEST)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
            ENDIF
C
          ELSE
C
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            IF (ORDER .EQ. 1) THEN
              CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
              CALL RECV0(BUFTMP, 1, SNDDEST)
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, 
     &                        SNDDEST)
              CALL SEND0(BUFTMP, 1, ME, RCVSRC)
              CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
            ELSE
              CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
              CALL SEND0(BUFTMP, 1, ME, RCVSRC)
              CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
              CALL RECV0(BUFTMP, 1, SNDDEST)
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, 
     &                        SNDDEST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         synchronous ordered SENDRECV 
          IF (ORDER .EQ. 1) THEN
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
          ELSE
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C       delayed-recv SENDRECV: send ... recv/send ... recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           delaying receive of SENDRECV
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
          ELSE
C           Do not block for send, enabling overlap of communication
C           with computation.
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C    
          IF (PROTOPT .EQ. 2) THEN
C           Post receive before send, increasing odds that receive will
C           be posted before message arrives.
            CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
          ELSE
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C    
          IF (PROTOPT .EQ. 4) THEN
C           Post receive before send to allow use of forcetypes.
            CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
          ELSE
C           Also do not block for send, enabling overlap of 
C           communication with computation.
            CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined SENDRECV option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SRBEGIN:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SREND(COMMOPT, PROTOPT, FORCETYPE, ORDER, ME, BASE, 
     &                 SNDDEST, RCVSRC, RCVLTH, RCVMSG)
C
C This subroutine completes the operation started in SRBEGIN. It
C completes any outstanding send and receive requests.
C
C Communication options (COMMOPT) for SRBEGIN/END include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: [send/recv]|[recv/send]
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SRBEGIN/END include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: SHIFT, SHIFTSUM
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C forcetype message type offset
      INTEGER FORCETYPE
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message type offset 
C (should be > NPROCS to guarantee correctness)
      INTEGER BASE
C message destination
      INTEGER SNDDEST
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      CHARACTER*1 RCVMSG(RCVLTH)
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm
C
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple SENDRECV: send/recv
C***********************************************************************
C
C       Choose communication protocol.
C
C       Complete send for nonblocking send protocols.
        IF ((PROTOPT .EQ. 1) .OR. (PROTOPT .EQ. 3)) THEN
          CALL SENDEND0(BASE+ME)
        ELSEIF (PROTOPT .EQ. 5) THEN
          CALL SENDEND0(BASE+ME+FORCETYPE)
        ELSEIF (PROTOPT .GT. 5) THEN
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SREND:',
     &           /,' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered SENDRECV:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
C
C       Complete send for nonblocking send protocols.
        IF ((PROTOPT .EQ. 1) .OR. (PROTOPT .EQ. 3)) THEN
          CALL SENDEND0(BASE+ME)
        ELSEIF (PROTOPT .EQ. 5) THEN
          CALL SENDEND0(BASE+ME+FORCETYPE)
        ELSEIF (PROTOPT .GT. 6) THEN
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C       delayed-recv SENDRECV: send ... recv/send ... recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           Delay receive of SENDRECV.
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
          ELSE
C           Also complete send.
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL SENDEND0(BASE+ME)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
          IF (PROTOPT .EQ. 2) THEN
C           Complete receive.
            CALL RECVEND0(BASE+RCVSRC)
          ELSE
C           Also complete send.
            CALL RECVEND0(BASE+RCVSRC)
            CALL SENDEND0(BASE+ME)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C           Complete forcetype receive.
            CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
          ELSE
C           Also complete forcetype send.
            CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
            CALL SENDEND0(BASE+ME+FORCETYPE)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined SENDRECV option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SREND:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SR1(PROTOPT, FORCETYPE, SAFEFORCE, ME, BASE, RCVSRC, 
     &               RCVLTH, RCVMSG)
C
C This subroutine begins a SENDRECV operation which will be completed 
C by SR2 and SR3. It posts a receive and sends handshaking messages 
C when forcetypes are used.
C
C Communication options (COMMOPT) for SR1/SR2/SR3 include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: 
C    IF (ORDER .EQ. 1) recv/send
C    IF (ORDER .NE. 1) recv/send
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SR1/SR2/SR3 include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: RINGSUM, SHIFT, SHIFTSUM
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C Communication protocol option 
      INTEGER PROTOPT
C forcetype message type offset
      INTEGER FORCETYPE
C Send handshaking message to guarantee correctness of using forcetype
C protocol? (not necessary if user has other guarantees)
      LOGICAL SAFEFORCE
C processor id
      INTEGER ME
C message type offset 
C (should be > NPROCS to guarantee correctness)
      INTEGER BASE
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received (eventually)
      CHARACTER*1 RCVMSG(RCVLTH)
C
C---- Local Variables --------------------------------------------------
C
C buffer for protocol messages
      CHARACTER*1 BUFTMP
C
C---- Executable Statements --------------------------------------------
C
C     (All communication algorithm options have the save code.)
C
C     Choose communication protocol.
      IF (PROTOPT .LE. 1) THEN
C       This procotol does not use nonblocking receive.
      ELSEIF (PROTOPT .LE. 3) THEN
C       Post receive before send, increasing odds that receive will
C       be posted before message arrives.
        CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC)
      ELSEIF (PROTOPT .LE. 5) THEN
C       Post receive before send to allow use of forcetypes.
        CALL RECVBEGIN0(RCVMSG, RCVLTH, BASE+RCVSRC+FORCETYPE)
        IF (SAFEFORCE) CALL SEND0(BUFTMP, 1, ME, RCVSRC)
      ELSEIF (PROTOPT .GT. 6) THEN
        WRITE (0,901) PROTOPT
  901   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR1:',
     &         /,' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &           ' PROTOPT = ',I3)
        STOP                                                   
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SR2(COMMOPT, PROTOPT, FORCETYPE, SAFEFORCE, ORDER, ME, 
     &               BASE, SNDDEST, SNDLTH, SNDMSG, RCVSRC, RCVLTH,
     &               RCVMSG)
C
C This subroutine continues the SENDRECV operation begun in SR1.
C It initiates the send and (sometimes) waits for the receive
C to complete.
C
C Communication options (COMMOPT) for SR1/SR2/SR3 include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: 
C    IF (ORDER .EQ. 1) send/recv
C    IF (ORDER .NE. 1) recv/send
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SR1/SR2/SR3 include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: RINGSUM, SHIFT, SHIFTSUM
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C Communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C forcetype message type offset
      INTEGER FORCETYPE
C Using handshaking message to guarantee correctness of using forcetype
C protocol? (not necessary if user has other guarantees)
      LOGICAL SAFEFORCE
C order of send/recv
      INTEGER ORDER
C processor id
      INTEGER ME
C message type offset 
C (should be > NPROCS to guarantee correctness)
      INTEGER BASE
C message destination
      INTEGER SNDDEST
C length of message to be sent
      INTEGER SNDLTH
C message to be sent
      CHARACTER*1 SNDMSG(SNDLTH)
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      CHARACTER*1 RCVMSG(RCVLTH)
C
C---- Local Variables --------------------------------------------------
C
C buffer for protocol messages
      CHARACTER*1 BUFTMP
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm.
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple SENDRECV: send/recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           simple SENDRECV
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
          ELSE
C           Do not block for send, enabling overlap of communication
C           with computation.
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
          IF ((PROTOPT .EQ. 0) .OR. (PROTOPT .EQ. 2)) THEN
C           Complete outstanding receive.
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECVEND0(BASE+RCVSRC)
          ELSE
C           Do not block for send, enabling overlap of communication
C           with computation.
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL RECVEND0(BASE+RCVSRC)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C           Send and receive forcetype messages.
            IF (SAFEFORCE) CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
            CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
          ELSE
C           Do not block for send, enabling overlap of communication
C           with computation.
            IF (SAFEFORCE) CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
            CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
          ENDIF
C
        ELSE
C
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR2:',
     &           /,' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered SENDRECV:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C
C           ordered SENDRECV
            IF (ORDER .EQ. 1) THEN
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            ELSE
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
          ELSE
C
C           Do not block for send, enabling overlap of communication
C           with computation.
            IF (ORDER .EQ. 1) THEN
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            ELSE
              CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
          IF ((PROTOPT .EQ. 0) .OR. (PROTOPT .EQ. 2)) THEN
C
C           ordered SENDRECV
            IF (ORDER .EQ. 1) THEN
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECVEND0(BASE+RCVSRC)
            ELSE
              CALL RECVEND0(BASE+RCVSRC)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
          ELSE
C
C           Do not block for send, enabling overlap of communication
C           with computation.
            IF (ORDER .EQ. 1) THEN
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
              CALL RECVEND0(BASE+RCVSRC)
            ELSE
              CALL RECVEND0(BASE+RCVSRC)
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C
C           Send and receive forcetype messages.
            IF (ORDER .EQ. 1) THEN
              IF (SAFEFORCE) CALL RECV0(BUFTMP, 1, SNDDEST)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
              CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
            ELSE
              CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
              IF (SAFEFORCE) CALL RECV0(BUFTMP, 1, SNDDEST)
              CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
            ENDIF
C
          ELSE
C
C           Do not block for send, enabling overlap of communication
C           with computation.
            IF (ORDER .EQ. 1) THEN
              IF (SAFEFORCE) CALL RECV0(BUFTMP, 1, SNDDEST)
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, 
     &                        SNDDEST)
              CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
            ELSE
              CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
              IF (SAFEFORCE) CALL RECV0(BUFTMP, 1, SNDDEST)
              CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, 
     &                        SNDDEST)
            ENDIF
C
          ENDIF
C
        ELSEIF (PROTOPT .EQ. 6) THEN
C
C         synchronous ordered SENDRECV 
          IF (ORDER .EQ. 1) THEN
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
          ELSE
            CALL SEND0(BUFTMP, 1, ME, RCVSRC)
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C       delayed-recv SENDRECV: send ... recv
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 3) THEN
C
          IF ((PROTOPT .EQ. 0) .OR. (PROTOPT .EQ. 2)) THEN
C           send of SENDRECV
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
          ELSE
C           Do not block for send, enabling overlap of communication
C           with computation.
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME, SNDDEST)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C           Send forcetype message.
            IF (SAFEFORCE) CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SEND0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
          ELSE
C           Also do not block for send, enabling overlap of 
C           communication with computation,
            IF (SAFEFORCE) CALL RECV0(BUFTMP, 1, SNDDEST)
            CALL SENDBEGIN0(SNDMSG, SNDLTH, BASE+ME+FORCETYPE, SNDDEST)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined SENDRECV option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR2:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE SR3(COMMOPT, PROTOPT, FORCETYPE, ME, BASE, RCVSRC,
     &               RCVLTH, RCVMSG)
C
C This subroutine completes the SENDRECV operation begun in SR1 and
C SR2. It waits until the message requested in SR1 has arrived and
C the send request in SR2 has completed.
C
C Communication options (COMMOPT) for SR1/SR2/SR3 include:
C  IF (COMMOPT .EQ. 0) simple SENDRECV: send/recv
C  IF (COMMOPT .EQ. 1) ordered SENDRECV: 
C    IF (ORDER .EQ. 1) send/recv
C    IF (ORDER .NE. 1) recv/send
C  IF (COMMOPT .EQ. 2) delayed-recv SENDRECV: send ... recv
C Communication protocol options (PROTOPT) for SR1/SR2/SR3 include:
C  IF (PROTOPT .EQ. 1, 3, .OR. 5)     nonblocking send   
C  IF (PROTOPT .EQ. 2, 3, 4, .OR. 5)  nonblocking receive
C  IF (PROTOPT .EQ. 4 .OR. 5)         forcetype          
C  IF (PROTOPT .EQ. 6 .AND. COMMOPT .EQ. 1) synchronous  
C
C called by: RINGSUM, SHIFT, SHIFTSUM
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C communication algorithm option
      INTEGER COMMOPT
C communication protocol option 
      INTEGER PROTOPT
C forcetype message type offset
      INTEGER FORCETYPE
C processor id
      INTEGER ME
C message type offset 
C (should be > NPROCS to guarantee correctness)
      INTEGER BASE
C source of message to be received
      INTEGER RCVSRC
C length of message to be received
      INTEGER RCVLTH
C
C     Output
C
C message received
      CHARACTER*1 RCVMSG(RCVLTH)
C
C---- Executable Statements --------------------------------------------
C
C     Choose communication algorithm
C
      IF (COMMOPT .EQ. 0) THEN
C***********************************************************************
C       simple SENDRECV: send/recv
C***********************************************************************
C
C       Choose communication protocol.
C
C       Complete send for nonblocking send protocols.
        IF ((PROTOPT .EQ. 1) .OR. (PROTOPT .EQ. 3)) THEN
          CALL SENDEND0(BASE+ME)
        ELSEIF (PROTOPT .EQ. 5) THEN
          CALL SENDEND0(BASE+ME+FORCETYPE)
        ELSEIF (PROTOPT .GT. 5) THEN
          WRITE (0,901) COMMOPT, PROTOPT
  901     FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR3:',
     &           /,' UNKNOWN COMMUNICATION PROTOCOL SPECIFIED',/, 
     &             ' COMMOPT = ',I3, ' PROTOPT = ',I3)
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 1) THEN
C***********************************************************************
C       ordered SENDRECV:
C       IF (ORDER .EQ. 1) send/recv
C       IF (ORDER .NE. 1) recv/send
C       (Deadlock avoidance is the responsibility of the calling
C        program.)
C***********************************************************************
C
C       Choose communication protocol.
C
C       Complete send for nonblocking send protocols.
        IF ((PROTOPT .EQ. 1) .OR. (PROTOPT .EQ. 3)) THEN
          CALL SENDEND0(BASE+ME)
        ELSEIF (PROTOPT .EQ. 5) THEN
          CALL SENDEND0(BASE+ME+FORCETYPE)
        ELSEIF (PROTOPT .GT. 6) THEN
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
        ENDIF
C
      ELSEIF (COMMOPT .EQ. 2) THEN
C***********************************************************************
C       delayed-recv SENDRECV: recvbegin ... send ... recvend
C***********************************************************************
C
C       Choose communication protocol.
        IF (PROTOPT .LE. 1) THEN
C
          IF (PROTOPT .EQ. 0) THEN
C           Receive message.
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
          ELSE
C           Also complete send.
            CALL RECV0(RCVMSG, RCVLTH, BASE+RCVSRC)
            CALL SENDEND0(BASE+ME)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 3) THEN
C
          IF ((PROTOPT .EQ. 0) .OR. (PROTOPT .EQ. 2)) THEN
C           Complete receive.
            CALL RECVEND0(BASE+RCVSRC)
          ELSE
C           Also complete send.
            CALL RECVEND0(BASE+RCVSRC)
            CALL SENDEND0(BASE+ME)
          ENDIF
C
        ELSEIF (PROTOPT .LE. 5) THEN
C
          IF (PROTOPT .EQ. 4) THEN
C           Complete forcetype receive.
            CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
          ELSE
C           Also complete forcetype send.
            CALL RECVEND0(BASE+RCVSRC+FORCETYPE)
            CALL SENDEND0(BASE+ME+FORCETYPE)
          ENDIF
C
        ELSE
C
C         protocol error
          WRITE (0,901) COMMOPT, PROTOPT
          STOP                                                   
C
        ENDIF
C
      ELSE
C***********************************************************************
C       undefined SENDRECV option
C***********************************************************************
C
        WRITE (0,900) COMMOPT
  900   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE SR3:',/,
     &           ' UNKNOWN COMMUNICATION OPTION SPECIFIED',/, 
     &           ' COMMOPT = ',I3)                                 
        STOP                                                   
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
