C#######################################################################
C PSTSWM Version 1.0 (8/1/93)                                          #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
      SUBROUTINE RFTLON(ISIGN, NSPECIES, TRIGS, WS, XX)
C
C IF (ISIGN .EQ. -1), then this routine performs a block, in-place, real
C FFT of a distributed real array XX of the form:
C                (MXLLON_P,NLVER_P,NLLAT_P,NSPECIES)
C producing a complex array of the form:
C                (MXLFC_S,NLVER_F,NLLAT_F,NSPECIES)
C If ISIGN=+1, then the inverse tranformation is performed.
C
C The algorithm used is determined by LTOPT and FTOPT.  Detailed
C descriptions of the various algorithms are provided in the text.
C In these descriptions, array dimensions that are distributed are 
C annotated with a "*".
C
C called by:  ADVECT, DZSC, EXPLIC, SHTRNS, SIMPLIC
C calls: PARFFT, REORDER1, REORDER2, REORDER3, REORDER4, REORDER5,
C        REORDER6, TRANSPOSE 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C parallel algorithm information
      INCLUDE 'algorithm.i'
C domain decomposition information
      INCLUDE 'physical.i'
      INCLUDE 'fourier.i'
      INCLUDE 'spectral.i'
C transform arrays
      INCLUDE 'trnsfm.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C direction of FFT (-1 is forward, +1 is inverse)
      INTEGER ISIGN
C number of species in XX
      INTEGER NSPECIES
C trigonometric function values used by PARFFT
      COMPLEX TRIGS(NTRIGS)
C
C     Input/Output
C
C On input, XX contains longitude data to transformed. On output, it
C contains the results of the transform.
C (Also organized as COMPLEX (MXLFC_S,NLVER_S,NLLAT_S,NSPECIES))
      REAL XX(MXLLON_P,NLVER_P,NLLAT_P,NSPECIES)
C
C     Work Space
C
C work array for communication buffers
C (big enough for REAL (MXLLON_P,NLVER_P,NLLAT_P,NSPECIES,BUFSWS2))
      REAL WS(MXLLON_P*NLVER_P*NLLAT_P*NSPECIES,2)
C
C---- Local Variables --------------------------------------------------
C
C bases for message types used in the distributed Fourier transform
      INTEGER BASE, BASE2
C
C---- External Functions -----------------------------------------------
C
C offset for message types
      EXTERNAL MSGBASE
      INTEGER MSGBASE
C
C---- Executable Statements --------------------------------------------
C
C     Pack first and last Fourier coefficients into the first complex
C     location before calculating the real inverse transform.
C     Both values are real, so no information is lost.
      IF (ISIGN .EQ. 1) THEN
        IF (FCDEX_S .EQ. 0) THEN
          CALL RFTPACK(2*NLFC_S, NLVER_S*NLLAT_S*NSPECIES, 2*MXLFC_S, 
     &                 XX)
        ENDIF
      ENDIF
C
      IF ((FTOPT .EQ. 0) .AND. (LTOPT .EQ. 0)) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       distributed FFT and distributed LT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C       Set message type offsets.
        BASE  = MSGBASE()
        BASE2 = MSGBASE()
C
        IF (ISIGN .EQ. -1) THEN
C         Forward distributed FFT
          CALL PARRFT(COMMFFT, PROTFFT, FORCETYPE, NPFC_F, MAPFC_F,
     &                FCDEX_F, BASE, BASE2, -1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
        ELSE
C         Inverse distributed FFT
          CALL PARRFT(COMMIFT, PROTIFT, FORCETYPE, NPFC_F, MAPFC_F,
     &                FCDEX_F, BASE, BASE2, +1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
        ENDIF
C
      ELSEIF(FTOPT .EQ. 1 .AND. LTOPT .EQ. 1) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       transpose FFT and transpose LT
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
        IF (ISIGN .EQ. -1) THEN
C         Forward Transpose FFT/Transpose LT:
C         1) The input real array (XX) of the form
C             (MXLLON_P*,NLVER_P,NLLAT_P*,NSPECIES)    
C            is reordered to give an array (1st half of WS) with
C            the form 
C             (NLLON_P*,NLLAT_P*,NSPECIES,NLVER_P).
C         2) This array is transposed to give (XX)
C             (MXLLON_F,NLVER_F*,NLLAT_F*,NSPECIES).   
C         3) This array is "FFT"ed to produce the output (XX)
C             (MXLFC_F,NLVER_F*,NLLAT_F*,NSPECIES).  
C         4) This is reordered to give (1st half of WS)
C             (NLLAT_F*,NLVER_F*,NSPECIES,NLFC_F-1).    
C         5) This is transposed to produce the output complex array 
C            (XX) of the form
C             (MXLFC_S*,NLVER_S*,NLLAT_S,NSPECIES).
C
          IF(NPLON_P .GT. 1) THEN
            CALL REORDER1(XX, WS, MXLLON_P, NLLON_P, NLVER_P,
     &                    NLLAT_P*NSPECIES)
            CALL TRANSPOSE(COMMFFT, BUFSFFT-1, PROTFFT, FORCETYPE,
     &                     NPLON_P, MAPLON_P, LONDEX_P, MSGBASE(),
     &                     -1, 1, NLON, NVER, NLLAT_P, NSPECIES,
     &                     NLLON_P, NLVER_F, MXLLON_F, WS,
     &                     WS(1,2), XX)
          ENDIF
C
C         serial real Fourier transform, thus message type offsets
C         are not necessary (and are set to 0)
          CALL PARRFT(COMMFFT, PROTFFT, FORCETYPE, NPFC_F, MAPFC_F,
     &                FCDEX_F, 0, 0, -1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
          IF(NPFC_S .GT. 1) THEN
            CALL REORDER3(XX, WS, MXLFC_F, NLFC_F, NLVER_F, NLLAT_F,
     &                    NSPECIES) 
            CALL TRANSPOSE(COMMFLT, BUFSFLT-1, PROTFLT, FORCETYPE,
     &                     NPFC_S, MAPFC_S, FCDEX_S, MSGBASE(),
     &                     -2, 2, NLAT, NFC-1, NLVER_F, NSPECIES,
     &                     NLLAT_F, NLFC_S-1, MXLFC_S, WS,
     &                     WS(1,2), XX)
          ENDIF
C
        ELSE
C         Inverse Transpose FFT/Transpose LT:
C         1) The input complex array (XX)
C             (MXLFC_S*,NLVER_S*,NLLAT_S,NSPECIES)
C            is reordered to the form (1st half of WS)
C             (NLFC_S-1*,NLVER_S*,NSPECIES,NLLAT_S).
C         2) This is transposed to give an array (XX) of the form
C             (MXLFC_F,NLVER_F*,NLLAT_F*,NSPECIES).   
C         3) This is "FFT"ed to give the real array (XX)
C             (MXLLON_F,NLVER_F*,NLLAT_F*,NSPECIES).    
C         4) This is reordered to give an array (1st half of WS)
C             (NLVER_F*,NLLAT_F*,NSPECIES,NLLON_F).
C         5) This array is transposed to give the output complex array
C            (XX) of the form
C             (MXLLON_P*,NLVER_P,NLLAT_P*,NSPECIES).
C
          IF(NPFC_S .GT. 1) THEN
            CALL REORDER4(XX, WS, MXLFC_S, NLFC_S, NLVER_S, NLLAT_S, 
     &                    NSPECIES)
            CALL TRANSPOSE(COMMILT, BUFSILT-1, PROTILT, FORCETYPE,
     &                     NPFC_S, MAPFC_S, FCDEX_S, MSGBASE(),
     &                     +2, 2, NFC-1, NLAT, NLVER_S, NSPECIES,
     &                     NLFC_S-1, NLLAT_F, MXLFC_F, WS,
     &                     WS(1,2), XX)
          ENDIF
C
C         serial real Fourier transform, thus message type offsets
C         are not necessary (and are set to 0)
          CALL PARRFT(COMMIFT, PROTIFT, FORCETYPE, NPFC_F, MAPFC_F,
     &                FCDEX_F, 0, 0, +1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
          IF(NPLON_P .GT. 1) THEN
            CALL REORDER2(XX, WS, MXLLON_F, NLLON_F,
     &                    NLVER_F*NLLAT_F*NSPECIES) 
            CALL TRANSPOSE(COMMIFT, BUFSIFT-1, PROTIFT, FORCETYPE,
     &                     NPLON_P, MAPLON_P, LONDEX_P, MSGBASE(),
     &                     +1, 1, NVER, NLON, NLLAT_F, NSPECIES,
     &                     NLVER_F, NLLON_P, MXLLON_P, WS,
     &                     WS(1,2), XX)
          ENDIF
C
        ENDIF
C
      ELSEIF(FTOPT .EQ. 1 .AND. LTOPT .EQ. 0) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       transpose FFT and distributed LT.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
        IF (ISIGN .EQ. -1) THEN
C         Forward Transpose FFT/Distributed LT:
C         1) The input real array (XX) with form
C             (MXLLON_P*,NLVER_P,NLLAT_P*,NSPECIES)
C            is reordered to give an array (1st half of WS) with the
C            form 
C             (NLLON_P*,NLLAT_P*,NSPECIES,NLVER_P).
C         2) This array is transposed to give (XX)
C             (MXLLON_F,NLVER_F*,NLLAT_F*,NSPECIES).  
C         3) This array is "FFT"ed to produce the output (XX)
C             (MXLFC_F,NLVER_F*,NLLAT_F*,NSPECIES)
C
          IF (NPLON_P .GT. 1) THEN
            CALL REORDER1(XX, WS, MXLLON_P, NLLON_P, NLVER_P,
     &                    NLLAT_P*NSPECIES)
            CALL TRANSPOSE(COMMFFT, BUFSFFT-1, PROTFFT, FORCETYPE,
     &                     NPLON_P, MAPLON_P, LONDEX_P, MSGBASE(),
     &                     -1, 1, NLON, NVER, NLLAT_P, NSPECIES,
     &                     NLLON_P, NLVER_F, MXLLON_F, WS,
     &                     WS(1,2), XX)
          ENDIF
C
C         serial real Fourier transform, thus message type offsets
C         are not necessary (and are set to 0)
          CALL PARRFT(COMMFFT, PROTFFT, FORCETYPE, NPFC_F, MAPFC_F,
     &                FCDEX_F, 0, 0, -1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
        ELSE
C         Inverse Transpose FFT/Distributed LT:
C         1) The input complex array (XX) with form
C             (MXLFC_F,NLVER_F*,NLLAT_F*,NSPECIES)
C            is "FFT"ed to give the real array (XX)
C             (MXLLON_F,NLVER_F*,NLLAT_F*,NSPECIES).
C         2) This is reordered to give an array (1st half of WS)
C             (NLVER_F*,NLLAT_F*,NSPECIES,NLLON_F).
C         3) This array is transposed to give the output complex array
C            (XX) of the form
C             (MXLLON_P*,NLVER_P,NLLAT_P*,NSPECIES).
C
C         serial real Fourier transform, thus message type offsets
C         are not necessary (and are set to 0)
          CALL PARRFT(COMMIFT, PROTIFT, FORCETYPE, NPFC_F, MAPFC_F,
     &                FCDEX_F, 0, 0, +1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
          IF (NPLON_P .GT. 1) THEN
            CALL REORDER2(XX, WS, MXLLON_F, NLLON_F,
     &                    NLVER_F*NLLAT_F*NSPECIES)
            CALL TRANSPOSE(COMMIFT, BUFSIFT-1, PROTIFT, FORCETYPE,
     &                     NPLON_P, MAPLON_P, LONDEX_P, MSGBASE(),
     &                     +1, 1, NVER, NLON, NLLAT_F, NSPECIES,
     &                     NLVER_F, NLLON_P, MXLLON_P, WS,
     &                     WS(1,2), XX)
          ENDIF
C
        ENDIF
C
      ELSEIF(FTOPT .EQ. 0 .AND. LTOPT .EQ. 1) THEN
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C       distributed FFT and transpose LT.
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C       Set message type offsets.
        BASE  = MSGBASE()
        BASE2 = MSGBASE()
C
        IF (ISIGN .EQ. -1) THEN
C         Forward Distributed FFT/Transpose LT:
C         1) The input real array (XX) with form
C             (MXLLON_P*,NLVER_P,NLLAT_P*,NSPECIES)
C            is "FFT"ed to give the complex array (XX)
C             (MXLFC_F*,NLVER_F,NLLAT_F*,NSPECIES).
C         2) This is reordered to give an array (1st half of WS)
C             (NLLAT_F*,NLFC_F-1*,NSPECIES,NLVER_F).
C         3) This is transposed to give the output complex array (XX)
C             (MXLFC_S*,NLVER_S*,NLLAT_S,NSPECIES)
C
          CALL PARRFT(COMMFFT, PROTFFT, FORCETYPE, NPFC_F, MAPFC_F,
     &                FCDEX_F, BASE, BASE2, -1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
          IF(NPVER_S .GT. 1) THEN
            CALL REORDER5(XX, WS, MXLFC_F, NLFC_F, NLVER_F, NSPECIES, 
     &                    NLLAT_F)
            CALL TRANSPOSE(COMMFLT, BUFSFLT-1, PROTFLT, FORCETYPE,
     &                     NPVER_S, MAPVER_S, VERDEX_S, MSGBASE(),
     &                     -3, 2, NLAT, NVER, NLFC_F-1, NSPECIES,
     &                     NLLAT_F, NLVER_S, MXLFC_S, WS,
     &                     WS(1,2), XX)
          ENDIF
C
        ELSE
C         Inverse Distributed FFT/Transpose LT:
C         1) The input complex array (XX) with form   
C             (MXLFC_S*,NLVER_S,NLLAT_S*,NSPECIES)
C            is reordered to give an array (1st half of WS) 
C             (NLVER_S,NLFC_S-1*,NSPECIES,NLLAT_S*).
C         2) This is transposed to give an array (XX)
C             (MXLFC_F,NLVER_F*,NLLAT_F*,NSPECIES)
C         3) This is "FFT"ed to give the output real array (XX)
C             (MXLLON_P,NLVER_P*,NLLAT_P*,NSPECIES).
C
          IF(NPVER_S .GT. 1) THEN
            CALL REORDER6(XX, WS, MXLFC_S, NLFC_F, NLVER_S, NLLAT_S,
     &                    NSPECIES)
            CALL TRANSPOSE(COMMILT, BUFSILT-1, PROTILT, FORCETYPE,
     &                     NPVER_S, MAPVER_S, VERDEX_S, MSGBASE(),
     &                     +3, 2, NVER, NLAT, NLFC_F-1, NSPECIES,
     &                     NLVER_S, NLLAT_F, MXLFC_F, WS,
     &                     WS(1,2), XX)
          ENDIF
C
          CALL PARRFT(COMMIFT, PROTIFT, FORCETYPE, NPFC_F, MAPFC_F,
     &                FCDEX_F, BASE, BASE2, +1, NLLON_F,
     &                NLVER_F*NLLAT_F*NSPECIES, MXLLON_F, IFAX,
     &                NTRIGS, TRIGS, WS, XX) 
C
        ENDIF
C
      ENDIF
C
C     Unpack the last Fourier coefficient from the imaginary part of 
C     the first complex location after calculating the real forward 
C     transform. Both values are real, so also zero out the imaginary
C     parts of both coefficients.
      IF (ISIGN .EQ. -1) THEN
        IF (FCDEX_S .EQ. 0) THEN
          CALL RFTUNPACK(2*NLFC_S, NLVER_S*NLLAT_S*NSPECIES, 
     &                   2*MXLFC_S, XX)
        ENDIF
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER1(FROM, TO, MXLLON, NLLON, NVER, COUNT)
C
C This routine reorders FROM(MXLLON,NVER,COUNT) into
C TO(NLLON,COUNT,NVER), where both arrays are real.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLLON, NLLON, NVER, COUNT
C array that is to be reordered
      REAL FROM(MXLLON,NVER,COUNT)
C
C     Output
C
C destination of reordered array
      REAL TO(NLLON,COUNT,NVER)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J, K
C
C---- Executable Statements --------------------------------------------
C
      DO K = 1,COUNT
        DO J = 1,NVER
          DO I = 1,NLLON
            TO(I,K,J) = FROM(I,J,K)
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER2(FROM, TO, MXLON, NLON, COUNT)
C
C This routine reorders FROM(MXLON,COUNT) into TO(COUNT,NLON), where
C both arrays are real.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLON, NLON, COUNT
C array that is to be reordered
      REAL FROM(MXLON,COUNT)
C
C     Output
C
C destination of reordered array
      REAL TO(COUNT,NLON)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J
C
C---- Executable Statements --------------------------------------------
C
      DO I = 1,NLON
        DO J = 1,COUNT
          TO(J,I) = FROM(I,J)
        ENDDO
      ENDDO
C      
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER3(FROM, TO, MXFC, NFC, NLVER, NLLAT, NSPECIES)
C
C This routine reorders FROM(MXFC,NLVER,NLLAT,NSPECIES) into
C TO(NLLAT,NLVER,NSPECIES,NFC-1), where both arrays are complex.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXFC, NFC, NLVER, NLLAT, NSPECIES
C array that is to be reordered
      COMPLEX FROM(MXFC,NLVER,NLLAT,NSPECIES)
C
C     Output
C
C destination of reordered array
      COMPLEX TO(NLLAT,NLVER,NSPECIES,NFC-1)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J, K, L
C
C---- Executable Statements --------------------------------------------
C
      DO L = 1,NSPECIES
        DO K = 1,NLLAT
          DO J = 1,NLVER
            DO I = 1,NFC-1
              TO(K,J,L,I) = FROM(I,J,K,L)
            ENDDO
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER4(FROM, TO, MXLFC, NLFC, NLVER, NLAT, NSPECIES)
C
C This routine reorders FROM(MXLFC,NLVER,NLAT,NSPECIES) into
C TO(NLFC-1,NLVER,NSPECIES,NLAT). Both arrays are complex.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLFC, NLFC, NLVER, NLAT, NSPECIES
C array that is to be reordered
      COMPLEX FROM(MXLFC,NLVER,NLAT,NSPECIES)
C
C     Output
C
C destination of reordered array
      COMPLEX TO(NLFC-1,NLVER,NSPECIES,NLAT)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J, K, L
C
C---- Executable Statements --------------------------------------------
C
      DO L = 1,NSPECIES
        DO K = 1,NLAT
          DO J = 1,NLVER
            DO I = 1,NLFC-1
              TO(I,J,L,K) = FROM(I,J,K,L)
            ENDDO
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER5(FROM, TO, MXLFC, NLFC, NLVER, NSPECIES, NLLAT)
C
C This routine reorders FROM(MXLFC,NLVER,NLLAT,NSPECIES) into
C TO(NLLAT,NLFC-1,NSPECIES,NLVER), where both arrays are complex.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLFC, NLFC, NLVER, NSPECIES, NLLAT
C array that is to be reordered
      COMPLEX FROM(MXLFC,NLVER,NLLAT,NSPECIES)
C
C     Output
C
C destination of reordered array
      COMPLEX TO(NLLAT,NLFC-1,NSPECIES,NLVER)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J, K, L
C
C---- Executable Statements --------------------------------------------
C
      DO L = 1,NSPECIES
        DO K = 1,NLLAT
          DO J = 1,NLVER
            DO I = 1,NLFC-1
              TO(K,I,L,J) = FROM(I,J,K,L)
            ENDDO
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE REORDER6(FROM, TO, MXLFC, NLFC, NLVER, NLAT, NSPECIES)
C
C This routine reorders FROM(MXLFC,NLVER,NLAT,NSPECIES) into
C TO(NLVER,NLFC-1,NSPECIES,NLAT). Both arrays are complex.
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C dimensions of input and output arrays
      INTEGER MXLFC, NLFC, NLVER, NLAT, NSPECIES
C array that is to be reordered
      COMPLEX FROM(MXLFC,NLVER,NLAT,NSPECIES)
C
C     Output
C
C destination of reordered array
      COMPLEX TO(NLVER,NLFC-1,NSPECIES,NLAT)
C
C---- Local Variables --------------------------------------------------
C
C loop indices
      INTEGER I, J, K, L
C
C---- Executable Statements --------------------------------------------
C
      DO L = 1,NSPECIES
        DO K = 1,NLAT
          DO J = 1,NLVER
            DO I = 1,NLFC-1
              TO(J,I,L,K) = FROM(I,J,K,L)
            ENDDO
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE RFTPACK(VECLTH, MVECS, JUMP, DATA)
C
C This routine copies element VECLTH-1 into element 2 for a sequence of 
C MVEC vectors. It is used to pack the first and last Fourier 
C coefficients into the first complex location before calculating the 
C real inverse Fourier transform. Both values are real, so no 
C information is lost. 
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C length of vector being modified
      INTEGER VECLTH
C number of vectors being modified
      INTEGER MVECS
C the offset between the start of successive vectors
      INTEGER JUMP
C
C     Input/Output
C
C data array to be modified
      REAL DATA(JUMP,MVECS)
C
C---- Local Variables --------------------------------------------------
C
C loop index
      INTEGER I
C
C---- Executable Statements --------------------------------------------
C
      DO I = 1,MVECS
        DATA(2,I) = DATA(VECLTH-1,I)
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE RFTUNPACK(VECLTH, MVECS, JUMP, DATA)
C
C This routine copies element 2 into element VECLTH-1 and zeroes out
C elements 2 and VECLTH for a sequence of MVEC vectors. It is used to 
C unpack the last Fourier coefficient from the imaginary part of the
C first coefficient after calculating the real forward Fourier 
C transform. Both values are real, so no information is lost. 
C
C called by: RFTLON
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C length of vector being modified
      INTEGER VECLTH
C number of vectors being modified
      INTEGER MVECS
C the offset between the start of successive vectors
      INTEGER JUMP
C
C     Input/Output
C
C data array to be modified
      REAL DATA(JUMP,MVECS)
C
C---- Local Variables --------------------------------------------------
C
C loop index
      INTEGER I
C
C---- Executable Statements --------------------------------------------
C
      DO I = 1,MVECS
        DATA(VECLTH-1,I) = DATA(2,I)
        DATA(VECLTH,I)   = 0.0
        DATA(2,I)        = 0.0
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
