C#######################################################################
C PSTSWM Version 1.0 (8/1/93)                                          #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
      PROGRAM PSTSWM
C
C This is the main program for a message-passing benchmark code 
C and parallel algorithm testbed that solves the nonlinear shallow 
C water equations on a sphere using the spectral transform method.
C It is based on the serial code STSWM 2.0 described in NCAR 
C technical note TN-343+STR "Description of a global shallow water 
C model based on the spectral transform method" by James J. Hack 
C and Ruediger Jakob. The embedded parallel algorithms and runtime 
C options are described in ORNL technical report ORNL/TM-12393 by 
C Patrick H. Worley and Ian T. Foster. The parallel algorithms are based 
C partly on previous work by John Drake, David Walker, and Patrick 
C Worley of Oak Ridge National Laboratory. Both the code development and
C the underlying parallel algorithm research were funded by the Computer 
C Hardware, Advanced Mathematics, and Model Physics (CHAMMP) program of
C the U.S. Department of Energy.
C
C The PICL communication and execution tracing library has been used
C to implement interprocessor communication and to collect 
C performance statistics. The interprocessor communication is well
C localized, and it is simple to modify PSTSWM to use a different
C message passing interface.
C     
C The main program calls INPUT to read problem and algorithm parameters
C and set up arrays for spectral transformations, and then calls
C INIT to set up the test case parameters. Routines ERRANL and
C NRGTCS are called once before the main timestepping loop for
C error normalization, once after the main timestepping for 
C calculating energetics data and errors, and periodically during 
C the timestepping, as requested. The prognostic fields are 
C initialized using routine ANLYTC, which provides the analytic
C solution. Each call to STEP advances the computed fields by a 
C timestep DT.
C     
C called by:
C calls: ANLYTC, DZSC, EPSLON, ERRANL, INIT, INPUT, NRGTCS, PSC,
C        SPCANL, SPEED, STEP, TDVMOD, TINIT, TEXIT, TOUTPUT, TSTART,
C        TSTOP, and PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C     
C---- Common Blocks ----------------------------------------------------
C     
C machine architecture information
      INCLUDE 'machine.i'
C problem resolution information
      INCLUDE 'problem.i'
C constants and timesteps
      INCLUDE 'consts.i'
C mountain data
      INCLUDE 'finit.i'
C time dependent fields
      INCLUDE 'tdvars.i'
C tracing  parameters
      INCLUDE 'tracing.i'
C transform arrays
      INCLUDE 'trnsfm.i'
C workspace
      INCLUDE 'wrkspc.i'
C     
C---- Local Variables ------------------------------------------------
C     
C single precision machine accuracy
      REAL EPS
C output and analysis flags
      LOGICAL ENERGY, CALLEA, CALLSA
C energetics and error analysis counters
      INTEGER ECTR, L2CTR
C max velocity and Courant number for parameter checking
      REAL MAXV, COUR
C
C---- External Functions -----------------------------------------------
C
C single precision machine epsilon function
      EXTERNAL EPSLON
      REAL EPSLON
C     
C---- Statement Functions ----------------------------------------------
C     
C time interval check
      REAL XTAU
      LOGICAL EVENT
      EVENT(XTAU) = MOD(NSTEP,MAX0(1,
     &     IFIX(XTAU * 3600.0/DT + 1000.0*EPS))) .EQ. 0
C     
C---- Executable Statements -------------------------------------------        
C     
C     Enable interprocessor communication.
      CALL OPEN0(NPROCS, ME, HOST)
C     
C     copyright message from stswm 2.0
      IF (ME .EQ. 0) WRITE(6,10)
 10   FORMAT(
     &/,'PARALLEL SPECTRAL TRANSFORM SHALLOW WATER MODEL, VERSION 1.0',
     &/,'(BASED ON SPECTRAL TRANSFORM SHALLOW WATER MODEL, VERSION 2.0',
     &/,'          COPYRIGHT (C) 1992',
     &/,'          UNIVERSITY CORPORATION FOR ATMOSPHERIC RESEARCH',
     &/,'          ALL RIGHTS RESERVED)',/)
C     
C     Compute single precision machine accuracy.
      EPS = EPSLON(1.0)
C
C     timestep counter
      NSTEP = 0
C     
C     model time
      TAU   = 0.0
C     
C     input routine:
C      Set up constants, read model parameters, and
C      set up arrays for spectral transform procedure.
      CALL INPUT(EPS)
C     
C     initialization routine for test cases 
      CALL INIT(EPS, WS(IALP), WS(IDALP), WS(IWTS), WS(IWTACSJ), 
     &          WS(ITRIGS), WS(IWS1), WS(IWS2), WS(IWS3), WS(IDSC), 
     &          WS(IZSC), WS(IPHICON), WS(IUCON), WS(IVCON), WS(IMOUNT), 
     &          WS(ITOPOSC), WS(IUIC12), WS(IVIC12), WS(IPIC12), 
     &          WS(IDIC12), WS(IEIC12))
C     
C     Initialize field variables at time = 0.0.
      CALL ANLYTC(0.0, ICOND, WS(IPHICON), WS(IUCON), WS(IMOUNT),
     &            WS(IUIC12), WS(IVIC12), WS(IPIC12), WS(IDIC12),
     &            WS(IEIC12), WS(ID), WS(IZ), WS(IH), WS(IU),
     &            WS(IV))
C     
C     Compute maximum wind speed, global mean geopotential, 
C     and Courant number.
      CALL SPEED(WS(IMOUNT), WS(IH), WS(IU), WS(IV), WS(IWS1), WS(IWS2),
     &           MAXV, PHIBAR, COUR)
C     
C     Write maximum wind speed and Courant number.
      IF (ME .EQ. 0) WRITE (6,129) MAXV, COUR
 129  FORMAT(' MAX. WIND      = ',1PE16.9,/,
     &       ' COURANT NUMBER = ',0PF6.4)
C     
C     warning for when timestep too long
C     (Courant number should be less than 1 at all times.)
      IF (COUR .GE. 1.0) THEN
        IF (ME .EQ. 0) WRITE(0,130) COUR
 130    FORMAT(/,' PSTSWM: WARNING FROM MAIN PROGRAM:',
     &         /,' TIMESTEP TOO LONG FOR EXPERIMENT',
     &         /,' COURANT NUMBER = ',0PF8.4)
      ENDIF
C     
C     Write mean geopotential.
      IF (ME .EQ. 0) WRITE(6,135) PHIBAR
 135  FORMAT(/,' GLOBAL MEAN STEADY GEOPOTENTIAL = ',1PE16.9)
C     
C     Compute initial conservation estimate.
      ECTR  = 1
      IF (EGYFRQ .LE. TAUE)
     &  CALL NRGTCS(WS(ID), WS(IZ), WS(IH), WS(IU), WS(IV), WS(IMOUNT),
     &              ECTR, WS(IWS1), WS(IWS2)) 
C     
C     Compute initial error estimate.
      L2CTR = 1
      IF (ERRFRQ .LE. TAUE)
     &  CALL ERRANL(WS(IPHICON), WS(IUCON), WS(IVCON), WS(IMOUNT),
     &              WS(IDIC12), WS(IEIC12), WS(IPIC12), WS(IUIC12),
     &              WS(IVIC12), WS(ID), WS(IZ), WS(IH), WS(IU), WS(IV),
     &              L2CTR, WS(IWS1), WS(IWS2))
C     
C     Compute initial spectral coefficients.
      CALL DZSC(WS(IALP), WS(IDALP), WS(IWTS), WS(IWTACSJ), WS(ITRIGS),
     &          WS(IU), WS(IV), WS(IWS1), WS(IWS2), WS(IWS3), WS(IDSC), 
     &          WS(IZSC))
      CALL PSC(PHIBAR, WS(IALP), WS(IWTS), WS(ITRIGS), WS(IMOUNT),
     &         WS(IH), WS(IWS1), WS(IWS2), WS(IWS3), WS(IPSC))
C     
C     Compute initial spectral analysis.
      IF (SPCFRQ .LE. TAUE) 
     &  CALL SPCANL(WS(IA2NNP1), WS(IDSC), WS(IZSC), WS(IPSC), WS(IWS1),
     &              WS(IWS2))
C     
C     spectral transform algorithm uses modified variables:
C      PHI  := G*(H-MOUNT)-PHIBAR    (use fluid depth)
C      UCOS := U*COS(RLAT)           (redefinition because of
C      VCOS := V*COS(RLAT)            multivalued u,v at pole)
      CALL TDVMOD(1, PHIBAR, -1, WS(IMOUNT), WS(ID), WS(IZ), WS(IH), 
     &            WS(IU), WS(IV), WS(IDSC), WS(IZSC), WS(IPSC), 
     &            WS(IDIV), WS(IZETA), WS(IPHI), WS(IUCOS), WS(IVCOS),
     &            WS(IDIVSC), WS(IZETASC), WS(IPHISC))
C
C     Prepare for performance data collection.
      CALL TINIT
C
C'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
C     main computational control
      DO WHILE ((TAU + EPS) .LT. TAUE)
C
C       Update time and iteration counters.
        NSTEP  = NSTEP + 1
        TAU    = NSTEP*DT/3600.0
C
C       Update analysis event switches.
        ENERGY = EVENT(EGYFRQ).AND.(.NOT. TIMING).AND.(.NOT. TRACING)
        CALLSA = EVENT(SPCFRQ).AND.(.NOT. TIMING).AND.(.NOT. TRACING)
        CALLEA = EVENT(ERRFRQ).AND.(.NOT. TIMING).AND.(.NOT. TRACING)
C
C       Compute next timestep (recording start and stop times).
        CALL TSTART
        CALL STEP(DT, WS(IALP), WS(IDALP), WS(IWTS), WS(IWTACSJ), 
     &            WS(IANNP1), WS(IA2NNP1),
     &            WS(ITRIGS), WS(IMOUNT), WS(ITOPOSC), 
     &            WS(IDIV), WS(IZETA), WS(IPHI), 
     &            WS(IUCOS), WS(IVCOS), WS(IWS1), WS(IWS2), WS(IWS3),
     &            WS(IDIVSC), WS(IZETASC), WS(IPHISC))
        CALL TSTOP
C
C       Uf necessary, transform back to standard variables.
        IF (ENERGY .OR. CALLSA .OR. CALLEA)
     &    CALL TDVMOD(LN, PHIBAR, +1, WS(IMOUNT), WS(ID), WS(IZ), 
     &                WS(IH), WS(IU), WS(IV), WS(IDSC), WS(IZSC),
     &                WS(IPSC), WS(IDIV), WS(IZETA), WS(IPHI),
     &                WS(IUCOS), WS(IVCOS), WS(IDIVSC), WS(IZETASC),
     &                WS(IPHISC))
C
C       Compute standard energetics information.
        IF (ENERGY) THEN 
          ECTR = ECTR + 1
          CALL NRGTCS(WS(ID), WS(IZ), WS(IH), WS(IU), WS(IV), 
     &                WS(IMOUNT), ECTR, WS(IWS1), WS(IWS2)) 
        ENDIF
C
C       Compute spectral analysis.
        IF (CALLSA)
     &    CALL SPCANL(WS(IA2NNP1), WS(IDSC), WS(IZSC), WS(IPSC),
     &                WS(IWS1), WS(IWS2))
C
C       Compute error analysis (compare with analytic solution).
        IF (CALLEA) THEN 
          L2CTR = L2CTR + 1
          CALL ERRANL(WS(IPHICON), WS(IUCON), WS(IVCON), WS(IMOUNT),
     &                WS(IDIC12), WS(IEIC12), WS(IPIC12), WS(IUIC12),
     &                WS(IVIC12), WS(ID), WS(IZ), WS(IH), WS(IU), 
     &                WS(IV), L2CTR, WS(IWS1), WS(IWS2))
        ENDIF      
C
      ENDDO
C
C'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
C     postprocessing code:
C     1) Finish tracing and timing logic.
      CALL TEXIT
C
C     2) Transform back to standard variables.
      CALL TDVMOD(LN, PHIBAR, +1, WS(IMOUNT), WS(ID), WS(IZ), WS(IH),
     &            WS(IU), WS(IV), WS(IDSC), WS(IZSC), WS(IPSC),
     &            WS(IDIV), WS(IZETA), WS(IPHI), WS(IUCOS),
     &            WS(IVCOS), WS(IDIVSC), WS(IZETASC), WS(IPHISC))
C
C     3) Calculate energetics.
      IF (EGYFRQ .LE. TAUE)
     &  CALL NRGTCS(WS(ID), WS(IZ), WS(IH), WS(IU), WS(IV), WS(IMOUNT),
     &              ECTR, WS(IWS1), WS(IWS2)) 
C
C     4) Calculate error measures.
      IF (ERRFRQ .LE. TAUE) THEN
        L2CTR = L2CTR + 1                                                      
        CALL ERRANL(WS(IPHICON), WS(IUCON), WS(IVCON), WS(IMOUNT),
     &              WS(IDIC12), WS(IEIC12), WS(IPIC12), WS(IUIC12),
     &              WS(IVIC12), WS(ID), WS(IZ), WS(IH), WS(IU), WS(IV),
     &              L2CTR, WS(IWS1), WS(IWS2))
      ENDIF
C
C     Output timing information.
      CALL TOUTPUT
C
C     normal termination of parallel shallow water model
      CALL CLOSE0
C
      STOP
      END                                                                       
