C#######################################################################
C PSTSWM Version 1.0 (8/1/93)                                          #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
      SUBROUTINE PRBINP
C                                                                              
C This routine inputs or determines necessary constants and problem and 
C output parameters.
C
C called by: INPUT
C calls: PICL routines
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C     
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C machine architecture information
      INCLUDE 'machine.i'
C problem resolution information
      INCLUDE 'problem.i'
C constants & timesteps
      INCLUDE 'consts.i'
C
C---- Local Variables --------------------------------------------------
C
C communication buffers
      REAL RBUF(11)
      INTEGER IBUF(12)
      CHARACTER CBUF*4
C temporary used in calculating the number of spectral coefficients for
C the given problem specifications
      INTEGER LMN
C minimum number of latitudes needed to guarantee the unaliased 
C evaluation of quadratic terms when using the spectral transform 
C method.
      INTEGER LATMIN
C base for message types used in broadcasting problem parameters
      INTEGER BASE
C
C---- External Functions -----------------------------------------------
C
C offset for message types
      EXTERNAL MSGBASE
      INTEGER MSGBASE
C
C---- Executable Statements --------------------------------------------
C
C     Initialize message type offset.
      BASE = MSGBASE()
C
      IF (ME .EQ. 0) THEN
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C Initialize data to default values.
C
C       The following physical constants are defined as:
C       A      => radius of the earth
C       OMEGA  => angular velocity of the earth
C       GRAV   => gravitational constant
C       HDC    => linear horizontal diffusion coefficient
C       ALPHA  => rotation angle of coordinate system
C
        A      = 6.37122E6
        OMEGA  = 7.292E-5
        GRAV   = 9.80616
        HDC    = 0.0E00
        ALPHA  = ATAN(1.0)
C
C       Other constants/variables required for model integration:
C       DT     => timestep in seconds
C       AFC    => asselin filter parameter
C       TAU    => current model time (in hours)
C       TAUE   => end of model run (in hours)
C       SITS   => logical flag for semi-implicit time differencing
C       NSTEP  => current model timestep (NSTEP=0 initially)
C       FORCED => logical flag .TRUE. if call to forcing in routine 
C                 COMP1
C       MOMENT => logical flag .TRUE. for momentum forcing
C       EGYFRQ => frequency of model energetics evaluation (in hours)
C       ERRFRQ => frequency of L2-error analysis (in hours)
C       SPCFRQ => frequency of spectral analysis (in hours)
C       ICOND  => type of initial conditions (default = 2)
C                 (are set up in subroutine INIT)
C       FTOPO  => logical flag .TRUE. if surface topography (case 5)
C
        DT     = 2400.0
        AFC    = 0.000
        TAUE   = 120.0
        SITS   = .TRUE.
        FORCED = .FALSE.
        MOMENT = .FALSE.
        EGYFRQ = 3.0
        ERRFRQ = 24.0
        SPCFRQ = 999.0
        ICOND  = 2
        FTOPO  = .FALSE.
C
C       Initialize (other) user selectable problem parameters.
C       See common block /PROBLEM/ (in problem.i) for definitions
C
        MM = MMX
        NN = NNX
        KK = KKX
        NLAT = NLATX
        NLON = NLONX
        NVER = NVERX
        NGRPHS = NGRPHSX
C
C       default experiment number
        CHEXP = '    '
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C       Read problem input data from external file.
        WRITE(6,200)
  200   FORMAT(/,' READING PARAMETERS FROM FILE problem:',/)
C
C       Open input file.
        OPEN(8, FILE='problem')
C
C       Read experiment label.
        READ(8,*) CHEXP
C
C       Read problem size parameters.
        READ(8,*) MM
        READ(8,*) NN
        READ(8,*) KK
        READ(8,*) NLAT
        READ(8,*) NLON
        READ(8,*) NVER
        READ(8,*) NGRPHS
C
C       Read physical constants.
        READ(8,*) A 
        READ(8,*) OMEGA
        READ(8,*) GRAV
        READ(8,*) HDC
        READ(8,*) ALPHA
C
C       Read model integration parameters
        READ(8,*) DT
        READ(8,*) EGYFRQ
        READ(8,*) ERRFRQ
        READ(8,*) SPCFRQ
        READ(8,*) TAUE
        READ(8,*) AFC
        READ(8,*) SITS
        READ(8,*) FORCED
        READ(8,*) MOMENT
        READ(8,*) ICOND
C
C       Close input file.
        CLOSE(8)
C
C       Send input parameters to other processors.
        IF (NPROCS .GT. 1) THEN
C
          RBUF(1)  = A
          RBUF(2)  = OMEGA
          RBUF(3)  = GRAV
          RBUF(4)  = HDC
          RBUF(5)  = ALPHA
          RBUF(6)  = DT
          RBUF(7)  = AFC
          RBUF(8)  = TAUE
          RBUF(9)  = EGYFRQ
          RBUF(10) = ERRFRQ
          RBUF(11) = SPCFRQ
          CALL BCAST0(RBUF, RBYTES*11, BASE, 0)

          IF (FTOPO) THEN   
            IBUF(1) = 1
          ELSE
            IBUF(1) = 0
          ENDIF
          IF (SITS) THEN   
            IBUF(2) = 1
          ELSE
            IBUF(2) = 0
          ENDIF
          IF (FORCED) THEN 
            IBUF(3) = 1
          ELSE
            IBUF(3) = 0
          ENDIF
          IF (MOMENT) THEN 
            IBUF(4) = 1
          ELSE
            IBUF(4) = 0
          ENDIF
          IBUF(5)  = MM
          IBUF(6)  = NN
          IBUF(7)  = KK
          IBUF(8)  = NLAT
          IBUF(9)  = NLON
          IBUF(10) = NVER
          IBUF(11) = NGRPHS
          IBUF(12) = ICOND
          CALL BCAST0(IBUF, IBYTES*12, BASE+1, 0)
   
          CBUF(1:4) = CHEXP
          CALL BCAST0(CBUF, 4, BASE+2, 0)
C
        ENDIF
C
      ELSE
C
C       Get problem parameters from node 0.
        CALL BCAST0(RBUF, RBYTES*11, BASE, 0)
        A      = RBUF(1)
        OMEGA  = RBUF(2)
        GRAV   = RBUF(3)
        HDC    = RBUF(4)
        ALPHA  = RBUF(5)
        DT     = RBUF(6)
        AFC    = RBUF(7)
        TAUE   = RBUF(8)
        EGYFRQ = RBUF(9)
        ERRFRQ = RBUF(10)
        SPCFRQ = RBUF(11)

        CALL BCAST0(IBUF, IBYTES*12, BASE+1, 0)
        IF (IBUF(1) .EQ. 1) THEN
          FTOPO = .TRUE.
        ELSE
          FTOPO = .FALSE.
        ENDIF
        IF (IBUF(2) .EQ. 1) THEN
          SITS    = .TRUE.
        ELSE
          SITS    = .FALSE.
        ENDIF
        IF (IBUF(3) .EQ. 1) THEN
          FORCED  = .TRUE.
        ELSE
          FORCED  = .FALSE.
        ENDIF
        IF (IBUF(4) .EQ. 1) THEN
          MOMENT  = .TRUE.
        ELSE
          MOMENT  = .FALSE.
        ENDIF
        MM = IBUF(5)
        NN = IBUF(6)
        KK = IBUF(7)
        NLAT = IBUF(8)
        NLON = IBUF(9)
        NVER = IBUF(10)
        NGRPHS = IBUF(11)
        ICOND = IBUF(12)

        CALL BCAST0(CBUF, 4, BASE+2, 0)
        CHEXP = CBUF(1:4)
C
      ENDIF
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C     Compute derived quantities.
C
C     Compute additional problem size parameters.
      NFC     = (NLON+2)/2
      LMN     = MM+NN-KK
      NALP    = (MM+1)*(NN+1)-(LMN**2+LMN)/2
      LRM     = (MM+1)*(KK+1)-(MM*MM+MM)/2-1
C
C     Compute Coriolis parameter for rotated coordinates,
C      F = 2.0*OMEGA*(-COS(LAMBDA)*COS(THETA)*SIN(ALPHA)
C                     +SIN(THETA)*COS(ALPHA)),
C     and transform F into spectral space:
C
C     wave M=0, N=1
      CORSC1 = SQRT(8.0/3.0)*OMEGA*COS(ALPHA)
C     wave M=1, N=1
      CORSC2 = - SQRT(4.0/3.0)*OMEGA*SIN(ALPHA)
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C     Record experiment number.
      IF (CHEXP .EQ. '    ') THEN
        IF (ME .EQ. 0) WRITE(0,590)
  590   FORMAT(/,' PSTSWM: ''EXPERIMENT'' HAS NOT BEEN DEFINED IN',/,
     &             ' ENVIRONMENT; ASSUMING EXPERIMENT=0000')
        CHEXP = '0000'
      ENDIF
C
      IF (ME .EQ. 0) WRITE(6,670) CHEXP
  670 FORMAT(/,' EXPERIMENT ',A4)
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C     Check that problem size fits in preallocated arrays.
      IF (MM .GT. MMX) THEN
        IF (ME. EQ. 0) WRITE(0,102) MM, MMX
 102    FORMAT (/,' PSTSWM: FATAL ERROR IN ROUTINE PRBINP:',/,
     &   I8, ' (MM) > ', I8, ' (MMX) ')
        STOP
      ENDIF
C
      IF (KK .GT. KKX) THEN
        IF (ME. EQ. 0) WRITE(0,103) KK, KKX
 103    FORMAT (/,' PSTSWM: FATAL ERROR IN ROUTINE PRBINP:',/,
     &   I8, ' (KK) > ', I8, ' (KKX) ')
        STOP
      ENDIF
C
      IF (NLON .GT. NLONX) THEN
        IF (ME. EQ. 0) WRITE(0,104) NLON, NLONX
 104    FORMAT (/,' PSTSWM: FATAL ERROR IN ROUTINE PRBINP:',/,
     &   I8, ' (NLON) > ', I8, ' (NLONX) ')
        STOP
      ENDIF
C
      IF (NLAT .GT. NLATX) THEN
        IF (ME. EQ. 0) WRITE(0,105) NLAT, NLATX
 105    FORMAT (/,' PSTSWM: FATAL ERROR IN ROUTINE PRBINP:',/,
     &   I8, ' (NLAT) > ', I8, ' (NLATX) ')
        STOP
      ENDIF
C
      IF (NPROCS .GT. NPROCSX) THEN
        IF (ME. EQ. 0) WRITE(0,106) NPROCS, NPROCSX
 106    FORMAT (/,' PSTSWM: FATAL ERROR IN ROUTINE PRBINP:',/,
     &   I8, ' (NPROCS) > ', I8, ' (NPROCSX) ')
        STOP
      ENDIF
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C     Check truncation parameters for consistency.
      IF ((KK .LT. NN) .OR. (KK .LT. MM)) THEN
        IF (ME. EQ. 0) WRITE(0,600) KK,NN,MM
  600   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE PRBINP ',/,
     &         ' TRUNCATION PARAMETER KK MUST BE GREATER OR EQUAL',
     &         ' THAN NN AND MM IN FILE ''PARAMS.i'':',/,
     &         ' KK = ',I4,' NN = ',I4,' MM = ',I4)
        STOP
      ENDIF
C
      IF (KK .GT. MM+NN) THEN
        IF (ME. EQ. 0) WRITE(0,605) KK,MM+NN
  605   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE PRBINP ',/,
     &         ' TRUNCATION PARAMETER KK MUST BE LESS THAN OR',
     &         ' EQUAL MM+NN IN FILE ''PARAMS.i'':',/,
     &         ' KK = ',I4,' MM+NN = ',I4)
        STOP
      ENDIF
C
C     Determine truncation type.
      IF (NN .EQ. KK) THEN
C
        IF (NN .EQ. MM) THEN
C
C         triangular truncation (MM .EQ. NN .EQ. KK)
          IF (ME .EQ. 0) WRITE(6,20) MM
   20     FORMAT (/,' SPECTRAL TRUNCATION TYPE: TRIANGULAR',
     &            /,' M = N = K = ',I4)
          IF (ME .EQ. 0) WRITE(STRUNC,'(''T-'',I4)') MM
C
        ELSE 
C
C         trapezoidal truncation (NN .GT. MM)
          IF (ME .EQ. 0) WRITE(6,24) MM,NN 
   24     FORMAT (/,' SPECTRAL TRUNCATION TYPE: TRAPEZOIDAL',
     &            /,' N = K > M, M = ',I4,' N = ',I4)
          STRUNC = 'TRAPEZ'
C
        ENDIF
C
      ELSE
C
C       (NN .LT. KK)
        IF (KK .EQ. NN+MM) THEN
          IF (NN .EQ. MM) THEN
C
C           rhomboidal truncation
            IF (ME .EQ. 0) WRITE(6,22) MM
   22       FORMAT (/,' SPECTRAL TRUNCATION TYPE: RHOMBOIDAL',
     &              /,' K = N + M, M = N = ',I4)
            IF (ME .EQ. 0) WRITE(STRUNC,'(''R-'',I4)') MM
C
          ELSE
C
C           parallelogramic truncation
            IF (ME .EQ. 0) WRITE(6,23) MM,NN
   23       FORMAT (/,' SPECTRAL TRUNCATION TYPE: PARALLELOGRAMMIC',
     &              /,' K = N + M, M = ',I4,' N = ',I4)
            STRUNC = 'PARALL'
          ENDIF
C
        ELSE
C
C         pentagonal truncation
          IF (ME .EQ. 0) WRITE(6,26) MM,NN,KK
   26     FORMAT (/,' SPECTRAL TRUNCATION TYPE: PENTAGONAL',
     &            /,' M = ',I4,' N = ',I4,' K = ',I4)
          STRUNC = 'PENTAG'
C
        ENDIF
C
      ENDIF 
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
C     Check gridpoint resolution.
      IF (NLON .LT. 3*MM+1) THEN
        IF (ME. EQ. 0) WRITE(0,610) NLON,MM
  610   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE PRBINP ',/,
     &         ' UNALIASED EVALUATION OF QUADRATIC TERMS REQUIRES',
     &         ' (NLON .GE. 3*MM+1) IN FILE ''PARAMS.i'':',/,
     &         ' NLON = ',I4,' MM = ',I4)
        STOP
      ENDIF
C
      IF (NN .EQ. KK) THEN
        LATMIN = (3*NN+1)/2
      ELSE
        LATMIN = (3*NN+2*MM+1)/2
      ENDIF
C
      IF (NLAT .LT. (3*NN+1)/2) THEN
        IF (ME .EQ. 0) WRITE(0,620) NLAT,LATMIN
  620   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE PRBINP ',/,
     &         ' UNALIASED EVALUATION OF QUADRATIC TERMS REQUIRES',
     &         ' (NLAT .GE. LATMIN) IN FILE ''PARAMS.i'':',/,
     &         ' NLAT = ',I4,' LATMIN = ',I4)
        STOP
      ENDIF
C
      IF (MOD(NLAT,2) .EQ. 1) THEN
        IF (ME .EQ. 0) WRITE(0,630) NLAT
  630   FORMAT (/,' PSTSWM: FATAL ERROR IN SUBROUTINE PRBINP ',/,
     &         ' NLAT MUST BE EVEN TO TAKE ADVANTAGE OF HEMISPHERIC',
     &         ' SYMMETRY PROPERTIES IN FILE ''PARAMS.i'':',/,
     &         ' NLAT = ',I4)
        STOP
      ENDIF
C
      IF (ME .EQ. 0) WRITE (6,640) NLAT, NLON, NVER
  640 FORMAT (/,' NUMBER OF GRIDPOINTS IN MODEL'
     &        /,' NORTH-SOUTH GAUSSIAN GRID:  NLAT = ',I4,
     &        /,' EAST-WEST EQUIDISTANT GRID: NLON = ',I4,
     &        /,' VERTICAL LAYERS:            NVER = ',I4)
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      RETURN
      END



