C#######################################################################
C PSTSWM Version 1.0 (8/1/93)                                          #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
      SUBROUTINE INIT(EPS, ALP, DALP, WTS, WTACSJ, TRIGS,
     &                WS1, WS2, WS3, DIVSC, ZETASC, PHICON, UCON, VCON, 
     &                MOUNT, TOPOSC, UIC12, VIC12, PIC12, DIC12, EIC12)
C                                                                             
C This routine initializes variables for the various test cases.
C For steady-state flows, the information is stored in common
C block /finit/, arrays UIC12, VIC12, PIC12, DIC12 and EIC12.
C The case is determined by the variable ICOND:
C  case 1: advection equation for solid body flow
C  case 2: solid body rotation steady state flow
C  case 3: jetstream steady state flow
C  case 4: forced low in jetstream
C  case 5: zonal flow over isolated mountain
C  case 6: Rossby-Haurwitz wave
C                                                                             
C called by: PSTSWM
C calls: DZSC, D01AHE, FU, FUNC2, GLAT_P, GLON_P, ROTATE,
C        SHTRNS, US 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C machine architecture information
      INCLUDE 'machine.i'
C problem resolution information
      INCLUDE 'problem.i'
C domain decomposition definition variables
      INCLUDE 'physical.i'
      INCLUDE 'spectral.i'
C constants & timesteps
      INCLUDE 'consts.i'
C initial conditions
      INCLUDE 'finit.i'
C transform arrays
      INCLUDE 'trnsfm.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C single precision machine accuracy
      REAL EPS
C associated Legendre polynomials
      REAL ALP(NFSPEC_S,NLLATH_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S,NLLATH_S)
C Gaussian weights
      REAL WTS(NLAT)
C 1.0/(A*COS(LAT)**2)
      REAL WTACSJ(NLAT)
C trigonometric function values used by REALFFT 
      COMPLEX TRIGS(NTRIGS)
C
C     Work Space
C
C work array 1
C (big enough for REAL (MXLLON_P,NLVER_P,NLLAT_P,2)
C             and COMPLEX (MXLFC_S,NLVER_S,NLLAT_S,2))
      REAL WS1(1)
C work array 2
C (big enough for REAL (MXLLON_P,NLVER_P,NLLAT_P,2,BUFSWS2)
C             and COMPLEX (8,NLFC_S,NLLATH_S))
      REAL WS2(1)
C work array 3
C (big enough for COMPLEX (MXLSPEC_S,2,BUFSWS3))
      REAL WS3(1)
C work array used to hold DIVSC of initial data for test
C case 3
      COMPLEX DIVSC(MXLSPEC_S)
C work array used to hold ZETASC of initial data for test
C case 3
      COMPLEX ZETASC(MXLSPEC_S)
C
C     Output
C
C balanced PHI (PHICON), UCON, VCON (used in ANLYTC for 
C geopotential that balances steady zonal flow)
      REAL PHICON(NLLAT_P)
      REAL UCON(NLLAT_P)
      REAL VCON(NLLAT_P)
C mountain height
      REAL MOUNT(NLLON_P,NLLAT_P)
C spectral coefficients of mountains
      COMPLEX TOPOSC(MXLSPEC_S)
C initial U, V wind
      REAL UIC12(NLLON_P,NLLAT_P)
      REAL VIC12(NLLON_P,NLLAT_P)
C initial height
      REAL PIC12(NLLON_P,NLLAT_P)
C initial divergence and vorticity
      REAL DIC12(NLLON_P,NLLAT_P)
      REAL EIC12(NLLON_P,NLLAT_P)
C
C---- Local Parameters -------------------------------------------------
C
      REAL PI
      PARAMETER (PI=3.141592653589793)
C
C---- Local Variables --------------------------------------------------
C
C longitude, sine and cosine of longitude, rotated longitude, and 
C longitude loop index
      REAL RLON, SINL, COSL, ROTLON
      INTEGER I
C latitude, sine and cosine of latitude, rotated latitude, and latitude
C loop index
      REAL RLAT, SINT, COST, ROTLAT
      INTEGER J
C sine and cosine of ALPHA (rotation angle)
      REAL SINA, COSA
C ETA  and PHI amplitudes
      REAL ETAAMP, PHIAMP
C max amplitude of flow (12 day rotation speed)
      REAL UBAR
C unnormalized PHI value for case #3
      REAL PHITMP
C mountain surface specifications for case #5
      REAL RADIUS, DIST, MOUNTA
C NAG integration routine arguments and relative error fro case #3
      INTEGER NLIMIT, IFAIL, NPTS
      REAL RELERR
C
C---- External Functions -----------------------------------------------
C
C actual function arguments for balance equations
      EXTERNAL FU, FUNC2, US
      REAL FU, FUNC2, US
C grid latitude and longitude
      EXTERNAL GLAT_P, GLON_P
      REAL GLAT_P, GLON_P
C quadrature routine
      EXTERNAL D01AHE
      REAL D01AHE
C     
C---- Statement Functions ----------------------------------------------
C
C zonal flow functions
      INCLUDE 'bubfnc.i'
C                                                                              
C---- Executable Statements --------------------------------------------
C
C     Determine which initial condition to use.
      IF (ICOND .EQ. 1) THEN
C       ## initial condition 1 ##
C       analytic specification for advection of cosine bell solution as
C       specified by Williamson and Rasch, 1989, Mon. Weather Review
C
        IF (ME .EQ. 0) WRITE (6,70) ALPHA
   70   FORMAT(/,' TEST CASE #1: ADVECTION OF COSINE BELL',
     &         /,' ROTATED BY AN ANGLE ALPHA = ',F5.3)
C
C       initial location and amplitude
        RLON0 = -90.0*(PI/180.0)
        RLAT0 = + 0.0*(PI/180.0)
        PHI0  = 1000.0
C
C       Use only explicit timestepping with advection equation !
        IF (SITS) THEN
          IF (ME .EQ. 0) WRITE(0,75)
   75     FORMAT(/,' PSTSWM: CANNOT RUN TEST CASE 1 WITH SEMI-IMPLICIT',
     &           /,' TIMESTEPPING. CHANGE PARAMETER SITS = .FALSE.',
     &           /,' IN INPUT FILE problem')
          STOP
        ENDIF
C
C       sine and cosine of rotation angle
        SINA = SIN(ALPHA)
        COSA = COS(ALPHA)

C       Choose velocity for one rotation in 12 days.
        SU0  = 2.0*PI*A/(3600.0*24*12)
        ETAAMP = 2.0*(SU0/A + OMEGA)
C
        DO I=1,NLLON_P
C         longitude = RLON = GLON_P(I)
          RLON = GLON_P(I)
          DO J=1,NLLAT_P
C           latitude = RLAT = GLAT_P(J)
            RLAT = GLAT_P(J)
C
C           Set up steady flow field.
            IF (ALPHA .NE. 0.0) THEN
              UIC12(I,J) = SU0*(COS(RLAT)*COSA+COS(RLON)*
     &                     SIN(RLAT)*SINA)
              VIC12(I,J) = -SU0*SIN(RLON)*SINA
            ELSE
              UIC12(I,J) = SU0*COS(RLAT)
              VIC12(I,J) = 0.0
            ENDIF
C
            DIC12(I,J) = 0.0
            EIC12(I,J) = ETAAMP*(-COS(RLON)*COS(RLAT)*SINA
     &                   +SIN(RLAT)*COSA)
          ENDDO
        ENDDO
C
      ELSEIF (ICOND .EQ. 2) THEN
C       ## initial condition 2 ##
C       analytic specification of U, V, PHI, DIV, ZETA field on Gaussian
C       grid. steady state solution (Williamson and Browning, 1973 JAM)
C
        IF (ME .EQ. 0) WRITE (6,90) ALPHA
   90 FORMAT(/,' TEST CASE #2: STEADY STATE NONLINEAR GEOSTROPHIC FLOW',
     &       /,' ROTATED BY AN ANGLE ALPHA = ', F5.3)
C
C       initial location and amplitude
        RLON0 = -90.0*(PI/180.0)
        RLAT0 = + 0.0*(PI/180.0)
C
C       sine and cosine of rotation angle
        SINA = SIN(ALPHA)
        COSA = COS(ALPHA)
C
C       max amplitude of flow (12 day rotation speed)
        UBAR = (2.0*PI*A)/(12.0*24.0*3600.0)
        PHI0 = 2.94E4
        ETAAMP = 2.0*(UBAR/A + OMEGA)
        PHIAMP = A*OMEGA*UBAR + (UBAR**2)/2.0
C
        DO I=1,NLLON_P                                                           
C         longitude = RLON = GLON_P(I)
          RLON = GLON_P(I)
          SINL = SIN(RLON)
          COSL = COS(RLON)
          DO J=1,NLLAT_P                                                         
C           latitude = RLAT = GLAT_P(J)
            RLAT = GLAT_P(J)
            SINT = SIN(RLAT)
            COST = COS(RLAT)
            UIC12(I,J) = UBAR*(COST*COSA + COSL*SINT*SINA)
            VIC12(I,J) = -UBAR*SINL*SINA
            PIC12(I,J) = (PHI0-PHIAMP*(- COSL*COST*SINA + 
     &                   SINT*COSA)**2)/GRAV
            DIC12(I,J) = 0.0
            EIC12(I,J) = ETAAMP*(- COSL*COST*SINA + SINT*COSA)
C
          ENDDO
C
        ENDDO
C
      ELSEIF (ICOND .EQ. 3) THEN
C       ## initial condition 3 ##
C       Initial U specified as bump (infinitely differentable) function
C       V=0; solve for PHI by integrating 1 dimensional balance equation.
C       See paper by Browning et. al., (Monthly Weather Review, 1989)
C
        IF (ME .EQ. 0) WRITE (6,120) ALPHA
  120 FORMAT(/,' TEST CASE #3: STEADY STATE NONLINEAR GEOSTROPHIC FLOW',
     &       /,' WITH COMPACT SUPPORT',
     &       /,' ROTATED BY AN ANGLE ALPHA = ',F5.3)
C
        PHI0   = 2.94E4
        NLIMIT = -1
        IFAIL  =  0
        DO J=1,NLLAT_P
C
C        latitude = RLAT = GLAT_P(J)
         RLAT = GLAT_P(J)
         DO I = 1, NLLON_P
C          longitude = RLON = GLON_P(I)
           RLON = GLON_P(I)
C          compute coordinates in rotated system
           CALL ROTATE(RLON, RLAT, ALPHA, ROTLON, ROTLAT)
C
C          Call numerical integration procedure from NAG for PHI.
C          What follows is a quick description of the argument list so 
C          the user can replace this routine with another one if 
C          necessary.
C
C          FUNCTION D01AHE (A, B, EPSR, NPTS, RELERR, F, NLIMIT, IFAIL)
C           A      - specifies the lower limit of integration
C           B      - specifies the upper limit of integration
C           EPSR   - specifies the relative accuracy required
C           NPTS   - number of points at which to evaluate the integral
C           RELERR - contains rough estimate of relative error on exit
C           F      - real function, supplied by the user
C           NLIMIT - specifies a limit to number of function evaluations
C                    (NLIMIT .LE. 0) => limit of 10,000
C           IFAIL  - must be preassigned when routine is called.
C                    Contains 0 on output if no error occurred.
C
            PHITMP = D01AHE(-0.5*PI, ROTLAT, 100.0*EPS, NPTS, 
     &                      RELERR, FU, NLIMIT, IFAIL)
            IF (IFAIL .NE. 0) THEN
              WRITE (0,140) IFAIL
  140         FORMAT(/,' PSTSWM: FATAL ERROR IN ROUTINE INIT:',/,
     &             ' FAILURE IN NAG INTEGRATION ROUTINE D01AHE',/,
     &             ' IFAIL = ',I4)
              STOP 
            ENDIF 
C
            IF (ALPHA .NE. 0.0) THEN
C             Rotate field variables.
              UIC12(I,J) = US(ROTLAT)*(COS(ALPHA)*SIN(ROTLON)
     &                     *SIN(RLON)+COS(RLON)*COS(ROTLON))
              VIC12(I,J) = US(ROTLAT)*(COS(ALPHA)*COS(RLON)
     &                     * SIN(ROTLON)*SIN(RLAT) 
     &                     - COS(ROTLON)*SIN(RLON)*SIN(RLAT) 
     &                     - SIN(ALPHA)*SIN(ROTLON)*COS(RLAT))
              PIC12(I,J) = (PHI0-PHITMP)/GRAV
            ELSE
C             no rotation -> original field
              UIC12(I,J) = US(RLAT)
              VIC12(I,J) = 0.0
              PIC12(I,J) = (PHI0-PHITMP)/GRAV
            ENDIF
          ENDDO
C
        ENDDO
C
C       Get spectral coefficients for ZETA and DIV (steady state).
        CALL DZSC(ALP, DALP, WTS, WTACSJ, TRIGS, UIC12, VIC12,
     &            WS1, WS2, WS3, DIVSC, ZETASC)
C
C       Inverse transform DIV and ZETA for initial condition information 
C       (destroys ZETASC and DIVSC!).
        CALL SHTRNS(+1, ALP, WTS, TRIGS, WS1, WS2, WS3, DIC12, DIVSC)
        CALL SHTRNS(+1, ALP, WTS, TRIGS, WS1, WS2, WS3, EIC12, ZETASC)
C
      ELSEIF (ICOND .EQ. 4) THEN
C       ## initial condition 4 ##
C       steady state nonlinear zonal geostrophic flow with compact
C       support (see Browning et al, Monthly Weather Review, 1989)
C
        IF (ME .EQ. 0) WRITE(6,200) 
  200   FORMAT(/,' TEST CASE #4: FORCED NONLINEAR SYSTEM WITH',
     &         ' ADVECTING LOW')
C
        IF (.NOT. FORCED) THEN
          IF (ME .EQ. 0) WRITE(0,205)
  205     FORMAT(/,' PSTSWM: FORCING TERMS MUST BE INCLUDED FOR',
     &           /,' TEST 4. SET PARAMETER FORCED = .TRUE. IN',
     &           /,' INPUT FILE problem')
          STOP
        ENDIF
C
C       Forced nonlinear solution:
C       Determine initial condition on time dependent variables for 
C       first two timesteps. (Note: no time truncation error to start 
C       leapfrog procedure)
C
C       constants for analytic stream function (forced case):
C       basic zonal flow amplitude
        SU0    = 20.0
        PHI0   = 1.0E5
C
C       Use a translating low instead of a translating high:
C       amplitude of low (To remove low, set to zero.)
        ALFA   = -0.03*(PHI0/(2.0*OMEGA*SIN(PI/4.0))) 
C*******ALFA   = 0.0
C
C       initial position of low (Greenwich)
        RLON0  = 0.0*(PI/180.0) 
        RLAT0  = +45.0*(PI/180.0)
C
C       areal extent of low
        SIGMA  = (2.0*A/1.0E6)**2
        NPWR   =  14
C
C       Balanced PHI (PHICON), UCON, VCON is used in ANLYTC for 
C       geopotential that balances steady zonal flow
        NLIMIT = -1
        IFAIL  = 0   
        DO J=1,NLLAT_P
C
C         latitude = rlat = GLAT_P(J)
          RLAT = GLAT_P(J)
C         Solve non-linear balance equation for PHI;
          PHICON(J) = D01AHE(-0.5*PI, RLAT, 100*EPS,           
     &                NPTS, RELERR, FUNC2, NLIMIT, IFAIL)
          IF (IFAIL .NE. 0) THEN
            WRITE (0,140) IFAIL
            STOP
          ENDIF
          PHICON(J) = (PHI0 - PHICON(J))/GRAV
          UCON(J) = BUBFNC(SIN(RLAT),COS(RLAT))
          VCON(J) = 0.0
C
        ENDDO
C
      ELSEIF (ICOND .EQ. 5) THEN
C       ## initial condition 5 ##
C       Zonal flow over an isolated mountain as used by Takacs.
C       Analytic specification of U, V, PHI, DIV, ZETA field on Gaussian
C       grid. Steady state solution (Williamson and Browning, 1973 JAM)
C
        IF (ME .EQ. 0) WRITE (6,280) 
  280   FORMAT(/,' TEST CASE #5: ZONAL FLOW OVER AN ISOLATED MOUNTAIN')
C
C       Set mountain surface.
        FTOPO = .TRUE.
        MOUNTA = 2000.0
        RADIUS = PI/9.0
C
        DO J = 1, NLLAT_P
C
          RLAT = GLAT_P(J)
          DO I = 1, NLLON_P
            RLON = GLON_P(I)
            DIST = SQRT((RLON -  1.5*PI)**2 + (RLAT - PI/6.0)**2)
            IF (DIST .LT. RADIUS) THEN
              MOUNT(I,J) = MOUNTA*(1.0 - DIST/RADIUS)
            ELSE
              MOUNT(I,J) = 0.0
            ENDIF
C           temporary copy for spectral transform
            PIC12(I,J)=MOUNT(I,J)  
          ENDDO
C
        ENDDO
C
C       Compute spectral coefficients.
        CALL SHTRNS(-1, ALP, WTS, TRIGS, WS1, WS2, WS3, PIC12, TOPOSC)
C
C       initial conditions
C
        PHI0 = 5960.0
        UBAR = 20.0
        ETAAMP = 2.0*(UBAR/A + OMEGA)
        PHIAMP = A*OMEGA*UBAR + (UBAR**2)/2.0
C
        DO I=1,NLLON_P
C
C         longitude = RLON = GLON_P(I)
          RLON = GLON_P(I)
          SINL = SIN(RLON)
          COSL = COS(RLON)
          DO J=1,NLLAT_P
C           latitude = RLAT = GLAT_P(J)
            RLAT = GLAT_P(J)
            SINT = SIN(RLAT)
            COST = COS(RLAT)
            UIC12(I,J) = UBAR*COST
            VIC12(I,J) = 0.0
C
C           free surface height (include mountains)
            PIC12(I,J) = PHI0-PHIAMP*SINT**2/GRAV
            DIC12(I,J) = 0.0
            EIC12(I,J) = ETAAMP*SINT
          ENDDO
C
        ENDDO
C
      ELSEIF (ICOND .EQ. 6) THEN
C       ## initial condition 6 ##
C       Rossby-Haurwitz wave as used by Philips in Monthly Weather 
C       Review, 1959
C
        R = 4
        K = 7.848E-6
        OMG = 7.848E-6
        PHI0 = 8000.0
        IF (ME .EQ. 0) WRITE(6,300) R
  300   FORMAT(/,' TEST CASE #6: ROSSBY-HAURWITZ WAVE, WAVENUMBER ',
     &         I2)
C
C       Latitude-dependent factors for geopotential (PHIA(NLAT), 
C       PHIB(NLAT) and PHIC(NLAT)) are computed in ANLYTC to save space.
C
      ELSE
C
         IF (ME .EQ. 0) WRITE (0,900) ICOND
  900    FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE INIT:',/,               
     &          ' ILLEGAL SPECIFICATION FOR INITIAL CONDITION CASE',/,
     &          ' ICOND = ',I3)
         STOP
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      REAL FUNCTION FU(RLATD)
C
C This function specifies the right hand side of the balance equation 
C for purposes of numerical integration. It requires the real function 
C US and is used for test case 3 geopotential field in subroutine INIT.
C                                                                             
C called by: D01AHE (from INIT)
C calls: US
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Common Blocks ----------------------------------------------------
C
C constants & timesteps
      INCLUDE 'consts.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C latitude
      REAL RLATD
C
C---- External Functions -----------------------------------------------
C
C zonal flow
      EXTERNAL US
      REAL US
C
C---- Executable Statements --------------------------------------------
C
      FU = (2.0*OMEGA*A*SIN(RLATD) + TAN(RLATD)*US(RLATD))*US(RLATD)
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      REAL FUNCTION FUNC2(THETA)  
C
C This function specifies the  right hand side of the non-linear balance 
C equation for purposes of numerical integration; It requires the real 
C function BUBFNC. Make sure that constants are consistant with those 
C in INPUT!! (Used for test case 4 in routine INIT.)
C                                                                              
C called by: D01AHE (from INIT)
C calls: BUBFNC
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Common Blocks ----------------------------------------------------
C
C constants & timesteps
      INCLUDE 'consts.i'
C initial conditions for case 4
      INCLUDE 'finit.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C latitude
      REAL THETA
C
C---- Local Variables --------------------------------------------------
C
C bump function value
      REAL SU
C     
C---- Statement Functions ----------------------------------------------
C
C zonal flow functions
      INCLUDE 'bubfnc.i'
C
C---- Executable Statements --------------------------------------------
C
      SU     = BUBFNC(SIN(THETA),COS(THETA))
      FUNC2  = (2.0*OMEGA*A*SIN(THETA) + TAN(THETA)*SU)*SU
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      REAL FUNCTION US(RLATD)  
C                                                                              
C This function specifies a zonally symmetric U field as a function of 
C latitude (bump function). It requires the real function BF2 and is
C used for test case 3. (See paper by Browning et al.)
C
C called by: FU, INIT
C calls: BF2
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Common Blocks ----------------------------------------------------
C
C constants
      INCLUDE 'consts.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C latitude
      REAL RLATD
C
C---- Local Parameters -------------------------------------------------
C
      REAL PI
      PARAMETER (PI=3.141592653589793)
C
C---- Local Variables --------------------------------------------------
C
C north-south extent of flow field
      REAL RLATE,RLATB
C flow profile temporaries
      REAL XE,X
C max amplitude of flow (12 day rotation speed)
      REAL UBAR
C
C---- External Functions -----------------------------------------------
C
C auxiliary function (See below.)
      EXTERNAL BF2
      REAL BF2
C
C---- Executable Statements --------------------------------------------
C
      UBAR  = (2.0*PI*A)/(12.0*24.0*3600.0)
      RLATB = -PI/6.
      RLATE =  PI/2.
      XE    = 3.0E-1
      X     = XE*(RLATD-RLATB)/(RLATE-RLATB)
      US    = UBAR*BF2(X)*BF2(XE-X)*EXP(4.0/XE)
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      REAL FUNCTION BF2(X)
C
C A function used in conjunction with the function US.
C
C called by: US
C calls:
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C scaled latitude
      REAL X
C
C---- Executable Statements --------------------------------------------
C
      IF (X .LE. 0.0) THEN
         BF2 = 0.0
      ELSE
         BF2 = EXP(-1.0/X)
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC

