C#######################################################################
C PSTSWM Version 1.0 (8/1/93)                                          #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
      SUBROUTINE FTDPIV (DTA, ALP, DALP, WTS, WTACSJ, A2NNP1,
     &                   F, WSA, WSB, DCOEF, VCOEF, PCOEF)
C                                                                             
C This routine performs a forward transform procedure used to evaluate 
C the explicit part of the right hand side for the divergence and 
C geopotential prognostic equations using semi-implicit timestepping 
C (the term M of eq. (8) and the term Q in eq. (9) in Ritchie's paper),
C and to evaluate the right hand side for the vorticity prognostic 
C equation. The complex coefficient vectors returned by this routine
C are zeroed within the forward transform procedure (i.e., the user 
C can not specify initial state).  
C
C called by: FTRNIM
C calls: FLTSUM, INDPIV, RSDPIV, TMPDPIV, TREESUM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C parallel algorithm information
      INCLUDE 'algorithm.i'
C domain decomposition information
      INCLUDE 'spectral.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C timestep
      REAL DTA
C associated Legendre polynomials
      REAL ALP(NFSPEC_S,NLLATH_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S,NLLATH_S)
C Gaussian weights
      REAL WTS(NLAT)
C 1.0/(A*COS(LAT)**2)
      REAL WTACSJ(NLAT)
C (N*(N+1))/A**2
      REAL A2NNP1(0:KK)
C
C     Input/Work Space
C
C arrays of Fourier coefficients 
C  F(MXLFC_S,NLVER_S,NLLAT_S,1): (UCOS*ZETA)^M
C  F(MXLFC_S,NLVER_S,NLLAT_S,2): (VCOS*ZETA)^M
C  F(MXLFC_S,NLVER_S,NLLAT_S,3): (UCOS*PHI)^M
C  F(MXLFC_S,NLVER_S,NLLAT_S,4): (VCOS*PHI)^M
C  F(MXLFC_S,NLVER_S,NLLAT_S,5): (UCOS**2+VCOS**2)/FAC:  
C  F(MXLFC_S,NLVER_S,NLLAT_S,6): (ZETA^M)^(TAU-1)
C  F(MXLFC_S,NLVER_S,NLLAT_S,7): (DIV^(TAU-1))^M
C  F(MXLFC_S,NLVER_S,NLLAT_S,8): (PHI^(TAU-1))^M
      COMPLEX F(MXLFC_S,NLVER_S,NLLAT_S,8)
C
C     Work Space
C
C work arrays
      COMPLEX WSA(14,MXLFC_S,NLVER_S,NLLATH_S)
      COMPLEX WSB(MXLSPEC_S,NLVER_S,3,BUFSFLT+NLLATH_S)
C (note: WSB will only have space allocated for 
C  WSB(MXLSPEC_S,NLVER_S,3,BUFSFLT+1) when (SUMOPT .EQ. 0))
C
C     Output
C
C computed divergence for new timelevel
      COMPLEX DCOEF(MXLSPEC_S,NLVER_S)
C computed vorticity for new timelevel
      COMPLEX VCOEF(MXLSPEC_S,NLVER_S)                        
C computed geopotential for new timelevel
      COMPLEX PCOEF(MXLSPEC_S,NLVER_S)                        
C (DCOEF, VCOEF, AND pcoef assumed to be contiguous)
C
C---- Local Variables --------------------------------------------------
C
C latitude indes and parallel stage
      INTEGER NL, P
C polynomial degree transition index
      INTEGER JNTRNS
C
C---- Executable Statements --------------------------------------------
C
C     Calculate intermediate quantities from arrays of Fourier 
C     coefficients for computational efficiency.
      CALL TMPDPIV(DTA, WTS, WTACSJ, F, WSA)
C
C     Calculate local contribution to vector sum.
      IF (SUMOPT .EQ. 0) THEN
C
C       Calculate local contribution to vector sum using in-place
C       summation ordering:
C       1) (P .EQ. NLTSTEPS) case: put contribution in output vectors.
C
C       Initialize contribution vector.
        CALL INDPIV(NLTSTEPS, JNTRNS, 1, ALP(1,1), DALP(1,1), A2NNP1, 
     &              WSA, DCOEF)
C
        DO NL=2,NLLATH_S
C         Add contribution to running sum.
          CALL RSDPIV(NLTSTEPS, JNTRNS, ALP(1,NL), DALP(1,NL), A2NNP1, 
     &                WSA(1,1,1,NL), DCOEF)
        ENDDO
C
C       Communicate partial results with rest of logical column.
        CALL FLTSUM(NLTSTEPS, NLTSTEPS, NLSPEC_S(NLTSTEPS), MXLSPEC_S,
     &              3*NLVER_S, WSB(1,1,1,BUFSFLT+1), WSB, DCOEF)
C
C       2) (P .LT. NLTSTEPS) case: use WSB for local contribution.
        DO P=NLTSTEPS-1,1,-1
C
C         Initialize contribution vector.
          CALL INDPIV(P, JNTRNS, 1, ALP(1,1), DALP(1,1), A2NNP1, 
     &                WSA, WSB(1,1,1,BUFSFLT+1))
C
          DO NL=2,NLLATH_S
C           Add contribution to running sum.
            CALL RSDPIV(P, JNTRNS, ALP(1,NL), DALP(1,NL), A2NNP1, 
     &                  WSA(1,1,1,NL), WSB(1,1,1,BUFSFLT+1))
          ENDDO
C
C         Communicate partial results with rest of logical column.
          CALL FLTSUM(P, NLTSTEPS, NLSPEC_S(P), MXLSPEC_S, 3*NLVER_S,
     &                WSB(1,1,1,BUFSFLT+1), WSB, DCOEF)
C
        ENDDO
C
      ELSE
C
C       Calculate local contribution to vector sum using binary tree
C       summation ordering:
C       1) (P .EQ. NLTSTEPS) case: put contribution in output vectors.
C
C       Compute individual components of sum.
        DO NL=1,NLLATH_S
          CALL INDPIV(NLTSTEPS, JNTRNS, NL, ALP(1,NL), DALP(1,NL), 
     &                A2NNP1, WSA(1,1,1,NL), WSB(1,1,1,BUFSFLT+NL))
        ENDDO
C
C       Compute local binary tree sum (doubling length because TREESUM
C       expects real vectors).
        CALL TREESUM(1, 2*NLSPEC_S(NLTSTEPS), 2*MXLSPEC_S, 3*NLVER_S, 
     &               NLLATH_S, WSB(1,1,1,BUFSFLT+1), DCOEF)
C
C       Communicate partial results with rest of logical column.
        CALL FLTSUM(NLTSTEPS, NLTSTEPS, NLSPEC_S(NLTSTEPS), MXLSPEC_S,
     &              3*NLVER_S, WSB(1,1,1,BUFSFLT+1), WSB, DCOEF)
C
C       2) (P .LT. NLTSTEPS) case: use WSB for local contribution,
        DO P=NLTSTEPS-1,1,-1
C
C         Compute individual components of sum.
          DO NL=1,NLLATH_S
            CALL INDPIV(P, JNTRNS, NL, ALP(1,NL), DALP(1,NL), A2NNP1, 
     &                  WSA(1,1,1,NL), WSB(1,1,1,BUFSFLT+NL))
          ENDDO
C
C         Compute local binary tree sum (doubling length because TREESUM
C         expects real vectors).
          CALL TREESUM(0, 2*NLSPEC_S(P), 2*MXLSPEC_S, 3*NLVER_S, 
     &                 NLLATH_S, WSB(1,1,1,BUFSFLT+1), 
     &                 WSB(1,1,1,BUFSFLT+1))
C
C         Communicate partial results with rest of logical column.
          CALL FLTSUM(P, NLTSTEPS, NLSPEC_S(P), MXLSPEC_S, 3*NLVER_S,
     &                WSB(1,1,1,BUFSFLT+1), WSB, DCOEF)
C
        ENDDO
C
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TMPDPIV (DTA, WTS, WTACSJ, F, WS)
C                                                                              
C This routine calculates intermediate quantities needed by FTDPIV.
C
C called by: FTDPIV
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C domain decomposition information
      INCLUDE 'spectral.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C timestep
      REAL DTA
C Gaussian weights
      REAL WTS(NLAT)
C 1.0/(A*COS(LAT)**2)
      REAL WTACSJ(NLAT)
C arrays of Fourier coefficients:
C  F(MXLFC_S,NLVER_S,NLLAT_S,1): (UCOS*ZETA)^M
C  F(MXLFC_S,NLVER_S,NLLAT_S,2): (VCOS*ZETA)^M
C  F(MXLFC_S,NLVER_S,NLLAT_S,3): (UCOS*PHI)^M
C  F(MXLFC_S,NLVER_S,NLLAT_S,4): (VCOS*PHI)^M
C  F(MXLFC_S,NLVER_S,NLLAT_S,5): (UCOS**2+VCOS**2)/FAC:  
C  F(MXLFC_S,NLVER_S,NLLAT_S,6): (ZETA^M)^(TAU-1)
C  F(MXLFC_S,NLVER_S,NLLAT_S,7): (DIV^(TAU-1))^M
C  F(MXLFC_S,NLVER_S,NLLAT_S,8): (PHI^(TAU-1))^M
      COMPLEX F(MXLFC_S,NLVER_S,NLLAT_S,8)
C
C     Output
C
C intermediate quantities
      COMPLEX WS(14,MXLFC_S,NLVER_S,NLLATH_S)
C
C---- Local Variables --------------------------------------------------
C
C latitude (northern, southern, & global) indices
      INTEGER NL, SL, GNL
C vertical layer index
      INTEGER JV
C untruncated and truncated wavenumber indices
      INTEGER IM, JM
C precalculated coefficients
      REAL FAC0, FAC1, FAC2, FAC3
C
C---- Executable Statements -------------------------------------------
C                                                                              
C     Calculate intermediate quantities.
      DO NL=1,NLLATH_S
        SL   = NLLAT_S-NL+1
        GNL  = LATTRUE_S(NL)
        FAC0 = WTS(GNL)
        FAC1 = DTA*WTS(GNL)
        FAC2 = DTA*WTACSJ(GNL)
        FAC3 = (DTA*WTACSJ(GNL))*WTS(GNL)
C
        DO JV=1,NLVER_S
          DO JM=1,NLMM_S
            IM = JMLTRUE_S(JM)
C
            WS(1,JM,JV,NL) = (F(IM,JV,NL,2)+F(IM,JV,SL,2))
     &                     * CMPLX(0.0,REAL(MTRUE_S(JM))*FAC3)
     &                     + (F(IM,JV,NL,7)+F(IM,JV,SL,7))*FAC0
            WS(8,JM,JV,NL) = (F(IM,JV,NL,2)-F(IM,JV,SL,2))
     &                     * CMPLX(0.0,REAL(MTRUE_S(JM))*FAC3)
     &                     + (F(IM,JV,NL,7)-F(IM,JV,SL,7))*FAC0
C
            WS(5,JM,JV,NL)  = (F(IM,JV,NL,2)-F(IM,JV,SL,2))*FAC3
            WS(12,JM,JV,NL) = (F(IM,JV,NL,2)+F(IM,JV,SL,2))*FAC3
C
          ENDDO
        ENDDO
C
        DO JV=1,NLVER_S
          DO JM=1,NLMM_S
            IM = JMLTRUE_S(JM)
C
            WS(2,JM,JV,NL) = (F(IM,JV,NL,5)+F(IM,JV,SL,5))*FAC1
            WS(9,JM,JV,NL) = (F(IM,JV,NL,5)-F(IM,JV,SL,5))*FAC1
C
          ENDDO
        ENDDO
C
        DO JV=1,NLVER_S
          DO JM=1,NLMM_S
            IM = JMLTRUE_S(JM)
C
            WS(3,JM,JV,NL)  = (F(IM,JV,NL,1)-F(IM,JV,SL,1))*FAC3
            WS(10,JM,JV,NL) = (F(IM,JV,NL,1)+F(IM,JV,SL,1))*FAC3
C
            WS(4,JM,JV,NL)  = ((F(IM,JV,NL,1)+F(IM,JV,SL,1))
     &                      *  CMPLX(0.0,-REAL(MTRUE_S(JM))*FAC2)
     &                      +  (F(IM,JV,NL,6)+F(IM,JV,SL,6)))*FAC0
            WS(11,JM,JV,NL) = ((F(IM,JV,NL,1)-F(IM,JV,SL,1))
     &                      *  CMPLX(0.0,-REAL(MTRUE_S(JM))*FAC2)
     &                      +  (F(IM,JV,NL,6)-F(IM,JV,SL,6)))*FAC0
C
          ENDDO
        ENDDO
C
        DO JV=1,NLVER_S
          DO JM=1,NLMM_S
            IM = JMLTRUE_S(JM)
C
            WS(6,JM,JV,NL)  = (F(IM,JV,NL,3)+F(IM,JV,SL,3))
     &                      * CMPLX(0.0,-REAL(MTRUE_S(JM))*FAC3)
     &                      + (F(IM,JV,NL,8)+F(IM,JV,SL,8))*FAC0
            WS(13,JM,JV,NL) = (F(IM,JV,NL,3)-F(IM,JV,SL,3))
     &                      * CMPLX(0.0,-REAL(MTRUE_S(JM))*FAC3)
     &                      + (F(IM,JV,NL,8)-F(IM,JV,SL,8))*FAC0
C
          ENDDO
        ENDDO
C
        DO JV=1,NLVER_S
          DO JM=1,NLMM_S
            IM = JMLTRUE_S(JM)
C
            WS(7,JM,JV,NL)  = (F(IM,JV,NL,4)-F(IM,JV,SL,4))*FAC3
            WS(14,JM,JV,NL) = (F(IM,JV,NL,4)+F(IM,JV,SL,4))*FAC3
C
          ENDDO
        ENDDO
C
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE INDPIV(P, JNTRNS, NL, ALP, DALP, A2NNP1, DATA, SUM)
C                                                                             
C This routine calculates a contribution to the forward transform 
C described in routine FTDPIV corresponding to a single latitude (NL) 
C and stage (P). INDPIV overwrites the output vector with these values,
C and is used to initialize the output vector as the first step in a
C running sum calculation (SUMOPT .EQ. 0), or when delaying the
C summation over latitude (NL) to allow a (reproducible) binary tree 
C sum ordering (SUMOPT .EQ. 1).
C 
C called by: FTDPIV
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C domain decomposition information
      INCLUDE 'spectral.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C parallel stage
      INTEGER P
C latitude index
      INTEGER NL
C associated Legendre polynomials
      REAL ALP(NFSPEC_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S)
C (N*(N+1))/A**2
      REAL A2NNP1(0:KK)
C data (computed in tmpdpiv from arrays of Fourier coefficients)
      COMPLEX DATA(14,MXLFC_S,NLVER_S)
C
C     Input/Output
C
C current polynomial degree transition index
      INTEGER JNTRNS
C
C     Output
C
C contribution to forward transform "sum"
      COMPLEX SUM(MXLSPEC_S,NLVER_S,3)
C
C---- Local Variables --------------------------------------------------
C
C vertical layer and wavenumber indices
      INTEGER JV, JM
C polynomial degree index and index bounds
      INTEGER JN, JNFIRST, JNLAST
C spectral coefficient index
      INTEGER L
C spectral coefficient offset for wavenumber index JM
      INTEGER IS
C true wavenumber
      INTEGER M
C
C---- Executable Statements --------------------------------------------
C
C     Initialize/update polynomial degree transition index.
      IF ((P .EQ. NLTSTEPS) .AND. (NL .EQ. 1)) THEN
C       First call in this transform: initialize offset.
        JNTRNS = NTRNS_S - JME_S(P)
      ELSEIF (NL .EQ. 1) THEN
C       First call for this value of p: update offset.
        JNTRNS = JNTRNS + (JMB_S(P+1)-1) - JME_S(P)     
      ENDIF
C
C     Vary M and N so that procedure moves along columns denoted
C     by index JM.  M is given by (JM-1); N is given by (JN+JM-2).
C     Take advantage of symmetric character of Legendre polynomials.
C     (Procedure assumes even number of Gaussian latitudes ...)
      DO JV=1,NLVER_S
C
        L = 1
        DO JM=JMB_S(P),JME_S(P)
C
          IS = LLCOL_S(JM,2)
          M  = MTRUE_S(JM)
C
          JNFIRST = JNB_S(JNTRNS+JM)
          JNLAST = JNE_S(JNTRNS+JM)
          L = L - JNFIRST
C
          IF (MOD(JNFIRST,2) .EQ. 0) THEN
C           Compute contributions for first JN.
C           N = M + JNFIRST - 1
C
            SUM(L+JNFIRST,JV,1)
     &           = ALP(IS+JNFIRST)*(DATA(8,JM,JV) 
     &           + DATA(9,JM,JV)*A2NNP1(M+JNFIRST-1))
     &           + DALP(IS+JNFIRST)*DATA(10,JM,JV)
            SUM(L+JNFIRST,JV,2)
     &           = ALP(IS+JNFIRST)*DATA(11,JM,JV) 
     &           + DALP(IS+JNFIRST)*DATA(12,JM,JV)
            SUM(L+JNFIRST,JV,3)
     &           = ALP(IS+JNFIRST)*DATA(13,JM,JV)
     &           + DALP(IS+JNFIRST)*DATA(14,JM,JV)
C
            JNFIRST = JNFIRST + 1
          ENDIF
C
C         Calculate paired JNs.
          DO JN=JNFIRST,JNLAST-1,2
C           N = M + JN - 1
C
            SUM(L+JN,JV,1)
     &           = ALP(IS+JN)*(DATA(1,JM,JV) 
     &           + DATA(2,JM,JV)*A2NNP1(M+JN-1))
     &           + DALP(IS+JN)*DATA(3,JM,JV)
            SUM(L+JN,JV,2)
     &           = ALP(IS+JN)*DATA(4,JM,JV) 
     &           + DALP(IS+JN)*DATA(5,JM,JV)
            SUM(L+JN,JV,3)
     &           = ALP(IS+JN)*DATA(6,JM,JV) 
     &           + DALP(IS+JN)*DATA(7,JM,JV)
C
            SUM(L+JN+1,JV,1) 
     &           = ALP(IS+JN+1)*(DATA(8,JM,JV) 
     &           + DATA(9,JM,JV)*A2NNP1(M+JN))
     &           + DALP(IS+JN+1)*DATA(10,JM,JV)
            SUM(L+JN+1,JV,2) 
     &           = ALP(IS+JN+1)*DATA(11,JM,JV) 
     &           + DALP(IS+JN+1)*DATA(12,JM,JV)
            SUM(L+JN+1,JV,3) 
     &           = ALP(IS+JN+1)*DATA(13,JM,JV) 
     &           + DALP(IS+JN+1)*DATA(14,JM,JV)
C
          ENDDO
C                                                                              
          IF (MOD((JNLAST-JNFIRST),2) .EQ. 0) THEN
C           Compute contributions for last JN.
C           N = M + JNLAST - 1
C
            SUM(L+JNLAST,JV,1)
     &           = ALP(IS+JNLAST)*(DATA(1,JM,JV) 
     &           + DATA(2,JM,JV)*A2NNP1(M+JNLAST-1))
     &           + DALP(IS+JNLAST)*DATA(3,JM,JV)
            SUM(L+JNLAST,JV,2)
     &           = ALP(IS+JNLAST)*DATA(4,JM,JV)  
     &           + DALP(IS+JNLAST)*DATA(5,JM,JV)
            SUM(L+JNLAST,JV,3)
     &           = ALP(IS+JNLAST)*DATA(6,JM,JV)
     &           + DALP(IS+JNLAST)*DATA(7,JM,JV)
C
          ENDIF
C
C         Update local spectral coefficient offset index.
          L = L + JNLAST + 1
C
        ENDDO
C
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE RSDPIV (P, JNTRNS, ALP, DALP, A2NNP1, DATA, SUM)
C                                                                             
C This routine calculates a contribution to the forward transform 
C described in routine FTDPIV corresponding to a single latitude and 
C stage (P). RSDPIV adds these values to the current contents of the 
C output vector, as part of a running sum calculation (SUMOPT .EQ. 0).
C 
C called by: FTDPIV
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C domain decomposition information
      INCLUDE 'spectral.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C parallel stage
      INTEGER P
C current polynomial degree transition index
      INTEGER JNTRNS
C associated Legendre polynomials
      REAL ALP(NFSPEC_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S)
C (N*(N+1))/A**2
      REAL A2NNP1(0:KK)
C data (computed in tmpdpiv from arrays of Fourier coefficients)
      COMPLEX DATA(14,MXLFC_S,NLVER_S)
C
C     Output
C
C contribution to forward transform SUM
      COMPLEX SUM(MXLSPEC_S,NLVER_S,3)
C
C---- Local Variables --------------------------------------------------
C
C vertical layer and wavenumber indices
      INTEGER JV, JM
C polynomial degree index and index bounds
      INTEGER JN, JNFIRST, JNLAST
C spectral coefficient index
      INTEGER L
C spectral coefficient offset for wavenumber index JM
      INTEGER IS
C true wavenumber
      INTEGER M
C
C---- Executable Statements --------------------------------------------
C                                                                              
C     Vary M and N so that procedure moves along columns denoted
C     by index JM.  M is given by (JM-1); N is given by (JN+JM-2).
C     Take advantage of symmetric character of Legendre polynomials.
C     (Procedure assumes even number of Gaussian latitudes ...)
C
      DO JV=1,NLVER_S
C
        L = 1
        DO JM=JMB_S(P),JME_S(P)
C
          IS = LLCOL_S(JM,2)
          M  = MTRUE_S(JM)
C
          JNFIRST = JNB_S(JNTRNS+JM)
          JNLAST = JNE_S(JNTRNS+JM)
          L = L - JNFIRST
C
          IF (MOD(JNFIRST,2) .EQ. 0) THEN
C           Compute contributions for first JN.
C           N = M + JNFIRST - 1
C
            SUM(L+JNFIRST,JV,1)
     &           = SUM(L+JNFIRST,JV,1)
     &           + ALP(IS+JNFIRST)*(DATA(8,JM,JV) 
     &           + DATA(9,JM,JV)*A2NNP1(M+JNFIRST-1))
     &           + DALP(IS+JNFIRST)*DATA(10,JM,JV)
            SUM(L+JNFIRST,JV,2)
     &           = SUM(L+JNFIRST,JV,2)
     &           + ALP(IS+JNFIRST)*DATA(11,JM,JV) 
     &           + DALP(IS+JNFIRST)*DATA(12,JM,JV)
            SUM(L+JNFIRST,JV,3)
     &           = SUM(L+JNFIRST,JV,3)
     &           + ALP(IS+JNFIRST)*DATA(13,JM,JV)
     &           + DALP(IS+JNFIRST)*DATA(14,JM,JV)
C
            JNFIRST = JNFIRST + 1
          ENDIF
C
C         Calculate paired JNs.
          DO JN=JNFIRST,JNLAST-1,2
C           N = M + JN - 1
C
            SUM(L+JN,JV,1)
     &           = SUM(L+JN,JV,1)
     &           + ALP(IS+JN)*(DATA(1,JM,JV) 
     &           + DATA(2,JM,JV)*A2NNP1(M+JN-1))
     &           + DALP(IS+JN)*DATA(3,JM,JV)
            SUM(L+JN,JV,2)
     &           = SUM(L+JN,JV,2)
     &           + ALP(IS+JN)*DATA(4,JM,JV) 
     &           + DALP(IS+JN)*DATA(5,JM,JV)
            SUM(L+JN,JV,3)
     &           = SUM(L+JN,JV,3)
     &           + ALP(IS+JN)*DATA(6,JM,JV) 
     &           + DALP(IS+JN)*DATA(7,JM,JV)
C
            SUM(L+JN+1,JV,1) 
     &           = SUM(L+JN+1,JV,1) 
     &           + ALP(IS+JN+1)*(DATA(8,JM,JV) 
     &           + DATA(9,JM,JV)*A2NNP1(M+JN))
     &           + DALP(IS+JN+1)*DATA(10,JM,JV)
            SUM(L+JN+1,JV,2) 
     &           = SUM(L+JN+1,JV,2) 
     &           + ALP(IS+JN+1)*DATA(11,JM,JV) 
     &           + DALP(IS+JN+1)*DATA(12,JM,JV)
            SUM(L+JN+1,JV,3) 
     &           = SUM(L+JN+1,JV,3) 
     &           + ALP(IS+JN+1)*DATA(13,JM,JV) 
     &           + DALP(IS+JN+1)*DATA(14,JM,JV)
C
          ENDDO
C                                                                              
          IF (MOD((JNLAST-JNFIRST),2) .EQ. 0) THEN
C           Compute contributions for last JN.
C           N = M + JNLAST - 1
C
            SUM(L+JNLAST,JV,1)
     &           = SUM(L+JNLAST,JV,1)
     &           + ALP(IS+JNLAST)*(DATA(1,JM,JV) 
     &           + DATA(2,JM,JV)*A2NNP1(M+JNLAST-1))
     &           + DALP(IS+JNLAST)*DATA(3,JM,JV)
            SUM(L+JNLAST,JV,2)
     &           = SUM(L+JNLAST,JV,2)
     &           + ALP(IS+JNLAST)*DATA(4,JM,JV)  
     &           + DALP(IS+JNLAST)*DATA(5,JM,JV)
            SUM(L+JNLAST,JV,3)
     &           = SUM(L+JNLAST,JV,3)
     &           + ALP(IS+JNLAST)*DATA(6,JM,JV)
     &           + DALP(IS+JNLAST)*DATA(7,JM,JV)
C
          ENDIF
C
C         Update local spectral coefficient offset index.
          L = L + JNLAST + 1
C
        ENDDO
C
      ENDDO
C
      RETURN
      END
