C#######################################################################
C PSTSWM Version 1.0 (8/1/93)                                          #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
      SUBROUTINE DZSC(ALP, DALP, WTS, WTACSJ, TRIGS, U, V, WS1, WS2, 
     &                WS3, DIVSC, ZETASC)
C
C This subroutine obtains divergence and vorticity spectral coefficients
C from the gridpoint velocities (unscaled by COS(THETA)).
C
C DZSC is used only during initialization. To save space, only a single
C vertical layer is assumed in the physical data, and only a single 
C vertical level of spectral coefficients is generated. In order to 
C produce a partitioning of the spectral coefficients that corresponds
C to that produced when there are vertical levels, vertical levels are 
C added before calling RFTLON (where all the spatial transforms are 
C hidden). Only one vertical level in the output from RFTLON is used in
C the rest of the calculation.
C
C called by: INIT, PSTSWM
C calls: ADDLVL, DZSC2, RFTLON, RMLVL, UCDC
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C domain decomposition definition variables
      INCLUDE 'physical.i'
      INCLUDE 'spectral.i'
C transform arrays
      INCLUDE 'trnsfm.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C associated Legendre polynomials
      REAL ALP(NFSPEC_S,NLLATH_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S,NLLATH_S)
C Gaussian weights
      REAL WTS(NLAT)
C 1.0/(A*COS(LAT)**2)
      REAL WTACSJ(NLAT)
C trigonometric function values used by RFTLON
      COMPLEX TRIGS(NTRIGS)
C eastward wind field
      REAL U(NLLON_P,NLLAT_P)
C northward wind field
      REAL V(NLLON_P,NLLAT_P)
C
C     Work Space
C
C work array 1
C (big enough for REAL (MXLLON_P,NLVER_P,NLLAT_P,2)
C             and COMPLEX (MXLFC_S,NLVER_S,NLLAT_S,2))
      REAL WS1(1)
C work array 2
C (big enough for REAL (MXLLON_P,NLVER_P,NLLAT_P,2,BUFSWS2)
C             and COMPLEX (8,NLFC_S,NLLATH_S))
      REAL WS2(1)
C work array 3
C (big enough for COMPLEX (MXLSPEC_S,2,BUFSWS3))
      REAL WS3(1)
C
C     Output
C
C computed divergence coefficients
      COMPLEX DIVSC(MXLSPEC_S)
C computed vorticity coefficients
      COMPLEX ZETASC(MXLSPEC_S)
C (DIVSC and ZETASC assumed to be contiguous)
C
C---- Executable Statements --------------------------------------------
C
C     Compute U*COS(PHI) and V*COS(PHI) fields.
C     (results in WS1)
      CALL UCVC(NLLON_P, NLLAT_P, U, V, WS1)
C
C     Add vertical levels before calling RFTLON.
      CALL ADDLVL(NLLON_P, 2*NLLAT_P, MXLLON_P, NLVER_P, WS1, WS1)
C
C     Fourier transform U*COS(PHI) and V*COS(PHI) fields.
C     (data and results in WS1)
      CALL RFTLON(-1, 2, TRIGS, WS2, WS1)
C
C     Remove levels from work array.
      CALL RMLVL(2*NLFC_S, 2*NLLAT_S, 2*MXLFC_S, NLVER_S, WS1, WS1)
C                                                                        
C     Compute vorticity and divergence coefficients from U and V Fourier
C     coefficients. 
C     (data in WS1)
      CALL DZSC2(ALP, DALP, WTS, WTACSJ, WS1, WS2, WS3, DIVSC, ZETASC)
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE UCVC(ISIZE, JSIZE, U, V, XCOS)
C                                                                              
C This subroutine scales the gridpoint velocities by COS(THETA).
C
C called by: DZSC
C calls: GLAT_P
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C size of arrays
      INTEGER ISIZE, JSIZE
C eastward wind field
      REAL U(ISIZE,JSIZE)
C northward wind field
      REAL V(ISIZE,JSIZE)
C
C     Output
C
C scaled wind fields
      REAL XCOS(ISIZE,JSIZE,2)
C
C---- Local Variables --------------------------------------------------
C
C longitude and latitude indices
      INTEGER I, J
C cosine of latitude
      REAL COSLAT
C
C---- Statement Function -----------------------------------------------
C
C grid latitude
      EXTERNAL GLAT_P
      REAL GLAT_P
C
C---- Executable Statements --------------------------------------------
C
C     Scale wind fields
      DO J=1,JSIZE
        COSLAT = COS(GLAT_P(J))
        DO I=1,ISIZE
          XCOS(I,J,1) = U(I,J)*COSLAT
          XCOS(I,J,2) = V(I,J)*COSLAT
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE DZSC2(ALP, DALP, WTS, WTACSJ, F, WSA, WSB, DCOEF, 
     &                 VCOEF)
C                                                                              
C This routine computes vorticity and divergence spectral coefficients 
C from (scaled) U and V Fourier coefficients. 
C
C called by: DZSC
C calls: FLTSUM, INDZSC, RSDZSC, TMPDZSC, TREESUM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C parallel algorithm information
      INCLUDE 'algorithm.i'
C domain decomposition definition variables
      INCLUDE 'spectral.i'
C constants 
      INCLUDE 'consts.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C associated Legendre polynomials
      REAL ALP(NFSPEC_S,NLLATH_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S,NLLATH_S)
C Gaussian weights
      REAL WTS(NLAT)
C 1.0/(A*COS(LAT)**2)
      REAL WTACSJ(NLAT)
C Fourier coefficients for scaled wind fields:
C  F(NLFC_S,NLLAT_S,1): (UCOS)^M
C  F(NLFC_S,NLLAT_S,2): (VCOS)^M
      COMPLEX F(NLFC_S,NLLAT_S,2)
C
C     Work Space
C
C work arrays
      COMPLEX WSA(8,NLFC_S,NLLATH_S)
      COMPLEX WSB(MXLSPEC_S,2,BUFSFLT+NLLATH_S)
C (note: WSB will have space allocated only for 
C  WSB(MXLSPEC_S,2,BUFSFLT+1) when (SUMOPT .EQ. 0))
C
C     Output
C
C computed divergence coefficients
      COMPLEX DCOEF(MXLSPEC_S)
C computed vorticity coefficients
      COMPLEX VCOEF(MXLSPEC_S)
C (DCOEF and VCOEF assumed to be contiguous)
C
C---- Local Variables --------------------------------------------------
C
C latitude index and parallel stage
      INTEGER NL, P
C polynomial degree transition index
      INTEGER JNTRNS
C
C---- Executable Statements --------------------------------------------
C
C     Calculate intermediate quantities from arrays of Fourier 
C     coefficients for computational efficiency.
      CALL TMPDZSC(WTS, WTACSJ, F, WSA)
C
C     Calculate local contribution to vector sum.
      IF (SUMOPT .EQ. 0) THEN
C
C       Calculate local contribution to vector sum using in-place
C       summation ordering:
C       1) (P .EQ. NLTSTEPS) case: put contribution in output vectors.
C
C       Initialize contribution vector.
        CALL INDZSC(NLTSTEPS, JNTRNS, 1, ALP(1,1), DALP(1,1),
     &              WSA, DCOEF)
C
        DO NL=2,NLLATH_S
C         Add contribution to running sum,
          CALL RSDZSC(NLTSTEPS, JNTRNS, ALP(1,NL), DALP(1,NL),
     &                WSA(1,1,NL), DCOEF)
        ENDDO
C
C       Communicate partial results with rest of logical column.
        CALL FLTSUM(NLTSTEPS, NLTSTEPS, NLSPEC_S(NLTSTEPS), MXLSPEC_S,
     &              2, WSB(1,1,BUFSFLT+1), WSB, DCOEF)
C
C       2) (P .LT. NLTSTEPS) case: use WSB for local contribution.
        DO P=NLTSTEPS-1,1,-1
C
C         Initialize contribution vector.
          CALL INDZSC(P, JNTRNS, 1, ALP(1,1), DALP(1,1), 
     &                WSA, WSB(1,1,BUFSFLT+1))
C
          DO NL=2,NLLATH_S
C           Add contribution to running sum.
            CALL RSDZSC(P, JNTRNS, ALP(1,NL), DALP(1,NL),
     &                  WSA(1,1,NL), WSB(1,1,BUFSFLT+1))
          ENDDO
C
C         Communicate partial results with rest of logical column.
          CALL FLTSUM(P, NLTSTEPS, NLSPEC_S(P), MXLSPEC_S, 2,
     &                WSB(1,1,BUFSFLT+1), WSB, DCOEF)
C
        ENDDO
C
      ELSE
C
C       Calculate local contribution to vector sum using binary tree
C       summation ordering:
C       1) (P .EQ. NLTSTEPS) case: put contribution in output vectors.
C
C       Compute individual components of sum.
        DO NL=1,NLLATH_S
          CALL INDZSC(NLTSTEPS, JNTRNS, NL, ALP(1,NL), DALP(1,NL), 
     &                WSA(1,1,NL), WSB(1,1,BUFSFLT+NL))
        ENDDO
C
C       Compute local binary tree sum (doubling length because TREESUM
C       expects real vectors).
        CALL TREESUM(1, 2*NLSPEC_S(NLTSTEPS), 2*MXLSPEC_S, 2, 
     &               NLLATH_S, WSB(1,1,BUFSFLT+1), DCOEF)
C
C       Communicate partial results with rest of logical column.
        CALL FLTSUM(NLTSTEPS, NLTSTEPS, NLSPEC_S(NLTSTEPS), MXLSPEC_S,
     &              2, WSB(1,1,BUFSFLT+1), WSB, DCOEF)
C
C       2) (P .LT. NLTSTEPS) case: use WSB for local contribution.
        DO P=NLTSTEPS-1,1,-1
C
C         Compute individual components of sum.
          DO NL=1,NLLATH_S
            CALL INDZSC(P, JNTRNS, NL, ALP(1,NL), DALP(1,NL),
     &                  WSA(1,1,NL), WSB(1,1,BUFSFLT+NL))
          ENDDO
C
C         Compute local binary tree sum (doubling length because TREESUM
C         expects real vectors).
          CALL TREESUM(0, 2*NLSPEC_S(P), 2*MXLSPEC_S, 2, NLLATH_S, 
     &                 WSB(1,1,BUFSFLT+1), WSB(1,1,BUFSFLT+1))
C
C         Communicate partial results with rest of logical column.
          CALL FLTSUM(P, NLTSTEPS, NLSPEC_S(P), MXLSPEC_S, 2,
     &                WSB(1,1,BUFSFLT+1), WSB, DCOEF)
C
        ENDDO
C
      ENDIF
C
      IF (P01_S .EQ. 1) THEN
C       (wave M=0, N=1)
        VCOEF(L01_S) = VCOEF(L01_S) + CORSC1
      ENDIF
C
      IF (P11_S .EQ. 1) THEN
C       (wave M=1, N=1)
        VCOEF(L11_S) = VCOEF(L11_S) + CORSC2
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TMPDZSC (WTS, WTACSJ, F, WS)
C                                                                              
C This routine calculates intermediate quantities needed by DZSC2.
C
C called by: DZSC2
C calls:
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C domain decomposition information
      INCLUDE 'spectral.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C Gaussian weights
      REAL WTS(NLAT)
C 1.0/(A*COS(LAT)**2)
      REAL WTACSJ(NLAT)
C Fourier coefficients for scaled wind fields
C  F(NLFC_S,NLLAT_S,1): (UCOS)^M
C  F(NLFC_S,NLLAT_S,2): (VCOS)^M
      COMPLEX F(NLFC_S,NLLAT_S,2)
C
C     Output
C
C intermediate quantities
      COMPLEX WS(8,NLFC_S,NLLATH_S)
C
C---- Local Variables --------------------------------------------------
C
C latitude (northern, southern, & global) indices
      INTEGER NL, SL, GNL
C untruncated and truncated wavenumber indices
      INTEGER IM, JM
C precalculated coefficients
      REAL FAC, MFAC
C
C---- Executable Statements --------------------------------------------
C
      DO NL=1,NLLATH_S
        SL  = NLLAT_S-NL+1
        GNL = LATTRUE_S(NL)
        FAC = WTS(GNL)*WTACSJ(GNL)
C
        DO JM=1,NLMM_S
          IM   = JMLTRUE_S(JM)
          MFAC = REAL(MTRUE_S(JM))*FAC
C
          WS(1,JM,NL) = (F(IM,NL,1) + F(IM,SL,1))*CMPLX(0.0,MFAC)
          WS(5,JM,NL) = (F(IM,NL,1) - F(IM,SL,1))*CMPLX(0.0,MFAC)
C
          WS(4,JM,NL) = (F(IM,NL,1) - F(IM,SL,1))*FAC
          WS(8,JM,NL) = (F(IM,NL,1) + F(IM,SL,1))*FAC
C
        ENDDO
C
        DO JM=1,NLMM_S
          IM   = JMLTRUE_S(JM)
          MFAC = REAL(MTRUE_S(JM))*FAC
C
          WS(2,JM,NL) = (F(IM,NL,2) - F(IM,SL,2))*FAC
          WS(6,JM,NL) = (F(IM,NL,2) + F(IM,SL,2))*FAC
C
          WS(3,JM,NL) = (F(IM,NL,2) + F(IM,SL,2))*CMPLX(0.0,MFAC)
          WS(7,JM,NL) = (F(IM,NL,2) - F(IM,SL,2))*CMPLX(0.0,MFAC)
C
        ENDDO
C
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE INDZSC (P, JNTRNS, NL, ALP, DALP, DATA, SUM)
C                                                                             
C This routine calculates a contribution to the forward transform 
C described in routine DZSC2 corresponding to a single latitude (NL) 
C and stage (P). INDZSC overwrites the output vector with these values,
C and is used to initialize the output vector as the first step in a
C running sum calculation (SUMOPT .EQ. 0), or when delaying the
C summation over latitude (NL) to allow a (reproducible) binary tree 
C sum ordering (SUMOPT .EQ. 1).
C 
C called by: DZSC2
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C domain decomposition information
      INCLUDE 'spectral.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C parallel stage
      INTEGER P
C latitude index
      INTEGER NL
C associated Legendre polynomials
      REAL ALP(NFSPEC_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S)
C data (computed in TMPDZSC from arrays of Fourier coefficients)
      COMPLEX DATA(8,NLFC_S)
C
C     Input/output
C
C current polynomial degree transition index
      INTEGER JNTRNS
C
C     Output
C
C contribution to forward transform "sum"
      COMPLEX SUM(MXLSPEC_S,2)
C
C---- Local Variables --------------------------------------------------
C
C wavenumber and wavenumber index
      INTEGER M, JM
C polynomial degree index and index bounds
      INTEGER JN, JNFIRST, JNLAST
C spectral coefficient index
      INTEGER L
C spectral coefficient offset for wavenumber index jm
      INTEGER IS
C
C---- Executable Statements --------------------------------------------
C
C     Initialize/update polynomial degree transition index.
      IF ((P .EQ. NLTSTEPS) .AND. (NL .EQ. 1)) THEN
C       First call in this transform: initialize offset.
        JNTRNS = NTRNS_S - JME_S(P)
      ELSEIF (NL .EQ. 1) THEN
C       First call for this value of P: update offset.
        JNTRNS = JNTRNS + (JMB_S(P+1)-1) - JME_S(P)     
      ENDIF
C
C     Vary M and N so that procedure moves along columns denoted
C     by index JM.  M is given by (JM-1); N is given by (JN+JM-2).
C     Take advantage of symmetric character of Legendre polynomials.
C     (Procedure assumes even number of Gaussian latitudes ...)
      L = 1
      DO JM=JMB_S(P),JME_S(P)
C
        IS = LLCOL_S(JM,2)
        M  = MTRUE_S(JM)
C
        JNFIRST = JNB_S(JNTRNS+JM)
        JNLAST = JNE_S(JNTRNS+JM)
        L = L - JNFIRST
C
        IF (MOD(JNFIRST,2) .EQ. 0) THEN
C         Compute contributions for first JN.
C
          SUM(L+JNFIRST,1) = ALP(IS+JNFIRST)*DATA(5,JM) 
     &                     - DALP(IS+JNFIRST)*DATA(6,JM)
          SUM(L+JNFIRST,2) = ALP(IS+JNFIRST)*DATA(7,JM)
     &                     + DALP(IS+JNFIRST)*DATA(8,JM)
C
          JNFIRST = JNFIRST + 1
        ENDIF
C
C       Calculate paired JNs.
        DO JN=JNFIRST,JNLAST-1,2
C
          SUM(L+JN,1) = ALP(IS+JN)*DATA(1,JM) 
     &                - DALP(IS+JN)*DATA(2,JM)
          SUM(L+JN,2) = ALP(IS+JN)*DATA(3,JM) 
     &                + DALP(IS+JN)*DATA(4,JM)
C
          SUM(L+JN+1,1) = ALP(IS+JN+1)*DATA(5,JM) 
     &                  - DALP(IS+JN+1)*DATA(6,JM)
          SUM(L+JN+1,2) = ALP(IS+JN+1)*DATA(7,JM) 
     &                  + DALP(IS+JN+1)*DATA(8,JM)
C
        ENDDO
C                                                                              
        IF (MOD((JNLAST-JNFIRST),2) .EQ. 0) THEN
C         Compute contributions for last JN.
C
          SUM(L+JNLAST,1) = ALP(IS+JNLAST)*DATA(1,JM)  
     &                    - DALP(IS+JNLAST)*DATA(2,JM)
          SUM(L+JNLAST,2) = ALP(IS+JNLAST)*DATA(3,JM)
     &                    + DALP(IS+JNLAST)*DATA(4,JM)
C
        ENDIF
C
C       Update local spectral coefficient offset index.
        L = L + JNLAST + 1
C
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE RSDZSC (P, JNTRNS, ALP, DALP, DATA, SUM)
C                                                                             
C This routine calculates a contribution to the forward transform 
C described in routine DZSC2 corresponding to a single latitude and 
C stage (P). RSDZSC adds these values to the current contents of the 
C output vector, as part of a running sum calculation (SUMOPT .EQ. 0).
C 
C called by: DZSC2
C calls: 
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C domain decomposition information
      INCLUDE 'spectral.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C parallel stage
      INTEGER P
C current polynomial degree transition index
      INTEGER JNTRNS
C associated Legendre polynomials
      REAL ALP(NFSPEC_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S)
C data (computed in TMPDZSC from arrays of Fourier coefficients)
      COMPLEX DATA(8,NLFC_S)
C
C     Output
C
C contribution to forward transform "sum"
      COMPLEX SUM(MXLSPEC_S,2)
C
C---- Local Variables --------------------------------------------------
C
C wavenumber and wavenumber index
      INTEGER M, JM
C polynomial degree index and index bounds
      INTEGER JN, JNFIRST, JNLAST
C spectral coefficient index
      INTEGER L
C spectral coefficient offset for wavenumber index jm
      INTEGER IS
C
C---- Executable Statements --------------------------------------------
C                                                                              
C     Vary M and N so that procedure moves along columns denoted
C     by index JM. M is given by (JM-1); N is given by (JN+JM-2).
C     Take advantage of symmetric character of Legendre polynomials.
C     (Procedure assumes even number of Gaussian latitudes ...)
      L = 1
      DO JM=JMB_S(P),JME_S(P)
C
        IS = LLCOL_S(JM,2)
        M  = MTRUE_S(JM)
C
        JNFIRST = JNB_S(JNTRNS+JM)
        JNLAST = JNE_S(JNTRNS+JM)
        L = L - JNFIRST
C
        IF (MOD(JNFIRST,2) .EQ. 0) THEN
C         Compute contributions for first JN.
C         N = M + JNFIRST - 1
C
          SUM(L+JNFIRST,1) = SUM(L+JNFIRST,1)
     &                     + ALP(IS+JNFIRST)*DATA(5,JM) 
     &                     - DALP(IS+JNFIRST)*DATA(6,JM)
          SUM(L+JNFIRST,2) = SUM(L+JNFIRST,2)
     &                     + ALP(IS+JNFIRST)*DATA(7,JM)
     &                     + DALP(IS+JNFIRST)*DATA(8,JM)
C
          JNFIRST = JNFIRST + 1
        ENDIF
C
C       Calculate paired JNs.
        DO JN=JNFIRST,JNLAST-1,2
C         N = M + JN - 1
C
          SUM(L+JN,1) = SUM(L+JN,1)
     &                + ALP(IS+JN)*DATA(1,JM) 
     &                - DALP(IS+JN)*DATA(2,JM)
          SUM(L+JN,2) = SUM(L+JN,2)
     &                + ALP(IS+JN)*DATA(3,JM) 
     &                + DALP(IS+JN)*DATA(4,JM)
C
          SUM(L+JN+1,1) = SUM(L+JN+1,1)
     &                  + ALP(IS+JN+1)*DATA(5,JM) 
     &                  - DALP(IS+JN+1)*DATA(6,JM)
          SUM(L+JN+1,2) = SUM(L+JN+1,2)
     &                  + ALP(IS+JN+1)*DATA(7,JM) 
     &                  + DALP(IS+JN+1)*DATA(8,JM)
C
        ENDDO
C                                                                              
        IF (MOD((JNLAST-JNFIRST),2) .EQ. 0) THEN
C         Compute contributions for last JN.
C         N = M + JNLAST - 1
C
          SUM(L+JNLAST,1) = SUM(L+JNLAST,1)
     &                    + ALP(IS+JNLAST)*DATA(1,JM)  
     &                    - DALP(IS+JNLAST)*DATA(2,JM)
          SUM(L+JNLAST,2) = SUM(L+JNLAST,2)
     &                    + ALP(IS+JNLAST)*DATA(3,JM)
     &                    + DALP(IS+JNLAST)*DATA(4,JM)
C
        ENDIF
C
C       Update local spectral coefficient offset index.
        L = L + JNLAST + 1
C
      ENDDO
C
      RETURN
      END

