C#######################################################################
C PSTSWM Version 1.0 (8/1/93)                                          #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
      SUBROUTINE DZPUV(TL, ALP, DALP, ANNP1, DIVSC, ZETASC, PHISC, WSA,
     &                 WSB, DIVFC, ZETAFC, PHIFC, UCOSFC, VCOSFC)
C
C This routine transforms divergence, geopotential, and vorticity
C from spectral space to Fourier space, and calculates capital U and V 
C momentum components from the vorticity and divergence spectral 
C coefficients.
C                                                                              
C called by: EXPLIC, SIMPLIC
C calls: DZPUV2, ILTCAST
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C parallel algorithm information
      INCLUDE 'algorithm.i'
C domain decomposition information
      INCLUDE 'spectral.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C time level offset for time dependent fields
      INTEGER TL
C associated Legendre polynomials
      REAL ALP(NFSPEC_S,NLLATH_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S,NLLATH_S)
C A/(N*(N+1))
      REAL ANNP1(0:KK)
C divergence spectral coefficients
      COMPLEX DIVSC(MXLSPEC_S,NLVER_S)
C vorticity spectral coefficients
      COMPLEX ZETASC(MXLSPEC_S,NLVER_S)                        
C geopotential spectral coefficients
      COMPLEX PHISC(MXLSPEC_S,NLVER_S)                        
C
C     Work Space
C
C work array for calculating intermediate quantities
      COMPLEX WSA(4,MXLSPEC_S,NLVER_S)
C work array for communicating spectral data
      COMPLEX WSB(MXLSPEC_S,NLVER_S,3,BUFSILT)
C
C     Work Space/Output
C
C divergence field
      COMPLEX DIVFC(MXLFC_S,NLVER_S,NLLAT_S,6)
C vorticity field
      COMPLEX ZETAFC(MXLFC_S,NLVER_S,NLLAT_S,6)
C geopotential field
      COMPLEX PHIFC(MXLFC_S,NLVER_S,NLLAT_S,6)
C eastward wind field (scaled by COS(THETA))
      COMPLEX UCOSFC(MXLFC_S,NLVER_S,NLLAT_S)
C northward wind field (scaled by COS(THETA))
      COMPLEX VCOSFC(MXLFC_S,NLVER_S,NLLAT_S)
C
C---- Local Variables --------------------------------------------------
C
C polynomial degree transition index
      INTEGER JNTRNS
C flag array to indicate whether a given Fourier coefficient
C has been "visited" in previous stages of the computation
      INTEGER JMFLAG(MMX+1)
C parallel stage / index of active buffer in multiple buffer algorithm
      INTEGER P, PM
C latitude and vertical layer indices
      INTEGER NL, JV
C untruncated and truncated wavenumber indices
      INTEGER IM, JM
C
C---- Executable Statements --------------------------------------------
C
C     Send local segment of data array on.
      CALL ILTCAST(1, NLTSTEPS, 3*MXLSPEC_S*NLVER_S, DIVSC, WSB, PM)
C
C     Calculate using local data.
      CALL DZPUV2(1, JNTRNS, JMFLAG, ALP, DALP, ANNP1, DIVSC, ZETASC,
     &            PHISC, WSA, DIVFC(1,1,1,TL), ZETAFC(1,1,1,TL),
     &            PHIFC(1,1,1,TL), UCOSFC, VCOSFC)
C
C     Finish calculation using external data.
      DO P=2,NLTSTEPS
C
C       Send current segment of data array on and get the next segment.
        CALL ILTCAST(P, NLTSTEPS, 3*MXLSPEC_S*NLVER_S, DIVSC, WSB, PM)
C
C       Calculate using new data.
        CALL DZPUV2(P, JNTRNS, JMFLAG, ALP, DALP, ANNP1, 
     &              WSB(1,1,1,PM), WSB(1,1,2,PM), WSB(1,1,3,PM),
     &              WSA, DIVFC(1,1,1,TL), ZETAFC(1,1,1,TL),
     &              PHIFC(1,1,1,TL), UCOSFC, VCOSFC)
C
      ENDDO
C
C     Zero the tail of the complex coefficient spectrum.
      DO NL=1,NLLAT_S
        DO JV=1,NLVER_S
          DO JM=NLMM_S+1,NLFC_S
            IM = JMLTRUE_S(JM)
            UCOSFC(IM,JV,NL)    = (0.0,0.0)  
            VCOSFC(IM,JV,NL)    = (0.0,0.0)
            DIVFC(IM,JV,NL,TL)  = (0.0,0.0)
            PHIFC(IM,JV,NL,TL)  = (0.0,0.0)
            ZETAFC(IM,JV,NL,TL) = (0.0,0.0)
          ENDDO
        ENDDO
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE DZPUV2(P, JNTRNS, JMFLAG, ALP, DALP, ANNP1, 
     &                  DIVSC, ZETASC, PHISC, WS, 
     &                  DIVFC, ZETAFC, PHIFC, UCOSFC, VCOSFC)
C                                                                              
C This routine computes contributions to the Fourier coefficients 
C of divergence, geopotential, vorticity, and capital U and V momentum 
C components using a segment of the divergence, geopotential, 
C and vorticity spectral coefficients.
C
C called by: DZPUV
C calls: TMPDZPUV
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C domain decomposition information
      INCLUDE 'spectral.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C parallel stage
      INTEGER P
C associated Legendre polynomials
      REAL ALP(NFSPEC_S,NLLATH_S)
C deriv. associated Legendre polynomials
      REAL DALP(NFSPEC_S,NLLATH_S)
C A/(N*(N+1))
      REAL ANNP1(0:KK)
C divergence spectral coefficients
      COMPLEX DIVSC(MXLSPEC_S,NLVER_S)
C vorticity spectral coefficients
      COMPLEX ZETASC(MXLSPEC_S,NLVER_S)                        
C geopotential spectral coefficients
      COMPLEX PHISC(MXLSPEC_S,NLVER_S)                        
C
C     Input/Output
C
C current polynomial degree transition index
      INTEGER JNTRNS
C flag array to indicate whether a given Fourier coefficient
C has been "visited" in previous stages of the computation
      INTEGER JMFLAG(MMX+1)
C
C     Work Space
C
C work array for calculating intermediate quantities
      COMPLEX WS(4,MXLSPEC_S,NLVER_S)
C
C     Work Space/Output
C
C divergence field
      COMPLEX DIVFC(MXLFC_S,NLVER_S,NLLAT_S)
C vorticity field
      COMPLEX ZETAFC(MXLFC_S,NLVER_S,NLLAT_S)
C geopotential field
      COMPLEX PHIFC(MXLFC_S,NLVER_S,NLLAT_S)
C eastward wind field (scaled by COS(THETA))
      COMPLEX UCOSFC(MXLFC_S,NLVER_S,NLLAT_S)
C northward wind field (scaled by COS(THETA))
      COMPLEX VCOSFC(MXLFC_S,NLVER_S,NLLAT_S)
C
C---- Local Variables --------------------------------------------------
C
C even and odd contributions to DIVFC
      COMPLEX DTMP1, DTMP2
C even and odd contributions to PHIFC
      COMPLEX PTMP1, PTMP2
C even and odd contributions to UCOSFC
      COMPLEX UTMP1, UTMP2
C even and odd contributions to VCOSFC
      COMPLEX VTMP1, VTMP2              
C even and odd contributions to ZETAFC
      COMPLEX ZTMP1, ZTMP2            
C
C latitude (northern and southern) indices
      INTEGER NL, SL
C wavenumber, untruncated wavenumber index, and truncated
C wavenumber index
      INTEGER M, IM, JM
C polynomial degree index and index bounds
      INTEGER JN, JNFIRST, JNLAST
C spectral coefficient and vertical layer indices
      INTEGER L, JV
C spectral coefficient offset for wavenumber index JM
      INTEGER IS
C
C---- Executable Statements --------------------------------------------
C
C     Initialize/update polynomial degree transition index
C     and flag array.
      IF (P .EQ. 1) THEN
C
C       First call in this transform: initialize offset
        JNTRNS = 1 - JMB_S(P)
C
C       and flag array.
        DO JM=1,NLMM_S
          JMFLAG(JM) = 0
        ENDDO
C
      ELSE
C
C       Update offset
        JNTRNS = JNTRNS + (JME_S(P-1)+1) - JMB_S(P)     
C
C       and flag array.
        DO JM=JMB_S(P-1),JME_S(P-1)
          JMFLAG(JM) = 1
        ENDDO
C
      ENDIF
C
C     Compute intermediate quantities for computational efficiency.
      CALL TMPDZPUV(P, JNTRNS, ANNP1, DIVSC, ZETASC, WS)
C
C     Compute contribution to Fourier coefficients.
      DO NL=1,NLLATH_S
        SL = NLLAT_S-NL+1
C
        DO JV=1,NLVER_S
C
          L = 1
          DO JM=JMB_S(P),JME_S(P)
            IS = LLCOL_S(JM,2)
            IM = JMLTRUE_S(JM)
            M  = MTRUE_S(JM)
C
            JNFIRST = JNB_S(JNTRNS+JM)
            JNLAST = JNE_S(JNTRNS+JM)
            L = L - JNFIRST
C
C           Initialize temporaries.
            IF (MOD(JNFIRST,2) .EQ. 0) THEN
              UTMP1 =  WS(2,L+JNFIRST,JV)*DALP(IS+JNFIRST,NL)        
              UTMP2 = -WS(1,L+JNFIRST,JV)*ALP(IS+JNFIRST,NL)         
              VTMP1 = -WS(4,L+JNFIRST,JV)*DALP(IS+JNFIRST,NL)         
              VTMP2 = -WS(3,L+JNFIRST,JV)*ALP(IS+JNFIRST,NL)         
              DTMP1 =  (0.0,0.0)
              DTMP2 =  DIVSC(L+JNFIRST,JV)*ALP(IS+JNFIRST,NL)
              ZTMP1 =  (0.0,0.0)
              ZTMP2 =  ZETASC(L+JNFIRST,JV)*ALP(IS+JNFIRST,NL) 
              PTMP1 =  (0.0,0.0)
              PTMP2 =  PHISC(L+JNFIRST,JV)*ALP(IS+JNFIRST,NL) 
C
              JNFIRST = JNFIRST + 1
            ELSE
              UTMP1 = (0.0,0.0)
              UTMP2 = (0.0,0.0)
              VTMP1 = (0.0,0.0)
              VTMP2 = (0.0,0.0)
              DTMP1 = (0.0,0.0)
              DTMP2 = (0.0,0.0)
              ZTMP1 = (0.0,0.0)
              ZTMP2 = (0.0,0.0)
              PTMP1 = (0.0,0.0)
              PTMP2 = (0.0,0.0)
            ENDIF
C
C           Compute contributions for paired JNs.
            DO JN=JNFIRST,JNLAST-1,2
              UTMP1 = UTMP1
     &              - WS(1,L+JN,JV)*ALP(IS+JN,NL)
     &              + WS(2,L+JN+1,JV)*DALP(IS+JN+1,NL)
              UTMP2 = UTMP2
     &              + WS(2,L+JN,JV)*DALP(IS+JN,NL)
     &              - WS(1,L+JN+1,JV)*ALP(IS+JN+1,NL)
              VTMP1 = VTMP1
     &              - WS(3,L+JN,JV)*ALP(IS+JN,NL)
     &              - WS(4,L+JN+1,JV)*DALP(IS+JN+1,NL)
              VTMP2 = VTMP2 
     &              - WS(4,L+JN,JV)*DALP(IS+JN,NL)
     &              - WS(3,L+JN+1,JV)*ALP(IS+JN+1,NL)
              DTMP1 = DTMP1 + DIVSC(L+JN,JV)*ALP(IS+JN,NL)
              DTMP2 = DTMP2 + DIVSC(L+JN+1,JV)*ALP(IS+JN+1,NL)
              ZTMP1 = ZTMP1 + ZETASC(L+JN,JV)*ALP(IS+JN,NL)
              ZTMP2 = ZTMP2 + ZETASC(L+JN+1,JV)*ALP(IS+JN+1,NL)
              PTMP1 = PTMP1 + PHISC(L+JN,JV)*ALP(IS+JN,NL)
              PTMP2 = PTMP2 + PHISC(L+JN+1,JV)*ALP(IS+JN+1,NL)
            ENDDO
C
            IF (MOD((JNLAST-JNFIRST),2) .EQ. 0) THEN
C             Compute contributions for last JN.
              UTMP1 = UTMP1 - WS(1,L+JNLAST,JV)*ALP(IS+JNLAST,NL)
              UTMP2 = UTMP2 + WS(2,L+JNLAST,JV)*DALP(IS+JNLAST,NL)
              VTMP1 = VTMP1 - WS(3,L+JNLAST,JV)*ALP(IS+JNLAST,NL)
              VTMP2 = VTMP2 - WS(4,L+JNLAST,JV)*DALP(IS+JNLAST,NL)
              DTMP1 = DTMP1 + DIVSC(L+JNLAST,JV)*ALP(IS+JNLAST,NL)
              ZTMP1 = ZTMP1 + ZETASC(L+JNLAST,JV)*ALP(IS+JNLAST,NL)
              PTMP1 = PTMP1 + PHISC(L+JNLAST,JV)*ALP(IS+JNLAST,NL)
            ENDIF
C
C           Combine contributions of even and odd wavenumbers to obtain
C           Fourier coefficients.
            IF (JMFLAG(JM) .EQ. 0) THEN
              UCOSFC(IM,JV,NL) = UTMP1 + UTMP2
              UCOSFC(IM,JV,SL) = UTMP1 - UTMP2
              VCOSFC(IM,JV,NL) = VTMP1 + VTMP2
              VCOSFC(IM,JV,SL) = VTMP1 - VTMP2
              DIVFC(IM,JV,NL)  = DTMP1 + DTMP2
              DIVFC(IM,JV,SL)  = DTMP1 - DTMP2
              PHIFC(IM,JV,NL)  = PTMP1 + PTMP2
              PHIFC(IM,JV,SL)  = PTMP1 - PTMP2
              ZETAFC(IM,JV,NL) = ZTMP1 + ZTMP2
              ZETAFC(IM,JV,SL) = ZTMP1 - ZTMP2
            ELSE
              UCOSFC(IM,JV,NL) = UCOSFC(IM,JV,NL) + UTMP1 + UTMP2
              UCOSFC(IM,JV,SL) = UCOSFC(IM,JV,SL) + UTMP1 - UTMP2
              VCOSFC(IM,JV,NL) = VCOSFC(IM,JV,NL) + VTMP1 + VTMP2
              VCOSFC(IM,JV,SL) = VCOSFC(IM,JV,SL) + VTMP1 - VTMP2
              DIVFC(IM,JV,NL)  = DIVFC(IM,JV,NL)  + DTMP1 + DTMP2
              DIVFC(IM,JV,SL)  = DIVFC(IM,JV,SL)  + DTMP1 - DTMP2
              PHIFC(IM,JV,NL)  = PHIFC(IM,JV,NL)  + PTMP1 + PTMP2
              PHIFC(IM,JV,SL)  = PHIFC(IM,JV,SL)  + PTMP1 - PTMP2
              ZETAFC(IM,JV,NL) = ZETAFC(IM,JV,NL) + ZTMP1 + ZTMP2
              ZETAFC(IM,JV,SL) = ZETAFC(IM,JV,SL) + ZTMP1 - ZTMP2
            ENDIF
C
C           Update local spectral coefficient offset index.
            L = L + JNLAST + 1
C
          ENDDO
C
        ENDDO
C
      ENDDO
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE TMPDZPUV (P, JNTRNS, ANNP1, DIVSC, ZETASC, WS)
C
C This routine calculates intermediate quantities needed by DZPUV2.
C
C called by: DZPUV2
C calls:
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
      INCLUDE 'params.i'
C
C---- Common Blocks ----------------------------------------------------
C
C problem resolution information
      INCLUDE 'problem.i'
C domain decomposition information
      INCLUDE 'spectral.i'
C constants 
      INCLUDE 'consts.i'
C
C---- Arguments --------------------------------------------------------
C
C     Input
C
C parallel stage
      INTEGER P
C polynomial degree transition index
      INTEGER JNTRNS
C A/(N*(N+1))
      REAL ANNP1(0:KK)
C divergence spectral coefficients
      COMPLEX DIVSC(MXLSPEC_S,NLVER_S)
C vorticity spectral coefficients
      COMPLEX ZETASC(MXLSPEC_S,NLVER_S)                        
C
C     Output
C
C intermediate quantities
      COMPLEX WS(4,MXLSPEC_S,NLVER_S)
C
C---- Local Variables --------------------------------------------------
C
C wavenumber and truncated wavenumber index
      INTEGER M, JM
C polynomial degree index and index bounds
      INTEGER JN, JNFIRST, JNLAST
C spectral coefficient and vertical layer indices
      INTEGER L, JV
C spectral coefficient offset for wavenumber index JM
      INTEGER IS
C
C---- Executable Statements -------------------------------------------
C                                                                              
C     Calculate intermediate quantities.
      DO JV=1,NLVER_S
C
        L = 1
        DO JM=JMB_S(P),JME_S(P)
          IS = LLCOL_S(JM,2)
          M  = MTRUE_S(JM)
C
          JNFIRST = JNB_S(JNTRNS+JM)
          JNLAST = JNE_S(JNTRNS+JM)
          L = L - JNFIRST
C
          DO JN=JNFIRST,JNLAST
C           N = M + JN - 1
C           (remember: ANNP1(0) "defined" to be 0.0, even though 
C            really undefined)
            WS(1,L+JN,JV)
     &            = CMPLX(0.0,ANNP1(M+JN-1)*REAL(M))*DIVSC(L+JN,JV)
            WS(2,L+JN,JV)
     &            = ANNP1(M+JN-1)*ZETASC(L+JN,JV)
            WS(3,L+JN,JV) 
     &            = CMPLX(0.0,ANNP1(M+JN-1)*REAL(M))*ZETASC(L+JN,JV)
            WS(4,L+JN,JV) 
     &            = ANNP1(M+JN-1)*DIVSC(L+JN,JV)
          ENDDO
C
          L = L + JNLAST + 1
        ENDDO
C
C       Add coriolis term for rotated coordinates.
C
        IF (P .EQ. P01_S) THEN
C         (wave M=0, N=1)
          WS(2,L01_S,JV) = WS(2,L01_S,JV) 
     &                   - CMPLX(ANNP1(1)*CORSC1,0.0)
        ENDIF
C
        IF (P .EQ. P11_S) THEN
C         (wave M=1, N=1)
          WS(2,L11_S,JV) = WS(2,L11_S,JV) 
     &                   - CMPLX(ANNP1(1)*CORSC2,0.0)
          WS(3,L11_S,JV) = WS(3,L11_S,JV) 
     &                   - CMPLX(0.0,ANNP1(1)*CORSC2)
        ENDIF
C
      ENDDO
C
      RETURN
      END
