/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	treestats/st-graph.c
 * Summary:
 *	Draw the splay tree graphically with # hits marked at each node.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		28/5/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "misc.h"
#include "readfile.h"

static void usage (void) __attribute__ ((noreturn));
static void get_max_hits (object *n);
static void draw (void);
static void prologue (void);
static void epilogue (void);
static void draw_rect (double, double, double, double);

static FILE *outfp;
static tree *t;
static struct ext_tree *et;
static unsigned max_hits, max_radius;
static int xpos = 0, ypos = 0, width = 0, height = 0;

int
main (int argc, char *argv[])
{
  int i;
  char *inputfile = NULL, *outputfile = NULL;
  int psize_set = 0, box = 0, append_mode = 0;

  if (argc < 2)
    usage ();

  for (i = 1; i < argc; ++i)
    {
      if (argv[i][0] == '-')
	{
	  switch (argv[i][1])
	    {
	    case 'o':
	      if (outputfile || i == argc-1)
		usage ();
	      outputfile = argv[++i];
	      break;
	    case 'p':
	      if (psize_set || i >= argc-4)
		usage ();
	      psize_set = 1;
	      xpos = atoi (argv[++i]);
	      ypos = atoi (argv[++i]);
	      width = atoi (argv[++i]);
	      height = atoi (argv[++i]);
	      break;
	    case 'b':
	      box = 1;
	      break;
	    case 'a':
	      append_mode = 1;
	      break;
	    default:
	      usage ();
	    }
	}
      else
	{
	  if (inputfile)
	    usage ();
	  inputfile = argv[i];
	}
    }

  if (outputfile == NULL)
    outputfile = "output.ps";
  if (inputfile == NULL)
    usage ();
  if (!psize_set)
    {
      /* Set paper to 8x11" default. */
      xpos = 72; ypos = 72; width = 8*72 - 144; height = 11*72 - 144;
    }

  max_radius = width / 17;

  if (readfile (inputfile, &t, &et) == -1)
    exit (1);

  max_hits = 0;
  get_max_hits (t->root);

  if (!append_mode)
    outfp = fopen (outputfile, "w");
  else
    outfp = fopen (outputfile, "a");
  if (outfp == NULL)
    {
      perror ("st-graph: fopen");
      exit (1);
    }
  prologue ();
  if (box)
    {
      draw_rect (xpos, ypos, width, height);
      xpos += width/20;		/* Reduce size by 10% */
      ypos += height/20;
      width -= width/10;
      height -= height/10;
    }
  draw ();
  epilogue ();
  fclose (outfp);

  exit (0);
}

static void
usage (void)
{
  fprintf (stderr,
	   "st-graph: Draw a splay tree from a dump file.\n"
	   "Usage:\n"
	   "  st-graph [options] [-o outputfile] inputfile\n"
	   "Options:\n"
	   "  -o outputfile       Output file name (default: output.ps)\n"
	   "  -p x y width height Specify position on page (default: 8x11\")\n"
	   "  -b                  Draw a box around the graph\n"
           "  -a                  Append to output file\n");
  exit (1);
}

static void
prologue (void)
{
  /* Write the standard PostScript prologue for the file.
   */
  char prologue[] =
    "gsave\n"
    "/line { newpath moveto lineto stroke } def\n"
    "/node { newpath 0 360 arc fill } def\n";
  fputs (prologue, outfp);
}

static void
epilogue (void)
{
  /* Write the standard PostScript epilogue for the file.
   */
  char epilogue[] =
    "grestore\n";
  fputs (epilogue, outfp);
}

static void draw_tree (object *, double, double, double, double,
		       double *, double *);
static void draw_line (double, double, double, double);
static void draw_node (double, double, unsigned);

#define SCALE 0.1

static void
draw (void)
{
  if (t->root)
    draw_tree (t->root, xpos, ypos, width, height, NULL, NULL);
  else
    {
      /* ... draw NULL root ... */
    }
}

/* Draw the tree recursively here. We are allowed to use all the paper we
 * want between coordinates (x,y) and (x+dx,y+dy). `n' is not NULL. Return
 * the coordinates of the root of our subtree.
 */
static void
draw_tree (object *n, 
	   double x, double y,
	   double dx, double dy,
	   double *nx_rtn, double *ny_rtn)
{
  double nx, ny;

  /* Decide where to put the root node of this subtree. */
  nx = x + dx/2;
  ny = y + (1 - SCALE/2) * dy;

  /* Return the root node, if required. */
  if (nx_rtn) *nx_rtn = nx;
  if (ny_rtn) *ny_rtn = ny;

  /* If the left subtree exists, draw it first. */
  if (n->left)
    {
      double nx_left, ny_left;

      draw_tree (n->left, x, y, dx/2, (1-SCALE) * dy,
		 &nx_left, &ny_left);

      /* Join left node to current node. */
      draw_line (nx, ny, nx_left, ny_left);
    }
  else
    {
      /* Draw NULL left node. */
      /* ... */
    }

  /* Same for right subtree. */
  if (n->right)
    {
      double nx_right, ny_right;

      draw_tree (n->right, x+dx/2, y, dx/2, (1-SCALE) * dy,
		 &nx_right, &ny_right);

      /* Join right node to current node. */
      draw_line (nx, ny, nx_right, ny_right);
    }
  else
    {
      /* Draw NULL right node. */
      /* ... */
    }

  /* Draw current node. */
  draw_node (nx, ny, n->hits);
}

static void
draw_line (double x1, double y1, double x2, double y2)
{
  /* Draw a line between (x1,y1) and (x2,y2). */
  fprintf (outfp, "%g %g %g %g line\n",
	   x1, y1, x2, y2);
}

static void
draw_node (double x, double y, unsigned hits)
{
  /* Draw a node at (x,y), weighted by `hits'. */
  double rad;

  if (hits == 0) return;	/* Don't draw it if it's unused. */
  rad = max_radius * (double) hits / max_hits;
  if (rad < 0.1) return;	/* Too small to draw. */
  fprintf (outfp,
	   "%g %g %g node\n", x, y, rad);
}

static void
draw_rect (double x, double y, double width, double height)
{
  fprintf (outfp,
	   "%g %g %g %g rectstroke\n",
	   x, y, width, height);
}

static void
get_max_hits (object *n)
{
  /* Search the tree for the node with the maximum number of hits, and
   * record this number.
   */
  if (n == NULL) return;
  if (n->hits > max_hits)
    max_hits = n->hits;
  get_max_hits (n->left);
  get_max_hits (n->right);
}
