/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	treestats/st-analyze.c
 * Summary:
 *	Do simple analysis on a list of splay tree dump files.
 * Other notes:
 *	Command line usage is `st-analyze [list of files ...]'
 * Author      	Date		Notes
 * RWMJ		28/5/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

#include "readfile.h"
#include "misc.h"

#define print_div(a,b) do {\
  if ((b) != 0) printf ("%2.2g\t", (double)(a)/(b)); else printf ("*\t");\
  } while (0)

int
main (int argc, char *argv[])
{
  int i;

  if (argc < 2)
    {
      fprintf (stderr, "st-analyze dumpfile [dumpfile [...]]\n");
      exit (1);
    }
  printf ("INDEX\tLOOKUPS\tNLOOKUP\tADDS\tDELS\tSPLAYS\tCOUNT\tHITS\tALD\tANLD\tASD\tALSD\tAASD\tADSD\tDEP5\tDEP10\tDEP50\n");
  for (i = 1; i < argc; ++i)
    {
      tree *t;
      struct ext_tree *et;
      int count, hits;
      int depth50, depth10, depth5;

      /* Read the file from disk. */
      if (readfile (argv[i], &t, &et) == -1)
	continue;

      /* Print some simple data about this splay tree. */
      count = counttree (t);
      hits = counttreehits (t);
      depth5 = calcbalance (t, 5);
      depth10 = calcbalance (t, 10);
      depth50 = calcbalance (t, 50);
      printf ("%u\t%u\t%u\t%u\t%u\t%u\t%d\t%d\t",
	      et->index,
	      et->look_up_operations,
	      et->nlook_up_operations,
	      et->add_operations,
	      et->del_operations,
	      et->splay_operations,
	      count,
	      hits);
      print_div (et->look_up_steps, et->look_up_operations);
      print_div (et->nlook_up_steps, et->nlook_up_operations);
      print_div (et->splay_steps, et->splay_operations);
      print_div (et->splay_look_up_steps, et->splay_look_up_operations);
      print_div (et->splay_add_steps, et->splay_add_operations);
      print_div (et->splay_del_steps, et->splay_del_operations);
      printf ("%d\t%d\t%d\t\n", depth5, depth10, depth50);

      /* Clean up. */
      deltree (t);
    }

  exit (0);
}
