/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rwmj@doc.ic.ac.uk>.		*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	treestats/misc.c
 * Summary:
 *	Miscellaneous functions to deal with the splay tree in memory.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		28/5/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>

#include "misc.h"

/* Delete a tree recursively. */

void
deltree (tree *t)
{
  object *n;

  n = t->root;
  delsubtree (n);
  free (t);
}

/* Delete a subtree recursively. */

void
delsubtree (object *n)
{
  if (n == NULL)
    return;
  if (n->left)
    delsubtree (n->left);
  if (n->right)
    delsubtree (n->right);
  free (n);
}

/* Count number of nodes in a tree. */

int
counttree (tree *t)
{
  return countsubtree (t->root);
}

/* Count number of nodes in a subtree. */

int
countsubtree (object *n)
{
  if (n == NULL)
    return 0;
  else
    return countsubtree (n->left) + countsubtree (n->right) + 1;
}

/* Count the number of hits received by the whole tree. */

int
counttreehits (tree *t)
{
  return countsubtreehits (t->root);
}

/* Count the number of hits received by a particular subtree. */

int
countsubtreehits (object *n)
{
  if (n == NULL)
    return 0;
  else
    return countsubtreehits (n->left)
      + countsubtreehits (n->right)
      + n->hits;
}

/* Calculate a measure of balance on the tree. The argument is a percentage
 * and we search the tree breadthfirst until we pass more than this percent
 * of the nodes.
 */

static int atdepth (object *, int);

int
calcbalance (tree *t, int pc)
{
  int count = pc * counttree (t) / 100;
  int sum = 0;
  int depth;

  for (depth = 0; sum < count; depth++)
    sum += atdepth (t->root, depth);

  return depth;
}

static int
atdepth (object *n, int d)
{
  /* Return # of nodes that are at depth `d' in subtree `n'. The root node
   * is at depth 0.
   */
  if (n == NULL && d > 0)
    return 0;
  else if (d == 0)
    return n != NULL;
  else
    return atdepth (n->left, d-1) + atdepth (n->right, d-1);
}
