/*----------------------------------------------------------------------*
 *	Bounds Checking for GCC.					*
 *	Written by Richard W.M. Jones.					*
 *----------------------------------------------------------------------*
 * File:
 *	lib/malloc.c
 * Summary:
 *	Replacement for malloc, free, realloc, memalign and some other
 *	memory allocation functions. These functions track heap memory
 *	as for other memory objects, and also provide strict checking
 *	of the actual use of the functions themselves.
 * Other notes:
 * Author      	Date		Notes
 * RWMJ		4/12/94		Initial implementation.
 * RWMJ		7/3/95		Implemented no-reuse mode fully.
 * RWMJ		7-12/3/95	Various bug-fixes and speed-ups.
 *----------------------------------------------------------------------*/

#define NO_FIX_MALLOC

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/types.h>

#include "getpagesize.h"		/* Nicked from GNU malloc library. */
#include "bounds-lib.h"

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

/* Some constants that can be tweaked:
 */
#define MALLOC_MIN_PAGE_REQ	16	/* min. # of pages to get from OS. */
#define MALLOC_MAX_SIZE		(1024*1024*1024) /* bytes: max. single malloc*/
#define MALLOC_ALIGNMENT	16	/* bytes: alignment of blocks */
#define MALLOC_PAGE_SIZE_K	64	/* size of our virtual `pages' */

/* Define _either_ USE_SBRK _or_ USE_MMAP to use `sbrk' or `mmap' respectively
 * to allocate memory. Define USE_MUNMAP if possible so that we can give
 * memory back to the operating system, otherwise you'll run out of VM
 * pretty quickly.
 */
#define MALLOC_USE_SBRK		0
#define MALLOC_USE_MMAP		1
#define MALLOC_USE_MUNMAP	1

/* Define DEBUG to trace calls to the library. Define INTERNAL_CHECKS to do
 * some simple integrity checks in the low-level routines.
 */
#define DEBUG			0	/* debug calls to malloc and free */
#define INTERNAL_CHECKS		1	/* do some integrity checks */

/* Standard definitions for MIN and MAX of integers.
 */
#ifdef MIN
#undef MIN
#endif
#ifdef MAX
#undef MAX
#endif

#define MIN(a,b) ((a)<(b) ? (a) : (b))
#define MAX(a,b) ((a)>(b) ? (a) : (b))

/* `sbrk' returns this when it fails.
 */
#define SBRK_FAILED ((void *) -1)

/* Some systems define MAP_ANON, others MAP_ANONYMOUS.
 */
#if MALLOC_USE_MMAP && defined(MAP_ANON) && !defined(MAP_ANONYMOUS)
#define MAP_ANONYMOUS MAP_ANON
#endif

/* Define the permissions and mapping type to use on each page. See
 * `do_sbrk_or_mmap' for more information.
 */
#if MALLOC_USE_MMAP
#define PERMISSIONS (PROT_READ|PROT_WRITE)
#define MAPPING_TYPE (MAP_PRIVATE|MAP_FIXED)
#endif /* MALLOC_USE_MMAP */

/* A certain well-known machine with broken header files needs the following:
 */
#ifdef sun
void *sbrk (size_t);
void *mmap (void *, size_t, int, int, int, off_t);
int   munmap (void *, size_t);
#endif /* sun */

static void *__bounds_heap_start, *__bounds_heap_extent;

/* Round up a number to the nearest multiple of n. n must be a power of 2.
 */
static inline unsigned
round_up (unsigned v, unsigned n)
{
  return (v + n - 1) & ~(n - 1);
}

/* Take the log of an allocation size (size >= 1).
 */
static inline int
get_log_alloc_size (size_t alloc_size)
{
  int log = 0;

  alloc_size --;
  while ((alloc_size /= 2) != 0)
    log ++;
  return log;
}

/* Local variables controlling how the allocator works.
 */
static int    initialized = 0;		/* Initialized yet? */
static size_t pagesize;			/* Size of our virtual `pages'. */
#if MALLOC_USE_MMAP && !defined(MAP_ANONYMOUS)
static int    dev_zero;			/* FD of "/dev/zero" */
#endif

/* The following structure is placed at the start of each quick allocation
 * page. It provides a way for `free' to find out what the size of the
 * allocation was.
 */
struct quick_table_entry;

typedef struct quick_page {
  struct quick_table_entry *list; /* Pointer to the master page entry. */
  int nr_free_slots;		/* Number of free allocations this page. */
  int next_free_slot;		/* Next unused slot in this page. */
} quick_page;

/* The following structure is the master page for quick memory allocations
 * (ie. allocations <= 32K in size).
 */
typedef struct quick_table_entry {
  quick_page *current;	/* Current page in use on this list. */
  int size;			/* Rounded up allocation size in this list. */
  int slots_per_page;	/* # of allocation slots per page. */
  int allocation_start;		/* Offset of allocated data in the page. */
} quick_table_entry;

static quick_table_entry *quick_table;

/*----------------------------------------------------------------------
 *	`get_pages'
 *	Get some adjacent pages of memory from the operating system.
 *	Always get at least MALLOC_MIN_PAGE_REQ pages. Returns NULL if
 *	we can't get more memory off the system.
 *----------------------------------------------------------------------*/

static inline int
do_sbrk_or_mmap (void *position, size_t size)
{
#if MALLOC_USE_SBRK
  return sbrk (size) != SBRK_FAILED;
#elif MALLOC_USE_MMAP
#if defined(MAP_ANONYMOUS)
  return mmap (position, size, PERMISSIONS, MAPPING_TYPE|MAP_ANONYMOUS,
	       -1, 0) == position;
#else
  return mmap (position, size, PERMISSIONS, MAPPING_TYPE, dev_zero, 0)
    == position;
#endif /* defined(MAP_ANONYMOUS) */
#else
#error "MALLOC_USE_SBRK or MALLOC_USE_MMAP must be defined as 1."
#endif
}

static void *
get_pages (int nr_pages)
{
  /* ALLOCATED_AHEAD is the number of pages we have already got from the
   * operating system. If possible, we satisfy the request from this before
   * asking the OS for more memory.
   */
  static int allocated_ahead = 0;
  size_t size = nr_pages * pagesize;
  void *ptr = __bounds_heap_extent;

  /* Can we satisfy this request immediately?
   */
  if (nr_pages <= allocated_ahead)
    {
      __bounds_heap_extent += size;
      allocated_ahead -= nr_pages;
      return ptr;
    }
  else
    {
      /* Need to get more from the operating system. Work out how many to get.
       */
      int pages_needed = MAX (MALLOC_MIN_PAGE_REQ,
			      nr_pages - allocated_ahead);
#if DEBUG
      printf ("memory allocator: increasing heap by %d pages\n",
	      pages_needed);
#endif
      if (do_sbrk_or_mmap (__bounds_heap_extent + allocated_ahead * pagesize,
			   pages_needed * pagesize))
	{
	  __bounds_heap_extent += size;
	  allocated_ahead = MAX (0,
				 MALLOC_MIN_PAGE_REQ
				 - (nr_pages - allocated_ahead));
	  return ptr;
	}
      else
	return NULL;
    }
}

/*----------------------------------------------------------------------
 *	`return_pages'
 *	Return one or more pages to the operating system. These pages
 *	have previously come from `get_pages'.
 *----------------------------------------------------------------------*/

static void
return_pages (void *pointer, int nr_pages)
{
#if MALLOC_USE_MUNMAP
  munmap (pointer, nr_pages * pagesize);
#endif
}

/*----------------------------------------------------------------------
 *	`init'
 *	Initialize the allocator.
 *----------------------------------------------------------------------*/

static int
init (void)
{
  int size, real_pagesize;
  quick_table_entry *list;
  void *eptr;

  /* Get the page size for this architecture, and check it is reasonable.
   */
  real_pagesize = getpagesize ();
  if (real_pagesize < 256 || real_pagesize > MALLOC_PAGE_SIZE_K * 1024)
    {
      printf ("Memory allocator: page size for this architecture is unreasonable.\n");
      return 0;
    }
  if (((MALLOC_PAGE_SIZE_K * 1024) % real_pagesize) != 0)
    {
      printf ("Memory allocator: page size for this architecture does not allow allocations\n"
	      "  of size %dK.\n", MALLOC_PAGE_SIZE_K);
      return 0;
    }

  /* Choose a suitable page size. We map and unmap pages in units of this
   * size, since it was found that mapping and unmapping single machine-sized
   * pages was hiddeously slow.
   */
  pagesize = MALLOC_PAGE_SIZE_K * 1024;

  /* Find the start of the heap. Sometimes it isn't page aligned, so align
   * it first (we rely on it being page aligned in get_pages and return_pages).
   */
  eptr = sbrk (0);
  if (((unsigned) eptr & (pagesize - 1)) != 0)
    {
      int required = pagesize - ((unsigned)eptr & (pagesize - 1));

      if (sbrk (required) == NULL)
	goto allocator_error;
      eptr = sbrk (0);
      if (eptr == NULL
	  || ((unsigned) eptr & (pagesize - 1)) != 0)
	{
	  allocator_error:
	  printf ("Memory allocator: could not realign heap to page size correctly.\n");
	  return 0;
	}
    }
  __bounds_heap_start = __bounds_heap_extent = eptr;

  /* If we are using mmap, open "/dev/zero". On systems that can manage
   * MAP_ANONYMOUS, we won't map from a file.
   */
#if MALLOC_USE_MMAP && !defined(MAP_ANONYMOUS)
  dev_zero = open ("/dev/zero", O_RDWR);
  if (dev_zero < 0)
    {
      printf ("Memory allocator: can't open /dev/zero (errno = %d).\n", errno);
      return 0;
    }
#endif /* MALLOC_USE_MMAP && !defined(MAP_ANONYMOUS) */

  /* Allocate a single page which is the jumping off point for the quick
   * memory allocator lists. Set up this page with preinitialized data.
   */
  quick_table = (quick_table_entry *) get_pages (1);
  if (quick_table == NULL)
    {
      printf ("Memory allocator: out of memory allocating quick table list.\n");
      return 0;
    }

  for (size = 1; size <= pagesize/2; size *= 2)
    {
      list = &quick_table[get_log_alloc_size (size)];
      list->current = NULL;
      list->size = size;
      /* Decide how many allocations we can fit in a page. The usable space
       * is (page size - header).
       */
      list->slots_per_page = (pagesize - sizeof (quick_page))
	/ size;
      list->allocation_start = pagesize - size * list->slots_per_page;
    }

  return initialized = 1;
}

/*----------------------------------------------------------------------
 * 	`__bounds_memalign'
 *	This is where allocation is actually done. We return a new block
 *	of memory, aligned to the requested alignment, of the requested
 *	size. Memory allocated by this call is not checked. You may
 *	request 0 bytes here: you will get a pointer that cannot be
 *	dereferenced.
 *----------------------------------------------------------------------*/

void *
__bounds_memalign (size_t alignment, size_t size)
{
  /* Notice that most (> 50%) calls to this function have alignment
   * MALLOC_ALIGNMENT and size sizeof (object). Many more have size
   * sizeof (function_context). If we handle these fast, then this
   * allocator will run fast overall.
   */
  size_t alloc_size;

  if (!initialized)
    if (!init ())
      return NULL;

#if DEBUG
  printf ("memory allocator: allocating %d bytes with alignment %d.\n",
	  size, alignment);
#endif

  if (alignment > pagesize)
    {
      /* We can't deal with alignments > the page size (say, 64K).
       */
      __bounds_errorf (NULL, 0, NULL, 0,
		       "in memalign, alignment may not be > %d bytes",
		       pagesize);
    }

  if (alignment < sizeof (int))
    alignment = sizeof (int);

  if (size > MALLOC_MAX_SIZE)
    {
      /* This may be caused by `malloc (-1)'.
       */
      __bounds_errorf (NULL, 0, NULL, 0,
		       "cannot allocate more than %d bytes of memory (requested %d bytes)",
		       MALLOC_MAX_SIZE, size);
    }

  /* ALLOC_SIZE is the real allocation size. For allocations less than 4 bytes,
   * we always allocate 4 bytes. We add a single byte of padding also.
   */
  alloc_size = MAX (size + 1, sizeof (int));

  if (__bounds_never_free_heap)
    {
      /* This is NO-REUSE HEAP mode.
       * If the requested size is < 1/2 the page size, this is a `small'
       * block. If, in addition, the alignment is less than the size, we
       * can allocate this anywhere, so this is a `quick' allocation.
       */
      if (alloc_size < pagesize / 2 && alignment <= alloc_size)
	{
	  /* Quick allocation from one of our pages dedicated to small
	   * memory requests.
	   * LOG_ALLOC_SIZE is the number of the list on which this request
	   * goes. eg. Sizes 1025 to 2048 go on the 2K list.
	   */
	  int log_alloc_size = get_log_alloc_size (alloc_size);
	  quick_table_entry *list = &quick_table [log_alloc_size];
	  quick_page *page = list->current;
	  char *data;
	  int slot;

	  /* If there isn't a current page in this list, or the current one
	   * in the list is full, we need to allocate another page in the
	   * list.
	   */
	  if (page == NULL
	      || page->next_free_slot == list->slots_per_page)
	    {
	      page = get_pages (pagesize / pagesize);
	      if (page == NULL)
		return NULL;
	      list->current = page;
	      page->list = list;
	      page->nr_free_slots = list->slots_per_page;
	      page->next_free_slot = 0;
	    }

	  /* Satisfy the allocation from free space in the current page in
	   * this list.
	   */
	  slot = page->next_free_slot;
	  data = (char *) page + list->allocation_start +
	    slot * list->size;
	  page->nr_free_slots --;
	  page->next_free_slot ++;

#if DEBUG
	  printf ("memory allocator: picked slot %d of page %p in list %d\n",
		  slot, page, log_alloc_size);
#endif

	  return (void *) data;
	}
      else
	{
	  /* The requested size is larger than half a page, or else there is
	   * a strange alignment request. We allocate nr. pages + 1 directly.
	   * The allocation starts at the start of the second page. The first
	   * page records the size of the allocation, so we can determine
	   * what to unmap in free. This wastes 4K per allocation, so a
	   * better method should be found.
	   */
	  int nr_pages;
	  void *pages;

	  nr_pages = round_up (alloc_size, pagesize) / pagesize + 1;
	  pages = get_pages (nr_pages);
	  if (pages == NULL)
	    return NULL;

	  /* Write the allocation size (in pages) at the beginning of the
	   * first page, followed by a magic number.
	   */
	  *(int *)pages = nr_pages;
#if INTERNAL_CHECKS
	  *((int *)pages + 1) = 0x12345678;
#endif

#if DEBUG
	  printf ("memory allocator: large request satisfied at address %p\n",
		  (char *) pages + pagesize);
#endif

	  return (void *) ((char *)pages + pagesize);
	}
    }

  return NULL;
}

/*----------------------------------------------------------------------
 *	`__bounds_malloc'
 *	Allocate memory aligned to the default alignment. Memory allocated
 *	by this call is not checked.
 *----------------------------------------------------------------------*/

void *
__bounds_malloc (size_t size)
{
  int alignment = MAX (sizeof (int),
		       MIN (MALLOC_ALIGNMENT, size));
  return __bounds_memalign (alignment, size);
}

/*----------------------------------------------------------------------
 *	`__bounds_free'
 *	Free memory previously allocated by `__bounds_memalign' et al. No
 *	checks are made in this call.
 *----------------------------------------------------------------------*/

void
__bounds_free (void *pointer)
{
  if (!initialized)
    if (!init ())
      return;

#if DEBUG
  printf ("memory allocator: freeing memory at address %p\n", pointer);
#endif

  if (pointer == NULL)
    return;

  if (__bounds_never_free_heap)
    {
      /* This is NO-REUSE HEAP mode.
       * We can tell from the alignment of the pointer passed what the
       * allocation strategy used was.
       */
      if (((unsigned) pointer & (pagesize - 1)) != 0)
	{
	  /* This page came from the quick allocation strategy. By rounding
	   * the pointer down to the nearest page, we can find out
	   * interesting things like which list this pointer is on.
	   */
	  quick_page *page = (quick_page *) ((unsigned) pointer
							 & ~(pagesize - 1));
	  quick_table_entry *list = page->list;

#if INTERNAL_CHECKS
	  /* Do a quick check here. This will detect some errors in this
	   * malloc/free library, hopefully :->
	   */
	  if (list < quick_table || list > quick_table + 32
	      || page->nr_free_slots < 0
	      || page->nr_free_slots >= list->slots_per_page
	      || list->slots_per_page <= 0
	      || list->slots_per_page > pagesize
	      || list->allocation_start <= 0
	      || list->allocation_start > pagesize)
	    __bounds_internal_error ("unreasonable page in free",
				     __FILE__, __LINE__);
#endif

	  /* Increment counter for the number of free slots in this page. If
	   * all the slots get freed, then unmap the page.
	   */
	  page->nr_free_slots ++;
	  if (page->nr_free_slots == list->slots_per_page
	      && list->current != page)
	    {
	      return_pages (page, pagesize / pagesize);
	    }

	  return;
	}
      else
	{
	  /* Large (>32K) allocation strategy was used. The previous page
	   * contains the size of the allocation and a magic number. Look for
	   * these now. Then unmap the memory.
	   */
	  int *page_ptr = (int *) ((char *) pointer - pagesize);
	  int nr_pages = page_ptr[0];
	  int magic = page_ptr[1];

#if INTERNAL_CHECKS
	  if (magic != 0x12345678)
	    __bounds_internal_error ("bad magic number in free",
				     __FILE__, __LINE__);
#endif

	  return_pages ((void *) page_ptr, nr_pages);
	  return;
	}
    }
}

/*----------------------------------------------------------------------
 *	`__bounds_realloc'
 *	Reallocate memory previously allocated, changing the allocation
 *	size. The memory may move. No checks are made in this call.
 *	(FIXME!)
 *----------------------------------------------------------------------*/

void *
__bounds_realloc (void *pointer, size_t new_size)
{
  void *new_pointer;

  printf ("__bounds_realloc is broken (doesn't memcpy old data to new area!).\n");
  ABORT ();

  if (!initialized)
    if (!init ())
      return NULL;

  new_pointer = __bounds_malloc (new_size);
  if (pointer)
    __bounds_free (pointer);
}

/*----------------------------------------------------------------------
 *	`memalign'
 *	This is the user's basic memory allocator. This works as for
 *	memalign in an ordinary C library, but with strict checks.
 *----------------------------------------------------------------------*/

void *
memalign (size_t alignment, size_t size)
{
  void *pointer = __bounds_memalign (alignment, size);

  if (pointer == NULL) return NULL;
  __bounds_add_heap_object (pointer, size, 1);
  return pointer;
}

/*----------------------------------------------------------------------
 *	`malloc'
 *	The user's memory allocator, with default memory alignment.
 *----------------------------------------------------------------------*/

void *
malloc (size_t size)
{
  void *pointer = __bounds_malloc (size);

  if (pointer == NULL) return NULL;
  __bounds_add_heap_object (pointer, size, 1);
  return pointer;
}

/*----------------------------------------------------------------------
 *	`free'
 *	The user's way of freeing previously allocated memory. We check
 *	here that the memory has been allocated properly.
 *----------------------------------------------------------------------*/

void
free (void *pointer)
{
  /* Various libraries seem to call free with a NULL pointer. Why, I'm
   * not sure, but this warning may be of interest.
   */
  if (pointer == NULL)
    {
      if (__bounds_warn_free_null)
	__bounds_warning (NULL, 0, "free called with a NULL pointer");
      return;
    }

  if (pointer == ILLEGAL)
    __bounds_error ("ILLEGAL pointer passed to free",
		    NULL, 0, pointer, NULL);

  if (!__bounds_is_heap_object (pointer))
    __bounds_error ("stale or invalid pointer passed to free",
		    NULL, 0, pointer, NULL);

  __bounds_delete_heap_object (pointer);
  __bounds_free (pointer);
}

/*----------------------------------------------------------------------
 *	`realloc'
 *	Reallocate a previously allocated block of memory. We make checks
 *	here as to the validity of the memory.
 *----------------------------------------------------------------------*/

void *
realloc (void *old_pointer, size_t new_size)
{
  void *new_pointer;
  object *obj;

  /* Allow `realloc (NULL, new_size)' to mean `malloc (new_size)'.
   */
  if (old_pointer == NULL)
    return malloc (new_size);

  /* Check the pointer is not ILLEGAL or invalid.
   */
  if (old_pointer == ILLEGAL)
    __bounds_error ("ILLEGAL pointer passed to realloc",
		    NULL, 0, old_pointer, NULL);
  if (!__bounds_is_heap_object (old_pointer))
    __bounds_error ("stale or invalid pointer passed to free",
		    NULL, 0, old_pointer, NULL);

  /* Reallocate the memory here.
   */
  new_pointer = __bounds_realloc (old_pointer, new_size);

  /* If the memory stayed in the same place, we just change the size field for
   * this object, else we use a new object.
   */
  if (old_pointer == new_pointer)
    {
      obj = __bounds_find_object (new_pointer);
      obj->size = new_size / obj->align;
    }
  else
    {
      __bounds_delete_heap_object (old_pointer);
      if (new_pointer != NULL)
	__bounds_add_heap_object (new_pointer, new_size, 1);
    }

  return new_pointer;
}

/*----------------------------------------------------------------------
 *	Allocate memory on a page boundary.
 *----------------------------------------------------------------------*/

void *
valloc (size_t size)
{
  return memalign (pagesize, size);
}

/*----------------------------------------------------------------------
 *	Allocate memory cleared to zeroes.
 *----------------------------------------------------------------------*/

void *
calloc (size_t nmemb, size_t size)
{
  int total_size = nmemb * size;
  void *result = malloc (total_size);

  /* We don't need to clear out the memory if we are mapping from /dev/zero,
   * since the memory will already be zero.
   */
  if (result != NULL
      && (!__bounds_never_free_heap || MALLOC_USE_SBRK))
    memset (result, 0, nmemb * size);

  return result;
}

/*----------------------------------------------------------------------
 *	Free memory allocated by calloc.
 *----------------------------------------------------------------------*/

#ifdef cfree			/* Some header files define cfree as free. */
#undef cfree
#endif

void
cfree (void *pointer)
{
  free (pointer);
}
