#include <float.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "spline.h"

#define MAXTRY (2 * nfns + 10)

/***********************************************************************
**
**     This procedure constructs a multivariate polynomial interpolant
**     to arbitrary linear functionals.
**
**     The parameters for this function have the following meaning:
**        nind   - the number of independent variables
**        ndep   - the number of dependent variables
**        nfns   - the number of functionals to interpolate
**        linfun - function box to evaluate the functionals at
**                 the given basis function
**        aux    - an auxiliary array to be passed to linfun
**        frhs   - the values of the dependent variables
**        incp   - the increment parameter for frhs per functional
**        incd   - the increment parameter for frhs per dependent var
**        bounds - the bounds on the independent variables, i.e.
**                     bound(2*i) <= x(i) <= bound(2*i+1)
**        poly   - on output, contains the interpolant
**
**     Thomas Grandine
**     May, 1997
**
** REVISION HISTORY:
** 
** 3/20/98 - Thanh Hoang
** Replace call to LINPACK routine dsvdc_ with call to LAPACK routine
** dgesvd_ to completely rid of the dependency on fortran.
**
***********************************************************************/

long int mvpifn (long int nind,
                 long int ndep,
                 long int nfns,
                 double (*linfun) (MVPOLY *monomial,
                                   long int ixfun,
                                   void *aux),
                 void *aux,
                 double *frhs,
                 long int incp,
                 long int incd,
                 double *bounds,
                 MVPOLY **poly)
{
  long int ier = 0;
  long int bytes;
  long int ix;
  long int iy;
  long int iz;
  long int iu;
  long int iv;
  long int nsize;
  long int nmono;
  long int nbase = 0;
  long int ndeg = 0;
  long int nrank;
  long int job = 11;
  char     jobu = 'A', jobvt = 'A';
  long int *dstart = NULL;
  long int *bstart = NULL;
  long int *binom = NULL;
  long int *multi = NULL;
  long int *piv = NULL;
  long int none = 0;
  long int one = 1;
  long int ncopy;
  double zero = 0.0;
  double **udeg = NULL;
  double *a = NULL;
  double *basis = NULL;
  double *u = NULL;
  double *v = NULL;
  double *s = NULL;
  double *e = NULL;
  double *rhs = NULL;
  double *work = NULL;
  long int lwork;
  double *xyzscl = NULL;
  double *bndscopy = NULL;
  double prev;
  double rank = sqrt (sqrt (DBL_EPSILON));
  MVPOLY *monomial = NULL;
  MVPOLY *interp = NULL;

/* Initialize the data structure for the monomial */

  B_ALLOC (monomial, MVPOLY, 1, bytes, ier, END);
  monomial->nind = nind;
  monomial->ndep = 1;
  monomial->nterm = 1;
  B_ALLOC (monomial->expon, long int, nind, bytes, ier, END);
  monomial->bounds = bounds;
  B_ALLOC (monomial->coefs, double, 1, bytes, ier, END);
  monomial->coefs[0] = 1.0;

/* Initialize the output data structure */

  B_ALLOC (interp, MVPOLY, 1, bytes, ier, END);
  interp->nind = nind;
  interp->ndep = ndep;
  *poly = interp;

/* Copy the bounds information */

  ncopy = 2 * nind;
  B_ALLOC (bndscopy, double, ncopy, bytes, ier, END);
  dcopy_ (&ncopy, bounds, &one, bndscopy, &one);

/* Initialize everything for the constant term */

  B_ALLOC (dstart, long int, MAXTRY, bytes, ier, END);
  dstart[0] = 0;
  B_ALLOC (bstart, long int, MAXTRY, bytes, ier, END);
  bstart[0] = 0;
  B_ALLOC (binom, long int, 1, bytes, ier, END);
  binom[0] = 1;
  B_ALLOC (multi, long int, nind, bytes, ier, END);
  for (ix = 0; ix < nind; ix++) {
    multi[ix] = 0;
    monomial->expon[ix] = 0;
  }
  B_ALLOC (udeg, double *, nfns, bytes, ier, END);
  udeg[0] = NULL;
  B_ALLOC (a, double, nfns, bytes, ier, END);
  for (ix = 0; ix < nfns; ix++) {
    a[ix] = linfun (monomial, ix, aux);
  }
  B_ALLOC (basis, double, nfns, bytes, ier, END);
  dcopy_ (&nfns, &zero, &none, basis, &one);
  nsize = 1;

/* Loop until all basis functions have been found */

  for (ndeg = 0; ndeg < MAXTRY; ndeg++) {
    dstart[ndeg+1] = nsize;
    iu = nfns - nbase;
    B_ALLOC (u, double, iu * iu, bytes, ier, END);
    udeg[ndeg] = u;
    iv = nsize - dstart[ndeg];
    B_ALLOC (v, double, iv * iv, bytes, ier, END);
    B_ALLOC (s, double, nfns + 1, bytes, ier, END);
    B_ALLOC (e, double, B_MAX (iu, iv), bytes, ier, END);
    lwork = 5 * B_MAX(iu, iv);
    B_ALLOC (work, double, lwork, bytes, ier, END);
    dgesvd_ (&jobu, &jobvt, &iu, &iv, &a[nbase], &nfns, s, u, &iu, v,
             &iv, work, &lwork, &ier);
    if (ier < 0) goto END;

    /* NOTE:  The output v is actually v transpose in the SVD */

    prev = s[0];
    for (ix = 1; ix < iu && ix < iv; ix++) {
      if (s[ix] < prev * rank) break;
      prev = s[ix];
    }
    if (s[0] < rank) {
      ix = 0;
    }

/* Store the basis functions for each degree */

    nrank = ix;
    bstart[ndeg+1] = bstart[ndeg] + nrank;
    for (ix = 0; ix < nrank; ix++) {
      dcopy_ (&iv, &v[ix], &iv,
              &basis[nfns * dstart[ndeg] + nbase + ix], &nfns);
    }
    B_FREE (v);
    B_FREE (s);
    B_FREE (e);
    B_FREE (a);
    B_FREE (work);
    nbase += nrank;
    if (nbase == nfns) break;

/* Bring in the polynomials of the next highest degree */

    nmono = iv * (nind + ndeg) / (ndeg + 1);
    ix = nmono + nsize;
    B_REALLOC (binom, long int, ix, bytes, ier, END);
    B_REALLOC (multi, long int, ix * nind, bytes, ier, END);

/* Determine the monomials of the next higher degree */

    for (ix = 0; ix < nind; ix++) {
      iu = dstart[ndeg+1];
      for (iy = dstart[ndeg]; iy < dstart[ndeg+1]; iy++) {
        for (iz = 0; iz < nind; iz++) {
          multi[nind * nsize + iz] = multi[nind * iy + iz];
	}
        multi[nind * nsize + ix] += 1;
        binom[nsize] = binom[iy];
        for (; iu < nsize; iu++) {
          for (iv = 0; iv < nind; iv++) {
            if (multi[nind * nsize + iv] != multi[nind*iu + iv]) break;
	  }
          if (iv == nind) {
            binom[iu] += binom[nsize];
            break;
	  }
	}
        if (iu == nsize) nsize += 1;
      }
    }

/* Now form the matrix */

    B_REALLOC (basis, double, nfns * nsize, bytes, ier, END);
    B_ALLOC (a, double, nfns * nmono, bytes, ier, END);
    ncopy = nfns * nmono;
    dcopy_ (&ncopy, &zero, &none, &basis[nfns * dstart[ndeg+1]], &one);
    for (ix = 0; ix < nfns; ix++) {
      for (iy = dstart[ndeg+1]; iy < nsize; iy++) {
        for (iz = 0; iz < nind; iz++) {
          monomial->expon[iz] = multi[iy * nind + iz];
	}
        a[(iy - dstart[ndeg+1]) * nfns + ix] = binom[iy] *
             linfun (monomial, ix, aux);
      }
    }

/* Multiply the matrix by all of the transformations */

    B_ALLOC (e, double, nfns, bytes, ier, END);
    for (ix = 0; ix <= ndeg; ix++) {
      u = udeg[ix];
      iu = nfns - bstart[ix];
      iv = nfns - bstart[ix+1];
      for (iy = 0; iy < nmono; iy++) {
        for (iz = 0; iz < iv; iz++) {
          e[iz] = ddot_ (&iu, &u[(iu - iv + iz) * iu], &one,
                         &a[iy * nfns + bstart[ix]], &one);
	}
        dcopy_ (&iv, e, &one, &a[iy * nfns + bstart[ix+1]], &one);
      }
    }
    B_FREE (e);
  }

/* Determine the Gramian matrix for this system */

  B_ALLOC (e, double, nsize * ndep, bytes, ier, END);
  B_ALLOC (a, double, nfns * nfns, bytes, ier, END);
  for (ix = 0; ix < nfns; ix++) {
    for (iy = 0; iy < nsize; iy++) {
      for (iz = 0; iz < nind; iz++) {
        monomial->expon[iz] = multi[iy * nind + iz];
      }
      e[iy] = linfun (monomial, ix, aux);
    }
    for (iy = 0; iy < nfns; iy++) {
      a[ix + iy * nfns] = ddot_ (&nsize, &basis[iy], &nfns, e, &one);
    }
  }

/* Solve the linear system for the coefficients */

  B_ALLOC (piv, long int, nfns, bytes, ier, END);
  B_ALLOC (rhs, double, nfns * ndep, bytes, ier, END);
  for (ix = 0; ix < ndep; ix++) {
    dcopy_ (&nfns, &frhs[ix * incd], &incp, &rhs[ix * nfns], &one);
  }
  dgesv_ (&nfns, &ndep, a, &nfns, piv, rhs, &nfns, &ier);
  if (ier < 0) goto END;
  for (ix = 0; ix < nsize; ix++) {
    for (iy = 0; iy < ndep; iy++) {
      e[ix + iy * nsize] = ddot_ (&nfns, &basis[nfns * ix], &one,
                                  &rhs[iy * nfns], &one);
    }
  }

/* Store results and exit the subroutine */

END:
  B_FREE (monomial->coefs);
  B_FREE (monomial->expon);
  B_FREE (monomial);
  B_FREE (xyzscl);
  B_FREE (dstart);
  B_FREE (bstart);
  B_FREE (binom);
  B_FREE (a);
  B_FREE (basis);
  B_FREE (rhs);
  B_FREE (piv);
  if (*udeg != NULL) {
    for (ix = 0; ix <= ndeg; ix++) {
      B_FREE (udeg[ix]);
    }
    B_FREE (udeg);
  }
  interp->nterm = nsize;
  interp->expon = multi;
  interp->bounds = bndscopy;
  interp->coefs = e;
  return ier;
}
