/****************************************************************

Generate a random instance for solving for a Schubert problem.

This creates the parameter homotopy file used by Bertini as
well as the polynomial system used by alphaCertified.

November 2, 2010
Jonathan D. Hauenstein

****************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <gmp.h>
#include <time.h>
#include "Complex.h"

#define NUMREAL 4

void createInputFiles(char *inputName, char *certifyName, int numberConj);
void print_function(FILE *OUT, int funcNumber, int numberConj, complex_rational *T, complex_rational **A);

int main()
{ 
  int numConj = (8 - NUMREAL) / 2;

  // seed the random number generator
  srand((unsigned int) time(NULL));

  createInputFiles("Shapiro.bertini", "Shapiro.poly", numConj);

  return 0;
}


void createRandomRat(char **str)
{ // create random rational number in [-1,1]

  // allocate
  *str = (char *)realloc(*str, 25 * sizeof(char));

  if (rand() % 2)
    sprintf(*str, "-%d/%d", rand(), RAND_MAX);
  else
    sprintf(*str, "%d/%d", rand(), RAND_MAX);

  return;
}

void createTangentNumbers(mpq_t *T)
{ // find 8 rational numbers in [-1,1]
  int i;
  char *str = NULL;

  // we need to choose 8 rational numbers 
  for (i = 0; i < 8; i++)
  {
    createRandomRat(&str); // in [-1,1]
    mpq_set_str(T[i], str, 10);
    mpq_canonicalize(T[i]);
  }

  free(str);

  return;
}

void createInputFile(char *name, complex_rational *T, int numberConj)
{
  int i;
  mpf_t temp;
  FILE *OUT = fopen(name, "w");

  mpf_init2(temp, 1024);

  // initial configuration data
    fprintf(OUT, "CONFIG\nMAXNUMBERSTEPS: 100000;\nUSERHOMOTOPY: 1;\nPRINTPATHMODULUS: 100;\nMAXNORM: 1e100;\nSECURITYLEVEL: 1;\nTRACKTOLBEFOREEG: 1e-8;\nTRACKTOLDURINGEG: 1e-8;\nFINALTOL: 1e-10;\nSHARPENDIGITS: 75;\nCONDNUMTHRESHOLD: 1e100;\nDEGREEBOUND: 2;\nCOEFFBOUND: 250;\nSAMPLEFACTOR: 0.01;\nMAXSTEPSIZE: 0.025;\nEND;\n");

  // input declarations
    fprintf(OUT, "INPUT\nvariable x11,x12,x13,x14,x21,x22,x23,x24,x31,x32,x33,x34,x41,x42,x43,x44,p11,p12,p21,p22,p31,p32,p41,p42,p51,p52,p61,p62,p71,p72,p81,p82;\nfunction f11,f12,f13,f14,f21,f22,f23,f24,f31,f32,f33,f34,f41,f42,f43,f44,f51,f52,f53,f54,f61,f62,f63,f64,f71,f72,f73,f74,f81,f82,f83,f84;\n");
    fprintf(OUT, "constant t11,t12,t13,t14,t15,t16,t17,t18;\nconstant t01,t02,t03,t04,t05,t06,t07,t08;\nconstant m11,m12,m21,m22,m31,m32,m41,m42,m51,m52,m61,m62,m71,m72,m81,m82;\n");
    fprintf(OUT, "pathvariable t;\nparameter s;\ns = t;\none_minus_s = 1-s;\n");

  // rational values
/*
  m11 = -7507711479/9159262168;
  m12 = -504431183/721436685;
  m21 = -2497561100/7963772073;
  m22 = 2958593309/13879520717;
  m31 = -3210388087/10601726659;
  m32 = -874591348/1083338117;
  m41 = -1989258080/9038571077;
  m42 = -8659693245/19494527939;
  m51 = -3329062227/16119080717;
  m52 = -5060359377/17496099145;
  m61 = 4809937445/11754463541;
  m62 = 595305349/8583693664;
  m71 = 299732645/1814290442;
  m72 = -2165265805/18733335799;
  m81 = -45605573/971757132;
  m82 = -441142253/10075887935;
*/

    fprintf(OUT, "m11 = -0.8196851822005844346740834550593682711583237214310077383517034403987812569402151433209199520753362062733348326622546786783928641881844646855838311887918874122132235924879577046734884988390912057142461303112248680859027901558369281083340642291788584602069226412823430822883287076579725651980049317003019975134748e0;\nm12 = -0.6992036771736940435736228190281174847658322226849331899444509118634575673123692067308720237868136688945891350118964355132564405149427631338154089017527574162658501348597209192377013652972193949355375517118317874284421785398950151807154081719589848691988819503959657942817254988911466291737021939770085298614944e0;\nm21 = -0.313615341713208270520024462282221823201092008450302618297950878584869815874249468867238887890256173081210683212882052055183171990813956586223358630269025782049149311458452133427852311664219072132151414474566416938688295380098078806480176369557933730683253822450274940207972976225081875218052840925398493634161e0;\nm22 = 0.2131624981384434645244241829679888650293370213065981909438580796204592746817397181741603505832354888223911572118877510948853033078404388824977608194739798233048741376074419962263888596780859576420775625259654273982665506763118007744099828198700184266600565498475058041876331744517831969744953549753039357778279e0;\nm31 = -0.302817474007844065091925701948999853005925343580394228309635953989935819944063321091515796643970048992375176836748890188492078156304029645327983738577917999121278797346514383473692990258031514864393939662692071299137990726044430080226613773140479531391774208541212682853795881854380490305376397084489291773958e0;\nm32 = -0.8073115256222448600504656663899143502582029060092602649556731141954271327462209104565273964231796710610894160940891180698666397981083868758584444786040884777619248118821614397234395473597094894797281465911902368704340530464322248157358982689612129654254563628540728249848888128801988788510429565177018506033052e0;\nm41 = -0.2200854607496494216040339270985853420202074713407710916648910476892187191902523775440351056720467055303768343647445767604931785613041321221664161949035918390665325737748801021017849102654127416339617063565632897661175298627349611536389979311770809013244207074347931246566442196934001053494177219048050198786969e0;\nm42 = -0.4442114870437950952016638108551021323604875211130907497682701389776904820490713806968475575560332115205880287409815591944506228035625171851974845606442258155364302612364990799175257731281861330943408385550443258671204492816829115422880515024303815741654928000357362231175799491104568879912286241280089506044233e0;\n");
    fprintf(OUT, "m51 = -0.2065292857234099052995020745760311710709079142332580702344031819392247293006715700535318561368055217736025187744582221015997658149824872547041542307080439195247191341426670020688376456127401871363183148951285197537856587680985926785715468540512799517858509647910959089963095443192822867728555465859449235240156e0;\nm52 = -0.2892278635975916561942870723255723754644967176767790315353748528383753623270863100724615835503142949296541609189652314353068899461817721655349021514704042333546115716184117451170284849228887928778366556289882066737682515573102052171755683086579811445164281503618559884423883104372977648670040158257230771996748e0;\nm61 = 0.4092009327539927072882823619453514964966787845005575826984480499142840465253351709723934223056517794681379581302322914746790484770452528472392324212152660757485095678174599563377896226527587134229386776912033641229701441463682362022649792316881031193629357993344087739341944673781178390232075330404055570331536e0;\nm62 = 0.6935305153033476195592247547384048863000057780253980881114538339144531707742919750124018405324116189242421687615342187029846921620058411255064099640730142440460699087688225309900807755573696577809280030709166743162096144441344779099982568995835963234579849516866449068096659419648180026197169750255430364342508e-1;\nm71 = 0.1652065391854167084897204126923356188854353232601111834551570657505563819753618037304304996167752505858155207103273710571639554500833334600127932548519703881017304064042464927453991404536099077349380645637552226050915832361531010038799509918820373744767818161718563471327596841300010552555179034559451203899359e0;\nm72 = -0.1155835686837783353397091368713792626762917078909294791956342062258679100935065665183616986419664616614605532059837710914242924685855624532490130483460939748032538857710143585730745486649032655820381581790701770359057022314149571925900648827631683665630532479198420842869769133315261937135363896969986727989469e0;\nm81 = -0.4693104017270027095617961464058490697035604571204731842400308722406165988396368157553177597877408735087112280643390225202895655207807623273507397319518721062496920269580280271099672258438335804259371260266706228815205649553184858950950308024083532015692991075469667867588133122134883389772744163404812551455501e-1;\nm82 = -0.4378197294827297024716263877343332123909732626457600632296035952289499138817089275219296094721799821208511684384866076817874016976152484371592010962555430604836317088316246740789103466740770177095067105864948268700648266985712559932317270259516835326930420053346891456866922766147259655880640756650098649593605e-1;\n");

  // old contant values
    fprintf(OUT, "t11 = -0.48910257408929780353901151102036842294234529147980116266636904864425322581652734191220418067839123336090563695795108653261466845655877558291592984494242653579349466467218472674279396368226688289625824765463167051827171392455058816610483289323493130401153046732388909110675291414325255483246563855512 + I*-0.87222627340571601273211399529828250835018256096884934497194562953957119672006311218295422793505511999577891399521565407414163013409929587057935837717635678869858437999723334943691891637497837294958274374009611560188927012830528886135677070857609951188251501692998224381265169501522923471788277810520;\nt12 = -0.99409888069746632499601446397719624683965916361512027530014548356341060890995054342321997429758935596186988765423198035932891647258188288112654566847356371276721456638208842666707023169734946824243632141193716565873288481867879629613746075266141615914669220147182692438145676316980883045322295170302 + I*0.1084777184312271797184124011437237582229516564670370975486704121148822654990510158818871063573295200987352080902215362733498265206466392857601307803146774862130216847247461168967810420525161218564966874283066438059368822098464893222198942492031649371405371243418098543955151884072374827478390464;\nt13 = -0.8031558670969203284073500981321632920617827715345343760966695619685177863550911467447552428586577046195720919193203623443905632647099912475852728997172995012528541573073322497219772340166660745272385385011541252804838200561835022902848384549466116253347924066682552261959266171959625252196514507474891 + I*-0.595768959536995268422288063447926663440534948316296416586648866352952883402016383695505600388274227068600791673663587983868541120654993940318693410334321037760731620075072393707682392852488375265665767810326381511782083247727275227269120237937125032494382076868141137643734848372231771802859045816957;\nt14 = 0.403115374680336979729133872751852584968366727379890293630277265269436608408387426835755558182529694527961766895856579771721260914832592582952968805984924728143247599423513188958844877592539943834383785577616311701479079774973868475647711195987912782754670103672386111504073009286488143381710155815 + I*-0.915149165271067377425140421109977912647157823118525519117159212806099607239852907616432099144066580274177275520372151818895085861197454837897368302685412917456527391544341240018582729531603771601428139663850662677105127641584765608523072536963782815762826167381748114664495986198157113958329083530014;\n");
    fprintf(OUT, "t15 = 0.999865012308603806623352738824918220849702880965097392593643832179715249567038751264606840324884968149784629507486319657514114355350824378567285900668454892682114908576111654329796259175029845711177455344006226561532984472749060286716170760551927713155674950844527628056237215093843372170672366558352 + I*0.01643037312770465399602848322047128769683795314381386483544266354308064663675241777739510538766528653458576851726397915820458375632807338708098194129736351861630547819399416495223961162175873702554557912916214965852199877053869751128606825175910683115385833782621606159063611743375360715531499335249340;\nt16 = 0.7568894756027079173676952450478393635814444925515069361481286936822310631511658964352890192446682263241307617980125183803677336464031231510607991386934891550043853859276555024776364853574198896262929399524250260857288678033079738864559167496988526838170954207177849458336132063237770571279827019538 + I*0.653542899679782296473717906337764750373689899717969468374052294181819916171159872151532736955014228875040638421663515038320802977029182382267218299974826791938048782751138324908066261040361943668054874922813512626457329164019077917452187064037296186614580856510729951268546429794616394982495962931822;\nt17 = -0.1050553592401558544722206534061168865439373573464609206194770895750208767702047954908313740459359444902730686487268418804722514113151127385411479796067683594709148068406170491711307044105140059874602910578644771959335199308429451404103637198440596089472652032468436275696245631403111008038423453860337 + I*0.9944663752460017880352438623958748798336884314879452544849909518700516978373805011630663070777260264438237615153385463929124910120266342043366093432306347127304572420336580151366744886972727492494784334166588390164429208684161340787310609530696684577104558757498878053950156813995797498838804481962;\nt18 = -0.72229398927641208505434633480036497157448858990753656252568519608829825119566088180272051565255302545990396200629158229745407105900709286456905694005167186039839336982558106430817102971861732117375388395999289659958771874080306303841472764784164248590141951536793093143664184088124232868471642296325 + I*0.691586142902795852904773710179141072535622424279468273272459445913498391308001517160170857270859215306104351694550880163790051627099837516061938992607172655352091354734016845292595975589666122382375671421444094935778699399756658675292566410162200802531028803554591853795682898317154372996519065886608;\n");

  // new constant values
  for (i = 1; i <= 8; i++)
  { // print T 
    fprintf(OUT, "t0%d = ", i);
    mpf_set_q(temp, T[i-1].re);
    mpf_out_str(OUT, 10, 0, temp);
    if (mpq_cmp_ui(T[i-1].im, 0, 1))
    {
      fprintf(OUT, "+I*");
      mpf_set_q(temp, T[i-1].im);
      mpf_out_str(OUT, 10, 0, temp);
    }
    fprintf(OUT, ";\n");
  }

  // setup the homotopy
    fprintf(OUT, "t1 = s*t11 + one_minus_s*t01;\nt2 = s*t12 + one_minus_s*t02;\nt3 = s*t13 + one_minus_s*t03;\nt4 = s*t14 + one_minus_s*t04;\nt5 = s*t15 + one_minus_s*t05;\nt6 = s*t16 + one_minus_s*t06;\nt7 = s*t17 + one_minus_s*t07;\nt8 = s*t18 + one_minus_s*t08;\n");
  for (i = 0; i < numberConj; i++)
  { // setup conjugate null vectors
    fprintf(OUT, "q%d1 = p%d1 + I*p%d1*one_minus_s;\n", 2*i+1, 2*i+1, 2*i+2);
    fprintf(OUT, "q%d2 = p%d2 + I*p%d2*one_minus_s;\n", 2*i+1, 2*i+1, 2*i+2);
    fprintf(OUT, "q%d3 = (m%d1*p%d1 + m%d2*p%d2 + 1)*s + one_minus_s*((m%d1 + I*m%d1)*q%d1 + (m%d2 + I*m%d2)*q%d2 + 1);\n", 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+2, 2*i+1, 2*i+1, 2*i+2, 2*i+1); 

    // print the functions
    fprintf(OUT, "f%d1 = q%d1*(x11 + t%d*(x12 + t%d*(x13 + t%d*(x14 - t%d)))) + q%d2*(x12 + t%d*(2*x13 + t%d*(3*x14 - 4*t%d))) + q%d3*(x13 + t%d*(3*x14 - 6*t%d));\n", 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1);
    fprintf(OUT, "f%d2 = q%d1*(x21 + t%d*(x22 + t%d*(x23 + t%d*(x24 - t%d^2)))) + q%d2*(x22 + t%d*(2*x23 + t%d*(3*x24 - 5*t%d^2))) + q%d3*(x23 + t%d*(3*x24 - 10*t%d^2));\n", 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1);
    fprintf(OUT, "f%d3 = q%d1*(x31 + t%d*(x32 + t%d*(x33 + t%d*(x34 - t%d^3)))) + q%d2*(x32 + t%d*(2*x33 + t%d*(3*x34 - 6*t%d^3))) + q%d3*(x33 + t%d*(3*x34 - 15*t%d^3));\n", 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1);
    fprintf(OUT, "f%d4 = q%d1*(x41 + t%d*(x42 + t%d*(x43 + t%d*(x44 - t%d^4)))) + q%d2*(x42 + t%d*(2*x43 + t%d*(3*x44 - 7*t%d^4))) + q%d3*(x43 + t%d*(3*x44 - 21*t%d^4));\n", 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1, 2*i+1);

    // setup conjugate null vectors
    fprintf(OUT, "q%d1 = p%d1*s + one_minus_s*(p%d1 - I*p%d1);\n", 2*i+2, 2*i+2, 2*i+1, 2*i+2);
    fprintf(OUT, "q%d2 = p%d2*s + one_minus_s*(p%d2 - I*p%d2);\n", 2*i+2, 2*i+2, 2*i+1, 2*i+2);
    fprintf(OUT, "q%d3 = (m%d1*p%d1 + m%d2*p%d2 + 1)*s + one_minus_s*((m%d1 - I*m%d1)*q%d1 + (m%d2 - I*m%d2)*q%d2 + 1);\n", 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+1, 2*i+2, 2*i+2, 2*i+1, 2*i+2, 2*i+2); 

    // print the functions
    fprintf(OUT, "f%d1 = q%d1*(x11 + t%d*(x12 + t%d*(x13 + t%d*(x14 - t%d)))) + q%d2*(x12 + t%d*(2*x13 + t%d*(3*x14 - 4*t%d))) + q%d3*(x13 + t%d*(3*x14 - 6*t%d));\n", 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2);
    fprintf(OUT, "f%d2 = q%d1*(x21 + t%d*(x22 + t%d*(x23 + t%d*(x24 - t%d^2)))) + q%d2*(x22 + t%d*(2*x23 + t%d*(3*x24 - 5*t%d^2))) + q%d3*(x23 + t%d*(3*x24 - 10*t%d^2));\n", 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2);
    fprintf(OUT, "f%d3 = q%d1*(x31 + t%d*(x32 + t%d*(x33 + t%d*(x34 - t%d^3)))) + q%d2*(x32 + t%d*(2*x33 + t%d*(3*x34 - 6*t%d^3))) + q%d3*(x33 + t%d*(3*x34 - 15*t%d^3));\n", 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2);
    fprintf(OUT, "f%d4 = q%d1*(x41 + t%d*(x42 + t%d*(x43 + t%d*(x44 - t%d^4)))) + q%d2*(x42 + t%d*(2*x43 + t%d*(3*x44 - 7*t%d^4))) + q%d3*(x43 + t%d*(3*x44 - 21*t%d^4));\n", 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2, 2*i+2);
  }
  for (i = 2*numberConj; i < 8; i++)
  { // print the other vectors
    fprintf(OUT, "p%d3 = m%d1*p%d1 + m%d2*p%d2 + 1;\n", i+1, i+1, i+1, i+1, i+1);

    // print the functions
    fprintf(OUT, "f%d1 = p%d1*(x11 + t%d*(x12 + t%d*(x13 + t%d*(x14 - t%d)))) + p%d2*(x12 + t%d*(2*x13 + t%d*(3*x14 - 4*t%d))) + p%d3*(x13 + t%d*(3*x14 - 6*t%d));\n", i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1);
    fprintf(OUT, "f%d2 = p%d1*(x21 + t%d*(x22 + t%d*(x23 + t%d*(x24 - t%d^2)))) + p%d2*(x22 + t%d*(2*x23 + t%d*(3*x24 - 5*t%d^2))) + p%d3*(x23 + t%d*(3*x24 - 10*t%d^2));\n", i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1);
    fprintf(OUT, "f%d3 = p%d1*(x31 + t%d*(x32 + t%d*(x33 + t%d*(x34 - t%d^3)))) + p%d2*(x32 + t%d*(2*x33 + t%d*(3*x34 - 6*t%d^3))) + p%d3*(x33 + t%d*(3*x34 - 15*t%d^3));\n", i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1);
    fprintf(OUT, "f%d4 = p%d1*(x41 + t%d*(x42 + t%d*(x43 + t%d*(x44 - t%d^4)))) + p%d2*(x42 + t%d*(2*x43 + t%d*(3*x44 - 7*t%d^4))) + p%d3*(x43 + t%d*(3*x44 - 21*t%d^4));\n", i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1, i+1);
  }

  // close the file
    fprintf(OUT, "END;");
  fclose(OUT);

  mpf_clear(temp);

  return;
}

void createCertifyFile(char *name, complex_rational *T, mpq_t **A, int numberConj)
{
  int j,k;
  FILE *OUT = fopen(name, "w");
  // convert A to complex_rational
  complex_rational **B = (complex_rational **)malloc(8 * sizeof(complex_rational *));
  for (j = 0; j < 8; j++)
  {
    B[j] = (complex_rational *)malloc(2 * sizeof(complex_rational));
    for (k = 0; k < 2; k++)
      B[j][k].init(A[j][k]);
  }

  // print the functions
    fprintf(OUT, "32 32\n\n");
  for (j = 0; j < 8; j++)
    print_function(OUT, j, numberConj, T, B);

  // close & clear
  fclose(OUT);
  for (j = 0; j < 8; j++)
  {
    for (k = 0; k < 2; k++)
      B[j][k].clear();
    free(B[j]);
  }
  free(B);

  return;
}

void createInputFiles(char *inputName, char *certifyName, int numberConj)
{
  int i, j;

  mpq_t *Q = (mpq_t *)malloc(8 * sizeof(mpq_t));
  complex_rational *T = (complex_rational *)malloc(8 * sizeof(complex_rational));

  for (i = 0; i < 8; i++)
  { // initialize 
    mpq_init(Q[i]);
    T[i].init();
  }

  createTangentNumbers(Q);

  // verify 0 <= numberConj <= 4
  if (numberConj < 0 || numberConj > 4)
  {
    printf("ERROR: The number of conjugates ones must be between 0 & 4!\n");
    exit(1);
  }

  // setup T from Q
  for (i = 0; i < numberConj; i++)
  {
    T[2*i].set_mpq(Q[2*i], Q[2*i+1]);
    T[2*i+1] = conjugate(T[2*i]);
  }
  for (i = 2*numberConj; i < 8; i++)
  {
    T[i] = Q[i];
  }

  createInputFile(inputName, T, numberConj);

  // setup A
  mpq_t **A = (mpq_t **)malloc(8 * sizeof(mpq_t *));
  for (i = 0; i < 8; i++)
  {
    A[i] = (mpq_t *)malloc(2 * sizeof(mpq_t));
    for (j = 0; j < 2; j++)
      mpq_init(A[i][j]);
  }

  // values for setting up null vectors
  mpq_set_str(A[0][0],"-7507711479/9159262168", 10);
  mpq_set_str(A[0][1],"-504431183/721436685", 10);
  mpq_set_str(A[1][0],"-2497561100/7963772073", 10);
  mpq_set_str(A[1][1],"2958593309/13879520717", 10);
  mpq_set_str(A[2][0],"-3210388087/10601726659", 10);
  mpq_set_str(A[2][1],"-874591348/1083338117", 10);
  mpq_set_str(A[3][0],"-1989258080/9038571077", 10);
  mpq_set_str(A[3][1],"-8659693245/19494527939", 10);
  mpq_set_str(A[4][0],"-3329062227/16119080717", 10);
  mpq_set_str(A[4][1],"-5060359377/17496099145", 10);
  mpq_set_str(A[5][0],"4809937445/11754463541", 10);
  mpq_set_str(A[5][1],"595305349/8583693664", 10);
  mpq_set_str(A[6][0],"299732645/1814290442", 10);
  mpq_set_str(A[6][1],"-2165265805/18733335799", 10);
  mpq_set_str(A[7][0],"-45605573/971757132", 10);
  mpq_set_str(A[7][1],"-441142253/10075887935", 10);

  createCertifyFile(certifyName, T, A, numberConj);

  // clear memory
  for (i = 0; i < 8; i++)
  {
    mpq_clear(Q[i]);
    T[i].clear();
    for (j = 0; j < 2; j++)
      mpq_clear(A[i][j]);
    free(A[i]);
  }
  free(A);
  free(T);
  free(Q);

  return;
}

void print_conj_function(FILE *OUT, int funcNumber, complex_rational *T, complex_rational **A)
{
  complex_rational temp, I;
  I.set_I();

  if (funcNumber == 0)
  {
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][0]+T[0]*T[0]+I*A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*I*A[0][0]*T[0]+I*T[0]*T[0]*T[0]-3*A[1][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[0][0]+I*T[0]*T[0]-A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[0][1]+2*I*T[0]-A[1][1]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[0][0]*T[0]+T[0]*T[0]*T[0]+3*I*A[1][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -4*T[0]*T[0]*T[0]-6*I*A[1][1]*T[0]*T[0]-6*A[0][1]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 6*A[1][1]*T[0]*T[0]-6*I*A[0][1]*T[0]*T[0]-4*I*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 6*A[1][0]*T[0]*T[0]-I*T[0]*T[0]*T[0]*T[0]-6*I*A[0][0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -T[0]*T[0]*T[0]*T[0]-6*I*A[1][0]*T[0]*T[0]-6*A[0][0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]*T[0]+3*I*A[1][1]*T[0]+3*A[0][1]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*I*A[0][1]*T[0]-3*A[1][1]*T[0]+3*I*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 2*T[0]+I*A[1][1]+A[0][1]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*I*A[0][1]*T[0]-3*A[1][1]*T[0]+3*I*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[0][0]+I*T[0]*T[0]-A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*I*A[0][0]*T[0]+I*T[0]*T[0]*T[0]-3*A[1][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 2*T[0]+I*A[1][1]+A[0][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][0]+T[0]*T[0]+I*A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*A[0][1]*T[0]*T[0]*T[0]-5*T[0]*T[0]*T[0]*T[0]-10*I*A[1][1]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*I*A[0][1]*T[0]*T[0]*T[0]-5*I*T[0]*T[0]*T[0]*T[0]+10*A[1][1]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 10*A[1][0]*T[0]*T[0]*T[0]-I*T[0]*T[0]*T[0]*T[0]*T[0]-10*I*A[0][0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -T[0]*T[0]*T[0]*T[0]*T[0]-10*I*A[1][0]*T[0]*T[0]*T[0]-10*A[0][0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]*T[0]+3*I*A[1][1]*T[0]+3*A[0][1]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[0][1]+2*I*T[0]-A[1][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[0][0]*T[0]+T[0]*T[0]*T[0]+3*I*A[1][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[0]*T[0]*T[0]*T[0]*T[0]-15*A[0][1]*T[0]*T[0]*T[0]*T[0]-15*I*A[1][1]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*I*A[0][1]*T[0]*T[0]*T[0]*T[0]-6*I*T[0]*T[0]*T[0]*T[0]*T[0]+15*A[1][1]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 15*A[1][0]*T[0]*T[0]*T[0]*T[0]-15*I*A[0][0]*T[0]*T[0]*T[0]*T[0]-I*T[0]*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*A[0][0]*T[0]*T[0]*T[0]*T[0]-T[0]*T[0]*T[0]*T[0]*T[0]*T[0]-15*I*A[1][0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*I*A[0][1]*T[0]-3*A[1][1]*T[0]+3*I*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 2*T[0]+I*A[1][1]+A[0][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[0][1]+2*I*T[0]-A[1][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]*T[0]+3*I*A[1][1]*T[0]+3*A[0][1]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][0]+T[0]*T[0]+I*A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[0][0]*T[0]+T[0]*T[0]*T[0]+3*I*A[1][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*I*A[0][0]*T[0]+I*T[0]*T[0]*T[0]-3*A[1][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[0][0]+I*T[0]*T[0]-A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[0]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*I*A[0][1]*T[0]-3*A[1][1]*T[0]+3*I*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 2*T[0]+I*A[1][1]+A[0][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[0][1]+2*I*T[0]-A[1][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]*T[0]+3*I*A[1][1]*T[0]+3*A[0][1]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*I*A[0][0]*T[0]+I*T[0]*T[0]*T[0]-3*A[1][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][0]+T[0]*T[0]+I*A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[0][0]+I*T[0]*T[0]-A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[0][0]*T[0]+T[0]*T[0]*T[0]+3*I*A[1][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*A[0][1]*T[0]*T[0]*T[0]*T[0]*T[0]-21*I*A[1][1]*T[0]*T[0]*T[0]*T[0]*T[0]-7*T[0]*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -7*I*T[0]*T[0]*T[0]*T[0]*T[0]*T[0]+21*A[1][1]*T[0]*T[0]*T[0]*T[0]*T[0]-21*I*A[0][1]*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*I*A[0][0]*T[0]*T[0]*T[0]*T[0]*T[0]-I*T[0]*T[0]*T[0]*T[0]*T[0]*T[0]*T[0]+21*A[1][0]*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*I*A[1][0]*T[0]*T[0]*T[0]*T[0]*T[0]-21*A[0][0]*T[0]*T[0]*T[0]*T[0]*T[0]-T[0]*T[0]*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]; temp.print_str(OUT);
  }
  else if (funcNumber == 1)
  {
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*A[1][0]+A[0][0]+T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -3*A[1][0]*T[1]-3*I*A[0][0]*T[1]-I*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*A[0][0]-A[1][0]-I*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -A[1][1]-2*I*T[1]-I*A[0][1]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]*T[1]+3*A[0][0]*T[1]-3*I*A[1][0]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -4*T[1]*T[1]*T[1]-6*A[0][1]*T[1]*T[1]+6*I*A[1][1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 6*A[1][1]*T[1]*T[1]+6*I*A[0][1]*T[1]*T[1]+4*I*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 6*I*A[0][0]*T[1]*T[1]+6*A[1][0]*T[1]*T[1]+I*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -T[1]*T[1]*T[1]*T[1]-6*A[0][0]*T[1]*T[1]+6*I*A[1][0]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[0][1]*T[1]+3*T[1]*T[1]-3*I*A[1][1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -3*A[1][1]*T[1]-3*I*A[0][1]*T[1]-3*I*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][1]+2*T[1]-I*A[1][1]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -3*A[1][1]*T[1]-3*I*A[0][1]*T[1]-3*I*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*A[0][0]-A[1][0]-I*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -3*A[1][0]*T[1]-3*I*A[0][0]*T[1]-I*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][1]+2*T[1]-I*A[1][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*A[1][0]+A[0][0]+T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 10*I*A[1][1]*T[1]*T[1]*T[1]-5*T[1]*T[1]*T[1]*T[1]-10*A[0][1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 5*I*T[1]*T[1]*T[1]*T[1]+10*A[1][1]*T[1]*T[1]*T[1]+10*I*A[0][1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 10*A[1][0]*T[1]*T[1]*T[1]+10*I*A[0][0]*T[1]*T[1]*T[1]+I*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 10*I*A[1][0]*T[1]*T[1]*T[1]-10*A[0][0]*T[1]*T[1]*T[1]-T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[0][1]*T[1]+3*T[1]*T[1]-3*I*A[1][1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -A[1][1]-2*I*T[1]-I*A[0][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]*T[1]+3*A[0][0]*T[1]-3*I*A[1][0]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*A[0][1]*T[1]*T[1]*T[1]*T[1]-6*T[1]*T[1]*T[1]*T[1]*T[1]+15*I*A[1][1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 15*I*A[0][1]*T[1]*T[1]*T[1]*T[1]+15*A[1][1]*T[1]*T[1]*T[1]*T[1]+6*I*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 15*I*A[0][0]*T[1]*T[1]*T[1]*T[1]+15*A[1][0]*T[1]*T[1]*T[1]*T[1]+I*T[1]*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*A[0][0]*T[1]*T[1]*T[1]*T[1]-T[1]*T[1]*T[1]*T[1]*T[1]*T[1]+15*I*A[1][0]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -3*A[1][1]*T[1]-3*I*A[0][1]*T[1]-3*I*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][1]+2*T[1]-I*A[1][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -A[1][1]-2*I*T[1]-I*A[0][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[0][1]*T[1]+3*T[1]*T[1]-3*I*A[1][1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*A[1][0]+A[0][0]+T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]*T[1]+3*A[0][0]*T[1]-3*I*A[1][0]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -3*A[1][0]*T[1]-3*I*A[0][0]*T[1]-I*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*A[0][0]-A[1][0]-I*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[1]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -3*A[1][1]*T[1]-3*I*A[0][1]*T[1]-3*I*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][1]+2*T[1]-I*A[1][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -A[1][1]-2*I*T[1]-I*A[0][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[0][1]*T[1]+3*T[1]*T[1]-3*I*A[1][1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -3*A[1][0]*T[1]-3*I*A[0][0]*T[1]-I*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*A[1][0]+A[0][0]+T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*A[0][0]-A[1][0]-I*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]*T[1]+3*A[0][0]*T[1]-3*I*A[1][0]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -7*T[1]*T[1]*T[1]*T[1]*T[1]*T[1]-21*A[0][1]*T[1]*T[1]*T[1]*T[1]*T[1]+21*I*A[1][1]*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 7*I*T[1]*T[1]*T[1]*T[1]*T[1]*T[1]+21*A[1][1]*T[1]*T[1]*T[1]*T[1]*T[1]+21*I*A[0][1]*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 21*I*A[0][0]*T[1]*T[1]*T[1]*T[1]*T[1]+I*T[1]*T[1]*T[1]*T[1]*T[1]*T[1]*T[1]+21*A[1][0]*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 21*I*A[1][0]*T[1]*T[1]*T[1]*T[1]*T[1]-21*A[0][0]*T[1]*T[1]*T[1]*T[1]*T[1]-T[1]*T[1]*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]; temp.print_str(OUT);
  }
  else if (funcNumber == 2)
  {
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -T[2]*T[2]*T[2]*T[2]-6*A[2][0]*T[2]*T[2]-6*I*A[3][0]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*I*A[3][1]*T[2]*T[2]-4*T[2]*T[2]*T[2]-6*A[2][1]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -6*I*A[2][0]*T[2]*T[2]-I*T[2]*T[2]*T[2]*T[2]+6*A[3][0]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -4*I*T[2]*T[2]*T[2]+6*A[3][1]*T[2]*T[2]-6*I*A[2][1]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -3*A[3][1]*T[2]+3*I*T[2]*T[2]+3*I*A[2][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[3][1]+2*T[2]+A[2][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 2*I*T[2]+I*A[2][1]-A[3][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][1]*T[2]+3*T[2]*T[2]+3*I*A[3][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[2]*T[2]*T[2]+3*I*A[2][0]*T[2]-3*A[3][0]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]+A[2][0]+I*A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[2][0]+I*T[2]*T[2]-A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][0]*T[2]+3*I*A[3][0]*T[2]+T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*A[2][0]*T[2]*T[2]*T[2]-T[2]*T[2]*T[2]*T[2]*T[2]-10*I*A[3][0]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*I*A[3][1]*T[2]*T[2]*T[2]-5*T[2]*T[2]*T[2]*T[2]-10*A[2][1]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -10*I*A[2][0]*T[2]*T[2]*T[2]+10*A[3][0]*T[2]*T[2]*T[2]-I*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -5*I*T[2]*T[2]*T[2]*T[2]+10*A[3][1]*T[2]*T[2]*T[2]-10*I*A[2][1]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[2]*T[2]*T[2]+3*I*A[2][0]*T[2]-3*A[3][0]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[2][0]+I*T[2]*T[2]-A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 2*I*T[2]+I*A[2][1]-A[3][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -3*A[3][1]*T[2]+3*I*T[2]*T[2]+3*I*A[2][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[3][1]+2*T[2]+A[2][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][1]*T[2]+3*T[2]*T[2]+3*I*A[3][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]+A[2][0]+I*A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][0]*T[2]+3*I*A[3][0]*T[2]+T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]+A[2][0]+I*A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][0]*T[2]+3*I*A[3][0]*T[2]+T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*A[2][0]*T[2]*T[2]*T[2]*T[2]-15*I*A[3][0]*T[2]*T[2]*T[2]*T[2]-T[2]*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*I*A[3][1]*T[2]*T[2]*T[2]*T[2]-15*A[2][1]*T[2]*T[2]*T[2]*T[2]-6*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[2]*T[2]*T[2]*T[2]*T[2]*T[2]-15*I*A[2][0]*T[2]*T[2]*T[2]*T[2]+15*A[3][0]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 15*A[3][1]*T[2]*T[2]*T[2]*T[2]-15*I*A[2][1]*T[2]*T[2]*T[2]*T[2]-6*I*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -3*A[3][1]*T[2]+3*I*T[2]*T[2]+3*I*A[2][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[2]*T[2]*T[2]+3*I*A[2][0]*T[2]-3*A[3][0]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][1]*T[2]+3*T[2]*T[2]+3*I*A[3][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 2*I*T[2]+I*A[2][1]-A[3][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[3][1]+2*T[2]+A[2][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[2][0]+I*T[2]*T[2]-A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[2]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[3][1]+2*T[2]+A[2][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*A[2][0]*T[2]*T[2]*T[2]*T[2]*T[2]-T[2]*T[2]*T[2]*T[2]*T[2]*T[2]*T[2]-21*I*A[3][0]*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -7*T[2]*T[2]*T[2]*T[2]*T[2]*T[2]-21*I*A[3][1]*T[2]*T[2]*T[2]*T[2]*T[2]-21*A[2][1]*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[2]*T[2]*T[2]*T[2]*T[2]*T[2]*T[2]-21*I*A[2][0]*T[2]*T[2]*T[2]*T[2]*T[2]+21*A[3][0]*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -7*I*T[2]*T[2]*T[2]*T[2]*T[2]*T[2]+21*A[3][1]*T[2]*T[2]*T[2]*T[2]*T[2]-21*I*A[2][1]*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][1]*T[2]+3*T[2]*T[2]+3*I*A[3][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 2*I*T[2]+I*A[2][1]-A[3][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -3*A[3][1]*T[2]+3*I*T[2]*T[2]+3*I*A[2][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[2]*T[2]*T[2]+3*I*A[2][0]*T[2]-3*A[3][0]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]+A[2][0]+I*A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][0]*T[2]+3*I*A[3][0]*T[2]+T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*A[2][0]+I*T[2]*T[2]-A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[2]; temp.print_str(OUT);
  }
  else if (funcNumber == 3)
  {
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*A[2][0]*T[3]*T[3]+6*I*A[3][0]*T[3]*T[3]-T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -4*T[3]*T[3]*T[3]-6*A[2][1]*T[3]*T[3]+6*I*A[3][1]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = 6*A[3][0]*T[3]*T[3]+6*I*A[2][0]*T[3]*T[3]+I*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 6*A[3][1]*T[3]*T[3]+6*I*A[2][1]*T[3]*T[3]+4*I*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -3*A[3][1]*T[3]-3*I*T[3]*T[3]-3*I*A[2][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 2*T[3]-I*A[3][1]+A[2][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -I*A[2][1]-A[3][1]-2*I*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][1]*T[3]+3*T[3]*T[3]-3*I*A[3][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]*T[3]*T[3]-3*A[3][0]*T[3]-3*I*A[2][0]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[2][0]+T[3]*T[3]-I*A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]*T[3]-I*A[2][0]-A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][0]*T[3]-3*I*A[3][0]*T[3]+T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 10*I*A[3][0]*T[3]*T[3]*T[3]-10*A[2][0]*T[3]*T[3]*T[3]-T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -5*T[3]*T[3]*T[3]*T[3]+10*I*A[3][1]*T[3]*T[3]*T[3]-10*A[2][1]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = 10*I*A[2][0]*T[3]*T[3]*T[3]+10*A[3][0]*T[3]*T[3]*T[3]+I*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 10*A[3][1]*T[3]*T[3]*T[3]+5*I*T[3]*T[3]*T[3]*T[3]+10*I*A[2][1]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]*T[3]*T[3]-3*A[3][0]*T[3]-3*I*A[2][0]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]*T[3]-I*A[2][0]-A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -I*A[2][1]-A[3][1]-2*I*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -3*A[3][1]*T[3]-3*I*T[3]*T[3]-3*I*A[2][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 2*T[3]-I*A[3][1]+A[2][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][1]*T[3]+3*T[3]*T[3]-3*I*A[3][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[2][0]+T[3]*T[3]-I*A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][0]*T[3]-3*I*A[3][0]*T[3]+T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[2][0]+T[3]*T[3]-I*A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][0]*T[3]-3*I*A[3][0]*T[3]+T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -T[3]*T[3]*T[3]*T[3]*T[3]*T[3]-15*A[2][0]*T[3]*T[3]*T[3]*T[3]+15*I*A[3][0]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[3]*T[3]*T[3]*T[3]*T[3]-15*A[2][1]*T[3]*T[3]*T[3]*T[3]+15*I*A[3][1]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = 15*A[3][0]*T[3]*T[3]*T[3]*T[3]+15*I*A[2][0]*T[3]*T[3]*T[3]*T[3]+I*T[3]*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 15*A[3][1]*T[3]*T[3]*T[3]*T[3]+15*I*A[2][1]*T[3]*T[3]*T[3]*T[3]+6*I*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -3*A[3][1]*T[3]-3*I*T[3]*T[3]-3*I*A[2][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]*T[3]*T[3]-3*A[3][0]*T[3]-3*I*A[2][0]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][1]*T[3]+3*T[3]*T[3]-3*I*A[3][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -I*A[2][1]-A[3][1]-2*I*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 2*T[3]-I*A[3][1]+A[2][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]*T[3]-I*A[2][0]-A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 2*T[3]-I*A[3][1]+A[2][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -T[3]*T[3]*T[3]*T[3]*T[3]*T[3]*T[3]-21*A[2][0]*T[3]*T[3]*T[3]*T[3]*T[3]+21*I*A[3][0]*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -7*T[3]*T[3]*T[3]*T[3]*T[3]*T[3]-21*A[2][1]*T[3]*T[3]*T[3]*T[3]*T[3]+21*I*A[3][1]*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = I*T[3]*T[3]*T[3]*T[3]*T[3]*T[3]*T[3]+21*A[3][0]*T[3]*T[3]*T[3]*T[3]*T[3]+21*I*A[2][0]*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 21*I*A[2][1]*T[3]*T[3]*T[3]*T[3]*T[3]+21*A[3][1]*T[3]*T[3]*T[3]*T[3]*T[3]+7*I*T[3]*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][1]*T[3]+3*T[3]*T[3]-3*I*A[3][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -I*A[2][1]-A[3][1]-2*I*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -3*A[3][1]*T[3]-3*I*T[3]*T[3]-3*I*A[2][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]*T[3]*T[3]-3*A[3][0]*T[3]-3*I*A[2][0]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[2][0]+T[3]*T[3]-I*A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*A[2][0]*T[3]-3*I*A[3][0]*T[3]+T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]*T[3]-I*A[2][0]-A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -I*T[3]; temp.print_str(OUT);
  }
  else if (funcNumber == 4)
  {
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 3*I*A[5][0]*T[4]+T[4]*T[4]*T[4]+3*A[4][0]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = I*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -3*A[5][0]*T[4]+I*T[4]*T[4]*T[4]+3*I*A[4][0]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -A[5][0]+I*A[4][0]+I*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 2*I*T[4]+I*A[4][1]-A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]*T[4]+A[4][0]+I*A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 2*T[4]+I*A[5][1]+A[4][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -T[4]*T[4]*T[4]*T[4]-6*I*A[5][0]*T[4]*T[4]-6*A[4][0]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -4*T[4]*T[4]*T[4]-6*I*A[5][1]*T[4]*T[4]-6*A[4][1]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[4]*T[4]*T[4]*T[4]+6*A[5][0]*T[4]*T[4]-6*I*A[4][0]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 6*A[5][1]*T[4]*T[4]-4*I*T[4]*T[4]*T[4]-6*I*A[4][1]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 3*I*T[4]*T[4]-3*A[5][1]*T[4]+3*I*A[4][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*A[4][1]*T[4]+3*T[4]*T[4]+3*I*A[5][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -3*A[5][0]*T[4]+I*T[4]*T[4]*T[4]+3*I*A[4][0]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 3*I*A[5][0]*T[4]+T[4]*T[4]*T[4]+3*A[4][0]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 3*I*T[4]*T[4]-3*A[5][1]*T[4]+3*I*A[4][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*A[4][1]*T[4]+3*T[4]*T[4]+3*I*A[5][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = I*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]*T[4]+A[4][0]+I*A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 2*T[4]+I*A[5][1]+A[4][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -A[5][0]+I*A[4][0]+I*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 2*I*T[4]+I*A[4][1]-A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -10*I*A[5][0]*T[4]*T[4]*T[4]-T[4]*T[4]*T[4]*T[4]*T[4]-10*A[4][0]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -10*A[4][1]*T[4]*T[4]*T[4]-10*I*A[5][1]*T[4]*T[4]*T[4]-5*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[4]*T[4]*T[4]*T[4]*T[4]+10*A[5][0]*T[4]*T[4]*T[4]-10*I*A[4][0]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -5*I*T[4]*T[4]*T[4]*T[4]+10*A[5][1]*T[4]*T[4]*T[4]-10*I*A[4][1]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -3*A[5][0]*T[4]+I*T[4]*T[4]*T[4]+3*I*A[4][0]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 3*I*A[5][0]*T[4]+T[4]*T[4]*T[4]+3*A[4][0]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 3*I*T[4]*T[4]-3*A[5][1]*T[4]+3*I*A[4][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*A[4][1]*T[4]+3*T[4]*T[4]+3*I*A[5][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = I*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -A[5][0]+I*A[4][0]+I*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 2*I*T[4]+I*A[4][1]-A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]*T[4]+A[4][0]+I*A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 2*T[4]+I*A[5][1]+A[4][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -15*A[4][0]*T[4]*T[4]*T[4]*T[4]-T[4]*T[4]*T[4]*T[4]*T[4]*T[4]-15*I*A[5][0]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -15*A[4][1]*T[4]*T[4]*T[4]*T[4]-15*I*A[5][1]*T[4]*T[4]*T[4]*T[4]-6*T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[4]*T[4]*T[4]*T[4]*T[4]*T[4]-15*I*A[4][0]*T[4]*T[4]*T[4]*T[4]+15*A[5][0]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -6*I*T[4]*T[4]*T[4]*T[4]*T[4]-15*I*A[4][1]*T[4]*T[4]*T[4]*T[4]+15*A[5][1]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -3*A[5][0]*T[4]+I*T[4]*T[4]*T[4]+3*I*A[4][0]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 3*I*A[5][0]*T[4]+T[4]*T[4]*T[4]+3*A[4][0]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = I*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -A[5][0]+I*A[4][0]+I*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 2*I*T[4]+I*A[4][1]-A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 3*I*T[4]*T[4]-3*A[5][1]*T[4]+3*I*A[4][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -21*I*A[5][0]*T[4]*T[4]*T[4]*T[4]*T[4]-21*A[4][0]*T[4]*T[4]*T[4]*T[4]*T[4]-T[4]*T[4]*T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -21*I*A[5][1]*T[4]*T[4]*T[4]*T[4]*T[4]-21*A[4][1]*T[4]*T[4]*T[4]*T[4]*T[4]-7*T[4]*T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -21*I*A[4][0]*T[4]*T[4]*T[4]*T[4]*T[4]-I*T[4]*T[4]*T[4]*T[4]*T[4]*T[4]*T[4]+21*A[5][0]*T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -7*I*T[4]*T[4]*T[4]*T[4]*T[4]*T[4]+21*A[5][1]*T[4]*T[4]*T[4]*T[4]*T[4]-21*I*A[4][1]*T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*A[4][1]*T[4]+3*T[4]*T[4]+3*I*A[5][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 2*T[4]+I*A[5][1]+A[4][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]*T[4]+A[4][0]+I*A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = I; temp.print_str(OUT);
  }
  else if (funcNumber == 5)
  {
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -3*I*A[5][0]*T[5]+3*A[4][0]*T[5]+T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[5]*T[5]*T[5]-3*A[5][0]*T[5]-3*I*A[4][0]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*A[4][0]-I*T[5]*T[5]-A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -I*A[4][1]-2*I*T[5]-A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = A[4][0]+T[5]*T[5]-I*A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -I*A[5][1]+A[4][1]+2*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -6*A[4][0]*T[5]*T[5]-T[5]*T[5]*T[5]*T[5]+6*I*A[5][0]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 6*I*A[5][1]*T[5]*T[5]-4*T[5]*T[5]*T[5]-6*A[4][1]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = I*T[5]*T[5]*T[5]*T[5]+6*A[5][0]*T[5]*T[5]+6*I*A[4][0]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 6*I*A[4][1]*T[5]*T[5]+4*I*T[5]*T[5]*T[5]+6*A[5][1]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -3*I*T[5]*T[5]-3*A[5][1]*T[5]-3*I*A[4][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*A[4][1]*T[5]+3*T[5]*T[5]-3*I*A[5][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[5]*T[5]*T[5]-3*A[5][0]*T[5]-3*I*A[4][0]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -3*I*A[5][0]*T[5]+3*A[4][0]*T[5]+T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -3*I*T[5]*T[5]-3*A[5][1]*T[5]-3*I*A[4][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*A[4][1]*T[5]+3*T[5]*T[5]-3*I*A[5][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = A[4][0]+T[5]*T[5]-I*A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -I*A[5][1]+A[4][1]+2*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*A[4][0]-I*T[5]*T[5]-A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -I*A[4][1]-2*I*T[5]-A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 10*I*A[5][0]*T[5]*T[5]*T[5]-T[5]*T[5]*T[5]*T[5]*T[5]-10*A[4][0]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -10*A[4][1]*T[5]*T[5]*T[5]-5*T[5]*T[5]*T[5]*T[5]+10*I*A[5][1]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = I*T[5]*T[5]*T[5]*T[5]*T[5]+10*A[5][0]*T[5]*T[5]*T[5]+10*I*A[4][0]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 10*I*A[4][1]*T[5]*T[5]*T[5]+10*A[5][1]*T[5]*T[5]*T[5]+5*I*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[5]*T[5]*T[5]-3*A[5][0]*T[5]-3*I*A[4][0]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -3*I*A[5][0]*T[5]+3*A[4][0]*T[5]+T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -3*I*T[5]*T[5]-3*A[5][1]*T[5]-3*I*A[4][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*A[4][1]*T[5]+3*T[5]*T[5]-3*I*A[5][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*A[4][0]-I*T[5]*T[5]-A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -I*A[4][1]-2*I*T[5]-A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = A[4][0]+T[5]*T[5]-I*A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -I*A[5][1]+A[4][1]+2*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -T[5]*T[5]*T[5]*T[5]*T[5]*T[5]+15*I*A[5][0]*T[5]*T[5]*T[5]*T[5]-15*A[4][0]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -15*A[4][1]*T[5]*T[5]*T[5]*T[5]-6*T[5]*T[5]*T[5]*T[5]*T[5]+15*I*A[5][1]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = 15*A[5][0]*T[5]*T[5]*T[5]*T[5]+15*I*A[4][0]*T[5]*T[5]*T[5]*T[5]+I*T[5]*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 15*A[5][1]*T[5]*T[5]*T[5]*T[5]+15*I*A[4][1]*T[5]*T[5]*T[5]*T[5]+6*I*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[5]*T[5]*T[5]-3*A[5][0]*T[5]-3*I*A[4][0]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -3*I*A[5][0]*T[5]+3*A[4][0]*T[5]+T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -I*A[4][0]-I*T[5]*T[5]-A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -I*A[4][1]-2*I*T[5]-A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -3*I*T[5]*T[5]-3*A[5][1]*T[5]-3*I*A[4][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -21*A[4][0]*T[5]*T[5]*T[5]*T[5]*T[5]+21*I*A[5][0]*T[5]*T[5]*T[5]*T[5]*T[5]-T[5]*T[5]*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -7*T[5]*T[5]*T[5]*T[5]*T[5]*T[5]-21*A[4][1]*T[5]*T[5]*T[5]*T[5]*T[5]+21*I*A[5][1]*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = 21*A[5][0]*T[5]*T[5]*T[5]*T[5]*T[5]+21*I*A[4][0]*T[5]*T[5]*T[5]*T[5]*T[5]+I*T[5]*T[5]*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 21*I*A[4][1]*T[5]*T[5]*T[5]*T[5]*T[5]+7*I*T[5]*T[5]*T[5]*T[5]*T[5]*T[5]+21*A[5][1]*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*A[4][1]*T[5]+3*T[5]*T[5]-3*I*A[5][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -I*A[5][1]+A[4][1]+2*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = A[4][0]+T[5]*T[5]-I*A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -I; temp.print_str(OUT);
  }
  else if (funcNumber == 6)
  {
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*T[6]*T[6]*T[6]+3*I*A[6][0]*T[6]-3*A[7][0]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]*T[6]*T[6]+3*I*A[7][0]*T[6]+3*A[6][0]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 3*I*A[6][1]*T[6]+3*I*T[6]*T[6]-3*A[7][1]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 3*A[6][1]*T[6]+3*I*A[7][1]*T[6]+3*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = I*A[7][0]+T[6]*T[6]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*A[6][0]-A[7][0]+I*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = I*A[7][1]+2*T[6]+A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 2*I*T[6]+I*A[6][1]-A[7][1]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -6*A[6][0]*T[6]*T[6]-6*I*A[7][0]*T[6]*T[6]-T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -6*A[6][1]*T[6]*T[6]-4*T[6]*T[6]*T[6]-6*I*A[7][1]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = 6*A[7][0]*T[6]*T[6]-6*I*A[6][0]*T[6]*T[6]-I*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -4*I*T[6]*T[6]*T[6]+6*A[7][1]*T[6]*T[6]-6*I*A[6][1]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*T[6]*T[6]*T[6]+3*I*A[6][0]*T[6]-3*A[7][0]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 3*I*A[6][1]*T[6]+3*I*T[6]*T[6]-3*A[7][1]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = I*A[7][1]+2*T[6]+A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 3*A[6][1]*T[6]+3*I*A[7][1]*T[6]+3*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = I*A[7][0]+T[6]*T[6]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*A[6][0]-A[7][0]+I*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 2*I*T[6]+I*A[6][1]-A[7][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]*T[6]*T[6]+3*I*A[7][0]*T[6]+3*A[6][0]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -10*A[6][0]*T[6]*T[6]*T[6]-10*I*A[7][0]*T[6]*T[6]*T[6]-T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -5*T[6]*T[6]*T[6]*T[6]-10*I*A[7][1]*T[6]*T[6]*T[6]-10*A[6][1]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[6]*T[6]*T[6]*T[6]*T[6]+10*A[7][0]*T[6]*T[6]*T[6]-10*I*A[6][0]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 10*A[7][1]*T[6]*T[6]*T[6]-5*I*T[6]*T[6]*T[6]*T[6]-10*I*A[6][1]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*T[6]*T[6]*T[6]+3*I*A[6][0]*T[6]-3*A[7][0]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]*T[6]*T[6]+3*I*A[7][0]*T[6]+3*A[6][0]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 3*I*A[6][1]*T[6]+3*I*T[6]*T[6]-3*A[7][1]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -15*A[6][0]*T[6]*T[6]*T[6]*T[6]-15*I*A[7][0]*T[6]*T[6]*T[6]*T[6]-T[6]*T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -15*A[6][1]*T[6]*T[6]*T[6]*T[6]-6*T[6]*T[6]*T[6]*T[6]*T[6]-15*I*A[7][1]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[6]*T[6]*T[6]*T[6]*T[6]*T[6]-15*I*A[6][0]*T[6]*T[6]*T[6]*T[6]+15*A[7][0]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 15*A[7][1]*T[6]*T[6]*T[6]*T[6]-6*I*T[6]*T[6]*T[6]*T[6]*T[6]-15*I*A[6][1]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = I*A[7][1]+2*T[6]+A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = I*A[7][0]+T[6]*T[6]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*A[6][0]-A[7][0]+I*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 2*I*T[6]+I*A[6][1]-A[7][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 3*A[6][1]*T[6]+3*I*A[7][1]*T[6]+3*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*A[6][0]-A[7][0]+I*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I*T[6]*T[6]*T[6]+3*I*A[6][0]*T[6]-3*A[7][0]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 2*I*T[6]+I*A[6][1]-A[7][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 3*I*A[6][1]*T[6]+3*I*T[6]*T[6]-3*A[7][1]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]*T[6]*T[6]+3*I*A[7][0]*T[6]+3*A[6][0]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = I*A[7][0]+T[6]*T[6]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 3*A[6][1]*T[6]+3*I*A[7][1]*T[6]+3*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = I*A[7][1]+2*T[6]+A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -21*A[6][0]*T[6]*T[6]*T[6]*T[6]*T[6]-21*I*A[7][0]*T[6]*T[6]*T[6]*T[6]*T[6]-T[6]*T[6]*T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -7*T[6]*T[6]*T[6]*T[6]*T[6]*T[6]-21*I*A[7][1]*T[6]*T[6]*T[6]*T[6]*T[6]-21*A[6][1]*T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -21*I*A[6][0]*T[6]*T[6]*T[6]*T[6]*T[6]+21*A[7][0]*T[6]*T[6]*T[6]*T[6]*T[6]-I*T[6]*T[6]*T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -21*I*A[6][1]*T[6]*T[6]*T[6]*T[6]*T[6]+21*A[7][1]*T[6]*T[6]*T[6]*T[6]*T[6]-7*I*T[6]*T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
  }
  else if (funcNumber == 7)
  {
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]*T[7]*T[7]-3*A[7][0]*T[7]-3*I*A[6][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]*T[7]*T[7]+3*A[6][0]*T[7]-3*I*A[7][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -3*I*A[6][1]*T[7]-3*I*T[7]*T[7]-3*A[7][1]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -3*I*A[7][1]*T[7]+3*A[6][1]*T[7]+3*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]*T[7]-I*A[7][0]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]*T[7]-I*A[6][0]-A[7][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = A[6][1]-I*A[7][1]+2*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -2*I*T[7]-A[7][1]-I*A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 6*I*A[7][0]*T[7]*T[7]-T[7]*T[7]*T[7]*T[7]-6*A[6][0]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 6*I*A[7][1]*T[7]*T[7]-6*A[6][1]*T[7]*T[7]-4*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = 6*A[7][0]*T[7]*T[7]+I*T[7]*T[7]*T[7]*T[7]+6*I*A[6][0]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 6*I*A[6][1]*T[7]*T[7]+4*I*T[7]*T[7]*T[7]+6*A[7][1]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]*T[7]*T[7]-3*A[7][0]*T[7]-3*I*A[6][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -3*I*A[6][1]*T[7]-3*I*T[7]*T[7]-3*A[7][1]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = A[6][1]-I*A[7][1]+2*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -3*I*A[7][1]*T[7]+3*A[6][1]*T[7]+3*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]*T[7]-I*A[7][0]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]*T[7]-I*A[6][0]-A[7][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -2*I*T[7]-A[7][1]-I*A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]*T[7]*T[7]+3*A[6][0]*T[7]-3*I*A[7][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -T[7]*T[7]*T[7]*T[7]*T[7]-10*A[6][0]*T[7]*T[7]*T[7]+10*I*A[7][0]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -5*T[7]*T[7]*T[7]*T[7]+10*I*A[7][1]*T[7]*T[7]*T[7]-10*A[6][1]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = 10*A[7][0]*T[7]*T[7]*T[7]+10*I*A[6][0]*T[7]*T[7]*T[7]+I*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 10*I*A[6][1]*T[7]*T[7]*T[7]+5*I*T[7]*T[7]*T[7]*T[7]+10*A[7][1]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]*T[7]*T[7]-3*A[7][0]*T[7]-3*I*A[6][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]*T[7]*T[7]+3*A[6][0]*T[7]-3*I*A[7][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -3*I*A[6][1]*T[7]-3*I*T[7]*T[7]-3*A[7][1]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -15*A[6][0]*T[7]*T[7]*T[7]*T[7]-T[7]*T[7]*T[7]*T[7]*T[7]*T[7]+15*I*A[7][0]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -15*A[6][1]*T[7]*T[7]*T[7]*T[7]-6*T[7]*T[7]*T[7]*T[7]*T[7]+15*I*A[7][1]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = 15*I*A[6][0]*T[7]*T[7]*T[7]*T[7]+15*A[7][0]*T[7]*T[7]*T[7]*T[7]+I*T[7]*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 15*A[7][1]*T[7]*T[7]*T[7]*T[7]+6*I*T[7]*T[7]*T[7]*T[7]*T[7]+15*I*A[6][1]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = A[6][1]-I*A[7][1]+2*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]*T[7]-I*A[7][0]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]*T[7]-I*A[6][0]-A[7][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -2*I*T[7]-A[7][1]-I*A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -3*I*A[7][1]*T[7]+3*A[6][1]*T[7]+3*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "21\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]*T[7]-I*A[6][0]-A[7][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I*T[7]*T[7]*T[7]-3*A[7][0]*T[7]-3*I*A[6][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -2*I*T[7]-A[7][1]-I*A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -3*I*A[6][1]*T[7]-3*I*T[7]*T[7]-3*A[7][1]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]*T[7]*T[7]+3*A[6][0]*T[7]-3*I*A[7][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[7]*T[7]-I*A[7][0]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -3*I*A[7][1]*T[7]+3*A[6][1]*T[7]+3*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = A[6][1]-I*A[7][1]+2*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -I; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -T[7]*T[7]*T[7]*T[7]*T[7]*T[7]*T[7]-21*A[6][0]*T[7]*T[7]*T[7]*T[7]*T[7]+21*I*A[7][0]*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -7*T[7]*T[7]*T[7]*T[7]*T[7]*T[7]+21*I*A[7][1]*T[7]*T[7]*T[7]*T[7]*T[7]-21*A[6][1]*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = 21*A[7][0]*T[7]*T[7]*T[7]*T[7]*T[7]+21*I*A[6][0]*T[7]*T[7]*T[7]*T[7]*T[7]+I*T[7]*T[7]*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 7*I*T[7]*T[7]*T[7]*T[7]*T[7]*T[7]+21*A[7][1]*T[7]*T[7]*T[7]*T[7]*T[7]+21*I*A[6][1]*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
  }

  return;
}

void print_std_function(FILE *OUT, int funcNumber, complex_rational *T, complex_rational **A)
{
  complex_rational temp, I;
  I.set_I();

  if (funcNumber == 0)
  {
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]*T[0]+A[0][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]*T[0]*T[0]+3*A[0][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][1]+2*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]*T[0]+3*A[0][1]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*A[0][1]*T[0]*T[0]-4*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*A[0][0]*T[0]*T[0]-T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]*T[0]+A[0][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]*T[0]*T[0]+3*A[0][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][1]+2*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]*T[0]+3*A[0][1]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*A[0][1]*T[0]*T[0]*T[0]-5*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*A[0][0]*T[0]*T[0]*T[0]-T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]*T[0]+A[0][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]*T[0]*T[0]+3*A[0][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][1]+2*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]*T[0]+3*A[0][1]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*A[0][1]*T[0]*T[0]*T[0]*T[0]-6*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*A[0][0]*T[0]*T[0]*T[0]*T[0]-T[0]*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]*T[0]+A[0][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[0]*T[0]*T[0]+3*A[0][0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[0][1]+2*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]*T[0]+3*A[0][1]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*A[0][1]*T[0]*T[0]*T[0]*T[0]*T[0]-7*T[0]*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -T[0]*T[0]*T[0]*T[0]*T[0]*T[0]*T[0]-21*A[0][0]*T[0]*T[0]*T[0]*T[0]*T[0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
  }
  else if (funcNumber == 1)
  {
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]+A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]*T[1]+3*A[1][0]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[1][1]+2*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]*T[1]+3*A[1][1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*A[1][1]*T[1]*T[1]-4*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -T[1]*T[1]*T[1]*T[1]-6*A[1][0]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]+A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]*T[1]+3*A[1][0]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[1][1]+2*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]*T[1]+3*A[1][1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*A[1][1]*T[1]*T[1]*T[1]-5*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*A[1][0]*T[1]*T[1]*T[1]-T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]+A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]*T[1]+3*A[1][0]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[1][1]+2*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]*T[1]+3*A[1][1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*A[1][1]*T[1]*T[1]*T[1]*T[1]-6*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*A[1][0]*T[1]*T[1]*T[1]*T[1]-T[1]*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]+A[1][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[1]*T[1]*T[1]+3*A[1][0]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[1][1]+2*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]*T[1]+3*A[1][1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*A[1][1]*T[1]*T[1]*T[1]*T[1]*T[1]-7*T[1]*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*A[1][0]*T[1]*T[1]*T[1]*T[1]*T[1]-T[1]*T[1]*T[1]*T[1]*T[1]*T[1]*T[1]; temp.print_str(OUT);
  }
  else if (funcNumber == 2)
  {
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]+A[2][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]*T[2]+3*A[2][0]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[2][1]+2*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]*T[2]+3*A[2][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*A[2][0]*T[2]*T[2]-T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -4*T[2]*T[2]*T[2]-6*A[2][1]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]+A[2][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]*T[2]+3*A[2][0]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[2][1]+2*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]*T[2]+3*A[2][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*A[2][0]*T[2]*T[2]*T[2]-T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -5*T[2]*T[2]*T[2]*T[2]-10*A[2][1]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]+A[2][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]*T[2]+3*A[2][0]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[2][1]+2*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]*T[2]+3*A[2][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*A[2][0]*T[2]*T[2]*T[2]*T[2]-T[2]*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[2]*T[2]*T[2]*T[2]*T[2]-15*A[2][1]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]+A[2][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = T[2]*T[2]*T[2]+3*A[2][0]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = A[2][1]+2*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]*T[2]+3*A[2][1]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*A[2][0]*T[2]*T[2]*T[2]*T[2]*T[2]-T[2]*T[2]*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*A[2][1]*T[2]*T[2]*T[2]*T[2]*T[2]-7*T[2]*T[2]*T[2]*T[2]*T[2]*T[2]; temp.print_str(OUT);
  }
  else if (funcNumber == 3)
  {
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]*T[3]+A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]*T[3]*T[3]+3*A[3][0]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = A[3][1]+2*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]*T[3]+3*A[3][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -6*A[3][0]*T[3]*T[3]-T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -4*T[3]*T[3]*T[3]-6*A[3][1]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]*T[3]+A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]*T[3]*T[3]+3*A[3][0]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = A[3][1]+2*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]*T[3]+3*A[3][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -10*A[3][0]*T[3]*T[3]*T[3]-T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -5*T[3]*T[3]*T[3]*T[3]-10*A[3][1]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -15*A[3][0]*T[3]*T[3]*T[3]*T[3]-T[3]*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -6*T[3]*T[3]*T[3]*T[3]*T[3]-15*A[3][1]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]*T[3]+A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]*T[3]*T[3]+3*A[3][0]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = A[3][1]+2*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]*T[3]+3*A[3][1]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = -T[3]*T[3]*T[3]*T[3]*T[3]*T[3]*T[3]-21*A[3][0]*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = -7*T[3]*T[3]*T[3]*T[3]*T[3]*T[3]-21*A[3][1]*T[3]*T[3]*T[3]*T[3]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]*T[3]+A[3][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 ");
    temp = T[3]*T[3]*T[3]+3*A[3][0]*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = A[3][1]+2*T[3]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 ");
    temp = 3*T[3]*T[3]+3*A[3][1]*T[3]; temp.print_str(OUT);
  }
  else if (funcNumber == 4)
  {
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -6*A[4][0]*T[4]*T[4]-T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -4*T[4]*T[4]*T[4]-6*A[4][1]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]*T[4]+A[4][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 3*A[4][0]*T[4]+T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 2*T[4]+A[4][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*T[4]*T[4]+3*A[4][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -10*A[4][0]*T[4]*T[4]*T[4]-T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -5*T[4]*T[4]*T[4]*T[4]-10*A[4][1]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]*T[4]+A[4][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 3*A[4][0]*T[4]+T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 2*T[4]+A[4][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*T[4]*T[4]+3*A[4][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -15*A[4][0]*T[4]*T[4]*T[4]*T[4]-T[4]*T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -6*T[4]*T[4]*T[4]*T[4]*T[4]-15*A[4][1]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]*T[4]+A[4][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 3*A[4][0]*T[4]+T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 2*T[4]+A[4][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*T[4]*T[4]+3*A[4][1]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = -T[4]*T[4]*T[4]*T[4]*T[4]*T[4]*T[4]-21*A[4][0]*T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = -7*T[4]*T[4]*T[4]*T[4]*T[4]*T[4]-21*A[4][1]*T[4]*T[4]*T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = T[4]*T[4]+A[4][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 ");
    temp = 3*A[4][0]*T[4]+T[4]*T[4]*T[4]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 2*T[4]+A[4][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 ");
    temp = 3*T[4]*T[4]+3*A[4][1]*T[4]; temp.print_str(OUT);
  }
  else if (funcNumber == 5)
  {
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -6*A[5][0]*T[5]*T[5]-T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -4*T[5]*T[5]*T[5]-6*A[5][1]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = T[5]*T[5]+A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = 3*A[5][0]*T[5]+T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 2*T[5]+A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 3*T[5]*T[5]+3*A[5][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -10*A[5][0]*T[5]*T[5]*T[5]-T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -5*T[5]*T[5]*T[5]*T[5]-10*A[5][1]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = T[5]*T[5]+A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = 3*A[5][0]*T[5]+T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 2*T[5]+A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 3*T[5]*T[5]+3*A[5][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -15*A[5][0]*T[5]*T[5]*T[5]*T[5]-T[5]*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -6*T[5]*T[5]*T[5]*T[5]*T[5]-15*A[5][1]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = T[5]*T[5]+A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = 3*A[5][0]*T[5]+T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 2*T[5]+A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 3*T[5]*T[5]+3*A[5][1]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = -T[5]*T[5]*T[5]*T[5]*T[5]*T[5]*T[5]-21*A[5][0]*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = -7*T[5]*T[5]*T[5]*T[5]*T[5]*T[5]-21*A[5][1]*T[5]*T[5]*T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = T[5]*T[5]+A[5][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 ");
    temp = 3*A[5][0]*T[5]+T[5]*T[5]*T[5]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 2*T[5]+A[5][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 ");
    temp = 3*T[5]*T[5]+3*A[5][1]*T[5]; temp.print_str(OUT);
  }
  else if (funcNumber == 6)
  {
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -6*A[6][0]*T[6]*T[6]-T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -4*T[6]*T[6]*T[6]-6*A[6][1]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]*T[6]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 3*A[6][0]*T[6]+T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 2*T[6]+A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 3*T[6]*T[6]+3*A[6][1]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -10*A[6][0]*T[6]*T[6]*T[6]-T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -5*T[6]*T[6]*T[6]*T[6]-10*A[6][1]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]*T[6]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 3*A[6][0]*T[6]+T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 2*T[6]+A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 3*T[6]*T[6]+3*A[6][1]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -15*A[6][0]*T[6]*T[6]*T[6]*T[6]-T[6]*T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -6*T[6]*T[6]*T[6]*T[6]*T[6]-15*A[6][1]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]*T[6]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 3*A[6][0]*T[6]+T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 2*T[6]+A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 3*T[6]*T[6]+3*A[6][1]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = -T[6]*T[6]*T[6]*T[6]*T[6]*T[6]*T[6]-21*A[6][0]*T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = -7*T[6]*T[6]*T[6]*T[6]*T[6]*T[6]-21*A[6][1]*T[6]*T[6]*T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = T[6]*T[6]+A[6][0]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 ");
    temp = 3*A[6][0]*T[6]+T[6]*T[6]*T[6]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 2*T[6]+A[6][1]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 ");
    temp = 3*T[6]*T[6]+3*A[6][1]*T[6]; temp.print_str(OUT);
  }
  else if (funcNumber == 7)
  {
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -6*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -T[7]*T[7]*T[7]*T[7]-6*A[7][0]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -4*T[7]*T[7]*T[7]-6*A[7][1]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = A[7][0]+T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = T[7]*T[7]*T[7]+3*A[7][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = A[7][1]+2*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 3*T[7]*T[7]+3*A[7][1]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -10*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -10*A[7][0]*T[7]*T[7]*T[7]-T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -5*T[7]*T[7]*T[7]*T[7]-10*A[7][1]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = A[7][0]+T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = T[7]*T[7]*T[7]+3*A[7][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = A[7][1]+2*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 3*T[7]*T[7]+3*A[7][1]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -15*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -15*A[7][0]*T[7]*T[7]*T[7]*T[7]-T[7]*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -6*T[7]*T[7]*T[7]*T[7]*T[7]-15*A[7][1]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = A[7][0]+T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = T[7]*T[7]*T[7]+3*A[7][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = A[7][1]+2*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 3*T[7]*T[7]+3*A[7][1]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "12\n");
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = -21*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = A[7][0]+T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = T[7]*T[7]*T[7]+3*A[7][0]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = A[7][1]+2*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 3*T[7]*T[7]+3*A[7][1]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 3*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = -T[7]*T[7]*T[7]*T[7]*T[7]*T[7]*T[7]-21*A[7][0]*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = -21*A[7][1]*T[7]*T[7]*T[7]*T[7]*T[7]-7*T[7]*T[7]*T[7]*T[7]*T[7]*T[7]; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 ");
    temp = 1; temp.print_str(OUT);
    fprintf(OUT, "0 0 0 0 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 ");
    temp = 1; temp.print_str(OUT);
  }

  return;
}

void print_function(FILE *OUT, int funcNumber, int numberConj, complex_rational *T, complex_rational **A)
{ // print the correct function to OUT

  if (funcNumber < 2*numberConj)
  { // print the complex conjugate function
    print_conj_function(OUT, funcNumber, T, A);
  }
  else
  { // print the standard function
    print_std_function(OUT, funcNumber, T, A);
  }

  return;
}



