function [w,dVdx,dVdy] = Faddeyeva_v2(z,sdgts)
    //%----------
    //%   Faddeyeva_v2 is a Scilab function that receives as input an array
    //%   of complex numbers, z=x+iy, and an integer, sdgts, representing
    //%   the desired number of significant figures in the output. z is usually
    //%   an array (with one or two dimensions)but can be a single scalar as well.
    //%   The function returns as output the Faddeyeva function,w, defined as
    //%   w(z)=exp(-z^2)*erfc(-i*z) where erfc(z) is the complex  complementary
    //%   error function. In addition, the function returns the derivatives of
    //%   the real part of the Faddeyeva function (dVdx and dVdy) with respect
    //%   to the real and imaginary parts of z, respectively.
    //%----------
    //%   The present version of the function is set to reproduce the highest
    //%   possible accuracy obtainable from algorithm 916 [Zaghloul and Ali,TOMS,
    //%   Vol. 38, No. 2, article 15:1-22 (2011)] through requesting a number of
    //%   significant figures, sdgts, equal to 13. The desired number of significant
    //%   figures can be reduced for marginal improvements of the efficiency at
    //%   the expense of accuracy.
    //%   The recommended range for sdgts is between 4 and 13. A number of
    //%   significant figures smaller than 4 is not recommended for accuracy
    //%   concerns, particularly regarding the computations of the derivatives, and
    //%   will be directly changed to 4 with a warning message.
    //%   Values of sdgts >13 are not recommended for performance concerns and
    //%   will be automatically changed to 13 with a warning message.
    //%   When sdgts is empty, a default value of 4 will be directly used.
    //%----------
    //%   The accompanying driver code “Faddeyeva_driver.m” can be run for
    //%   computation of the Faddeyeva function, w(z)=V+iL and the partial
    //%   derivatives of its real part, V(x,y), on an array of the complex
    //%   variable z.The partial derivatives of the imaginary part, L(x,y), are
    //%   simply given by Eq. (23) in the original article of algorithm 916,
    //%   TOMS. Vol. 38, No. 2, article 15:1-22 (2011) and as commented by the
    //%   end of this function.
    //%   An example of generating an array of z is included in the driver code.
    //%----------
    //%   Author: M. R. Zaghloul
    //%   United Arab Emirates University, June7,2014
    //%--------

    // Reset outputs
    w=%nan.*z;

    //// Machine-related parameters            
    Rmin=2.23D-308;
    sqrt_log_Rmin=2.661567596467717D+001;

    // Check inputs, at least one input argument is needed

    [lhs,rhs]=argn(0)
    if rhs<1 
        error('Faddeyeva_v2.Sci needs at least one input argument');
        w=%nan+i*%nan;
        return
    elseif rhs==1, sdgts=[];
        //    % Set the default value of the # of significant digits "sdgts"
        if isempty(sdgts),  sdgts=4; end
    elseif rhs==2
        //     Set the maximum & minimum values of the # of significant digits
        //   "sdgts" with warning messages when "sdgts" goes beyond the limits
        if ~isempty(sdgts) & sdgts > 13;
            sdgts=13;
            warning(['sdgts > 13 is not recommended for performance concerns.'...
            'The maximum value of sdgts=13 has been used']);

        elseif (rhs==2 & ~isempty(sdgts) & sdgts < 4);
            sdgts = 4;
            warning(['sdgts <4 is not recommended for accuracy concerns'...
            'The value of sdgts has been set to the minimum value sdgts=4   ']);
        end
    end

    xx=real(z);     //vector of the real part of the input comlex array
    yy=abs(imag(z));     //vector of the imaginary part of the input comlex array

    select sdgts
    case  4; Ncycles= 7; bigz_border=1.60e2;
    case  5; Ncycles= 8; bigz_border=5.00e2;
    case  6; Ncycles= 9; bigz_border=1.60e3;
    case  7; Ncycles= 9; bigz_border=1.00e4;
    case  8; Ncycles= 9; bigz_border=1.16e4;
    case  9; Ncycles=10; bigz_border=4.30e4;
    case 10; Ncycles=11; bigz_border=1.50e5;
    case 11; Ncycles=11; bigz_border=5.00e5;
    case 12; Ncycles=11; bigz_border=1.70e6;
    case 13; Ncycles=13; bigz_border=1.00e8;
    end

    // Faddeyeva_v2 cannot calculate w for y negative & exp(y^2-x^2)>=the largest 
    // floating point number due to unavoidable over flow problems 
    idx_neg=find(imag(z)<0) 
    if ~isempty(idx_neg)
        neg_y_idx=find((yy.^2-xx.^2)>=sqrt_log_Rmin^2, 1);
        if ~isempty(neg_y_idx)
            es = sprintf(['Input array, z, has points with imag(z)<0 & exp(imag(z)^2-real(z)^2)>=the largest floating point number. Faddeyeva cannot calculate w for these points due to unavoidable overflow problems     ']);
            error(es) 
        end
    end

    //------------Repeatedly used parameters & constants
    a=.5;                                   // a
    a_sqr=.25;                              // a^2
    inv_a_sqr=4;                            // 1/(a^2)
    two_a_sqr=.5;                           // 2*a^2
    exp_two_a_sqr=1.648721270700128D+000;   // exp(2*a^2)
    a_pi=1.591549430918954D-001;            // a/pi
    two_a_pi=3.183098861837908D-001;        // 2*a/pi
    two_a_pi_half_a=7.957747154594770D-002; // 0.5*a*(2*a/pi)
    exp_minus_a_sqr=7.788007830714049D-001; // exp(-a^2)

    //---Pre-calculated values for the fixed # of computational cycles
    nn=1:1:Ncycles;
    exp_2a_sqr_n_1=exp(-2*a_sqr.*(nn-1));                 //  exp(-2*a^2*(n-1))
    inv_asqr_exp_asqr_nsqr=inv_a_sqr.*exp(-0.25.*nn.*nn); // exp(-a^2*n^2)/()a^2)
    //------------

    xsqr_pls_ysqr=(xx.^2+yy.^2);

    index_x0=find(xx==0 & xsqr_pls_ysqr<bigz_border);
    if ~isempty(index_x0);
        w(index_x0)=erfcx((yy(index_x0)))+%i.*0;
    end

    index_x1=find(xx~=0.0 & abs(xx)<sqrt_log_Rmin & xsqr_pls_ysqr<bigz_border );
    if ~isempty(index_x1);
        erfcxy(index_x1)=erfcx((yy(index_x1)));   
        //-------Calculating Faddeyeva fn for values of 0<|x|<sqrt(-log(Rmin))
        //-------while x^2+y^2<bigz_border
        for ix=1:prod(size(index_x1));
            idx=index_x1(ix);
            xsign=sign(xx(idx));
            x=xsign*xx(idx);
            erfcx_y=erfcxy(idx);
            y=max(Rmin,(yy(idx)));
            x_sqr=x*x;
            two_x=2*x;
            two_yx=y*two_x;
            cos_2yx=cos(two_yx);
            exp_x_sqr=exp(-x_sqr);
            two_a_pi_y=two_a_pi/y;
            V_old=exp_x_sqr*(erfcx_y*cos_2yx+two_a_pi_y*sin(two_yx*.5)^2);
            L_old=exp_x_sqr*(-erfcx_y+.5*two_a_pi_y);
            n3=ceil(two_x);
            n3_3=n3-1;
            two_a_sqr_n3=two_a_sqr*n3;
            y_sqr_a_sqr=inv_a_sqr*y*y;
            Sigma1=0;
            Sigma2_3=0;
            Sigma4_5=0;
            exp1=exp(-x);
            exp3=exp(-two_a_sqr_n3+x+two_a_sqr);
            exp2=exp_two_a_sqr/(exp3*exp3);
            del2_tmp=1.0;
            del3_tmp=exp(-((a_sqr*n3*n3-x*n3-two_a_sqr_n3)+x_sqr+x+a_sqr));
            del3_3_tmp=exp3*exp_minus_a_sqr ;    
            for n=1:Ncycles;
                den1=inv_asqr_exp_asqr_nsqr(n)*exp_x_sqr/(n*n+y_sqr_a_sqr);
                del2_tmp=del2_tmp*exp1;
                del3_tmp=del3_tmp*exp3;
                exp3_den=del3_tmp*inv_asqr_exp_asqr_nsqr(n)/((n3_3+n)^2+y_sqr_a_sqr);
                Sigma1=Sigma1+den1;
                if (n3_3>=n)
                    del3_3_tmp=del3_3_tmp*exp2;
                    exp3_3_den=del3_3_tmp*del3_tmp*inv_asqr_exp_asqr_nsqr(n)/((n3-n)^2+y_sqr_a_sqr);
                    Sigma2_3=Sigma2_3+del2_tmp*den1+exp3_3_den+exp3_den;
                    Sigma4_5=Sigma4_5-n*del2_tmp*den1+(n3-n)*exp3_3_den+(n3_3+n)*exp3_den;
                else
                    Sigma2_3=Sigma2_3+del2_tmp*den1+exp3_den;
                    if x>=5.0D-4
                        Sigma4_5= Sigma4_5-n*del2_tmp*den1+(n3_3+n)*exp3_den;
                    else
                        Sigma4_5=Sigma4_5+n*n*two_x*den1*...
                        (1+1.666666666666667D-001*n*n*x_sqr+...
                        8.333333333333333D-003*(n*n*x_sqr)^2);
                    end
                end
            end
            if ((y < 5D0) & (two_yx>Rmin))
                w(idx)=V_old+y*two_a_pi*(-cos_2yx*Sigma1+.5*Sigma2_3)+...
                %i* xsign*(sin(two_yx)*(L_old+two_a_pi*y*Sigma1)+two_a_pi_half_a*Sigma4_5);
            elseif ((y < 5D0) & (two_yx<=Rmin))
                w(idx)=V_old+y*two_a_pi*(-cos_2yx*Sigma1+.5*Sigma2_3)+...
                %i* xsign*(y*(two_x*L_old+two_x*two_a_pi*y*Sigma1)+two_a_pi_half_a*Sigma4_5);
            else
                w(idx)=V_old+y*two_a_pi*(-cos_2yx*Sigma1+.5*Sigma2_3)+...
                %i* xsign*two_a_pi_half_a*Sigma4_5;
            end
        end
    end

    if  sdgts>5 then

        index_x2=find(abs(xx)>=sqrt_log_Rmin & xsqr_pls_ysqr<bigz_border);
        if ~isempty(index_x2);
            //-------Calculating Faddeyeva for values of x>=sqrt(-log(Rmin)) 
            //-------while x^2+y^2<bigz_border
            for ix=1:prod(size(index_x2));
                idx=index_x2(ix);
                xsign=sign(xx(idx));
                x=xsign*xx(idx);
                y=max(Rmin,(yy(idx)));
                n3=ceil(2.*x);
                n3_3=n3-1;
                y_sqr_a_sqr=inv_a_sqr*y*y;
                Sigma3=0;
                Sigma5=0;
                del3_3_tmp=exp((x-two_a_sqr*n3)+a_sqr);
                exp2=1/(del3_3_tmp*del3_3_tmp);
                afactor=del3_3_tmp;
                exp3_den=inv_a_sqr*exp(-(a*n3_3-x)^2);
                for n=1:Ncycles;
                    del3_3_tmp=del3_3_tmp*exp2;
                    exp3_den=exp3_den*afactor*exp_2a_sqr_n_1(n);
                    exp3_3_den=exp3_den*del3_3_tmp/((n3-n)^2+y_sqr_a_sqr);
                    den1=exp3_den/((n3_3+n)^2+y_sqr_a_sqr);
                    Sigma3=Sigma3+exp3_3_den+den1;
                    Sigma5=Sigma5+(n3-n)*exp3_3_den+(n3_3+n)*den1;
                end
                w(idx)=y*a_pi*Sigma3+%i*xsign*two_a_pi_half_a*Sigma5;
            end
        end
    end
    //------- Fix up for negative y points
    if~isempty(idx_neg);
        w(idx_neg)=2.*exp(-(xx(idx_neg)+%i.*yy(idx_neg)).^2)-(real(w(idx_neg))-%i*imag(w(idx_neg)));
    end
    //-------Calculating Faddeyeva fn for the region of  x^2+y^2>bigz_border
    index_bigz=find((xx.^2+yy.^2)>=bigz_border); 
    if ~isempty(index_bigz);
        w(index_bigz)=5.641895835477563e-001.*((yy(index_bigz))+%i.*(xx(index_bigz)))./(xsqr_pls_ysqr(index_bigz));
    end

    //    % Calculations of partial derivatives
    dVdx=-2*real(z.*w); // % Partial derivative of real(w) w.r.t. x 
    dVdy=2*imag(z.*w)-1.128379167095513e+000; //  % Partial derivative of real(w) w.r.t. y
    //    %dLdx=-dVdy; dLdy=dVdx; % Partial derivatives of imag(w) w.r.t. x&y

endfunction
