    function [w,dVdx,dVdy] = Faddeyeva_v2(z,sdgts)

    %----------
    %   Faddeyeva_v2 is a Matlab function that receives as input an array
    %   of complex numbers, z=x+iy, and an integer, sdgts, representing
    %   the desired number of significant figures in the output. z is usually
    %   an array (with one or two dimensions)but can be a single scalar as well.
    %   The function returns as output the Faddeyeva function,w, defined as
    %   w(z)=exp(-z^2)*erfc(-i*z) where erfc(z) is the complex  complementary
    %   error function. In addition, the function returns the derivatives of
    %   the real part of the Faddeyeva function (dVdx and dVdy) with respect
    %   to the real and imaginary parts of z, respectively.
    %----------
    %   The present version of the function is set to reproduce the highest
    %   possible accuracy obtainable from algorithm 916 [Zaghloul and Ali,TOMS,
    %   Vol. 38, No. 2, article 15:1-22 (2011)] through requesting a number of
    %   significant figures, sdgts, equal to 13. The desired number of
    %   significant figures can be reduced for marginal improvements of
    %   the efficiency at the expense of accuracy.
    %   The recommended range for sdgts is between 4 and 13. A number of
    %   significant figures smaller than 4 is not recommended for accuracy
    %   concerns, particularly regarding the computations of the derivatives,
    %   and it will be directly changed to 4 with a warning message.
    %   Values of sdgts >13 are not recommended for performance concerns and
    %   will be automatically changed to 13 with a warning message.
    %   When sdgts is empty, a default value of 4 will be directly used.
    %----------
    %   The accompanying driver code Faddeyeva_driver.m can be run for
    %   computation of the Faddeyeva function, w(z)=V+iL and the partial
    %   derivatives of its real part, V(x,y), on an array of the complex
    %   variable z.The partial derivatives of the imaginary part, L(x,y), are
    %   simply given by Eq. (23) in the original article of algorithm 916,
    %   TOMS. Vol. 38, No. 2, article 15:1-22 (2011) and as commented in the
    %   last line in this file Faddeyeva_v2.m .
    %   An example of generating an array of z is included in the driver code.
    %----------
    %   Author: M. R. Zaghloul
    %   United Arab Emirates University, March 6,2015
    %----------

    %---------- Reset outputs
    w=NaN(size(z));

    %---------- Machine-related parameters
    Rmin=2.23e-308;
    sqrt_log_Rmin=2.661567596467717e+001;

    %---------- Check input, at least one input argument is needed
    if nargin<1
        error('Faddeyeva_v2.m needs at least one input argument');
        w=NaN+i*NaN;
        return
    elseif nargin==1, sdgts=[];
        % Set the default value of the # of significant digits "sdgts"
        if isempty(sdgts),  sdgts=4; end
    elseif nargin==2
        % Set the maximum & minimum values of the # of significant digits
        % "sdgts" with warning messages when "sdgts" goes beyond
        % the recommended limits
        if ~isempty(sdgts) && sdgts > 13;
            sdgts=13;
            ws1=sprintf(['sdgts > 13 is not recommended for performance concerns.'...
                'The maximum value of sdgts=13 has been used']);
            warning(ws1);
        elseif nargin==2 && ~isempty(sdgts) && sdgts < 4;
            sdgts = 4;
            ws2=sprintf(['sdgts <4 is not recommended for accuracy concerns'...
                'The value of sdgts has been set to the minimum value sdgts=4']);
            warning(ws2);
        end
    end

    %----------- Moving Boundary for Asymptotic Expression for large |z|
    switch sdgts
        case  4; Ncycles= 7; bigz_border=1.07e2;    
        case  5; Ncycles= 8; bigz_border=1.10e2;  
        case  6; Ncycles= 9; bigz_border=1.80e2;  
        case  7; Ncycles= 9; bigz_border=3.80e2;  
        case  8; Ncycles=10; bigz_border=8.10e2;  
        case  9; Ncycles=10; bigz_border=1.75e3;  
        case 10; Ncycles=11; bigz_border=3.80e3;  
        case 11; Ncycles=11; bigz_border=8.20e3;   
        case 12; Ncycles=12; bigz_border=1.75e4;  
        case 13; Ncycles=13; bigz_border=3.80e4;  
    end

    %-----------
    xx=real(z);     %vector of the real part of the input complex array
    yy=abs(imag(z)); %vector of abs. of imaginary part of input complex array
    %-----------

    % Faddeyeva_v2 cannot calculate w for y negative & exp(y^2-x^2)>=the largest
    % floating point number, due to unavoidable over flow problems
    idx_neg=find(imag(z)<0);
    if ~isempty(idx_neg);
        neg_y_idx=find((yy.^2-xx.^2)>=sqrt_log_Rmin^2, 1);
        if ~isempty(neg_y_idx)
            error(['Input array, z, has points with imag(z)<0 & ',...
                'exp(imag(z)^2-real(z)^2)>=the largest floating point number. ', ...
                'Faddeyeva_v2 cannot calculate w for these points ',...
                'due to unavoidable overflow problems ']);
        end
    end

    %--------- Repeatedly used parameters & constants
    a=0.5;                                  % a
    a_sqr=0.25;                             % a^2
    inv_a_sqr=4.0;                          % 1/(a^2)
    two_a_sqr=0.5;                          % 2*a^2
    exp_two_a_sqr=1.648721270700128;        % exp(2*a^2)
    a_pi=1.591549430918954e-001;            % a/pi
    two_a_pi=3.183098861837908e-001;        % 2*a/pi
    two_a_pi_half_a=7.957747154594770e-002; % 0.5*a*(2*a/pi)
    exp_minus_a_sqr=7.788007830714049e-001; % exp(-a^2)

    %--------- Pre-calculated values for the fixed # of computational cycles
    nn=1:1:Ncycles;
    exp_2a_sqr_n_1=exp(-2*a_sqr.*(nn-1));                 %  exp(-2*a^2*(n-1))
    inv_asqr_exp_asqr_nsqr=inv_a_sqr.*exp(-0.25.*nn.*nn); %(1/a^2)*exp(-a^2*n^2)
    %------------

    xsqr_pls_ysqr=(xx.^2+yy.^2);

    %  --------------
    %  For x=0, use the asymptotic expressions for x--->0 from eq. (6)
    %  in the original article of the algorithm 916, TOMS. Vol. 38, No. 2,
    %  article 15:1-22 (2011).
    index_x0=find(xx==0.0 & xsqr_pls_ysqr<bigz_border);
    if ~isempty(index_x0);
        w(index_x0)=erfcx((yy(index_x0)))+1i.*0.0;
    end

    index_x1=find(xx~=0.0 & abs(xx)<sqrt_log_Rmin & xsqr_pls_ysqr<bigz_border);
    if ~isempty(index_x1);
        erfcxy(index_x1)=erfcx((yy(index_x1)));
    %-------Calculating Faddeyeva fn for values of 0<|x|<sqrt(-log(Rmin))
    %-------while |z|<bigz_border
        for ix=1:numel(index_x1);
            idx=index_x1(ix);
            xsign=sign(xx(idx));
            x=xsign*xx(idx);
            y=max(Rmin,(yy(idx)));
            erfcx_y=erfcxy(idx);
            x_sqr=x*x;
            two_x=2.0*x;
            two_yx=y*two_x;
            cos_2yx=cos(two_yx);
            exp_x_sqr=exp(-x_sqr);
            two_a_pi_y=two_a_pi/y;
            V_old=exp_x_sqr*(erfcx_y*cos_2yx+two_a_pi_y*sin(two_yx*0.5)^2);
            L_old=exp_x_sqr*(-erfcx_y+0.5*two_a_pi_y);
            n3=ceil(two_x);
            n3_3=n3-1;
            two_a_sqr_n3=two_a_sqr*n3;
            y_sqr_a_sqr=inv_a_sqr*y*y;
            sigma1=0.0;
            sigma2_3=0.0;
            sigma4_5=0.0;
            exp1=exp(-x);
            exp3=exp(-two_a_sqr_n3+x+two_a_sqr);
            exp2=exp_two_a_sqr/(exp3*exp3);
            del2_tmp=1.0;
            del3_tmp=exp(-((a_sqr*n3*n3-x*n3-two_a_sqr_n3)+x_sqr+x+a_sqr));
            del3_3_tmp=exp3*exp_minus_a_sqr ;
            n=1;
            while (n<=Ncycles)
                den1=inv_asqr_exp_asqr_nsqr(n)*exp_x_sqr/(n*n+y_sqr_a_sqr);
                del2_tmp=del2_tmp*exp1;
                del3_tmp=del3_tmp*exp3;
                exp3_den=del3_tmp*inv_asqr_exp_asqr_nsqr(n)/((n3_3+n)^2+y_sqr_a_sqr);
                sigma1=sigma1+den1;
                if (n3_3>=n)
                    del3_3_tmp=del3_3_tmp*exp2;
                    exp3_3_den=del3_3_tmp*del3_tmp*inv_asqr_exp_asqr_nsqr(n)/...
                        ((n3-n)^2+y_sqr_a_sqr);
                    sigma2_3=sigma2_3+del2_tmp*den1+exp3_3_den+exp3_den;
                    sigma4_5=sigma4_5-n*del2_tmp*den1+(n3-n)*exp3_3_den+(n3_3+n)*...
                        exp3_den;
                else
                    sigma2_3=sigma2_3+del2_tmp*den1+exp3_den;
                    if x>=5e-4
                        sigma4_5= sigma4_5-n*del2_tmp*den1+(n3_3+n)*exp3_den;
                    else
                        sigma4_5=sigma4_5+n*n*two_x*den1*...
                            (1+1.666666666666667e-001*n*n*x_sqr+...
                            8.333333333333333e-003*(n*n*x_sqr)^2);
                    end
                end
                n=n+1;
            end
            if ((y < 5.e0) && (two_yx>Rmin))
                w(idx)=V_old+y*two_a_pi*(-cos_2yx*sigma1+.5*sigma2_3)+...
                    1i* xsign*(sin(two_yx)*(L_old+two_a_pi*y*sigma1)+...
                    two_a_pi_half_a*sigma4_5);
            elseif ((y < 5.e0) && (two_yx<=Rmin))
                w(idx)=V_old+y*two_a_pi*(-cos_2yx*sigma1+.5*sigma2_3)+...
                    1i* xsign*(y*(two_x*L_old+two_x*two_a_pi*y*sigma1)+...
                    two_a_pi_half_a*sigma4_5);
            else
                w(idx)=V_old+y*two_a_pi*(-cos_2yx*sigma1+.5*sigma2_3)+...
                    1i* xsign*two_a_pi_half_a*sigma4_5;
            end
        end
    end
    if sdgts>5
        index_x2=find(abs(xx)>=sqrt_log_Rmin & xsqr_pls_ysqr<bigz_border);
        if ~isempty(index_x2);
            %-------Calculating Faddeyeva for values of x>=sqrt(-log(Rmin))
            %-------while |z|<bigz_border
            for ix2=1:numel(index_x2);
                idx2=index_x2(ix2);
                xsign=sign(xx(idx2));
                x=xsign*xx(idx2);
                y=max(Rmin,(yy(idx2)));
                n3=ceil(2*x);
                n3_3=n3-1;
                y_sqr_a_sqr=inv_a_sqr*y*y;
                sigma3=0;
                sigma5=0;
                del3_3_tmp=exp((x-two_a_sqr*n3)+a_sqr);
                exp2=1/(del3_3_tmp*del3_3_tmp);
                factor=del3_3_tmp;
                exp3_den=inv_a_sqr*exp(-(a*n3_3-x)^2);
                n=1;
                while(n<=Ncycles);
                    del3_3_tmp=del3_3_tmp*exp2;
                    exp3_den=exp3_den*factor*exp_2a_sqr_n_1(n);
                    exp3_3_den=exp3_den*del3_3_tmp/((n3-n)^2+y_sqr_a_sqr);
                    den1=exp3_den/((n3_3+n)^2+y_sqr_a_sqr);
                    sigma3=sigma3+exp3_3_den+den1;
                    sigma5=sigma5+(n3-n)*exp3_3_den+(n3_3+n)*den1;
                    n=n+1;
                end
                w(idx2)=y*a_pi*sigma3+1i*xsign*two_a_pi_half_a*sigma5;
            end
        end
    end
    %------- Fix up for negative y points
    if~isempty(idx_neg);
%          w(idx_neg)=2.*exp(-(z(idx_neg)).^2)-(real(w(idx_neg))-1i*imag(w(idx_neg)));
         w(idx_neg)=2.*exp(-(z(idx_neg)).^2)-(real(w(idx_neg))-1i*imag(w(idx_neg)));
    end

    %------ Asymptotic expression for large |z| (|z|>bigz_border)
    index_bigz=find((xx.^2+yy.^2)>=bigz_border);
    if~isempty(index_bigz);
%         w(index_bigz)=(-1i*z(index_bigz))* 5.641895835477563e-001./...
%             (0.5-z(index_bigz).^2);
 
tt=z(index_bigz).^2;
w(index_bigz)=(1i* 5.641895835477563e-001).*...
        2*(tt - 1.)./(z(index_bigz).*(2.*tt - 3));

    end

    %-------Calculations of partial derivatives
    dVdx=-2*real(z.*w);  % Partial derivative of real(w) w.r.t. x
    dVdy=2*imag(z.*w)-1.128379167095513e+000; % Partial derivative of real(w) w.r.t. y
    %dLdx=-dVdy; dLdy=dVdx; % Partial derivatives of imag(w) w.r.t. x&y
