
#include <numeric>

#include <functions/complex/e_float_complex.h>
#include <functions/constants/constants.h>
#include <functions/elementary/elementary.h>
#include <functions/gamma/gamma.h>
#include <functions/gamma/gamma_util.h>
#include <functions/hypergeometric/hypergeometric.h>
#include <functions/integer/integer.h>
#include <functions/tables/tables.h>
#include <utility/util_coefficient_expansion.h>
#include <utility/util_digit_scale.h>

namespace Gamma_Series
{
  static const std::vector<e_float>& CoefOneOverGammaAtZero(void)
  {
    // Series coefficients of 1/gamma[x] to 400 decimal digits of precision.
    // Extracted from Mathematica Version 4.1 using:
    // N[Series[1/Gamma[x], {x, 0, 20}], 400]
    // Table[SeriesCoefficient[%, n], {n, 20}]
    // Export["c:\\temp\\coef.txt", %, "list"]
    // The coefficient of (x^1) is 1 and the coefficient of (x^2) is EulerGamma.
    static const std::tr1::array<e_float, 20u> coef_data =
    {{
      ef::one(),
      ef::euler_gamma(),
      e_float("-0.6558780715202538810770195151453904812797663804785843472923624456838708383537221020861828159940213640004823766794673960425880273800045120663276349042352029670719647868920256583927459606866272636428716503560447876477020029264993733572853936743624613834732228286244522007087071544720609051364563240296088675489599938359316660318321422152685114387721454550793121583896628767836242712406766501355141085654"),
      e_float("-0.04200263503409523552900393487542981871139450040110609352206581297618009687597598854710770129478771323353200022200001805792037621099075181447805816907283782783355156941422840681585671757922536196664416350181665682109004334254906673031085386942302647163933719562136186164185652922234123296618703050633022456154116377616684761434108546150201363905266261778775672100038613880494186238891616981359256812529"),
      e_float("+0.1665386113822914895017007951021052357177815022471743405704689031789938660564742483194719146580416266239559340512887953328435534067790239085961674942407332694546242602880866376070387262496367207086080494302533434060985324494851329241260152920664205323542319634585425345992159996851601411139446646838555765301874799329345794442502091118859615974367643203447574836048889488156231419555751071696734321971"),
      e_float("-0.04219773455554433674820830128918739130165268418982248637691887327545901118558899606734728429375531504062334826057082510869875342573608881031728336193513216807653051215558592232817158189354800086887093374690761985231094669862796057544336120242135379457098619992409318732513064699294082343694452964488092134178703074119479383139206604543459710454255056686229455745478027704185028721684902564483861681488"),
      e_float("-0.009621971527876973562114921672348198975362942252113002105138862627311673514460739536466888015658820015733562800356029803723332017419507197553494169877116743012525221376253108885704616876673111525831349233429832639845702227478746003004271641976963298258697182291199410219284964853091797576513483088948247179443482321981474420510403801562538507188880431476146363008815197790256291597231466854822676154536"),
      e_float("+0.007218943246663099542395010340446572709904800880238318001094781173622594974158542714089090120849888863763580197688859795311415307822774690748268422748882778219306457154065720813703287027418373611063189696287389264012126740635036226069672622811192532111244693818497291437270392053464350242231258452208196814361249731802322058940324639712443939052265996811453388534005459217602808725999963971215780208927"),
      e_float("-0.001165167591859065112113971084018388666809333795384057443407505275620025848166530722148800109172188515797192888894863669066772517370809868046375622203294817029684730913245113885467980578363470095176347017392442241646234586862652283891629280315418462703268811259855101866764711768855009672612320853542794740964104932696508211319286122820153051919898017312123494771712287419491362072058225406140411120019"),
      e_float("-0.0002152416741149509728157299630536478064782419233783387503502674890856394637167847391861851194625624815259563219739244945163417211076747477123391086742668973557906175454607228737337091983400046734865215719165942151250609611490595370582548259601169491801275521476262588189925692003213654485091597871400858799644166761772424150734750207019101578462267487952510338819128595535888433239855683688898584219257"),
      e_float("+0.0001280502823881161861531986263281643233948920996936772149005458380412035520434687587389223798955199755551386484801312995164552307622434192355599981862167102840027210758568419015795376653975972537042214781394208013779609060164253049743961101368403030633183813736201875808775924737002119102356796180501539005893241175032007956168654433971365386229244346745439587609848542929397402454550229243419558216667"),
      e_float("-0.00002013485478078823865568939142102181838229483329797911526116267090822918618897047762330812383650301882610598611158717931945723532269982416612357113737088330997041202300337442465482349805785560019857678505838147808340261428265467157736711014454674223004617947229087707010849731551472310234902900406148808090505873712222916856506434771262514141240239945017462827997814312257914315173799509409580592227999"),
      e_float("-1.250493482142670657345359473833092242322655621153959815349923157491212455610792211796228757892295094621846077970315353463508834829915102416793382769258699176937711876791097906964868348037193273974207229913211507712885193827801031655018418659441463872772541489470618445691009164835001388077418133290491428524980227554901591552867473525328143941529948218328641439202885771220946627449251945182703333714e-6"),
      e_float("+1.133027231981695882374129620330744943324004838621075654295505395460408427303676342059099599341571794151723102554723472282247098515047076970814877245054820361527216773963022956952446219594483308210208907304801101488348848062734625098840906296035467239877276901851067511872545950865225464347561051817490348962059288584407364395282537157756108088904191766662293100072409206551732618605079716933286705185e-6"),
      e_float("-2.056338416977607103450154130020572836512579026293379453468317253324568037109991871188900376836419475497313953593845935602678421832119064686665246359768127970426304592335249241703427667413130015011643569198543828768405669892358923289301890386377129451006676551195702074350994748934923779785037553155730018542197343217850356691279814564656597210371502101337995794398983674689948759643317988389766813875e-7"),
      e_float("+6.116095104481415817862498682855342867275865719712320867324029277235074352503861972826211688750122568389248723593760275052110202607289094841928198408739156977044243956049097412356065163620990532543731396877525356196920079422184202257312549571022890099774170257148393843273692290361740154736183615097192651911573292083206887421491192856772933647750427490115120897580598446855104580940274924647813616162e-9"),
      e_float("+5.002007644469222930055665048059991303044612742494481718953378877374721317711613245138149284064041492607495463035818621192634138138674255799031521408077064669154375661227867987031920077145517025614542160750699253046327614481165280828729590903826540895400924922302845714850348986296988595579237764267997702727230547855413274977607953320860481079313147166661821990367245206182829153066329757101435599756e-9"),
      e_float("-1.181274570487020144588126565436505577738759504932587590961892631696433914362731439517968288144559739788048758350566671400704726777156064156164018114386575307315494202805215438614394288438629133949868735821855227356344177768519582180435858700409517489758611765199834757506730273652724786075640254440409769591442946310523471108835614728780635461979668551745785999641219440682865730142763528023172889291e-9"),
      e_float("+1.043426711691100510491540332312250191400709823125812121087107392734758807124820988847060670016098018403228867991155660260908069802732864181290140345583710171861863717571042850540110559715149698077318582760140410483437385783369743279091227248338835597955083612786539728965176160729156453537619398985580910769450857209100810336623880736523569415690948751514525776483744186241841939479643365941249571376e-10"),
      e_float("+7.782263439905071254049937311360777226068086181392938819435507326929869575078978038827379507664107281913393462406889059435100758544819098632173849066677032230561159253228543787258397911069334455291546933865105252140021359958145198606994108363499673176861344191053751641915087563642143554534989078722037333135067999258209158657882450793530257918827738106564148852026754272154918712839939196672793348702e-12")
    }};

    static const std::vector<e_float> coef(coef_data.begin(), coef_data.end());

    return coef;
  }
}

namespace GammaTemplate
{
  template<typename T> static inline T gamma(const T& x)
  {
    using ef::abs;
    using efz::abs;
    using ef::exp;
    using efz::exp;
    using ef::log;
    using efz::log;
    using ef::real;
    using efz::real;
    using ef::sin;
    using efz::sin;

    if(!ef::isfinite(x))
    {
      return x;
    }

    const bool b_neg = ef::isneg(x);

    if(ef::isint(x))
    {
      if(b_neg || ef::iszero(x))
      {
        return std::numeric_limits<e_float>::quiet_NaN();
      }
      else
      {
        const INT64 nx = ef::to_int64(x);

        // Possible special handling for (relatively) small, pure integers.
        if(static_cast<std::size_t>(nx) < Tables::A000142().size())
        {
          return ef::factorial(static_cast<UINT32>(static_cast<INT32>(nx - static_cast<INT64>(1))));
        }
      }
    }

    // Make a local, unsigned copy of the input argument.

    T xx(!b_neg ? x : -x);

    T G;

    if(ef::small_arg(xx))
    {
      static const T tz(ef::zero());

      G = ef::one() / std::accumulate(Gamma_Series::CoefOneOverGammaAtZero().begin(),
                                      Gamma_Series::CoefOneOverGammaAtZero().end(),
                                      tz,
                                      Util::coefficient_expansion<T, T>(xx, xx));
    }
    else
    {
      // Check if the argument should be scaled up for the Bernoulli series expansion.
      static const INT32   min_arg_n = static_cast<INT32>(static_cast<double>(240.0) * Util::DigitScale());
      static const e_float min_arg_x = e_float(min_arg_n);

      const e_float rx = real(xx);

      const INT32 n_recur = rx < min_arg_x ? static_cast<INT32>((min_arg_n - ef::to_int32(rx)) + 1u)
                                           : static_cast<INT32>(0);

      // Scale the argument up and use downward recursion later for the final result.
      if(n_recur != static_cast<INT32>(0))
      {
        xx += n_recur;
      }

            T one_over_x_pow_two_n_minus_one = ef::one() / xx;
      const T one_over_x2                    = one_over_x_pow_two_n_minus_one * one_over_x_pow_two_n_minus_one;
            T sum                            = (ef::bernoulli(static_cast<UINT32>(2u)) / static_cast<INT32>(2)) * one_over_x_pow_two_n_minus_one;

      // Perform the Bernoulli series expansion.
      for(INT32 n2 = static_cast<INT32>(4); n2 < ef::max_iteration(); n2 += static_cast<INT32>(2))
      {
        one_over_x_pow_two_n_minus_one *= one_over_x2;

        const T term = (ef::bernoulli(static_cast<UINT32>(n2)) * one_over_x_pow_two_n_minus_one) / static_cast<INT32>(n2 * (n2 - static_cast<INT32>(1)));

        const INT64 order_check = static_cast<INT64>(abs(term).order() - abs(sum).order());

        if((n2 > static_cast<INT32>(20)) && (order_check < -ef::tol()))
        {
          break;
        }

        sum += term;
      }

      static const e_float half_ln_two_pi = ef::log(ef::two_pi()) / static_cast<INT32>(2);

      G = exp(((((xx - ef::half()) * log(xx)) - xx) + half_ln_two_pi) + sum);

      // Rescale the result using downward recursion if necessary.
      for(INT32 k = static_cast<INT32>(0); k < n_recur; k++)
      {
        G /= --xx;
      }
    }

    // Return the result, accounting for possible negative arguments.
    return !b_neg ? G : -ef::pi() / (xx * G * sin(ef::pi() * xx));
  }

  template<typename T> static inline T gamma_near_n(const INT32 n, const T& x)
  {
    using ef::sin;
    using efz::sin;

    if(n >= static_cast<INT32>(0))
    {
      return GammaTemplate::gamma<T>(n + x);
    }
    else
    {
      const INT32 nn = static_cast<INT32>(-n);

      const bool b_neg = (static_cast<INT32>(nn % static_cast<INT32>(2)) != static_cast<INT32>(0));

      const T n_minus_x = nn - x;
      const T Gn        = ef::pi() / (GammaTemplate::gamma<T>(n_minus_x + ef::one()) * sin(ef::pi() * x));

      return !b_neg ? Gn : -Gn;
    }
  }
}

e_float ef::gamma(const e_float& x)
{
  if(x.has_its_own_gamma())
  {
    return e_float::my_gamma(x);
  }
  else
  {
    return GammaTemplate::gamma<e_float>(x);
  }
}

ef_complex efz::gamma(const ef_complex& z)
{
  return GammaTemplate::gamma<ef_complex>(z);
}

e_float ef::gamma_near_n(const INT32 n, const e_float& x)
{
  return GammaTemplate::gamma_near_n<e_float>(n, x);
}

namespace Gamma_Series
{
  static e_float lower_incomplete_gamma(const e_float& a, const e_float& x)
  {
    // Abramowitz and Stegun 13.6.10 (Special Cases), page 509
    const e_float pow_term  = ef::pow(x, a);
    const e_float h1f1_term = ef::hyperg_1f1(a, a + ef::one(), -x);

    return (pow_term * h1f1_term) / a;
  }
}

e_float ef::incomplete_gamma(const e_float& a, const e_float& x)
{
  // Abramowitz and Stegun 13.6.10 (Special Cases), page 509
  const e_float gamma_a  = ef::gamma(a);
  const e_float gamma_ax = Gamma_Series::lower_incomplete_gamma(a, x);

  return gamma_a - gamma_ax;
}

e_float ef::gen_incomplete_gamma(const e_float& a, const e_float& x0, const e_float& x1)
{
  if(ef::iszero(x0))
  {
    return Gamma_Series::lower_incomplete_gamma(a, x1);
  }
  else
  {
    const e_float term0 = ef::incomplete_gamma(a, x0);
    const e_float term1 = ef::incomplete_gamma(a, x1);

    return term0 - term1;
  }
}
