% FORWARD_SUB_DERIV  Derivative of forward substitution
%
%  [X, Xd] = FORWARD_SUB_DERIV(L, Ld, Y, Yd) solves a lower triangular system
%  LX=Y and returns the derivative of X in Xd.
%
%  Xd = FORWARD_SUB_DERIV(L, Ld, X, Yd) finds the derivative if X is given.
%
%  The parameters are:
%    L   nn        A lower triangular matrix
%    Ld  nnnPar   Ld (:,:,i) contains derivatives of L with respect to a
%                   parameter theta(i), i=1...,nPar
%    X   nm        Solution to LX = Y
%    Yd  nmnPar   Derivatives of Y. May also be nnPar when m=1
%    Xd  nmnPar   Returns derivatives of X. Is nnPar if Yd is nnPar.
%
%  METHOD:
%    From LX = Y it follows that Ld(:,:,i)X + LXd(:,:,i) = Yd, and thus Xd
%    may be obtained with forward substitution. 
%
function [varargout] = forward_sub_deriv(L, Ld, varargin)
  LT.LT = true;
  if nargout == 2
    [Y, Yd] = deal(varargin{:});
    X = linsolve(L, Y, LT);
  else
    [X, Yd] = deal(varargin{:});
  end
  [n, m] = size(X);
  nPar = size(Ld, 3);
  LdX= reshape(reshape(permute(Ld, [1,3,2]), n*nPar, n)*X, n,nPar, m);
  RHS = reshape(Yd - permute(LdX, [1,3,2]), [n,m*nPar]);
  Xd = reshape(linsolve(L,RHS,LT), [n,m,nPar]);
  if nargout == 2, varargout = {X, Xd}; else varargout = {Xd}; end
end
