function [Q,nfun]=da2glob(f,a,tolabs,tolrel,trace)
%DA2GLOB  Numerically evaluates an integral using adaptively
%   a sequence of 5/9/17/33 point rules, the three first being Newton-Cotes rules
%   while the fourth rule is an equidistant rule using 33 points of polynomial degree
%   27 (Thus this rule is not interpolatory). The routine is globally adaptive and the
%   fact that the routine makes use of four rules implies that it is doubly adaptive.
%
%   [Q,NFUN]=DA2GLOB(F,A) approximates the sum of the integrals of F(x) from
%   A(i) to A(i+1), i=1,...,length(A)-1, to a relative precision 10^-3. Note that length(A)
%   has to be at least 2. Q is the estimate of the integral
%   and NFUN is the number of function values used by the code. F is a function handle.
%   The function F must return a vector of output values if
%   given a vector of input values.
%
%   [Q,NFUN]=DA2GLOB(F,A,TOLABS) integrates to an absolute
%   error of TOLABS.
%
%   [Q,NFUN]=DA2GLOB(F,A,TOLABS,TOLREL) integrates to an absolute
%   error of max(TOLABS,EST*TOLREL), where EST is the current estimate of the
%   integral. The code also estimates the problems NOISE level
%   and it will not allow the effective absolute error to be below this
%   NOISE level: thus max([TOLABS,EST*TOLREL,NOISE])
%
%   [Q,NFUN]=DA2GLOB(F,A,TOLABS,TOLREL,TRACE) displays the left
%   and right end points of the current interval, the partial integral 
%   and the rule used: 5, 9, 17 or 33 nodes.
%
%   The code also counts the number of function values and it may stop due
%   to the fact that the maximum number (10 000) has been reached.
%   Furthermore the code may stop processing an interval if the interval
%   size becomes too small giving a warning that this has happened. Finally
%   the code checks that the tolerance is not below the problem's noise
%   level. In such a case the stopping criterion is modified and a warning
%   given.
%
%   Modified by
%   Terje O. Espelid 24/09/06
 
%   Reference:  Berntsen and Espelid, TOMS 1991
%               Espelid, in Espelid and Genz, 1992.
%               Espelid, Bit 2003
%               Espelid, Report 266 February 2004.
%   Check the number of input variables:
  if(nargin<5), trace=0; end
  if(nargin<4), tolrel=0; end
  if(nargin<3), tolabs=0;tolrel=0.001;end
% The weights of the rules and nullrules
w1=[1.5555555555555556e-01   7.1111111111111114e-01   2.6666666666666666e-01   7.1111111111111114e-01   1.5555555555555556e-01];
nw1=[1.2710311885185779e-01  -5.0841247540743117e-01   7.6261871311114682e-01  -5.0841247540743117e-01   1.2710311885185779e-01
  -3.3628324334270127e-01   6.7256648668540253e-01   0.0000000000000000e+00  -6.7256648668540253e-01   3.3628324334270127e-01
   5.6842242780997809e-01  -2.8421121390498905e-01  -5.6842242780997809e-01  -2.8421121390498905e-01   5.6842242780997809e-01
  -6.7256648668540253e-01  -3.3628324334270127e-01   0.0000000000000000e+00   3.3628324334270127e-01   6.7256648668540253e-01];
w2=[6.9770723104056437e-02   4.1537918871252205e-01  -6.5467372134038804e-02   7.4045855379188708e-01  -3.2028218694885363e-01   7.4045855379188708e-01  -6.5467372134038804e-02   4.1537918871252205e-01   6.9770723104056437e-02];
nw2=[1.1018547692345270e-02  -8.8148381538762158e-02   3.0851933538566756e-01  -6.1703867077133512e-01   7.7129833846416884e-01  -6.1703867077133512e-01   3.0851933538566756e-01  -8.8148381538762158e-02   1.1018547692345270e-02
  -4.2674651711845396e-02   2.5604791027107238e-01  -5.9744512396583560e-01   5.9744512396583560e-01   0.0000000000000000e+00  -5.9744512396583560e-01   5.9744512396583560e-01  -2.5604791027107238e-01   4.2674651711845396e-02
   1.1236757938944257e-01  -4.7756221240513091e-01   6.1802168664193424e-01   2.8091894847360643e-02  -5.6183789694721287e-01   2.8091894847360643e-02   6.1802168664193424e-01  -4.7756221240513091e-01   1.1236757938944257e-01
  -2.3112700627433047e-01   6.3559926725440874e-01  -2.3112700627433047e-01  -5.2003576411724362e-01   0.0000000000000000e+00   5.2003576411724362e-01   2.3112700627433047e-01  -6.3559926725440874e-01   2.3112700627433047e-01
   3.9111964345081668e-01  -5.8667946517622505e-01  -3.0730829128278453e-01   2.5143405650409645e-01   5.0286811300819290e-01   2.5143405650409645e-01  -3.0730829128278453e-01  -5.8667946517622505e-01   3.9111964345081668e-01
  -5.5619114160254801e-01   2.7809557080127401e-01   5.1646320291665171e-01   3.5755144817306656e-01   0.0000000000000000e+00  -3.5755144817306656e-01  -5.1646320291665171e-01  -2.7809557080127401e-01   5.5619114160254801e-01
   6.6477556470172228e-01   1.6619389117543057e-01  -1.8993587562906350e-01  -4.0361373571175990e-01  -4.7483968907265878e-01  -4.0361373571175990e-01  -1.8993587562906350e-01   1.6619389117543057e-01   6.6477556470172228e-01
  -6.4550259924248821e-01  -4.8412694943186613e-01  -3.2275129962124410e-01  -1.6137564981062205e-01   0.0000000000000000e+00   1.6137564981062205e-01   3.2275129962124410e-01   4.8412694943186613e-01   6.4550259924248821e-01];
w3=[3.0797894233299011e-02   2.6128238288028033e-01  -3.6795289329867603e-01   1.7037379778090087e+00  -3.9501480717783930e+00   8.5525299934402952e+00  -1.3934614237197881e+01   1.9180342211078734e+01  -2.0951950514333333e+01   1.9180342211078734e+01  -1.3934614237197881e+01   8.5525299934402952e+00  -3.9501480717783930e+00   1.7037379778090087e+00  -3.6795289329867603e-01   2.6128238288028033e-01   3.0797894233299011e-02];
nw3=[1.7047365484036969e-03  -2.7275784774459150e-02   2.0456838580844364e-01  -9.5465246710607032e-01   3.1026205180947288e+00  -7.4462892434273487e+00   1.3651530279616807e+01  -1.9502186113738293e+01   2.1939959377955578e+01  -1.9502186113738293e+01   1.3651530279616807e+01  -7.4462892434273487e+00   3.1026205180947288e+00  -9.5465246710607032e-01   2.0456838580844364e-01  -2.7275784774459150e-02   1.7047365484036969e-03
  -9.4915714022159627e-03   1.3288199963102348e-01  -8.5424142619943655e-01   3.3220499907755867e+00  -8.6373299760165256e+00   1.5547193956829744e+01  -1.9002125947236355e+01   1.3572947105168828e+01   0.0000000000000000e+00  -1.3572947105168828e+01   1.9002125947236355e+01  -1.5547193956829744e+01   8.6373299760165256e+00  -3.3220499907755867e+00   8.5424142619943655e-01  -1.3288199963102348e-01   9.4915714022159627e-03
   3.6721149063598142e-02  -4.4524393239612747e-01   2.4144155509315781e+00  -7.6150482870636651e+00   1.5037310541543439e+01  -1.7961232035732444e+01   9.1894675531654357e+00   7.2202959346299851e+00  -1.5753372948283605e+01   7.2202959346299851e+00   9.1894675531654357e+00  -1.7961232035732444e+01   1.5037310541543439e+01  -7.6150482870636651e+00   2.4144155509315781e+00  -4.4524393239612747e-01   3.6721149063598142e-02
  -1.1389885682659154e-01   1.1817006395758871e+00  -5.2962968424365071e+00   1.3027181749541409e+01  -1.7768221664948282e+01   9.4393677595037744e+00   8.1437682631012951e+00  -1.4251594460427265e+01   0.0000000000000000e+00   1.4251594460427265e+01  -8.1437682631012951e+00  -9.4393677595037744e+00   1.7768221664948282e+01  -1.3027181749541409e+01   5.2962968424365071e+00  -1.1817006395758871e+00   1.1389885682659154e-01
   3.0014984205120593e-01  -2.6263111179480521e+00   9.4697275167155475e+00  -1.7063518520611058e+01   1.2696338318766013e+01   5.4777346174345078e+00  -1.4032005115893879e+01  -1.1555768918971430e+00   1.3866922702765715e+01  -1.1555768918971430e+00  -1.4032005115893879e+01   5.4777346174345078e+00   1.2696338318766013e+01  -1.7063518520611058e+01   9.4697275167155475e+00  -2.6263111179480521e+00   3.0014984205120593e-01
  -6.9333963835801982e-01   5.0267123780956435e+00  -1.4040127676749902e+01   1.6466816411002974e+01  -6.9333963835801982e-01  -1.4040127676749902e+01   1.9066840054845546e+00   1.3346788038391882e+01   0.0000000000000000e+00  -1.3346788038391882e+01  -1.9066840054845546e+00   1.4040127676749902e+01   6.9333963835801982e-01  -1.6466816411002974e+01   1.4040127676749902e+01  -5.0267123780956435e+00   6.9333963835801982e-01
   1.4311829358292953e+00  -8.4081997479971111e+00   1.7174195229951543e+01  -9.5327006261487011e+00  -1.0938326723838186e+01   7.8970629852009342e+00   1.1858372896871305e+01  -3.0412637386372525e+00  -1.2880646422463659e+01  -3.0412637386372525e+00   1.1858372896871305e+01   7.8970629852009342e+00  -1.0938326723838186e+01  -9.5327006261487011e+00   1.7174195229951543e+01  -8.4081997479971111e+00   1.4311829358292953e+00
  -2.6735921944284664e+00   1.2365363899231658e+01  -1.6709951215177917e+01  -1.6709951215177918e+00   1.3367960972142335e+01   6.3497814617676083e+00  -8.6891746318925165e+00  -1.1696965850624542e+01   0.0000000000000000e+00   1.1696965850624542e+01   8.6891746318925165e+00  -6.3497814617676083e+00  -1.3367960972142335e+01   1.6709951215177918e+00   1.6709951215177917e+01  -1.2365363899231658e+01   2.6735921944284664e+00
   4.5591565530058782e+00  -1.5957047935520574e+01   1.1397891382514695e+01   1.1397891382514695e+01  -4.3838043778902680e+00  -1.2800708783439580e+01  -6.4880304792775965e+00   6.1373261290463752e+00   1.2274652258092750e+01   6.1373261290463752e+00  -6.4880304792775965e+00  -1.2800708783439580e+01  -4.3838043778902680e+00   1.1397891382514695e+01   1.1397891382514695e+01  -1.5957047935520574e+01   4.5591565530058782e+00
  -7.1376364508808008e+00   1.7844091127202002e+01  -2.1412909352642404e+00  -1.3561509256673522e+01  -8.1808294706249178e+00   4.1178671832004623e+00   1.1804552591841325e+01   9.6083567608010778e+00   0.0000000000000000e+00  -9.6083567608010778e+00  -1.1804552591841325e+01  -4.1178671832004623e+00   8.1808294706249178e+00   1.3561509256673522e+01   2.1412909352642404e+00  -1.7844091127202002e+01   7.1376364508808008e+00
   1.0294045683708612e+01  -1.6727824236026496e+01  -7.7205342627814595e+00   6.4337785523178832e+00   1.2669594687641368e+01   9.2052523902394316e+00   1.9796241699439637e-01  -8.4134027222618464e+00  -1.1877745019663783e+01  -8.4134027222618464e+00   1.9796241699439637e-01   9.2052523902394316e+00   1.2669594687641368e+01   6.4337785523178832e+00  -7.7205342627814595e+00  -1.6727824236026496e+01   1.0294045683708612e+01
  -1.3692360757232027e+01   1.1980815662578024e+01   1.3692360757232027e+01   5.1346352839620097e+00  -4.7396633390418552e+00  -1.0927557142790944e+01  -1.1585843717657870e+01  -7.2411523235361672e+00   0.0000000000000000e+00   7.2411523235361672e+00   1.1585843717657870e+01   1.0927557142790944e+01   4.7396633390418552e+00  -5.1346352839620097e+00  -1.3692360757232027e+01  -1.1980815662578024e+01   1.3692360757232027e+01
   1.6769648614663378e+01  -4.1924121536658445e+00  -1.2577236460997533e+01  -1.2577236460997533e+01  -7.7398378221523281e+00  -9.6747972776904101e-01   5.4823851240245656e+00   9.9972905202800906e+00   1.1609756733228490e+01   9.9972905202800906e+00   5.4823851240245656e+00  -9.6747972776904101e-01  -7.7398378221523281e+00  -1.2577236460997533e+01  -1.2577236460997533e+01  -4.1924121536658445e+00   1.6769648614663378e+01
  -1.8797050108382589e+01  -4.6992625270956472e+00   4.6992625270956472e+00   1.0069848272347816e+01   1.2083817926817378e+01   1.1412494708660859e+01   8.7272018360347730e+00   4.6992625270956472e+00   0.0000000000000000e+00  -4.6992625270956472e+00  -8.7272018360347730e+00  -1.1412494708660859e+01  -1.2083817926817378e+01  -1.0069848272347816e+01  -4.6992625270956472e+00   4.6992625270956472e+00   1.8797050108382589e+01
   1.8987887997058074e+01   1.1867429998161299e+01   5.6963663991174229e+00   4.7469719992645187e-01  -3.7975775994116150e+00  -7.1204579988967787e+00  -9.4939439985290370e+00  -1.0918035598308393e+01  -1.1392732798234846e+01  -1.0918035598308393e+01  -9.4939439985290370e+00  -7.1204579988967787e+00  -3.7975775994116150e+00   4.7469719992645187e-01   5.6963663991174229e+00   1.1867429998161299e+01   1.8987887997058074e+01];
w4=[1.4695535312430933e-02   1.4401180552105430e-01  -3.0306061552847280e-01   1.5082956203676328e+00  -4.4611013899969842e+00   1.1036738075229430e+01  -2.0246745867038545e+01   2.7624501654321350e+01  -2.4354757093571234e+01   7.4467704632794574e+00   1.3501113227549911e+01  -1.9331291092852421e+01   3.9483050394538197e+00   1.5326402464090805e+01  -1.4231233070221510e+01  -5.4208814966182697e+00   1.7596473481403098e+01  -5.4208814966182697e+00  -1.4231233070221510e+01   1.5326402464090805e+01   3.9483050394538197e+00  -1.9331291092852421e+01   1.3501113227549911e+01   7.4467704632794574e+00  -2.4354757093571234e+01   2.7624501654321350e+01  -2.0246745867038545e+01   1.1036738075229430e+01  -4.4611013899969842e+00   1.5082956203676328e+00  -3.0306061552847280e-01   1.4401180552105430e-01   1.4695535312430933e-02];
nw4=[5.8715078831162645e-08  -1.8788825225972046e-06   2.9122679100256674e-05  -2.9122679100256673e-04   2.1113942347686090e-03  -1.1823807714704208e-02   5.3207134716168940e-02  -1.9762650037434179e-01   6.1758281366981804e-01  -1.6468875031195149e+00   3.7878412571748843e+00  -7.5756825143497686e+00   1.3257444400112094e+01  -2.0396068307864759e+01   2.7680378417816460e+01  -3.3216454101379753e+01   3.5292482482715990e+01  -3.3216454101379753e+01   2.7680378417816460e+01  -2.0396068307864759e+01   1.3257444400112094e+01  -7.5756825143497686e+00   3.7878412571748843e+00  -1.6468875031195149e+00   6.1758281366981804e-01  -1.9762650037434179e-01   5.3207134716168940e-02  -1.1823807714704208e-02   2.1113942347686090e-03  -2.9122679100256673e-04   2.9122679100256674e-05  -1.8788825225972046e-06   5.8715078831162645e-08
   4.6603649039042801e-07  -1.3981094711712842e-05   2.0225983682944576e-04  -1.8781270562734248e-03   1.2569004145829843e-02  -6.4520887948593189e-02   2.6394908706242670e-01  -8.8234409103725508e-01   2.4509558084368197e+00  -5.7188968863525798e+00   1.1274396718809369e+01  -1.8790661198015616e+01   2.6306925677221862e+01  -3.0354145012179075e+01   2.7463274058638209e+01  -1.6477964435182926e+01   0.0000000000000000e+00   1.6477964435182926e+01  -2.7463274058638209e+01   3.0354145012179075e+01  -2.6306925677221862e+01   1.8790661198015616e+01  -1.1274396718809369e+01   5.7188968863525798e+00  -2.4509558084368197e+00   8.8234409103725508e-01  -2.6394908706242670e-01   6.4520887948593189e-02  -1.2569004145829843e-02   1.8781270562734248e-03  -2.0225983682944576e-04   1.3981094711712842e-05  -4.6603649039042801e-07
   2.5941169713712034e-06  -7.2797407509104397e-05   9.8025195055689352e-04  -8.4235842029744285e-03   5.1814243856925575e-02  -2.4246903280016285e-01   8.9516394110205377e-01  -2.6664070668800939e+00   6.4913561768485888e+00  -1.2978128062611834e+01   2.1214265426542749e+01  -2.7814727731222547e+01   2.7751464514244784e+01  -1.7656926071870146e+01  -3.0820541604550561e-01   1.7845093589034775e+01  -2.5149561949313256e+01   1.7845093589034775e+01  -3.0820541604550561e-01  -1.7656926071870146e+01   2.7751464514244784e+01  -2.7814727731222547e+01   2.1214265426542749e+01  -1.2978128062611834e+01   6.4913561768485888e+00  -2.6664070668800939e+00   8.9516394110205377e-01  -2.4246903280016285e-01   5.1814243856925575e-02  -8.4235842029744285e-03   9.8025195055689352e-04  -7.2797407509104397e-05   2.5941169713712034e-06
   1.1691232321542031e-05  -3.0616414642038194e-04   3.8259557772246299e-03  -3.0311711899658006e-02   1.7054000587433360e-01  -7.2282432161771792e-01   2.3869667522003519e+00  -6.2552031404138191e+00   1.3090555918107396e+01  -2.1722554438601829e+01   2.7751317407199895e+01  -2.4995193418128675e+01   1.0644340923309549e+01   9.5696718877831053e+00  -2.2780059283676206e+01   1.8751878068879805e+01   0.0000000000000000e+00  -1.8751878068879805e+01   2.2780059283676206e+01  -9.5696718877831053e+00  -1.0644340923309549e+01   2.4995193418128675e+01  -2.7751317407199895e+01   2.1722554438601829e+01  -1.3090555918107396e+01   6.2552031404138191e+00  -2.3869667522003519e+00   7.2282432161771792e-01  -1.7054000587433360e-01   3.0311711899658006e-02  -3.8259557772246299e-03   3.0616414642038194e-04  -1.1691232321542031e-05
   4.5241558967149093e-05  -1.1027629998242592e-03   1.2747356515917244e-02  -9.2720933272403092e-02   4.7449675410899761e-01  -1.8076040501785993e+00   5.2805464198507774e+00  -1.1969581694292478e+01   2.0929303400664619e+01  -2.7287443554651361e+01   2.3767281767212349e+01  -7.4577015838115095e+00  -1.3413226523862413e+01   2.2705033677997015e+01  -1.1301506525198945e+01  -1.0816011694045546e+01   2.1954889408808871e+01  -1.0816011694045546e+01  -1.1301506525198945e+01   2.2705033677997015e+01  -1.3413226523862413e+01  -7.4577015838115095e+00   2.3767281767212349e+01  -2.7287443554651361e+01   2.0929303400664619e+01  -1.1969581694292478e+01   5.2805464198507774e+00  -1.8076040501785993e+00   4.7449675410899761e-01  -9.2720933272403092e-02   1.2747356515917244e-02  -1.1027629998242592e-03   4.5241558967149093e-05
   1.5522490188229926e-04  -3.5119634050870211e-03   3.7418589956570554e-02  -2.4870096225170243e-01   1.1501539322615948e+00  -3.9016261036399991e+00   9.9402265465181259e+00  -1.9029175320746088e+01   2.6538951692463691e+01  -2.4154669033742763e+01   7.6327558038613033e+00   1.3999467890348040e+01  -2.1884870999228660e+01   6.9693946747849482e+00   1.5365058964657567e+01  -1.9385822058212817e+01   0.0000000000000000e+00   1.9385822058212817e+01  -1.5365058964657567e+01  -6.9693946747849482e+00   2.1884870999228660e+01  -1.3999467890348040e+01  -7.6327558038613033e+00   2.4154669033742763e+01  -2.6538951692463691e+01   1.9029175320746088e+01  -9.9402265465181259e+00   3.9016261036399991e+00  -1.1501539322615948e+00   2.4870096225170243e-01  -3.7418589956570554e-02   3.5119634050870211e-03  -1.5522490188229926e-04
   4.8185679100586412e-04  -1.0088876561685280e-02   9.8687379551491344e-02  -5.9613942673884168e-01   2.4724421681278552e+00  -7.3825502391665569e+00   1.6091840692152612e+01  -2.5081900159591495e+01   2.5487465225487519e+01  -1.0523310289681163e+01  -1.2223152411073594e+01   2.1618538930469395e+01  -6.1057667224516852e+00  -1.6491446666201796e+01   1.6879676545447197e+01   5.8692537283905315e+00  -2.0208063469901575e+01   5.8692537283905315e+00   1.6879676545447197e+01  -1.6491446666201796e+01  -6.1057667224516852e+00   2.1618538930469395e+01  -1.2223152411073594e+01  -1.0523310289681163e+01   2.5487465225487519e+01  -2.5081900159591495e+01   1.6091840692152612e+01  -7.3825502391665569e+00   2.4724421681278552e+00  -5.9613942673884168e-01   9.8687379551491344e-02  -1.0088876561685280e-02   4.8185679100586412e-04
   1.3722780701947185e-03  -2.6502120230635501e-02   2.3695590241914691e-01  -1.2928989772270643e+00   4.7648371954404585e+00  -1.2338354097465723e+01   2.2386851966397579e+01  -2.6656941417187973e+01   1.5348266209610093e+01   8.0550647703186744e+00  -2.1558516149655606e+01   8.2014503201049713e+00   1.5625931183651909e+01  -1.6242861476963139e+01  -7.7016063331055165e+00   1.9254015832763791e+01   0.0000000000000000e+00  -1.9254015832763791e+01   7.7016063331055165e+00   1.6242861476963139e+01  -1.5625931183651909e+01  -8.2014503201049713e+00   2.1558516149655606e+01  -8.0550647703186744e+00  -1.5348266209610093e+01   2.6656941417187973e+01  -2.2386851966397579e+01   1.2338354097465723e+01  -4.7648371954404585e+00   1.2928989772270643e+00  -2.3695590241914691e-01   2.6502120230635501e-02  -1.3722780701947185e-03
   3.6217967807219459e-03  -6.4286892857814526e-02   5.2279468208179114e-01  -2.5580692246161996e+00   8.2829978716501831e+00  -1.8232965984229885e+01   2.6381507168330124e+01  -2.0878710690369619e+01  -1.2493768706775299e+00   2.0396423529067544e+01  -1.2483407910478503e+01  -1.2782236360100359e+01   1.7364689557565221e+01   6.5116572372290422e+00  -1.8784518498202655e+01  -1.9625616341405758e+00   1.9064884445937022e+01  -1.9625616341405758e+00  -1.8784518498202655e+01   6.5116572372290422e+00   1.7364689557565221e+01  -1.2782236360100359e+01  -1.2483407910478503e+01   2.0396423529067544e+01  -1.2493768706775299e+00  -2.0878710690369619e+01   2.6381507168330124e+01  -1.8232965984229885e+01   8.2829978716501831e+00  -2.5580692246161996e+00   5.2279468208179114e-01  -6.4286892857814526e-02   3.6217967807219459e-03
   8.9267041267638743e-03  -1.4505894205991293e-01   1.0669571117965111e+00  -4.6421021145654429e+00   1.3018458548085839e+01  -2.3678427749598690e+01   2.5290263831281820e+01  -7.8267559421518200e+00  -1.6361397535637412e+01   1.7548616913319684e+01   7.2778629342860608e+00  -1.9084342664489306e+01  -2.5966307760192344e+00   1.8896613778815670e+01   6.9102271998418607e-01  -1.8657613439573019e+01   0.0000000000000000e+00   1.8657613439573019e+01  -6.9102271998418607e-01  -1.8896613778815670e+01   2.5966307760192344e+00   1.9084342664489306e+01  -7.2778629342860608e+00  -1.7548616913319684e+01   1.6361397535637412e+01   7.8267559421518200e+00  -2.5290263831281820e+01   2.3678427749598690e+01  -1.3018458548085839e+01   4.6421021145654429e+00  -1.0669571117965111e+00   1.4505894205991293e-01  -8.9267041267638743e-03
   2.0670095438142064e-02  -3.0617578867747930e-01   2.0239190625984023e+00  -7.7498272140203399e+00   1.8472374556657293e+01  -2.6565097730985453e+01   1.7488564809808715e+01   8.0791117210474273e+00  -2.0923643243975334e+01   1.2375256840198827e+00   1.9327197389815126e+01  -3.8915521428854851e+00  -1.8374933105637439e+01   3.3477751591281439e+00   1.8212173348301871e+01  -1.2746036736260380e+00  -1.8246957854014862e+01  -1.2746036736260380e+00   1.8212173348301871e+01   3.3477751591281439e+00  -1.8374933105637439e+01  -3.8915521428854851e+00   1.9327197389815126e+01   1.2375256840198827e+00  -2.0923643243975334e+01   8.0791117210474273e+00   1.7488564809808715e+01  -2.6565097730985453e+01   1.8472374556657293e+01  -7.7498272140203399e+00   2.0239190625984023e+00  -3.0617578867747930e-01   2.0670095438142064e-02
   4.5180960042357270e-02  -6.0711915056917587e-01   3.5802267207758272e+00  -1.1913654401169083e+01   2.3514440705426384e+01  -2.4728506248444443e+01   4.1020944505643007e+00   1.9351488280667098e+01  -1.1473501262280354e+01  -1.5474613879946869e+01   1.1851609107529159e+01   1.5197013568029222e+01  -9.4515276581155589e+00  -1.6547014623265930e+01   5.2570987038384445e+00   1.7742708125454751e+01   0.0000000000000000e+00  -1.7742708125454751e+01  -5.2570987038384445e+00   1.6547014623265930e+01   9.4515276581155589e+00  -1.5197013568029222e+01  -1.1851609107529159e+01   1.5474613879946869e+01   1.1473501262280354e+01  -1.9351488280667098e+01  -4.1020944505643007e+00   2.4728506248444443e+01  -2.3514440705426384e+01   1.1913654401169083e+01  -3.5802267207758272e+00   6.0711915056917587e-01  -4.5180960042357270e-02
   9.3587837517216846e-02  -1.1347525298962542e+00   5.9182986120262564e+00  -1.6841659679660768e+01   2.6478174968664145e+01  -1.7062772657287653e+01  -1.0453934086747147e+01   1.9553121764139465e+01   5.7979868837534045e+00  -1.8212112501680124e+01  -7.5038264866713851e+00   1.5696640074708744e+01   1.1815386409932231e+01  -1.1032837496587840e+01  -1.5945418200077786e+01   4.0187639366049703e+00   1.7630706302525031e+01   4.0187639366049703e+00  -1.5945418200077786e+01  -1.1032837496587840e+01   1.1815386409932231e+01   1.5696640074708744e+01  -7.5038264866713851e+00  -1.8212112501680124e+01   5.7979868837534045e+00   1.9553121764139465e+01  -1.0453934086747147e+01  -1.7062772657287653e+01   2.6478174968664145e+01  -1.6841659679660768e+01   5.9182986120262564e+00  -1.1347525298962542e+00   9.3587837517216846e-02
   1.8430030171174749e-01  -2.0042657811152540e+00   9.1503613507126893e+00  -2.1800198994814970e+01   2.5600880202876404e+01  -4.6120432982724102e+00  -1.9903604019366966e+01   8.0422019726227543e+00   1.8192701677350168e+01  -4.4143663617590043e+00  -1.8352051052972740e+01  -2.7813406349955421e+00   1.6365569533452028e+01   1.1004434686286709e+01  -9.9563932875927357e+00  -1.6593988812654562e+01   0.0000000000000000e+00   1.6593988812654562e+01   9.9563932875927357e+00  -1.1004434686286709e+01  -1.6365569533452028e+01   2.7813406349955421e+00   1.8352051052972740e+01   4.4143663617590043e+00  -1.8192701677350168e+01  -8.0422019726227543e+00   1.9903604019366966e+01   4.6120432982724102e+00  -2.5600880202876404e+01   2.1800198994814970e+01  -9.1503613507126893e+00   2.0042657811152540e+00  -1.8430030171174749e-01
   3.4596507459410830e-01  -3.3515366601304244e+00   1.3227584021376350e+01  -2.5616063234815780e+01   1.9769845155267888e+01   9.0804046563256602e+00  -1.932712659253413e+01   -8.2949439973067314e+00   1.5928696399744720e+01   1.3137557097418776e+01  -9.0186629748404634e+00  -1.7330844293948672e+01  -1.7927456062254210e+00   1.5625407726987476e+01   1.2560860448813177e+01  -6.3687093641070831e+00  -1.7151375713238878e+01  -6.3687093641070831e+00   1.2560860448813177e+01   1.5625407726987476e+01  -1.7927456062254210e+00  -1.7330844293948672e+01  -9.0186629748404634e+00   1.3137557097418776e+01   1.5928696399744720e+01  -8.2949439973067314e+00  -1.9327126592534139e+01   9.0804046563256602e+00   1.9769845155267888e+01  -2.5616063234815780e+01   1.3227584021376350e+01  -3.3515366601304244e+00   3.4596507459410830e-01];

% w1 and nw1 are the weights and nullrules associated with Newton-Cotes 5 point rule.
% w2 and nw2 are the weights and nullrules associated with Newton-Cotes 9 point rule.
% w3 and nw3 are the weights and nullrules associated with Newton-Cotes 17 point rule.
% w4 and nw4 are the weights and nullrules associated with 33 point rule (degree 27).
% The coordinates for the four new points in a 5 point approximation to get a 9 point approximation.
  z4=(-3:2:3)/4;
% The coordinates for the eight new points in a 9 point approximation to get a 17 point approximation.
  z8=(-7:2:7)/8;
% The coordinates for the sixteen new points in a 17 point approximation to get a 33 point approximation.
  z16=(-15:2:15)/16;
% Define the number of intervals given
  nint=length(a)-1;
  if nint < 1, warning('No interval is given.');Q=0;nfun=0;return,end
% Initialize: the function count, the maximum number of intervals, too small intervals
% (logical: interval) and the maximum number of function values allowed.
  nfun=0;int_max=2000;interval=0;nmax=10000;singular=0;
% Initialize several dynamic vectors for speed: The choice of value of
% int_max is connected to the fact that we have a bound 10 000 on the
% number of function values and if we use the 5 point rule everywhere
% (not very likely) then 2000 intervals times 4 f-evaluation per interval
% implies approx 8 000 function evaluations. Therefore, this will do in
% most situations.
  is=zeros(1,int_max);error=is;isa=is;lp=is;rp=is;flag=is;
  fu=zeros(int_max,33);
  for i=1:nint  
% Compute y and count the number of function values
     x=linspace(a(i),a(i+1),9); y=feval(f,x);nfun=nfun+9;
% Check if some of the function values are infinite. In case there are: put
% the value(s) to zero.
     IN=find(isinf(y)); y(IN)=0;singular=max(1-isempty(IN),singular);
% Check if some of the function values are NaN: put
% those function values to 1 (any number might do).
     y(isnan(y))=1;
% Apply the degree 9 Cotes rule on the interval and
% initialize the datastructure by storing: error estimate, integral, absolute value, 
% function values, left and right point of the interval and
% signalling by a flag that a nine point rule has been used.
% flag is used as follows: 0: the 5 point rule has been used
%                          1: the 9 point rule has been used
%                          2: the 9 point rule has been used
%                          3: the 9 point rule has been used
%                          4: the 17 point rule has been used
%                          5: the 17 point rule has been used
%                          6: the 17 point rule has been used
%                          7: the 33 point rule has been used
% The 9 point rule is used either in combination with the 5 point rule (flag=1 or 2)
% or in combination with the 17 point rule (flag=3). The 17 point rule is used either
% combination with the 9 point rule (flag=4 or 5) or in combination with the 33 point rule
% (flag=6). The code decides how to proceed.
     [error(i),is(i),isa(i),rmax]=cotes9(a(i),a(i+1),y(1:9),32,w2,nw2,interval);
     fu(i,1:33)=0;fu(i,1:4:33)=y;lp(i)=a(i);rp(i)=a(i+1);
     flag(i)=1+ (rmax<1/4);
  end
% Compute global values
  Q=sum(is);isabs=sum(isa);total=sum(error);noise=50*eps*isabs;

% global error test and check the number of function evaluations.

  while (total> max([abs(Q)*tolrel,tolabs,noise]))&&(nfun<nmax-3)

%  Find the interval with the greatest error estimate.
      [err,i]=max(error);
%  Prepare to process this interval: either by subdivision or  by using a new rule
      aa=lp(i);bb=rp(i);cc=(aa+bb)/2;total=total-err;Q=Q-is(i);isabs=isabs-isa(i);
      if (flag(i)==1) || (flag(i)==2)
%  On this interval a 9 point rule has been used.
%  Subdivide the interval and use all nine function values
%  to compute: the left and the right part  of the integral using the 5 point rule
         z(1:5)=fu(i,1:4:17);[Lerr,Linte,Linteabs]=cotes5(aa,cc,z(1:5),32,w1,nw1);
         zz(1:5)=fu(i,17:4:33);[Rerr,Rinte,Rinteabs]=cotes5(cc,bb,zz(1:5),32,w1,nw1);
%  Check if this subdivision, in the case flag(i)==2, implies a reduction of the error.
         if (flag(i)==2)&&(Lerr + Rerr >= err)
%  With no error reduction we choose instead to apply the 17 point rule.
%  Compute f in 8 new points
             hh=(bb-aa)/2;x8=cc+hh*z8;y8=feval(f,x8);nfun=nfun+8;
             IN=find(isinf(y8)); y8(IN)=0;singular=max(1-isempty(IN),singular);
             y8(isnan(y8))=1;
             fu(i,3:4:31)=y8(1:8);y(1:17)=fu(i,1:2:33);
             [err,inte,inteabs]=cotes17(aa,bb,y(1:17),32,w3,nw3,interval);
             error(i)=err;is(i)=inte;isa(i)=inteabs;total=total+err;
             Q=Q+inte;isabs=isabs+inteabs;flag(i)=4;
         else
%  Store the left part in the datastructure        
             error(i)=Lerr;is(i)=Linte;isa(i)=Linteabs;
             total=total+Lerr;Q=Q+Linte;isabs=isabs+Linteabs;
             flag(i)=0;fu(i,1:33)=0;fu(i,1:8:33)=z(1:5);lp(i)=aa;rp(i)=cc;
%  and then the right part of the integral. Increase the size of the datastructure
             nint=nint+1;i=nint;
             error(i)=Rerr;is(i)=Rinte;isa(i)=Rinteabs;
             total=total+Rerr;Q=Q+Rinte;isabs=isabs+Rinteabs;
             flag(i)=0;fu(i,1:33)=0;fu(i,1:8:33)=zz(1:5);lp(i)=cc;rp(i)=bb;
         end
      elseif flag(i)==0
%  On this interval a 5 point rule has been used only. Compute f in 4 new
%  points and apply the 9 point rule
         hh=(bb-aa)/2;x4=cc+hh*z4;y4=feval(f,x4);nfun=nfun+4;
         IM=find(isinf(y4)); y4(IM)=0;singular=max(1-isempty(IM),singular);
         y4(isnan(y4))=1;
         fu(i,5:8:29)=y4(1:4);y(1:9)=fu(i,1:4:33);
         [err,inte,inteabs,rmax]=cotes9(aa,bb,y(1:9),32,w2,nw2,interval);
         error(i)=err;is(i)=inte;isa(i)=inteabs;flag(i)=1+(rmax<1/4);
         total=total+err;Q=Q+inte;isabs=isabs+inteabs;
         
      elseif flag(i)==3
%  Apply the 17 point rule: compute f in 8 new points.
             hh=(bb-aa)/2;x8=cc+hh*z8;y8=feval(f,x8);nfun=nfun+8;
             IN=find(isinf(y8)); y8(IN)=0;singular=max(1-isempty(IN),singular);
             y8(isnan(y8))=1;
             fu(i,3:4:31)=y8(1:8);y(1:17)=fu(i,1:2:33);
             [err,inte,inteabs,rmax]=cotes17(aa,bb,y(1:17),32,w3,nw3,interval);
             error(i)=err;is(i)=inte;isa(i)=inteabs;flag(i)=4+(rmax<1/8);
             Q=Q+inte;isabs=isabs+inteabs;total=total+err;
      elseif (flag(i)==4||flag(i)==5)
%  On this interval a 17 point rule has been used.
%  Subdivide the interval and use all 17 function values
%  to compute: the left and the right part  of the integral using the 9 point rule
          z(1:9)=fu(i,1:2:17);[Lerr,Linte,Linteabs]=cotes9(aa,cc,z(1:9),32,w2,nw2,interval);
          zz(1:9)=fu(i,17:2:33);[Rerr,Rinte,Rinteabs]=cotes9(cc,bb,zz(1:9),32,w2,nw2,interval);
%  Check if this subdivision, in the case flag(i)==5, implies a reduction of the error.
         if (flag(i)==5)&&(Lerr + Rerr >= err)
%  With no error reduction we choose instead to apply the 33 point rule.
%  Compute f in 16 new points
             hh=(bb-aa)/2;x16=cc+hh*z16;y16=feval(f,x16);nfun=nfun+16;
             IN=find(isinf(y16)); y16(IN)=0;singular=max(1-isempty(IN),singular);
             fu(i,2:2:32)=y16;y=fu(i,1:33);
             [err,inte,inteabs]=rule33(aa,bb,y(1:33),32,w4,nw4,interval);
             error(i)=err;is(i)=inte;isa(i)=inteabs;total=total+err;
             Q=Q+inte;isabs=isabs+inteabs;flag(i)=7;
         else
%  Store the left part in the datastructure and update global variables
            error(i)=Lerr;is(i)=Linte;isa(i)=Linteabs;flag(i)=3;
            fu(i,1:33)=0;fu(i,1:4:33)=z(1:9);lp(i)=aa;rp(i)=cc;
            total=total+Lerr;Q=Q+Linte;isabs=isabs+Linteabs;
%  and then the right part of the integral. Increase the size of the datastructure
             nint=nint+1;i=nint;
             error(i)=Rerr;is(i)=Rinte;isa(i)=Rinteabs;flag(i)=3;
             fu(i,1:33)=0;fu(i,1:4:33)=zz(1:9);lp(i)=cc;rp(i)=bb;
             total=total+Rerr;Q=Q+Rinte;isabs=isabs+Rinteabs;
         end
     elseif flag(i)==6
%  On this interval a 17 point rule has been used.
%  Apply the 33 point rule.
%  Compute f in 16 new points
             hh=(bb-aa)/2;x16=cc+hh*z16;y16=feval(f,x16);nfun=nfun+16;
             IMN=find(isinf(y16)); y16(IMN)=0;singular=max(1-isempty(IMN),singular);
             y16(isnan(y16))=1;
             fu(i,2:2:32)=y16;y=fu(i,1:33);
             [err,inte,inteabs]=rule33(aa,bb,y(1:33),32,w4,nw4,interval);
             error(i)=err;is(i)=inte;isa(i)=inteabs;flag(i)=7;
             Q=Q+inte;isabs=isabs+inteabs;total=total+err;
     elseif flag(i)==7
%  On this interval a 33 point rule has been used.
%  Subdivide the interval and apply the 17 point rule on each part.
         z(1:17)=fu(i,1:17);[Lerr,Linte,Linteabs]=cotes17(aa,cc,z(1:17),32,w3,nw3,interval);
         zz(1:17)=fu(i,17:33);[Rerr,Rinte,Rinteabs]=cotes17(cc,bb,zz(1:17),32,w3,nw3,interval);
%  Store the left part in the datastructure
             error(i)=Lerr;is(i)=Linte;isa(i)=Linteabs;
             flag(i)=6;fu(i,1:33)=0;fu(i,1:2:33)=z(1:17);lp(i)=aa;rp(i)=cc;
             total=total+Lerr;Q=Q+Linte;isabs=isabs+Linteabs;
%  and then the right part of the integral. Increase the size of the datastructure
             nint=nint+1;i=nint;
             error(i)=Rerr;is(i)=Rinte;isa(i)=Rinteabs;
             flag(i)=6;fu(i,1:33)=0;fu(i,1:2:33)=zz(1:17);lp(i)=cc;rp(i)=bb;
             total=total+Rerr;Q=Q+Rinte;isabs=isabs+Rinteabs;
     end

% Redefine the global noise level
      noise=50*eps*isabs;
  end
  if (nfun>=nmax-3) 
       warning(['Stopping: maximum number of f-evaluations, ',...
                'required tolerance may not be met.']);
  end
  if (interval==1)
         warning(['Interval too small, ',...
                'required tolerance may not be met.']);
  end
  if max(abs(Q)*tolrel,tolabs) < noise
       warning(['Stopping: the tolerance is below the noise level of the problem, ',...
                'required tolerance may not be met.']);
  end
  if singular==1
      warning('Singularity probably detected, required tolerance may not be met.')
  end
  if trace
     format long
     for i=1:nint
         if flag(i)==0
             points=5;
         elseif (flag(i)> 0) && (flag(i) < 4)
             points=9;
         elseif (flag(i)>3)&&(flag(i)<7)
             points=17;
         else
             points=33;
         end
         disp([lp(i),rp(i),is(i),points])
     end
  end
  Q=sum(is);

function [err,Q,Qabs,rmax]=cotes9(a,b,y,D,w2,nw2,interval)
%
%   [err,Q,Qabs,rmax] = COTES9(A,B,Y,D,W2,NW2,INTERVAL) applies a 9 point Newton-Cotes rule to 
%   approximate the integral of F(X) from A to B.
%   The argument Y contains the 9 function values to be used in the
%   approximation and D is a constant to be used in the error estimate.
%   W2: is the rule's weights, NW2: is the nullrules weights
%   INTERVAL: logical signalling that too small intervals has been found.

%
%   See also COTES5
%   Modified by
%   T O Espelid, 10/07/2006

  h=(b-a)/2;x=linspace(a,b,9);Q=(h*w2)*y';
  Qabs=(h*abs(w2))*abs(y');
% define the local noise level
  noise=50*Qabs*eps;
%  Compute the error estimates
  e=(h*nw2)*y';E=[norm(e(1:2)),norm(e(3:4)),norm(e(5:6)),norm(e(7:8))];
  Emin=min(E(2:4));
  if Emin==0,
     rmax=2;
  else
     r=E(1:3)./E(2:4);
     if sum(isinf(r))>0
        rmax=2;
     else
        rmax=max(r);
     end
  end
  if rmax > 1
     err=D*max(E(1:4));
  elseif 0.25 < rmax
         err=D*rmax*E(1);
      else
         err=D*4*rmax*rmax*E(1);
  end
  
% If the highest degree null rules are on the local noise level, then put the error to zero.
  if (E(1)<noise) && (E(2) < noise), err=0; end;
%check if interval has become too small: unable to distinguish between the
%endpoints and the two points close to the endpoints.
  if ((x(2) <= a) || (b<=x(8)))
       interval =1;
%In order to avoid handling this interval again: put the error to zero.
       err=0;
   end

function [err,Q,Qabs]=cotes5(a,b,y,C,w1,nw1)
%
%
%   [err,Q,Qabs] = COTES5(A,B,Y,C,W1,NW1) applies a 5 point Newton-Cotes rule to 
%   approximate the integral of F(X) from A to B. 
%   The argument Y contains the 5 function values to be used in the
%   approximation. C is a constant.
%   W1: is the rule's weights, NW1: is the nullrules weights
%
%   See also COTES9
%   Modified by
%   T O Espelid, 10/07/2006

  h=(b-a)/2;Q=(h*w1)*y';Qabs=(h*w1)*abs(y');
% define the local noise level 
  noise=50*Qabs*eps;E=abs((h*nw1)*y');
  Emin=min(E(2:4));
  if Emin==0,
     rmax=2;
  else
     r=E(1:3)./E(2:4);
     if sum(isinf(r))>0
        rmax=2;
     else
        rmax=max(r);
     end
  end
  if rmax > 1
     err=C*max(E(1:4));
  elseif 0.5 < rmax
         err=C*rmax*E(2);
      else
         err=C*(2*rmax)^3*rmax*E(2);
  end
% If the highest degree null rules are on the local noise level, then put the error to zero.
  if (E(1)<noise) && (E(2) < noise), err=0; end;

function [err,Q,Qabs,rmax]=cotes17(a,b,y,D,w3,nw3,interval)
%
%   [err,Q,Qabs,rmax] = COTES17(A,B,Y,D,W3,NW3,INTERVAL) applies a 17 point Newton-Cotes rule to
%   approximate the integral of F(X) from A to B.
%   The argument Y contains the 17 function values to be used in the
%   approximation and D is a constant to be used in the error estimate.
%   W3: is the rule's weights, NW3: is the nullrules weights
%   INTERVAL: logical signalling that too small intervals has been found.

%
%   See also COTES5 and COTES9
%   Modified by
%   T O Espelid, 10/07/2006

  h=(b-a)/2;x=linspace(a,b,17);Q=(h*w3)*y';
  Qabs=(h*abs(w3))*abs(y');
% define the local noise level
  noise=50*Qabs*eps;
%  Compute the error estimates
  e=(h*nw3)*y';
  E=[norm(e(1:3)),norm(e(4:6)),norm(e(7:9)),norm(e(10:12)),norm(e(13:15))];
  Emin=min(E(2:5));
  if Emin==0,
     rmax=2;
  else
     r=E(1:4)./E(2:5);
     if sum(isinf(r))>0
        rmax=2;
     else
        rmax=max(r);
     end
  end
  if rmax > 1
     err=D*max(E(1:5));
  elseif 0.125 < rmax
         err=D*rmax*E(1);
      else
         err=D*(8*rmax)^(2/3)*rmax*E(1);
  end

% If the highest degree null rules are on the local noise level, then put the error to zero.
  if (E(1)<noise) && (E(2) < noise), err=0; end;
%check if interval has become too small: unable to distinguish between the
%endpoints and the two points close to the endpoints.
  if ((x(2) <= a)|| (b<=x(16)))
       interval =1;
%In order to avoid handling this interval again: put the error to zero.
       err=0;
   end

function [err,Q,Qabs]=rule33(a,b,y,D,w4,nw4,interval)
%
%   [err,Q,Qabs,rmax] = RULE33(A,B,Y,D) applies a 33 point degree 27 rule to
%   approximate the integral of F(X) from A to B.
%   The argument Y contains the 17 function values to be used in the
%   approximation and D is a constant to be used in the error estimate.
%   W4: is the rule's weights, NW4: is the nullrules weights
%   INTERVAL: logical signalling that too small intervals has been found.
%
%   See also COTES5, COTES9 and COTES17
%   Modified by
%   T O Espelid, 10/07/2006

  h=(b-a)/2;x=linspace(a,b,33); Q=(h*w4)*y';
  Qabs=(h*abs(w4))*abs(y');
% define the local noise level
  noise=50*Qabs*eps;
%  Compute the error estimates
  e=(h*nw4)*y';
  E=[norm(e(1:3)),norm(e(4:6)),norm(e(7:9)),norm(e(10:12)),norm(e(13:15))];
  Emin=min(E(2:5));
  if Emin==0,
     rmax=2;
  else
     r=E(1:4)./E(2:5);
     if sum(isinf(r))>0
        rmax=2;
     else
        rmax=max(r);
     end
  end
  if rmax > 1
     err=D*max(E(1:5));
  else
% Note: the rule has local order 29
         err=D*norm(e(1:4));
  end

% If the highest degree null rules are on the local noise level, then put the error to zero.
  if (E(1)<noise) && (E(2) < noise), err=0; end;
% check if interval has become too small: unable to distinguish between the
% endpoints and the two points close to the endpoints.
  if ((x(2) <= a) || (b<=x(32)))
       interval =1;
% In order to avoid handling this interval again: put the error to zero.
       err=0;
   end
