function [Q,nfun]=da3glob(f,a,tolabs,tolrel,trace)
%DA3GLOB  Numerically evaluates an integral using adaptively
%   a sequence of 5/9/17/33 point rules, the two first being Newton-Cotes rules
%   while the third and the fourth rule  are equidistant rules using 17 and 33 points 
%   respectively. These two rules have polynomial degrees 15 og 23
%   respectively (They are not interpolatory). All four rules are specified
%   on the interval [-1,1] and their weights have a 2-norm between 1 and 4.
%   The routine is globally adaptive and the
%   fact that the routine makes use of four rules implies that it is doubly adaptive.
%
%   [Q,NFUN]=DA3GLOB(F,A) approximates the sum of the integrals of F(x) from
%   A(i) to A(i+1), i=1,...,length(A)-1, to a relative precision 10^-3. Note that length(A)
%   has to be at least 2. Q is the estimate og the integral
%   and NFUN is the number of function values used by the code. F is a function handle.
%   The function F must return a vector of output values if
%   given a vector of input values.
%
%   [Q,NFUN]=DA3GLOB(F,A,TOLABS) integrates to an absolute
%   error of TOLABS.
%
%   [Q,NFUN]=DA3GLOB(F,A,TOLABS,TOLREL) integrates to an absolute
%   error of max(TOLABS,EST*TOLREL), where EST is the current estimate of the
%   integral. The code also estimates the problems NOISE level
%   and it will not allow the effective absolute error to be below this
%   NOISE level: thus max([TOLABS,EST*TOLREL,NOISE])
%
%   [Q,NFUN]=DA3GLOB(F,A,TOLABS,TOLREL,TRACE) displays the left
%   and right end points of the current interval, the partial integral 
%   and the rule used: 5, 9, 17 or 33 nodes.
%
%   The code also counts the number of function values and it may stop due
%   to the fact that the maximum number (10 000) has been reached.
%   Furthermore the code may stop processing an interval if the interval
%   size becomes too small giving a warning that this has happened. Finally
%   the code checks that the tolerance is not below the problem's noise
%   level. In such a case the stopping criterion is modified and a warning
%   given.
%
%   Modified by
%   Terje O. Espelid 24/09/06

%   Reference:  Berntsen and Espelid, TOMS 1991
%               Espelid, in Espelid and Genz, 1992.
%               Espelid, Bit 2003
%               Espelid, Report 266 February 2004
%   Check the number of input variables:
  if(nargin<5), trace=0; end
  if(nargin<4), tolrel=0; end
  if(nargin<3), tolabs=0;tolrel=0.001;end
%The weights of the rules and nullrules
w1=[1.5555555555555556e-01   7.1111111111111114e-01   2.6666666666666666e-01   7.1111111111111114e-01   1.5555555555555556e-01];
nw1=[1.2710311885185779e-01  -5.0841247540743117e-01   7.6261871311114682e-01  -5.0841247540743117e-01   1.2710311885185779e-01
  -3.3628324334270127e-01   6.7256648668540253e-01   0.0000000000000000e+00  -6.7256648668540253e-01   3.3628324334270127e-01
   5.6842242780997809e-01  -2.8421121390498905e-01  -5.6842242780997809e-01  -2.8421121390498905e-01   5.6842242780997809e-01
  -6.7256648668540253e-01  -3.3628324334270127e-01   0.0000000000000000e+00   3.3628324334270127e-01   6.7256648668540253e-01];
w2=[6.9770723104056437e-02   4.1537918871252205e-01  -6.5467372134038804e-02   7.4045855379188708e-01  -3.2028218694885363e-01   7.4045855379188708e-01  -6.5467372134038804e-02   4.1537918871252205e-01   6.9770723104056437e-02];
nw2=[1.1018547692345270e-02  -8.8148381538762158e-02   3.0851933538566756e-01  -6.1703867077133512e-01   7.7129833846416884e-01  -6.1703867077133512e-01   3.0851933538566756e-01  -8.8148381538762158e-02   1.1018547692345270e-02
  -4.2674651711845396e-02   2.5604791027107238e-01  -5.9744512396583560e-01   5.9744512396583560e-01   0.0000000000000000e+00  -5.9744512396583560e-01   5.9744512396583560e-01  -2.5604791027107238e-01   4.2674651711845396e-02
   1.1236757938944257e-01  -4.7756221240513091e-01   6.1802168664193424e-01   2.8091894847360643e-02  -5.6183789694721287e-01   2.8091894847360643e-02   6.1802168664193424e-01  -4.7756221240513091e-01   1.1236757938944257e-01
  -2.3112700627433047e-01   6.3559926725440874e-01  -2.3112700627433047e-01  -5.2003576411724362e-01   0.0000000000000000e+00   5.2003576411724362e-01   2.3112700627433047e-01  -6.3559926725440874e-01   2.3112700627433047e-01
   3.9111964345081668e-01  -5.8667946517622505e-01  -3.0730829128278453e-01   2.5143405650409645e-01   5.0286811300819290e-01   2.5143405650409645e-01  -3.0730829128278453e-01  -5.8667946517622505e-01   3.9111964345081668e-01
  -5.5619114160254801e-01   2.7809557080127401e-01   5.1646320291665171e-01   3.5755144817306656e-01   0.0000000000000000e+00  -3.5755144817306656e-01  -5.1646320291665171e-01  -2.7809557080127401e-01   5.5619114160254801e-01
   6.6477556470172228e-01   1.6619389117543057e-01  -1.8993587562906350e-01  -4.0361373571175990e-01  -4.7483968907265878e-01  -4.0361373571175990e-01  -1.8993587562906350e-01   1.6619389117543057e-01   6.6477556470172228e-01
  -6.4550259924248821e-01  -4.8412694943186613e-01  -3.2275129962124410e-01  -1.6137564981062205e-01   0.0000000000000000e+00   1.6137564981062205e-01   3.2275129962124410e-01   4.8412694943186613e-01   6.4550259924248821e-01];
w5=[3.2587004342807348e-02   2.3265662112814697e-01  -1.5325968015767599e-01   7.0183631648434164e-01  -6.9396767247322522e-01   7.3769703510789264e-01   3.9257951974485816e-01  -1.2870774416966075e+00   2.0738965950389239e+00  -1.2870774416966075e+00   3.9257951974485816e-01   7.3769703510789264e-01  -6.9396767247322522e-01   7.0183631648434164e-01  -1.5325968015767599e-01   2.3265662112814697e-01   3.2587004342807348e-02];
nw5=[1.3599340539007795e-04  -2.1758944862412472e-03   1.6319208646809354e-02  -7.6156307018443656e-02   2.4750799780994190e-01  -5.9401919474386056e-01   1.0890351903637443e+00  -1.5557645576624919e+00   1.7502351273703032e+00  -1.5557645576624919e+00   1.0890351903637443e+00  -5.9401919474386056e-01   2.4750799780994190e-01  -7.6156307018443656e-02   1.6319208646809354e-02  -2.1758944862412472e-03   1.3599340539007795e-04
  -7.5717923611077240e-04   1.0600509305550813e-02  -6.8146131249969516e-02   2.6501273263877034e-01  -6.8903310486080283e-01   1.2402595887494450e+00  -1.5158728306937663e+00   1.0827663076384046e+00   0.0000000000000000e+00  -1.0827663076384046e+00   1.5158728306937663e+00  -1.2402595887494450e+00   6.8903310486080283e-01  -2.6501273263877034e-01   6.8146131249969516e-02  -1.0600509305550813e-02   7.5717923611077240e-04
   2.9293876028360947e-03  -3.5518824684387648e-02   1.9260723488647324e-01  -6.0748175413813510e-01   1.1995842233613807e+00  -1.4328367112372047e+00   7.3307924760973275e-01   5.7599083740764712e-01  -1.2567072816166847e+00   5.7599083740764712e-01   7.3307924760973275e-01  -1.4328367112372047e+00   1.1995842233613807e+00  -6.0748175413813510e-01   1.9260723488647324e-01  -3.5518824684387648e-02   2.9293876028360947e-03
  -9.0861508333292679e-03   9.4268814895791156e-02  -4.2250601374981100e-01   1.0392285015620351e+00  -1.4174395299993658e+00   7.5301475031216314e-01   6.4965978458304263e-01  -1.1369046230203246e+00   0.0000000000000000e+00   1.1369046230203246e+00  -6.4965978458304263e-01  -7.5301475031216314e-01   1.4174395299993658e+00  -1.0392285015620351e+00   4.2250601374981100e-01  -9.4268814895791156e-02   9.0861508333292679e-03
   2.3944109830964542e-02  -2.0951096102093975e-01   7.5543666516693120e-01  -1.3612226438903341e+00   1.0128358458498001e+00   4.3698000441510282e-01  -1.1193871345975923e+00  -9.2184822849213477e-02   1.1062178741905617e+00  -9.2184822849213477e-02  -1.1193871345975923e+00   4.3698000441510282e-01   1.0128358458498001e+00  -1.3612226438903341e+00   7.5543666516693120e-01  -2.0951096102093975e-01   2.3944109830964542e-02
  -5.5310375436324374e-02   4.0100022191335166e-01  -1.1200351025855686e+00   1.3136214166127038e+00  -5.5310375436324374e-02  -1.1200351025855686e+00   1.5210353244989203e-01   1.0647247271492442e+00   0.0000000000000000e+00  -1.0647247271492442e+00  -1.5210353244989203e-01   1.1200351025855686e+00   5.5310375436324374e-02  -1.3136214166127038e+00   1.1200351025855686e+00  -4.0100022191335166e-01   5.5310375436324374e-02
   1.1417097929990812e-01  -6.7075450338696019e-01   1.3700517515988975e+00  -7.6046027283688800e-01  -8.7259248464929773e-01   6.2997915363699308e-01   9.4598811419923867e-01  -2.4261333101230473e-01  -1.0275388136991732e+00  -2.4261333101230473e-01   9.4598811419923867e-01   6.2997915363699308e-01  -8.7259248464929773e-01  -7.6046027283688800e-01   1.3700517515988975e+00  -6.7075450338696019e-01   1.1417097929990812e-01
  -2.1328275473716010e-01   9.8643274065936559e-01  -1.3330172171072507e+00  -1.3330172171072507e-01   1.0664137736858006e+00   5.0654654250075526e-01  -6.9316895289577041e-01  -9.3311205197507552e-01   0.0000000000000000e+00   9.3311205197507552e-01   6.9316895289577041e-01  -5.0654654250075526e-01  -1.0664137736858006e+00   1.3330172171072507e-01   1.3330172171072507e+00  -9.8643274065936559e-01   2.1328275473716010e-01
   3.6370149154738113e-01  -1.2729552204158339e+00   9.0925372886845279e-01   9.0925372886845279e-01  -3.4971297264171264e-01  -1.0211618801138007e+00  -5.1757519950973474e-01   4.8959816169839770e-01   9.7919632339679541e-01   4.8959816169839770e-01  -5.1757519950973474e-01  -1.0211618801138007e+00  -3.4971297264171264e-01   9.0925372886845279e-01   9.0925372886845279e-01  -1.2729552204158339e+00   3.6370149154738113e-01
  -5.6939677177717563e-01   1.4234919294429391e+00  -1.7081903153315270e-01  -1.0818538663766337e+00  -6.5261629995999360e-01   3.2849813756375523e-01   9.4169466101609822e-01   7.6649565431542876e-01   0.0000000000000000e+00  -7.6649565431542876e-01  -9.4169466101609822e-01  -3.2849813756375523e-01   6.5261629995999360e-01   1.0818538663766337e+00   1.7081903153315270e-01  -1.4234919294429391e+00   5.6939677177717563e-01
   8.2119570269050934e-01  -1.3344430168720778e+00  -6.1589677701788204e-01   5.1324731418156844e-01   1.0107024033113960e+00   7.3433846490593624e-01   1.5792225051740563e-02  -6.7116956469897393e-01  -9.4753350310443385e-01  -6.7116956469897393e-01   1.5792225051740563e-02   7.3433846490593624e-01   1.0107024033113960e+00   5.1324731418156844e-01  -6.1589677701788204e-01  -1.3344430168720778e+00   8.2119570269050934e-01
  -1.0922923949445911e+00   9.5575584557651716e-01   1.0922923949445911e+00   4.0960964810422162e-01  -3.7810121363466609e-01  -8.7173335365770244e-01  -9.2424741110696162e-01  -5.7765463194185096e-01   0.0000000000000000e+00   5.7765463194185096e-01   9.2424741110696162e-01   8.7173335365770244e-01   3.7810121363466609e-01  -4.0960964810422162e-01  -1.0922923949445911e+00  -9.5575584557651716e-01   1.0922923949445911e+00
   1.3377795087684237e+00  -3.3444487719210592e-01  -1.0033346315763176e+00  -1.0033346315763176e+00  -6.1743669635465714e-01  -7.7179587044332143e-02   4.3735099325121546e-01   7.9752239945809877e-01   9.2615504453198561e-01   7.9752239945809877e-01   4.3735099325121546e-01  -7.7179587044332143e-02  -6.1743669635465714e-01  -1.0033346315763176e+00  -1.0033346315763176e+00  -3.3444487719210592e-01   1.3377795087684237e+00
  -1.4995131405615487e+00  -3.7487828514038718e-01   3.7487828514038718e-01   8.0331061101511536e-01   9.6397273321813837e-01   9.1041869248379748e-01   6.9620252954643336e-01   3.7487828514038718e-01   0.0000000000000000e+00  -3.7487828514038718e-01  -6.9620252954643336e-01  -9.1041869248379748e-01  -9.6397273321813837e-01  -8.0331061101511536e-01  -3.7487828514038718e-01   3.7487828514038718e-01   1.4995131405615487e+00
   1.5147370145277248e+00   9.4671063407982814e-01   4.5442110435831745e-01   3.7868425363193119e-02  -3.0294740290554495e-01  -5.6802638044789688e-01  -7.5736850726386240e-01  -8.7097378335344178e-01  -9.0884220871663490e-01  -8.7097378335344178e-01  -7.5736850726386240e-01  -5.6802638044789688e-01  -3.0294740290554495e-01   3.7868425363193119e-02   4.5442110435831745e-01   9.4671063407982814e-01   1.5147370145277248e+00];
w6=[1.5719820110560612e-02   1.2431335118942319e-01  -1.2642677329136229e-01   5.3212453603657750e-01  -7.6465819010968195e-01   9.8324562571214047e-01  -3.5551576249713174e-01  -3.1542179983747937e-01   6.3627224435759000e-01   1.3778904251757138e-01  -4.6429465456793817e-01   9.1711547266897966e-02   5.4468348497948627e-01   1.9483082196529555e-02  -3.9853034620283323e-01   8.0527089406565444e-02   5.1795540546616858e-01   8.0527089406565444e-02  -3.9853034620283323e-01   1.9483082196529555e-02   5.4468348497948627e-01   9.1711547266897966e-02  -4.6429465456793817e-01   1.3778904251757138e-01   6.3627224435759000e-01  -3.1542179983747937e-01  -3.5551576249713174e-01   9.8324562571214047e-01  -7.6465819010968195e-01   5.3212453603657750e-01  -1.2642677329136229e-01   1.2431335118942319e-01   1.5719820110560612e-02];
nw6=[1.9082044442343243e-09  -6.1062542215498379e-08   9.4646940434022490e-07  -9.4646940434022490e-06   6.8619031814666314e-05  -3.8426657816213131e-04   1.7291996017295909e-03  -6.4227413778527665e-03   2.0071066805789895e-02  -5.3522844815439716e-02   1.2310254307551136e-01  -2.4620508615102271e-01   4.3085890076428973e-01  -6.6285984732967651e-01   8.9959550709027525e-01  -1.0795146085083303e+00   1.1469842715401009e+00  -1.0795146085083303e+00   8.9959550709027525e-01  -6.6285984732967651e-01   4.3085890076428973e-01  -2.4620508615102271e-01   1.2310254307551136e-01  -5.3522844815439716e-02   2.0071066805789895e-02  -6.4227413778527665e-03   1.7291996017295909e-03  -3.8426657816213131e-04   6.8619031814666314e-05  -9.4646940434022490e-06   9.4646940434022490e-07  -6.1062542215498379e-08   1.9082044442343243e-09
   1.5145903230336724e-08  -4.5437709691010172e-07   6.5733220019661384e-06  -6.1037990018256996e-05   4.0848501012218146e-04  -2.0968897186271979e-03   8.5781852125658108e-03  -2.8675647710577141e-02   7.9654576973825386e-02  -1.8586067960559258e-01   3.6641105407959673e-01  -6.1068509013266137e-01   8.5495912618572578e-01  -9.8649129944506830e-01   8.9253974711696649e-01  -5.3552384827017985e-01   0.0000000000000000e+00   5.3552384827017985e-01  -8.9253974711696649e-01   9.8649129944506830e-01  -8.5495912618572578e-01   6.1068509013266137e-01  -3.6641105407959673e-01   1.8586067960559258e-01  -7.9654576973825386e-02   2.8675647710577141e-02  -8.5781852125658108e-03   2.0968897186271979e-03  -4.0848501012218146e-04   6.1037990018256996e-05  -6.5733220019661384e-06   4.5437709691010172e-07  -1.5145903230336724e-08
   8.4307227924677153e-08  -2.3658715836362526e-06   3.1857593752037381e-05  -2.7376137667666260e-04   1.6839314937605204e-03  -7.8800964792864123e-03   2.9092285060897498e-02  -8.6656612176052375e-02   2.1096513795696681e-01  -4.2178128923458552e-01   6.8945075735145467e-01  -9.0396177827690716e-01   9.0190576210190276e-01  -5.7383938628005682e-01  -1.0016489057713440e-02   5.7995471644160823e-01  -8.1734550710941667e-01   5.7995471644160823e-01  -1.0016489057713440e-02  -5.7383938628005682e-01   9.0190576210190276e-01  -9.0396177827690716e-01   6.8945075735145467e-01  -4.2178128923458552e-01   2.1096513795696681e-01  -8.6656612176052375e-02   2.9092285060897498e-02  -7.8800964792864123e-03   1.6839314937605204e-03  -2.7376137667666260e-04   3.1857593752037381e-05  -2.3658715836362526e-06   8.4307227924677153e-08
   3.7995795830732984e-07  -9.9501490331732006e-06   1.2434124185607370e-04  -9.8511224902893529e-04   5.5424467378290205e-03  -2.3491351972426158e-02   7.7574971463222189e-02  -2.0329030752814195e-01   4.2543512633713415e-01  -7.0596984190477097e-01   9.0190098121218198e-01  -8.1232862348905444e-01   3.4593462293069993e-01   3.1100853119250432e-01  -7.4033810786542653e-01   6.0942466196239387e-01   0.0000000000000000e+00  -6.0942466196239387e-01   7.4033810786542653e-01  -3.1100853119250432e-01  -3.4593462293069993e-01   8.1232862348905444e-01  -9.0190098121218198e-01   7.0596984190477097e-01  -4.2543512633713415e-01   2.0329030752814195e-01  -7.7574971463222189e-02   2.3491351972426158e-02  -5.5424467378290205e-03   9.8511224902893529e-04  -1.2434124185607370e-04   9.9501490331732006e-06  -3.7995795830732984e-07
   1.4703232219690726e-06  -3.5839128535496144e-05   4.1428135395779381e-04  -3.0133740849215911e-03   1.5420856669019678e-02  -5.8746035100885975e-02   1.7161455535672279e-01  -3.8900414406924583e-01   6.8018966437426076e-01  -8.8682536239980603e-01   7.7242223971082036e-01  -2.4237077747819641e-01  -4.3592172528552409e-01   7.3789982119293351e-01  -3.6729210634188997e-01  -3.5151381884058774e-01   7.1352058749731240e-01  -3.5151381884058774e-01  -3.6729210634188997e-01   7.3789982119293351e-01  -4.3592172528552409e-01  -2.4237077747819641e-01   7.7242223971082036e-01  -8.8682536239980603e-01   6.8018966437426076e-01  -3.8900414406924583e-01   1.7161455535672279e-01  -5.8746035100885975e-02   1.5420856669019678e-02  -3.0133740849215911e-03   4.1428135395779381e-04  -3.5839128535496144e-05   1.4703232219690726e-06
   5.0447151485460276e-06  -1.1413668023585388e-04   1.2160814746997066e-03  -8.0826304060444415e-03   3.7379305091391979e-02  -1.2680048156139168e-01   3.2305133281528975e-01  -6.1843665443381279e-01   8.6249983092931626e-01  -7.8501209087594881e-01   2.4805993343831242e-01   4.5497421407019639e-01  -7.1124503166056774e-01   2.2650109915188088e-01   4.9935509558953006e-01  -6.3002745705221086e-01   0.0000000000000000e+00   6.3002745705221086e-01  -4.9935509558953006e-01  -2.2650109915188088e-01   7.1124503166056774e-01  -4.5497421407019639e-01  -2.4805993343831242e-01   7.8501209087594881e-01  -8.6249983092931626e-01   6.1843665443381279e-01  -3.2305133281528975e-01   1.2680048156139168e-01  -3.7379305091391979e-02   8.0826304060444415e-03  -1.2160814746997066e-03   1.1413668023585388e-04  -5.0447151485460276e-06
   1.5660053403417579e-05  -3.2788236813405556e-04   3.2072799696225228e-03  -1.9374169738534178e-02   8.0352870629714870e-02  -2.3992840436559193e-01   5.2297506085232526e-01  -8.1514653998849740e-01   8.2832715860597994e-01  -3.4200120075745732e-01  -3.9724503854340260e-01   7.0258940098842049e-01  -1.9843371459558751e-01  -5.3596201259969367e-01   5.4857924816691295e-01   1.9074718576226288e-01  -6.5674980414348727e-01   1.9074718576226288e-01   5.4857924816691295e-01  -5.3596201259969367e-01  -1.9843371459558751e-01   7.0258940098842049e-01  -3.9724503854340260e-01  -3.4200120075745732e-01   8.2832715860597994e-01  -8.1514653998849740e-01   5.2297506085232526e-01  -2.3992840436559193e-01   8.0352870629714870e-02  -1.9374169738534178e-02   3.2072799696225228e-03  -3.2788236813405556e-04   1.5660053403417579e-05
   4.4598204829132682e-05  -8.6130283076262491e-04   7.7009230862820520e-03  -4.2018432460679773e-02   1.5485431840324512e-01  -4.0098902346742654e-01   7.2755910858154405e-01  -8.6633442540785366e-01   4.9880934123731890e-01   2.6178471866684316e-01  -7.0063869702302994e-01   2.6654216023440247e-01   5.0783328445637477e-01  -5.2788314474634290e-01  -2.5029753387259751e-01   6.2574383468149386e-01   0.0000000000000000e+00  -6.2574383468149386e-01   2.5029753387259751e-01   5.2788314474634290e-01  -5.0783328445637477e-01  -2.6654216023440247e-01   7.0063869702302994e-01  -2.6178471866684316e-01  -4.9880934123731890e-01   8.6633442540785366e-01  -7.2755910858154405e-01   4.0098902346742654e-01  -1.5485431840324512e-01   4.2018432460679773e-02  -7.7009230862820520e-03   8.6130283076262491e-04  -4.4598204829132682e-05
   1.1770619831679677e-04  -2.0892850201231425e-03   1.6990510029615685e-02  -8.3135698022446594e-02   2.6919240619119433e-01  -5.9256033399405028e-01   8.5738297940407271e-01  -6.7854543198025541e-01  -4.0603990399227866e-02   6.6287139180328247e-01  -4.0570318439770559e-01  -4.1541492773491179e-01   5.6434187684184678e-01   2.1162491010428941e-01  -6.1048545611502858e-01  -6.3782062579182078e-02   6.1959717934062597e-01  -6.3782062579182078e-02  -6.1048545611502858e-01   2.1162491010428941e-01   5.6434187684184678e-01  -4.1541492773491179e-01  -4.0570318439770559e-01   6.6287139180328247e-01  -4.0603990399227866e-02  -6.7854543198025541e-01   8.5738297940407271e-01  -5.9256033399405028e-01   2.6919240619119433e-01  -8.3135698022446594e-02   1.6990510029615685e-02  -2.0892850201231425e-03   1.1770619831679677e-04
   2.9011246899689138e-04  -4.7143276211994850e-03   3.4675458895184898e-02  -1.5086550272843990e-01   4.2309200554718596e-01  -7.6953453803905714e-01   8.2191823292072996e-01  -2.5436482024826695e-01  -5.3173549474683746e-01   5.7031940432973738e-01   2.3652613046245527e-01  -6.2022955963981830e-01  -8.4388925050812386e-02   6.1412848473564108e-01   2.2457819210844179e-02  -6.0636111869279274e-01   0.0000000000000000e+00   6.0636111869279274e-01  -2.2457819210844179e-02  -6.1412848473564108e-01   8.4388925050812386e-02   6.2022955963981830e-01  -2.3652613046245527e-01  -5.7031940432973738e-01   5.3173549474683746e-01   2.5436482024826695e-01  -8.2191823292072996e-01   7.6953453803905714e-01  -4.2309200554718596e-01   1.5086550272843990e-01  -3.4675458895184898e-02   4.7143276211994850e-03  -2.9011246899689138e-04
   6.7176556283317682e-04  -9.9505273994664311e-03   6.5776141783379163e-02  -2.5186468325054784e-01   6.0034096736776055e-01  -8.6334956132054708e-01   5.6836774739447604e-01   2.6256622997815110e-01  -6.8000571271542687e-01   4.0218834022028714e-02   6.2812219089221955e-01  -1.2647308394794243e-01  -5.9717417931961037e-01   1.0880066184216856e-01   5.9188458593963178e-01  -4.1423846191955133e-02  -5.9301506127430514e-01  -4.1423846191955133e-02   5.9188458593963178e-01   1.0880066184216856e-01  -5.9717417931961037e-01  -1.2647308394794243e-01   6.2812219089221955e-01   4.0218834022028714e-02  -6.8000571271542687e-01   2.6256622997815110e-01   5.6836774739447604e-01  -8.6334956132054708e-01   6.0034096736776055e-01  -2.5186468325054784e-01   6.5776141783379163e-02  -9.9505273994664311e-03   6.7176556283317682e-04
   1.4683537936738965e-03  -1.9731004102492986e-02   1.1635519658580411e-01  -3.8718654096938560e-01   7.6420505858138232e-01  -8.0366145225225738e-01   1.3331558122819537e-01   6.2891163010955298e-01  -3.7288183096147898e-01  -5.0291556387815994e-01   3.8517010656404355e-01   4.9389372213891419e-01  -3.0716891539703683e-01  -5.3776793749560858e-01   1.7085251881019037e-01   5.7662725098439249e-01   0.0000000000000000e+00  -5.7662725098439249e-01  -1.7085251881019037e-01   5.3776793749560858e-01   3.0716891539703683e-01  -4.9389372213891419e-01  -3.8517010656404355e-01   5.0291556387815994e-01   3.7288183096147898e-01  -6.2891163010955298e-01  -1.3331558122819537e-01   8.0366145225225738e-01  -7.6420505858138232e-01   3.8718654096938560e-01  -1.1635519658580411e-01   1.9731004102492986e-02  -1.4683537936738965e-03
   3.0415479469960336e-03  -3.6878768857326903e-02   1.9234111472878546e-01  -5.4734372309066925e-01   8.6052462427424592e-01  -5.5452976073184790e-01  -3.3974651624713814e-01   6.3546459600737282e-01   1.8843105654349487e-01  -5.9188260846133589e-01  -2.4386980884082446e-01   5.1013128052224943e-01   3.8399289086557925e-01  -3.5856052589220605e-01  -5.1821641868279866e-01   1.3060739007452651e-01   5.7298725968179365e-01   1.3060739007452651e-01  -5.1821641868279866e-01  -3.5856052589220605e-01   3.8399289086557925e-01   5.1013128052224943e-01  -2.4386980884082446e-01  -5.9188260846133589e-01   1.8843105654349487e-01   6.3546459600737282e-01  -3.3974651624713814e-01  -5.5452976073184790e-01   8.6052462427424592e-01  -5.4734372309066925e-01   1.9234111472878546e-01  -3.6878768857326903e-02   3.0415479469960336e-03
   5.9896480052655590e-03  -6.5137422057262959e-02   2.9738119310014044e-01  -7.0849324288090798e-01   8.3201307657125301e-01  -1.4988860943321461e-01  -6.4685505669249799e-01   2.6136668554456427e-01   5.9125176844561889e-01  -1.4346422891143787e-01  -5.9643052648873296e-01  -9.0391883418730265e-02   5.3187108214499257e-01   3.5763745178715017e-01  -3.2357674209313586e-01  -5.3929457015522642e-01   0.0000000000000000e+00   5.3929457015522642e-01   3.2357674209313586e-01  -3.5763745178715017e-01  -5.3187108214499257e-01   9.0391883418730265e-02   5.9643052648873296e-01   1.4346422891143787e-01  -5.9125176844561889e-01  -2.6136668554456427e-01   6.4685505669249799e-01   1.4988860943321461e-01  -8.3201307657125301e-01   7.0849324288090798e-01  -2.9738119310014044e-01   6.5137422057262959e-02  -5.9896480052655590e-03
   1.1243655054754947e-02  -1.0892290834293854e-01   4.2988845656929997e-01  -8.3250651589894231e-01   6.4250797475016519e-01   2.9510764296973652e-01  -6.2811989002353441e-01  -2.6958064802826581e-01   5.1767297031576320e-01   4.2696263615286872e-01  -2.9310107577524569e-01  -5.6324192630554437e-01  -5.8263144685846403e-02   5.0781627243232030e-01   4.0822034490927445e-01  -2.0697919094295544e-01  -5.5740930630182062e-01  -2.0697919094295544e-01   4.0822034490927445e-01   5.0781627243232030e-01  -5.8263144685846403e-02  -5.6324192630554437e-01  -2.9310107577524569e-01   4.2696263615286872e-01   5.1767297031576320e-01  -2.6958064802826581e-01  -6.2811989002353441e-01   2.9510764296973652e-01   6.4250797475016519e-01  -8.3250651589894231e-01   4.2988845656929997e-01  -1.0892290834293854e-01   1.1243655054754947e-02];

% w1 and nw1 are the weights and nullrules associated with Newton-Cotes 5 point rule.
% w2 and nw2 are the weights and nullrules associated with Newton-Cotes 9 point rule.
% w5 and nw5 are the weights and nullrules associated with the 17 point rule (degree 15)
% w6 and nw6 are the weights and nullrules associated with 33 point rule (degree 23)
% The coordinates for the four new points in a 5 point approximation to get a 9 point approximation.
  z4=(-3:2:3)/4;
% The coordinates for the eight new points in a 9 point approximation to get a 17 point approximation.
  z8=(-7:2:7)/8;
% The coordinates for the sixteen new points in a 17 point approximation to get a 33 point approximation.
  z16=(-15:2:15)/16;
% Define the number of intervals given
  nint=length(a)-1;
  if nint < 1, warning('No interval is given.');Q=0;nfun=0;return,end
% Initialize: the function count, the maximum number of intervals, too small intervals
% (logical: interval) and the maximum number of function values allowed.
  nfun=0;int_max=2000;interval=0;nmax=10000;singular=0;
% Initialize several dynamic vectors for speed: The choice of value of
% int_max is connected to the fact that we have a bound 10 000 on the
% number of function values and if we use the 5 point rule everywhere
% (not very likely) then 2000 intervals times 4 f-evaluation per interval
% implies approx 8 000 function evaluations. Therefore, this will do in
% most situations.
  is=zeros(1,int_max);error=is;isa=is;lp=is;rp=is;flag=is;
  fu=zeros(int_max,33);
  for i=1:nint  
% Compute y and count the number of function values
     x=linspace(a(i),a(i+1),9); y=feval(f,x);nfun=nfun+9;
% Check if some of the function values are infinite. In case there are: put
% the value(s) to zero.
     IN=find(isinf(y)); y(IN)=0;singular=max(1-isempty(IN),singular);
% Check if some of the function values are NaN: put
% those function values to 1 (any number might do).
     y(isnan(y))=1;
% Apply the degree 9 Cotes rule on the interval and
% initialize the datastructure by storing: error estimate, integral, absolute value, 
% function values, left and right point of the interval and
% signalling by a flag that a nine point rule has been used.
% flag is used as follows: 0: the 5 point rule has been used
%                          1: the 9 point rule has been used
%                          2: the 9 point rule has been used
%                          3: the 9 point rule has been used
%                          4: the 17 point rule has been used
%                          5: the 17 point rule has been used
%                          6: the 17 point rule has been used
%                          7: the 33 point rule has been used
% The 9 point rule is used either in combination with the 5 point rule (flag=1 or 2)
% or in combination with the 17 point rule (flag=3). The 17 point rule is used either
% combination with the 9 point rule (flag=4 or 5) or in combination with the 33 point rule
% (flag=6). The code decides how to proceed.
     [error(i),is(i),isa(i),rmax]=cotes9(a(i),a(i+1),y(1:9),32,w2,nw2,interval);
     fu(i,1:33)=0;fu(i,1:4:33)=y;lp(i)=a(i);rp(i)=a(i+1);
     flag(i)=1+ (rmax<1/4);
  end
% Compute global values
  Q=sum(is);isabs=sum(isa);total=sum(error);noise=50*eps*isabs;

% global error test and check the number of function evaluations.

  while (total> max([abs(Q)*tolrel,tolabs,noise]))&&(nfun<nmax-3)

%  Find the interval with the greatest error estimate.
      [err,i]=max(error);
%  Prepare to process this interval: either by subdivision or  by using a new rule
      aa=lp(i);bb=rp(i);cc=(aa+bb)/2;total=total-err;Q=Q-is(i);isabs=isabs-isa(i);
      if (flag(i)==1) || (flag(i)==2)
%  On this interval a 9 point rule has been used.
%  Subdivide the interval and use all nine function values
%  to compute: the left and the right part  of the integral using the 5 point rule
         z(1:5)=fu(i,1:4:17);[Lerr,Linte,Linteabs]=cotes5(aa,cc,z(1:5),32,w1,nw1);
         zz(1:5)=fu(i,17:4:33);[Rerr,Rinte,Rinteabs]=cotes5(cc,bb,zz(1:5),32,w1,nw1);
%  Check if this subdivision, in the case flag(i)==2, implies a reduction of the error.
         if (flag(i)==2)&&(Lerr + Rerr >= err)
%  With no error reduction we choose instead to apply the 17 point rule.
%  Compute f in 8 new points
             hh=(bb-aa)/2;x8=cc+hh*z8;y8=feval(f,x8);nfun=nfun+8;
             IN=find(isinf(y8)); y8(IN)=0;singular=max(1-isempty(IN),singular);
             y(isnan(y))=1;
             fu(i,3:4:31)=y8(1:8);y(1:17)=fu(i,1:2:33);
             [err,inte,inteabs]=rule17(aa,bb,y(1:17),32,w5,nw5,interval);
             error(i)=err;is(i)=inte;isa(i)=inteabs;total=total+err;
             Q=Q+inte;isabs=isabs+inteabs;flag(i)=4;
         else
%  Store the left part in the datastructure        
             error(i)=Lerr;is(i)=Linte;isa(i)=Linteabs;
             total=total+Lerr;Q=Q+Linte;isabs=isabs+Linteabs;
             flag(i)=0;fu(i,1:33)=0;fu(i,1:8:33)=z(1:5);lp(i)=aa;rp(i)=cc;
%  and then the right part of the integral. Increase the size of the datastructure
             nint=nint+1;i=nint;
             error(i)=Rerr;is(i)=Rinte;isa(i)=Rinteabs;
             total=total+Rerr;Q=Q+Rinte;isabs=isabs+Rinteabs;
             flag(i)=0;fu(i,1:33)=0;fu(i,1:8:33)=zz(1:5);lp(i)=cc;rp(i)=bb;
         end
      elseif flag(i)==0
%  On this interval a 5 point rule has been used only. Compute f in 4 new
%  points and apply the 9 point rule
         hh=(bb-aa)/2;x4=cc+hh*z4;y4=feval(f,x4);nfun=nfun+4;
         IM=find(isinf(y4)); y4(IM)=0;singular=max(1-isempty(IM),singular);
         y4(isnan(y4))=1;
         fu(i,5:8:29)=y4(1:4);y(1:9)=fu(i,1:4:33);
         [err,inte,inteabs,rmax]=cotes9(aa,bb,y(1:9),32,w2,nw2,interval);
         error(i)=err;is(i)=inte;isa(i)=inteabs;flag(i)=1+(rmax<1/4);
         total=total+err;Q=Q+inte;isabs=isabs+inteabs;
         
      elseif flag(i)==3
%  Apply the 17 point rule: compute f in 8 new points.
             hh=(bb-aa)/2;x8=cc+hh*z8;y8=feval(f,x8);nfun=nfun+8;
             IN=find(isinf(y8)); y8(IN)=0;singular=max(1-isempty(IN),singular);
             y8(isnan(y8))=1;
             fu(i,3:4:31)=y8(1:8);y(1:17)=fu(i,1:2:33);
             [err,inte,inteabs,rmax]=rule17(aa,bb,y(1:17),32,w5,nw5,interval);
             error(i)=err;is(i)=inte;isa(i)=inteabs;flag(i)=4+(rmax<1/8);
             Q=Q+inte;isabs=isabs+inteabs;total=total+err;
      elseif (flag(i)==4||flag(i)==5)
%  On this interval a 17 point rule has been used.
%  Subdivide the interval and use all 17 function values
%  to compute: the left and the right part  of the integral using the 9 point rule
          z(1:9)=fu(i,1:2:17);[Lerr,Linte,Linteabs]=cotes9(aa,cc,z(1:9),32,w2,nw2,interval);
          zz(1:9)=fu(i,17:2:33);[Rerr,Rinte,Rinteabs]=cotes9(cc,bb,zz(1:9),32,w2,nw2,interval);
%  Check if this subdivision, in the case flag(i)==5, implies a reduction of the error.
         if (flag(i)==5)&&(Lerr + Rerr >= err)
%  With no error reduction we choose instead to apply the 33 point rule.
%  Compute f in 16 new points
             hh=(bb-aa)/2;x16=cc+hh*z16;y16=feval(f,x16);nfun=nfun+16;
             IMN=find(isinf(y16)); y16(IMN)=0;singular=max(1-isempty(IMN),singular);
             y16(isnan(y16))=1;
             fu(i,2:2:32)=y16;y=fu(i,1:33);
             [err,inte,inteabs]=rule33(aa,bb,y(1:33),32,w6,nw6,interval);
             error(i)=err;is(i)=inte;isa(i)=inteabs;total=total+err;
             Q=Q+inte;isabs=isabs+inteabs;flag(i)=7;
         else
%  Store the left part in the datastructure and update global variables
            error(i)=Lerr;is(i)=Linte;isa(i)=Linteabs;flag(i)=3;
            fu(i,1:33)=0;fu(i,1:4:33)=z(1:9);lp(i)=aa;rp(i)=cc;
            total=total+Lerr;Q=Q+Linte;isabs=isabs+Linteabs;
%  and then the right part of the integral. Increase the size of the datastructure
             nint=nint+1;i=nint;
             error(i)=Rerr;is(i)=Rinte;isa(i)=Rinteabs;flag(i)=3;
             fu(i,1:33)=0;fu(i,1:4:33)=zz(1:9);lp(i)=cc;rp(i)=bb;
             total=total+Rerr;Q=Q+Rinte;isabs=isabs+Rinteabs;
         end
     elseif flag(i)==6
%  On this interval a 17 point rule has been used.
%  Apply the 33 point rule.
%  Compute f in 16 new points
             hh=(bb-aa)/2;x16=cc+hh*z16;y16=feval(f,x16);nfun=nfun+16;
             IMN=find(isinf(y16)); y16(IMN)=0;singular=max(1-isempty(IMN),singular);
             y16(isnan(y16))=1;
             fu(i,2:2:32)=y16;y=fu(i,1:33);
             [err,inte,inteabs]=rule33(aa,bb,y(1:33),32,w6,nw6,interval);
             error(i)=err;is(i)=inte;isa(i)=inteabs;flag(i)=7;
             Q=Q+inte;isabs=isabs+inteabs;total=total+err;
     elseif flag(i)==7
%  On this interval a 33 point rule has been used.
%  Subdivide the interval and apply the 17 point rule on each part.
         z(1:17)=fu(i,1:17);[Lerr,Linte,Linteabs]=rule17(aa,cc,z(1:17),32,w5,nw5,interval);
         zz(1:17)=fu(i,17:33);[Rerr,Rinte,Rinteabs]=rule17(cc,bb,zz(1:17),32,w5,nw5,interval);
%  Store the left part in the datastructure
             error(i)=Lerr;is(i)=Linte;isa(i)=Linteabs;
             flag(i)=6;fu(i,1:33)=0;fu(i,1:2:33)=z(1:17);lp(i)=aa;rp(i)=cc;
             total=total+Lerr;Q=Q+Linte;isabs=isabs+Linteabs;
%  and then the right part of the integral. Increase the size of the datastructure
             nint=nint+1;i=nint;
             error(i)=Rerr;is(i)=Rinte;isa(i)=Rinteabs;
             flag(i)=6;fu(i,1:33)=0;fu(i,1:2:33)=zz(1:17);lp(i)=cc;rp(i)=bb;
             total=total+Rerr;Q=Q+Rinte;isabs=isabs+Rinteabs;
     end

% Redefine the global noise level
      noise=50*eps*isabs;
  end
  if (nfun>=nmax-3) 
       warning(['Stopping: maximum number of f-evaluations, ',...
                'required tolerance may not be met.']);
  end
  if (interval==1)
         warning(['Interval too small, ',...
                'required tolerance may not be met.']);
  end
  if max(abs(Q)*tolrel,tolabs) < noise
       warning(['Stopping: the tolerance is below the noise level of the problem, ',...
                'required tolerance may not be met.']);
  end
  if singular==1
      warning('Singularity probably detected, required tolerance may not be met.')
  end
  if trace
     format long
     for i=1:nint
         if flag(i)==0
             points=5;
         elseif (flag(i)> 0) && (flag(i) < 4)
             points=9;
         elseif (flag(i)>3)&&(flag(i)<7)
             points=17;
         else
             points=33;
         end
         disp([lp(i),rp(i),is(i),points])
     end
  end
  Q=sum(is);
  
function [err,Q,Qabs,rmax]=cotes9(a,b,y,D,w2,nw2,interval)
%
%   [err,Q,Qabs,rmax] = COTES9(A,B,Y,D,W2,NW2,INTERVAL) applies a 9 point Newton-Cotes rule to 
%   approximate the integral of F(X) from A to B.
%   The argument Y contains the 9 function values to be used in the
%   approximation and D is a constant to be used in the error estimate.
%   W2: is the rule's weights, NW2: is the nullrules weights
%   INTERVAL: logical signalling that too small intervals has been found.

%
%   See also COTES5
%   Modified by
%   T O Espelid, 10/07/2006

  h=(b-a)/2;x=linspace(a,b,9);Q=(h*w2)*y';
  Qabs=(h*abs(w2))*abs(y');
% define the local noise level
  noise=50*Qabs*eps;
%  Compute the error estimates
  e=(h*nw2)*y';E=[norm(e(1:2)),norm(e(3:4)),norm(e(5:6)),norm(e(7:8))];
  Emin=min(E(2:4));
  if Emin==0,
     rmax=2;
  else
     r=E(1:3)./E(2:4);
     if sum(isinf(r))>0
        rmax=2;
     else
        rmax=max(r);
     end
  end
  if rmax > 1
     err=D*max(E(1:4));
  elseif 0.25 < rmax
         err=D*rmax*E(1);
      else
         err=D*4*rmax*rmax*E(1);
  end
  
% If the highest degree null rules are on the local noise level, then put the error to zero.
  if (E(1)<noise) && (E(2) < noise), err=0; end;
% check if interval has become too small: unable to distinguish between the
% endpoints and the two points close to the endpoints.
  if ((x(2) <= a) || (b<=x(8)))
       interval =1;
% In order to avoid handling this interval again: put the error to zero.
       err=0;
   end

function [err,Q,Qabs]=cotes5(a,b,y,C,w1,nw1)
%
%
%   [err,Q,Qabs] = COTES5(A,B,Y,C,W1,NW1) applies a 5 point Newton-Cotes rule to 
%   approximate the integral of F(X) from A to B. 
%   The argument Y contains the 5 function values to be used in the
%   approximation. C is a constant.
%   W1: is the rule's weights, NW1: is the nullrules weights
%
%   See also COTES9
%   Modified by
%   T O Espelid, 10/07/2006

  h=(b-a)/2;Q=(h*w1)*y';Qabs=(h*w1)*abs(y');
% define the local noise level 
  noise=50*Qabs*eps;E=abs((h*nw1)*y');
  Emin=min(E(2:4));
  if Emin==0,
     rmax=2;
  else
     r=E(1:3)./E(2:4);
     if sum(isinf(r))>0
        rmax=2;
     else
        rmax=max(r);
     end
  end
  if rmax > 1
     err=C*max(E(1:4));
  elseif 0.5 < rmax
         err=C*rmax*E(2);
      else
         err=C*(2*rmax)^3*rmax*E(2);
  end
% If the highest degree null rules are on the local noise level, then put the error to zero.
  if (E(1)<noise) && (E(2) < noise), err=0; end;

function [err,Q,Qabs,rmax]=rule17(a,b,y,D,w3,nw3,interval)
%
%   [err,Q,Qabs,rmax] = RULE17(A,B,Y,D,W3,NW3,INTERVAL) applies a 17 point non-interpolatory rule to
%   approximate the integral of F(X) from A to B. The rule has degree of
%   precision 15 and will have local order 17.
%   The argument Y contains the 17 function values to be used in the
%   approximation and D is a constant to be used in the error estimate.
%   W3: is the rule's weights, NW3: is the nullrules weights
%   INTERVAL: logical signalling that too small intervals has been found.

%
%   See also COTES5 and COTES9
%   Modified by
%   T O Espelid, 10/07/2006

  h=(b-a)/2;x=linspace(a,b,17);Q=(h*w3)*y';
  Qabs=(h*abs(w3))*abs(y');
% define the local noise level
  noise=50*Qabs*eps;
%  Compute the error estimates
  e=(h*nw3)*y';
  E=[norm(e(1:3)),norm(e(4:6)),norm(e(7:9)),norm(e(10:12)),norm(e(13:15))];
  Emin=min(E(2:5));
  if Emin==0,
     rmax=2;
  else
     r=E(1:4)./E(2:5);
     if sum(isinf(r))>0
        rmax=2;
     else
        rmax=max(r);
     end
  end
  if rmax > 1
     err=D*max(E(1:5));
  elseif 0.125 < rmax
         err=D*E(1);
      else
% Note: the rules local order is 17 while E(1) has order 14.          
         err=D*8*rmax*E(1);
  end

% If the highest degree null rules are on the local noise level, then put the error to zero.
  if (E(1)<noise) && (E(2) < noise), err=0; end;
% check if interval has become too small: unable to distinguish between the
% endpoints and the two points close to the endpoints.
  if ((x(2) <= a) || (b<=x(16)))
       interval =1;
% In order to avoid handling this interval again: put the error to zero.
       err=0;
   end

function [err,Q,Qabs]=rule33(a,b,y,D,w4,nw4,interval)
%
%   [err,Q,Qabs,rmax] = RULE33(A,B,Y,D,W4,NW4,INTERVAL) applies a 33 point degree 23 rule to
%   approximate the integral of F(X) from A to B.
%   The argument Y contains the 17 function values to be used in the
%   approximation and D is a constant to be used in the error estimate.
%   W4: is the rule's weights, NW4: is the nullrules weights
%   INTERVAL: logical signalling that too small intervals has been found.

%
%   See also COTES5, COTES9 and RULE17
%   Modified by
%   T O Espelid, 10/07/2006

  h=(b-a)/2;x=linspace(a,b,33); Q=(h*w4)*y';
  Qabs=(h*abs(w4))*abs(y');
% define the local noise level
  noise=50*Qabs*eps;
%  Compute the error estimates
  e=(h*nw4)*y';
  E=[norm(e(1:3)),norm(e(4:6)),norm(e(7:9)),norm(e(10:12)),norm(e(13:15))];
  Emin=min(E(2:5));
  if Emin==0,
     rmax=2;
  else
     r=E(1:4)./E(2:5);
     if sum(isinf(r))>0
        rmax=2;
     else
        rmax=max(r);
     end
  end
  if rmax > 1
     err=D*max(E(1:5));
 elseif 1/8 < rmax
% Note: the rule has local order 25
% While the E-sequence have orders 30, 27, 24, 21 and 18.
         err=D*E(3);
     else
         err=D*(8*rmax)^(1/3)*E(3);
  end

% If the highest degree null rules are on the local noise level, then put the error to zero.
  if (E(1)<noise) && (E(2) < noise) && (E(3) < noise), err=0; end;
% check if interval has become too small: unable to distinguish between the
% endpoints and the two points close to the endpoints.
  if ((x(2) <= a) || (b<=x(32)))
       interval =1;
% In order to avoid handling this interval again: put the error to zero.
       err=0;
