// $Id: APPSPACK_List.hpp,v 1.9 2004/11/23 22:26:01 tgkolda Exp $ 
// $Source: /space/CVS-Acro/acro/packages/appspack/appspack/src/APPSPACK_List.hpp,v $ 

//@HEADER
// ************************************************************************
// 
//          APPSPACK: Asynchronous Parallel Pattern Search
//                 Copyright (2003) Sandia Corporation
// 
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
// 
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as
// published by the Free Software Foundation; either version 2.1 of the
// License, or (at your option) any later version.
//  
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//                                                                                 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.                                                                           .
// 
// Questions? Contact Tammy Kolda (tgkolda@sandia.gov) 
// 
// ************************************************************************
//@HEADER

/*!
  \file APPSPACK_List.hpp
  \brief Class definition for APPSPACK::List
*/
#ifndef APPSPACK_LIST_HPP
#define APPSPACK_LIST_HPP

#include "APPSPACK_Point.hpp"

namespace APPSPACK
{

/*!  
 \brief Manipulates a list of Point objects.

 These lists are used in several ways.

 Each iteration of APPS, a list of trial points is generated by
 Solver::generateTrialPoints. This list is passed to the
 Evaluator. The evaluator empties the list into its queue. Then it
 refills and returns the list with evaluated points.

 The queue in the Evaluator is also a list of trial points.
*/
class List
{
public:

  //! Constructor 
  List();

  //! Desctructor: deletes the contents of the current list 
  ~List();


  //@{ \name Accessors

  /*! Return true of the list is empty */
  bool isEmpty() const;

  /*! Return true of the list is \b not empty */
  bool isNotEmpty() const;

  /*! Return size */
  int size() const;

  /*! Return a const reference to the best Point in the list. */
  /*! Moves the best Point to the end of the list (if it's not already there). */
  const Point& best();
    
  //@}

  //@{ \name Manipulators

  //! Push the given Point onto the list.
  /*! The Point is pushed onto the \e front of the list. The list takes
    ownership for the given pointer. 
  */
  void push(Point* trialPointPtr);

  //! Pop the next Point from the list.
  /*! If the list is empty, returns NULL. Otherwise, pops a Point off
    the \e end of the list. Ownership of the pointer is passed on to
    the calling object.
   */
  Point* pop();

  //! Pop the Point with the given tag.
  /*! If the Point with the given tag is not in the list, returns
    NULL. Otherwise, it pops the Point with the given tag off the
    list. Ownership of the pointer is passed on to the calling object.
   */
  Point* pop(int tag);

  /*! 
    \brief Insert the source list into this list and empty the source list.
    
    Insert the given source list at the \e front of this list. This
    list takes ownership of all the pointers in the source list, and
    the source list is emptied.
  */
  void insertList(List& source);

  //! Pop the best Point off the list.
  /*! The best trial Point is defined as the Point with the smallest
    function value.

    If the list is empty, returns NULL. Otherwise, it pops the best
    Point off the list. Ownership of the pointer is passed on to the
    calling object.
  */
  Point* popBest();

  /*! \brief Prune the list except for the most recently added n points. 

  The most recently added points are at the front of the list.
  */
  void prune(int n = 0);

  //@}

  //@{ \name Printing

  //! %Print out the given trial point list, preceeded by the specified label.
  /*! Prints the list out in reverse order; i.e., starts with the end
    of the list. This means that the objects that are first pushed
    onto the will be printed first. 
  */
  void print(const string label) const;

  //@}

private:

  //! Move the best point to the end of the list.
  void moveBestToEndOfList();
    
private:

  //! A list of pointers to trial points
  typedef list<Point*> TPL;

  //! The list of trial points
  /*! Add to the front, remove from the back. */
  TPL tpl;

  //! True if the best pointer is valid
  bool isValidBest;
};


}
#endif
