      PROGRAM ZFIG12
*
*     This code is part of a package for solving rank deficient least
*     squares problems, written by:
*     ==================================================================
*     L. Foster                   and   R. Kommu
*     Department of Mathematics         Department of Physics
*     San Jose State University         San Jose State University
*     San Jose, CA 95192                San Jose, CA 95192
*     foster@math.sjsu.edu              rkommu@email.sjsu.edu
*     ==================================================================
*     03/05/2004
*
*  Purpose
*  =======
*
*  xFIG12 is a program that generates timing data that can be used to
*  creates plots similar to those in Figures 1 and 2 in Foster
*  and Kommu's paper.  The program collects  run times
*  for the least squares routines xGELSD, xGELSY, and xGELSZ
*  for a set of matrices described by a short data file.
*  This input file specifies the dimensions m and n
*  of the matrices, the ranks of the matrices, scaling
*  of the matrices (normal, up or down), the minimum amount
*  of time for each timing run and a seed used, along with m, n, and
*  the rank, to initialize a random number generator.  The matrices
*  are generated by the routine xQRT25 which is identical to LAPACK's
*  xQRT15 except in xQRT25 an arbitrary rank can be specified whereas
*  xQRT15 allows only two rank choices. xQRT15 is used to generate
*  matrices in LAPACK's testing and timing routines.
*
*  A sample input file is:
*
*  The first line is for comments and is not used by xFIG12.F.
*  100 100   1  1 2 1212
*  100 100  20  1 2 1212    Each line includes:
*  100 100  40  1 2 1212    M N RANK SCALING(1,2,3) MIN_TIME SEED
*  100 100  60  1 2 1212    The final line should be:
*  100 100  80  1 2 1212     0 0 0 0 0 0
*  100 100 100  1 2 1212
*   0 0 0 0 0 0
*
*  This file will generate 100 by 100 matrices with ranks of 1, 20,
*  40, 60, 80, and 100.  The matrices will be scaled normally,
*  each timing run will take at least 2 seconds and the
*  random number seed in each case is set at 1212.  Note that the
*  fourth number in each line specifies the scaling factor as
*  defined in xQRT15 and xQRT25.  A scale factor of 1 is normal
*  scaling, 2 is scaled up, and 3 is scaled down. The routine xQRT25
*  uses a random number generator which is initialized using the
*  seed value as well m, n and the rank.  The line of 0's is used
*  to signal the end of data
*
*  The program writes the results to a file zfig12.out.  In
*  zfig12.out for each matrix m, n, the rank and times for xGELSD,
*  xGELSY and xGELSZ are printed.  A matlab program plotfig12.m
*  can be used to read zfig12.out and plot the times.
*
*
*  Internal Parameters
*  ===================
*
*  LDA     INTEGER
*          The leading dimension of the A matrix.
*
*  LDB     INTEGER
*          The leading dimension of the B matrix.
*
*  LEVEL   INTEGER
*          Input and output level.
*          Level = 0: No monitor output except for errors. File
*                     output.
*          Level = 1: Some monitor output. File output.
*          Level = 2: Periodic monitor output.  File output.
*          Level = 3: Prompts for interactive input.
*                     Periodic monitor output. File output.
*
*  LWORK   INTEGER
*          The allocated size of the work space WORK.  LWORK is set to
*          500*LDA which, according to the xGELSD, xGELSY and xGELSZ
*          documentation is an overestimate of the space needed for
*          WORK.
*
*  MAXIC   INTEGER
*          The maximum number of timing runs.  The allocated size
*          of the arrays TIMES and RANKS.
*
*  NRHS    INTEGER
*          The number of columns in B.
*
      INTEGER            LDA, LDB, LWORK, LEVEL, MAXIC, NRHS
      PARAMETER          ( LEVEL = 2 )
      PARAMETER          ( LDA = 2068, LDB = 2068, LWORK = 500*LDA,
     $                   MAXIC = 5000, NRHS = 1 )
*
      INTEGER            I, IC, INFO, IREP, ISEEDT, J, M, N, NOUT,
     $                   NREP, RANK, SCALE
      INTEGER            IPVT( LDA )
      INTEGER            ISEED( 4 )
      INTEGER            IWORK( LWORK )
      INTEGER            MVALS( MAXIC ), NVALS( MAXIC ), RANKS( MAXIC )
*
*
      CHARACTER*80       LINE
*
      DOUBLE PRECISION   EPS, NORMA, NORMB, TREP, TIMEC, TIMED, TIMEY,
     $                   TIMEZ, RCOND, S1, S2
      COMPLEX*16         A( LDA, LDA ), A0( LDA, LDA ), B( LDB, NRHS ),
     $                   B0( LDB, NRHS )
      COMPLEX*16         WORK( LWORK )
      DOUBLE PRECISION   RWORK( LWORK )
      DOUBLE PRECISION   S( LDA )
      DOUBLE PRECISION   TIMES( MAXIC, 3 )
*
*
*     .. External Functions ..
      DOUBLE PRECISION   DLAMCH, DSECND
      EXTERNAL           DLAMCH, DSECND
*     ..
*     .. External Subroutines ..
      EXTERNAL           ZGELSD, ZGELSY, ZGELSZ, ZQRT25
*     ..
*     .. Intrinsic Functions ..
      INTRINSIC          SQRT
*     ..
*     Executable statements
*
*
      IF( LEVEL.NE.0 ) THEN
         WRITE( *, * )'xFIG12 is running'
         WRITE( *, * )
      END IF
      IC = 0
      EPS = DLAMCH( 'Epsilon' )
*
*     Threshold for rank estimation
*
      RCOND = SQRT( EPS ) - ( SQRT( EPS )-EPS ) / 2
*
*
      IF( LEVEL.EQ.0 .OR. LEVEL.EQ.1 .OR. LEVEL.EQ.2 ) THEN
*        The first line of the input file is for comments.
*        Read and ignore first line.
         READ( *, FMT = '( A80 )' )LINE
      END IF
*
   10 CONTINUE
*     Top of loop
*
*
      IF( LEVEL.EQ.3 ) THEN
*        Prompt the keyboard input.
         WRITE( *, * )
         WRITE( *, * )'ENTER ROWS, COLS, RANK, SCALE(1-3),',
     $      ' MIN_TIME AND SEED   (ALL 0 TO QUIT)'
         READ( *, * )M, N, RANK, SCALE, TREP, ISEEDT
      ELSE
         READ( *, * )M, N, RANK, SCALE, TREP, ISEEDT
      END IF
*
      IF( M.GT.LDA .OR. N.GT.LDA ) THEN
         WRITE( *, * )
         WRITE( *, * )'M OR N IS TOO LARGE, INCREASE LDA '
         GO TO 380
      END IF
      IF( M.EQ.0 ) THEN
*        Exit loop when M = 0
         GO TO 380
      END IF
      IC = IC + 1
      IF( IC.GT.MAXIC ) THEN
         WRITE( *, * )
         WRITE( *, * )'THE MAXIMIUM NUMBER OF TIMING RUNS HAS BEEN',
     $      ' REACHED. INCREASE MAXIC.'
         GO TO 380
      END IF
*
*     Initialize seed for random number geneerator
      ISEED( 1 ) = ISEEDT
      ISEED( 2 ) = 10000*M + N
      ISEED( 3 ) = RANK
      ISEED( 4 ) = 13
      CALL ZQRT25( SCALE, RANK, M, N, NRHS, A0, LDA, B0, LDB, S, NORMA,
     $             NORMB, ISEED, WORK, LWORK )
*
*     Time xGELSD **********************************
*
      S1 = DSECND( )
      S2 = S1
      IREP = 0
*
   20 CONTINUE
*     Top of loop
*
      IF( S2-S1.GE.TREP )
     $   GO TO 70
      DO 50 I = 1, M
         DO 30 J = 1, N
            A( I, J ) = A0( I, J )
   30    CONTINUE
         DO 40 J = 1, NRHS
            B( I, J ) = B0( I, J )
   40    CONTINUE
   50 CONTINUE
      DO 60 J = 1, N
         IPVT( J ) = 0
   60 CONTINUE
*
      CALL ZGELSD( M, N, NRHS, A, LDA, B, LDB, S, RCOND, RANK, WORK,
     $             LWORK, RWORK, IWORK, INFO )
      IREP = IREP + 1
      S2 = DSECND( )
      GO TO 20
*
*     End of loop
   70 CONTINUE
      NREP = IREP
      TIMED = S2 - S1
      TIMED = TIMED / NREP
*
*     Adjust for time to make copies
*
      S1 = DSECND( )
      IREP = 0
   80 CONTINUE
*     Top of loop
*
      IF( IREP.GE.NREP )
     $   GO TO 130
*           Top of loop
      DO 110 I = 1, M
         DO 90 J = 1, N
            A( I, J ) = A0( I, J )
   90    CONTINUE
         DO 100 J = 1, NRHS
            B( I, J ) = B0( I, J )
  100    CONTINUE
  110 CONTINUE
      DO 120 J = 1, N
         IPVT( J ) = 0
  120 CONTINUE
      IREP = IREP + 1
      S2 = DSECND( )
      GO TO 80
*
*     End of loop
  130 CONTINUE
      TIMEC = S2 - S1
      TIMEC = TIMEC / NREP
*
      TIMED = TIMED - TIMEC
*
*     Time xGELSY **********************************
*
      S1 = DSECND( )
      S2 = S1
      IREP = 0
  140 CONTINUE
*     Top of loop
*
      IF( S2-S1.GE.TREP )
     $   GO TO 190
      DO 170 I = 1, M
         DO 150 J = 1, N
            A( I, J ) = A0( I, J )
  150    CONTINUE
         DO 160 J = 1, NRHS
            B( I, J ) = B0( I, J )
  160    CONTINUE
  170 CONTINUE
      DO 180 J = 1, N
         IPVT( J ) = 0
  180 CONTINUE
*
      CALL ZGELSY( M, N, NRHS, A, LDA, B, LDB, IPVT, RCOND, RANK, WORK,
     $             LWORK, RWORK, INFO )
      IREP = IREP + 1
      S2 = DSECND( )
      GO TO 140
*
*     End of loop
  190 CONTINUE
      NREP = IREP
      TIMEY = S2 - S1
      TIMEY = TIMEY / NREP
*
*        Adjust for time to make copies
*
      S1 = DSECND( )
      IREP = 0
  200 CONTINUE
*     Top of loop
*
      IF( IREP.GE.NREP )
     $   GO TO 250
*           Top of loop
      DO 230 I = 1, M
         DO 210 J = 1, N
            A( I, J ) = A0( I, J )
  210    CONTINUE
         DO 220 J = 1, NRHS
            B( I, J ) = B0( I, J )
  220    CONTINUE
  230 CONTINUE
      DO 240 J = 1, N
         IPVT( J ) = 0
  240 CONTINUE
      IREP = IREP + 1
      S2 = DSECND( )
      GO TO 200
*
*     End of loop
  250 CONTINUE
      TIMEC = S2 - S1
      TIMEC = TIMEC / NREP
*
      TIMEY = TIMEY - TIMEC
*
*     Time xGELSZ **********************************
*
      S1 = DSECND( )
      S2 = S1
      IREP = 0
  260 CONTINUE
*     Top of loop
*
      IF( S2-S1.GE.TREP )
     $   GO TO 310
*           Top of loop
      DO 290 I = 1, M
         DO 270 J = 1, N
            A( I, J ) = A0( I, J )
  270    CONTINUE
         DO 280 J = 1, NRHS
            B( I, J ) = B0( I, J )
  280    CONTINUE
  290 CONTINUE
      DO 300 J = 1, N
         IPVT( J ) = 0
  300 CONTINUE
*
      CALL ZGELSZ( M, N, NRHS, A, LDA, B, LDB, IPVT, RCOND, RANK, WORK,
     $             LWORK, RWORK, INFO )
      IREP = IREP + 1
      S2 = DSECND( )
      GO TO 260
*
*     End of loop
  310 CONTINUE
      NREP = IREP
      TIMEZ = S2 - S1
      TIMEZ = TIMEZ / NREP
*
*     Adjust for time to make copies
*
      S1 = DSECND( )
      IREP = 0
  320 CONTINUE
*     Top of loop
*
      IF( IREP.GE.NREP )
     $   GO TO 370
*           Top of loop
      DO 350 I = 1, M
         DO 330 J = 1, N
            A( I, J ) = A0( I, J )
  330    CONTINUE
         DO 340 J = 1, NRHS
            B( I, J ) = B0( I, J )
  340    CONTINUE
  350 CONTINUE
      DO 360 J = 1, N
         IPVT( J ) = 0
  360 CONTINUE
      IREP = IREP + 1
      S2 = DSECND( )
      GO TO 320
*
*     End of loop
  370 CONTINUE
      TIMEC = S2 - S1
      TIMEC = TIMEC / NREP
*
      TIMEZ = TIMEZ - TIMEC
*
*     save times and ranks
*
      MVALS( IC ) = M
      NVALS( IC ) = N
      RANKS( IC ) = RANK
      TIMES( IC, 1 ) = TIMED
      TIMES( IC, 2 ) = TIMEY
      TIMES( IC, 3 ) = TIMEZ
      IF( LEVEL.EQ.2 .OR. LEVEL.EQ.3 ) THEN
*        Monitor output.
         WRITE( *, * )'      M       N    RANK  TIME xGELSD',
     $      '     TIME xGELSY     TIME xGELSZ'
         WRITE( *, '(3I8,3G16.8)' )M, N, RANKS( IC ),
     $      REAL( TIMES( IC, 1 ) ), REAL( TIMES( IC, 2 ) ),
     $      REAL( TIMES( IC, 3 ) )
      END IF
*
      GO TO 10
*
*     End of loop
  380 CONTINUE
*
*     Define the output unit
      NOUT = 11
      OPEN( NOUT, FILE = 'fig12.out' )
*
      DO 390 I = 1, IC
         WRITE( NOUT, '(3I8,3G16.8)' )MVALS( I ), NVALS( I ),
     $      RANKS( I ), REAL( TIMES( I, 1 ) ), REAL( TIMES( I, 2 ) ),
     $      REAL( TIMES( I, 3 ) )
  390 CONTINUE
*
      IF( LEVEL.NE.0 ) THEN
         WRITE( *, * )
         WRITE( *, * )'The results have been written to file ',
     $      'fig12.out.'
         WRITE( *, * )'Run plotfig12 from Matlab to plot the times.'
         WRITE( *, * )
      END IF
*
      END
