// File: driver.cc -- Allows setting up params of all the SVD algos
// Author: Suvrit Sra
// Date: 15 Nov, 2003

// (c) Suvrit Sra 2004 All Rights Reserved

#include <iostream>
#include <cstdlib>
#include "driver.h"

#include "s_runbls1.h"
#include "s_runbls2.h"
#include "s_runlas1.h"
#include "s_runlas2.h"
#include "s_runsis1.h"
#include "s_runsis2.h"
#include "s_runtms1.h"
#include "s_runtms2.h"

using namespace ssvd;
/*
 * Process cmd line arguments passed to a driver object and take
 * appropriate action. This is a sample only, and can be used for a quick
 * and dirty implementation. If the user wants to have his own driver, one
 * way would be to use the other constructor that is available.
 */
Driver::Driver(char** argv)
{
  m_progname = argv[0];
  cmdok = 0;
  verbose = 0;

  while (*++argv) {
    if (*argv[0] == '-') {
      ++*argv;
      switch (*argv[0]) {
      case 'h': 
	showHelp(1);
	exit(1);
      case 'f':
	++argv;
	if (*argv) {
	  if (*argv[0] != '-') {
	    strcpy(fname, *argv);
	    ++cmdok;
	  } else {
	    --argv;
	  }
	} else {
	  std::cerr << "Option -f requires an argument. Use -h for usage\n";
	  exit(1);
	}
	break;
      case 'z':
	ascii = true;
	break;
      case 'o':
	++argv;
	if (*argv) {
	  strcpy(optionsFile, *argv);
	  cmdok++;
	} else {
	  std::cerr << "Option -o requires an argument. Use -h for usage\n";
	  exit(1);
	}
	break;
      case 't':
	++argv;
	if (*argv) {
	  strcpy(txx, *argv);
	} else {
	  std::cerr << "Option -t requires an argument. Use -h for usage\n";
	  exit(1);
	}
	break;
      case 'a':
	++argv;
	if (*argv) {
	  algo = atoi(*argv);
	  ++cmdok;
	} else {
	  std::cerr << "Option -a requires an argument. Use -h for usage\n";
	  exit(1);
	}
	break;
      case 'v':
	verbose = 1;
	break;
      default:
	std::cerr << *argv << " Unknown option. Use -h for usage\n";
	break;
      }
    }
  }

  if (cmdok < 3) {
    showHelp(1);
    exit(1);
  }
}


void Driver::showHelp(int level)
{
  std::cout << m_progname << ":  -f <file> -o <options file> [-t txx|txn] [-a 0-7] [-h] [-v]\n";
  if (level > 0) {
    std::cout
      << "\t-f <matrix prefix>\n"
      << "\t-t <txx|tfn|lfn etc.>\n"
      << "\t-o <options file>\n"
      << "\t-a <0--7>\n"
      << "\t\t 0: BLS1\n"
      << "\t\t 1: BLS2\n"
      << "\t\t 2: LAS1\n"
      << "\t\t 3: LAS2\n"
      << "\t\t 4: SIS1\n"
      << "\t\t 5: SIS2\n"
      << "\t\t 6: TMS1\n"
      << "\t\t 7: TMS2\n"
      << "\t-h This help message\n"
      << "\t-v Be verbose\n"
      << "\t-z Output singular vectors to a text file\n";
    std::cout.flush();
  }
}

/*
 * The main driver routine that delegates execution to appropriate object
 * based on user choice of algorithm.
 */
int Driver::execute()
{
  
  // Open the options file and parse it accordingly ...

  switch (algo) {
  case BLS1:
    algoname = "bls1";
    svd = new s_runbls1(optionsFile);
    break;
  case BLS2:
    algoname = "bls2";
    svd = new s_runbls2(optionsFile);
    break;
  case LAS1:
    algoname = "las1";
    svd = new s_runlas1(optionsFile);
    break;
  case LAS2:
    algoname = "las2";
    svd = new s_runlas2(optionsFile);
    break;
  case SIS1:
    algoname = "sis1";
    svd = new s_runsis1(optionsFile);
    break;
  case SIS2:
    algoname = "sis2";
    svd = new s_runsis2(optionsFile);
    break;
  case TMS1:
    algoname = "tms1";
    svd = new s_runtms1(optionsFile);
    break;
  case TMS2:
    algoname = "tms2";
    svd = new s_runtms2(optionsFile);
    break;
  default:
    std::cerr << m_progname << ": Invalid algorithm specification\n";
    showHelp(1);
    return -1;
  }

  if (verbose) {
    std::cout << "Running Algorithm: " << algoname << std::endl
	      << "Option file: " << optionsFile << std::endl
	      << "";
      
  }

  int result;
  svd->init(fname, txx);
  result = svd->runIt();
  
  if (result != 0) {
    std::cerr << m_progname << ": Error during SVD\n";
  }
  return result;
}


Driver::Driver(int alg, OptionsStruct* opt)
{
  algo = alg;
  options   = opt;

  m_progname = "SVD::Driver";

  if (options == 0) {
    options = new OptionsStruct();
    algo = LAS2;		// Default algo to use
    // Setup default values
    strcpy(options->name, "Default");
    options->lanmax = 80;
    options->nums   = 10;
    strcpy(options->out1, "of1");
    strcpy(options->out2, "of2");
    options->endl = -1e-30;
    options->endr = 1e-30;
    options->vectors = false;
    options->kappa = 1e-6;
    strcpy(options->txx, "txx");
    strcpy(options->prefix, "data");
  } 
}

/*
 * This function 
 */
int Driver::executeWithOptions()
{

  switch (algo) {
  case BLS1:
    algoname = "bls1";
    svd = new s_runbls1(options);
    break;
  case BLS2:
    algoname = "bls2";
    svd = new s_runbls2(options);
    break;
  case LAS1:
    algoname = "las1";
    svd = new s_runlas1(options);
    break;
  case LAS2:
    algoname = "las2";
    svd = new s_runlas2(options);
    break;
  case SIS1:
    algoname = "sis1";
    svd = new s_runsis1(options);
    break;
  case SIS2:
    algoname = "sis2";
    svd = new s_runsis2(options);
    break;
  case TMS1:
    algoname = "tms1";
    svd = new s_runtms1(options);
    break;
  case TMS2:
    algoname = "tms2";
    svd = new s_runtms2(options);
    break;
  default:
    std::cerr << m_progname << ": Invalid algorithm specification\n";
    showHelp(1);
    return -1;
  }

  /*if (verbose) {
    std::cout << "Running Algorithm: " << algoname << std::endl
	      << "Option file: " << optionsFile << std::endl
	      << "";
      
	      }*/

  int result;
  if (options->sA == 0) {
    svd->init(options->prefix, options->txx);
  } else {
    svd->extern_init(options->sA);
  }

  DBG("driver::executeWithOptions() calling svd->runIt()\n");
  result = svd->runIt();
  
  if (result != 0) {
    std::cerr <<  m_progname << ": Error during SVD. Return value =" << result << std::endl;
  }
  return result;
}


void Driver::setOptions(OptionsStruct* o)
{
  if (options)
    delete options;
  options = o;
}
