*DECK SRMGEN
      SUBROUTINE SRMGEN( NELTMX, FACTOR, IERR, N, NELT, ISYM, 
     $     IA, JA, A, F, SOLN, DSUM, ITMP, IDIAG )
C***BEGIN PROLOGUE  SRMGEN
C***SUBSIDIARY
C***PURPOSE  This routine generates a "Random" symmetric or 
C            non-symmetric matrix of size N for use in the SLAP
C            Quick Checks.
C***LIBRARY   SLATEC(SLAP)
C***AUTHOR  Seager, Mark K., (LLNL)
C             seager@lll-crg.llnl.gov
C             Lawrence Livermore National Laboratory
C             PO BOX 808, L-300
C             Livermore, CA 94550
C             (415)423-3141
C***DESCRIPTION
C
C *Usage:
C       INTEGER NELTMX, IERR, N, NELT, ISYM, 
C       INTEGER IA(NELTMX), JA(NELTMX), ITMP(N), IDIAG(N)
C       REAL    FACTOR, A(NELTMX), F(N), SOLN(N), DSUM(N)
C
C       CALL SRMGEN( NELTMX, FACTOR, IERR, N, NELT, ISYM, 
C      $     IA, JA, A, F, SOLN, DSUM, ITMP, IDIAG )
C
C *Arguments:
C  
C NELTMX :IN       Integer.
C         Maximum number of non-zeros that can be created by this
C         routine for storage in the IA, JA, A arrays,  see below.
C FACTOR :IN       Real.
C         Non-zeros in the upper triangle are set to FACTOR times
C         the coresponding entry in the lower triangle when a non-
C         symmetric matrix is requested (See ISYM, below).
C IERR   :OUT      Integer.
C         Return error flag.  
C             IERR = 0 => everything went OK. 
C                  = 1 => Ran out of space trying to create matrix.
C                         Set NELTMX to something larger and retry.
C N      :IN       Integer.
C         Size of the linear system to generate (number of unknowns).
C NELT   :OUT      Integer.
C         Number of non-zeros stored in the IA, JA, A arrays, see below.
C ISYM   :IN       Integer.
C         Flag to indicate the type of matrix to generate:
C             ISYM = 0 => Non-Symmetric Matrix (See FACTOR, above).
C                  = 1 => Symmetric Matrix.
C IA     :OUT      Integer IA(NELTMX).
C         Stores the row indicies for the non-zeros.
C JA     :OUT      Integer JA(NELTMX).
C         Stores the column indicies for the non-zeros.
C A      :OUT      Real A(NELTMX).
C         Stores the values of the non-zeros.
C F      :OUT      Real F(N).
C         The right hand side of the linear system.  Obtained by mult-
C         iplying the matrix time SOLN, see below.
C SOLN   :OUT      Real SOLN(N).
C         The true solution to the linear system.  Each component is
C         chosen at random (0.0<SOLN(I)<1.0, I=1,N)
C DSUM   :WORK     Real DSUM(N).
C ITMP   :WORK     Integer ITMP(N).
C IDIAG  :WORK     Integer IDIAG(N).
C
C *Description
C         The matrix is generated by choosing a random number of 
C         entries for each column and then chosing negative random 
C         numbers for each off diagionals.   The diagionals elements 
C         are chosen to be positive and large enough so the matrix 
C         is slightly diagionally domainate.  The lower triangle of 
C         the matrix is generated and if isym.eq.0 (all matrix elements 
C         stored) the upper triangle elements are chosen so that they
C         are FACTOR times the coresponding lower triangular element.
C
C***ROUTINES CALLED  RAND, SMPL
C***REVISION HISTORY  (YYMMDD)
C   881120  DATE WRITTEN
C***END PROLOGUE  SRMGEN
C
      INTEGER NELTMX, IERR, N, NELT, ISYM
      INTEGER IA(NELTMX), JA(NELTMX)
      INTEGER ITMP(N), IDIAG(N)
      REAL    FACTOR, A(NELTMX)
      REAL    F(N), SOLN(N), DSUM(N)
C
C         Start by setting the random number generator seed.
C         This is done for reproducablility in debuggin.  Remove
C         the seed seeting call for production testing.
C
C***FIRST EXECUTABLE STATEMENT  SRMGEN
      ISEED = RAND( 1234.0 )
      IERR = 0
      DO 10 I = 1, N
         IDIAG(I) = 0
         DSUM(I) = -1.0
 10   CONTINUE
C
C         Set the matrix elements.
C         Loop over the columns.
      NELT = 0
CVD$ NOCONCUR
      DO 30 ICOL = 1, N
         NL = N+1-ICOL
C
C         To keep things sparse divide by two, three or four or ...
C
         INUM = (IFIX( RAND(0.0)*NL ) + 1)/3
         CALL SMPL( NL, INUM, ITMP )
C
C         Set up this column (and row, if non-sym structure).
CVD$ NOVECTOR
CVD$ NOCONCUR
         DO 20 IROW = 1, INUM
            NELT = NELT + 1
            IF( NELT.GT.NELTMX ) THEN
               IERR = 1
               RETURN
            ENDIF
            IA(NELT) = N+1-ITMP(IROW)
            JA(NELT) = ICOL
            IF( IA(NELT).EQ.ICOL ) THEN
               IDIAG(ICOL) = NELT
            ELSE
               A(NELT) = -RAND(0.0)
               DSUM(ICOL) = DSUM(ICOL) + A(NELT)
               IF( ISYM.EQ.0 ) THEN
C
C         Copy this element into upper triangle.
C
                  NELT = NELT + 1
                  IF( NELT.GT.NELTMX ) THEN
                     IERR = 1
                     RETURN
                  ENDIF
                  IA(NELT) = ICOL
                  JA(NELT) = IA(NELT-1)
                  A(NELT)  = A(NELT-1)*FACTOR
                  DSUM(JA(NELT)) = DSUM(JA(NELT)) + A(NELT)
               ELSE
                  DSUM(IA(NELT)) = DSUM(IA(NELT)) + A(NELT)
               ENDIF
            ENDIF
 20      CONTINUE
         IF( IDIAG(ICOL).EQ.0 ) THEN
C
C         Add a diagional to the column.
C
            NELT = NELT + 1
            IF( NELT.GT.NELTMX ) THEN
               IERR = 1
               RETURN
            ENDIF
            IDIAG(ICOL) = NELT
            A(NELT) = 0.0
            IA(NELT) = ICOL
            JA(NELT) = ICOL
         ENDIF
 30   CONTINUE
C
C         Clean up the diagionals.
C
CVD$ NODEPCHK
CLLL. OPTION ASSERT (NOHAZARD)
CDIR$ IVDEP
      DO 40 I = 1, N
         A(IDIAG(I)) = -1.0001*DSUM(I)
 40   CONTINUE
C
C         Set a random soln and determine the right-hand side.
C
CVD$ NOVECTOR
CVD$ NOCONCUR
      DO 50 I = 1, N
         SOLN(I) = RAND(0.0)
         F(I) = 0.0
 50   CONTINUE
C
CVD$ NOVECTOR
CVD$ NOCONCUR
      DO 60 K = 1, NELT
         F(IA(K)) = F(IA(K)) + A(K)*SOLN(JA(K))
         IF( ISYM.NE.0 .AND. IA(K).NE.JA(K) ) THEN
            F(JA(K)) = F(JA(K)) + A(K)*SOLN(IA(K))
         ENDIF
 60   CONTINUE
      RETURN
      END
*DECK SMPL
      SUBROUTINE SMPL( N, M, INDX )
C
C         This routine picks m "random" integers in the range 1 to
C         n with out any repetitions.
C
      INTEGER N, M, INDX(M)
C
C..       Check the input
      IF( N*M.LT.0 .OR. M.GT.N ) RETURN
C
C..       Set the indeicies.
      INDX(1) = IFIX( RAND(0.0)*N ) + 1
CVD$ NOCONCUR
      DO 30 I = 2, M
 10      ID = IFIX( RAND(0.0)*N ) + 1
C
C..       Check to see if id has already been chosen.
CVD$ NOVECTOR
CVD$ NOCONCUR
         DO 20 J = 1, I-1
            IF( ID.EQ.INDX(J) ) GOTO 10
 20      CONTINUE
         INDX(I) = ID
 30   CONTINUE
C
C$$$      write(3,1000) m, n, (indx(i),i=1,m)
C$$$ 1000 format(' SMPL: m, n = ',2i5,' INDX follows'/(10i5))
      RETURN
      END
      FUNCTION RAND(R)
C***BEGIN PROLOGUE  RAND
C***DATE WRITTEN   770401   (YYMMDD)
C***REVISION DATE  861211   (YYMMDD)
C***CATEGORY NO.  L6A21
C***KEYWORDS  LIBRARY=SLATEC(FNLIB),TYPE=SINGLE PRECISION(RAND-S),
C             RANDOM NUMBER,SPECIAL FUNCTIONS,UNIFORM
C***AUTHOR  FULLERTON, W., (LANL)
C***PURPOSE  Generates a uniformly distributed random number.
C***DESCRIPTION
C
C      This pseudo-random number generator is portable among a wide
C variety of computers.  RAND(R) undoubtedly is not as good as many
C readily available installation dependent versions, and so this
C routine is not recommended for widespread usage.  Its redeeming
C feature is that the exact same random numbers (to within final round-
C off error) can be generated from machine to machine.  Thus, programs
C that make use of random numbers can be easily transported to and
C checked in a new environment.
C      The random numbers are generated by the linear congruential
C method described, e.g., by Knuth in Seminumerical Methods (p.9),
C Addison-Wesley, 1969.  Given the I-th number of a pseudo-random
C sequence, the I+1 -st number is generated from
C             X(I+1) = (A*X(I) + C) MOD M,
C where here M = 2**22 = 4194304, C = 1731 and several suitable values
C of the multiplier A are discussed below.  Both the multiplier A and
C random number X are represented in double precision as two 11-bit
C words.  The constants are chosen so that the period is the maximum
C possible, 4194304.
C      In order that the same numbers be generated from machine to
C machine, it is necessary that 23-bit integers be reducible modulo
C 2**11 exactly, that 23-bit integers be added exactly, and that 11-bit
C integers be multiplied exactly.  Furthermore, if the restart option
C is used (where R is between 0 and 1), then the product R*2**22 =
C R*4194304 must be correct to the nearest integer.
C      The first four random numbers should be .0004127026,
C .6750836372, .1614754200, and .9086198807.  The tenth random number
C is .5527787209, and the hundredth is .3600893021 .  The thousandth
C number should be .2176990509 .
C      In order to generate several effectively independent sequences
C with the same generator, it is necessary to know the random number
C for several widely spaced calls.  The I-th random number times 2**22,
C where I=K*P/8 and P is the period of the sequence (P = 2**22), is
C still of the form L*P/8.  In particular we find the I-th random
C number multiplied by 2**22 is given by
C I   =  0  1*P/8  2*P/8  3*P/8  4*P/8  5*P/8  6*P/8  7*P/8  8*P/8
C RAND=  0  5*P/8  2*P/8  7*P/8  4*P/8  1*P/8  6*P/8  3*P/8  0
C Thus the 4*P/8 = 2097152 random number is 2097152/2**22.
C      Several multipliers have been subjected to the spectral test
C (see Knuth, p. 82).  Four suitable multipliers roughly in order of
C goodness according to the spectral test are
C    3146757 = 1536*2048 + 1029 = 2**21 + 2**20 + 2**10 + 5
C    2098181 = 1024*2048 + 1029 = 2**21 + 2**10 + 5
C    3146245 = 1536*2048 +  517 = 2**21 + 2**20 + 2**9 + 5
C    2776669 = 1355*2048 + 1629 = 5**9 + 7**7 + 1
C
C      In the table below LOG10(NU(I)) gives roughly the number of
C random decimal digits in the random numbers considered I at a time.
C C is the primary measure of goodness.  In both cases bigger is better.
C
C                   LOG10 NU(I)              C(I)
C       A       I=2  I=3  I=4  I=5    I=2  I=3  I=4  I=5
C
C    3146757    3.3  2.0  1.6  1.3    3.1  1.3  4.6  2.6
C    2098181    3.3  2.0  1.6  1.2    3.2  1.3  4.6  1.7
C    3146245    3.3  2.2  1.5  1.1    3.2  4.2  1.1  0.4
C    2776669    3.3  2.1  1.6  1.3    2.5  2.0  1.9  2.6
C   Best
C    Possible   3.3  2.3  1.7  1.4    3.6  5.9  9.7  14.9
C
C             Input Argument --
C R      If R=0., the next random number of the sequence is generated.
C        If R .LT. 0., the last generated number will be returned for
C          possible use in a restart procedure.
C        If R .GT. 0., the sequence of random numbers will start with
C          the seed R mod 1.  This seed is also returned as the value of
C          RAND provided the arithmetic is done exactly.
C
C             Output Value --
C RAND   a pseudo-random number between 0. and 1.
C***REFERENCES  (NONE)
C***ROUTINES CALLED  (NONE)
C***END PROLOGUE  RAND
      SAVE IA1, IA0, IA1MA0, IC, IX1, IX0
      DATA IA1, IA0, IA1MA0 /1536, 1029, 507/
      DATA IC /1731/
      DATA IX1, IX0 /0, 0/
C***FIRST EXECUTABLE STATEMENT  RAND
      IF (R.LT.0.) GO TO 10
      IF (R.GT.0.) GO TO 20
C
C           A*X = 2**22*IA1*IX1 + 2**11*(IA1*IX1 + (IA1-IA0)*(IX0-IX1)
C                   + IA0*IX0) + IA0*IX0
C
      IY0 = IA0*IX0
      IY1 = IA1*IX1 + IA1MA0*(IX0-IX1) + IY0
      IY0 = IY0 + IC
      IX0 = MOD (IY0, 2048)
      IY1 = IY1 + (IY0-IX0)/2048
      IX1 = MOD (IY1, 2048)
C
 10   RAND = IX1*2048 + IX0
      RAND = RAND / 4194304.
      RETURN
C
 20   IX1 = AMOD(R,1.)*4194304. + 0.5
      IX0 = MOD (IX1, 2048)
      IX1 = (IX1-IX0)/2048
      GO TO 10
C
      END
*DECK MATGEN1
      SUBROUTINE MATGEN1( NMAX, NELTMAX, FACTOR, SCALE,
     $     N1, N2, N, NELT, ISYM, 
     $     IA, JA, A, A0, A1, A2, A3, A4, F, SOLN )
C----------------------------------------------------------------------
C         This routine generates a "stiffness" matrix for the Elliptic
C         Partial Differential Equation:
C         
C         -d/dx(p(x,y)du/dx) - d/dy(p(x,y)du/dy) = f(x,y)
C         
C         0 .lt. x .lt. 1 ,   0 .lt. y .lt. 1
C         
C         with boundary conditions
C         
C         u(x,0) = u(x,1) = u(0,y) = u(1,y) = 0
C         
C using the finite element method with piecewise bilinear finite
C elements.
C----------------------------------------------------------------------
      INTEGER NMAX, NELTMAX, N1, N2, N, NELT, ISYM
      INTEGER IA(NELTMAX), JA(NELTMAX)
      REAL    FACTOR, A(NELTMAX)
      REAL    A0(NMAX), A1(NMAX), A2(NMAX), A3(NMAX), A4(NMAX)
      REAL    F(NMAX), SOLN(NMAX)
C         
C..       set problem size.
      N = N1*N2
      N1P1 = N1 + 1
      N2P1 = N2 + 1
      H1 = 1./N1P1
      H2 = 1./N2P1
      H1H2 = H1*H2
      NZ = N1P1*N2P1
C         
C..       zero out matrix and right-hand side.
      DO 10 I=1,N
         A0(I) = 0.
         A1(I) = 0.
         A2(I) = 0.
         A3(I) = 0.
         A4(I) = 0.
         F(I) = 0.
 10   CONTINUE
C         
C         
C..       loop over zones.
CVD$ NOCONCUR
      DO 101 J=1,N2P1
CVD$ NOVECTOR
CVD$ NOCONCUR
         DO 100 K=1,N1P1
C         
C..       calculate coords of zone.
            XL = (K-1)*H1
            XU = XL + H1
            YL = (J-1)*H2
            YU = YL + H2
C         
C..       set gauss quadrature pts.
            X1 = (XL+XU)/2. - H1/(2.*SQRT(3.))
            X2 = (XL+XU)/2. + H1/(2.*SQRT(3.))
            Y1 = (YL+YU)/2. - H2/(2.*SQRT(3.))
            Y2 = (YL+YU)/2. + H2/(2.*SQRT(3.))
C         
C..       evaluate p at gauss quadrature pts.
            P11 = P(X1,Y1)
            P12 = P(X1,Y2)
            P21 = P(X2,Y1)
            P22 = P(X2,Y2)
C         
C..       calculate node indices of the vertices of this zone.
            ILL = (J-2)*N1 + (K-1)
            ILR = ILL + 1
            IUL = ILL + N1
            IUR = IUL + 1
C         
C..       evaluate integral involving lower left basis function.
            IF(J .EQ. 1 .OR. K .EQ. 1) GO TO 50
C         
            X1T = H1/2. - H1/(2.*SQRT(3.))
            X2T = H1/2. + H1/(2.*SQRT(3.))
            Y1T = H2/2. - H2/(2.*SQRT(3.))
            Y2T = H2/2. + H2/(2.*SQRT(3.))
C         
            BII11 = (1./H1 - Y1T/H1H2)**2 + (1./H2 - X1T/H1H2)**2
            BII12 = (1./H1 - Y2T/H1H2)**2 + (1./H2 - X1T/H1H2)**2
            BII21 = (1./H1 - Y1T/H1H2)**2 + (1./H2 - X2T/H1H2)**2
            BII22 = (1./H1 - Y2T/H1H2)**2 + (1./H2 - X2T/H1H2)**2
C         
            V1 = .5 * H1 * (P11*BII11 + P21*BII21)
            V2 = .5 * H1 * (P12*BII12 + P22*BII22)
            V = .5 * H2 * (V1 + V2)
C         
            A0(ILL) = A0(ILL) + V
C         
C..       evaluate integral involving lower right basis function.
 50         IF(J .EQ. 1 .OR. K .EQ. N1P1) GO TO 60
C         
            X1T = -H1/2. - H1/(2.*SQRT(3.))
            X2T = -H1/2. + H1/(2.*SQRT(3.))
            Y1T = H2/2. - H2/(2.*SQRT(3.))
            Y2T = H2/2. + H2/(2.*SQRT(3.))
C         
            BII11 = (1./H1 - Y1T/H1H2)**2 + (1./H2 + X1T/H1H2)**2
            BII12 = (1./H1 - Y2T/H1H2)**2 + (1./H2 + X1T/H1H2)**2
            BII21 = (1./H1 - Y1T/H1H2)**2 + (1./H2 + X2T/H1H2)**2
            BII22 = (1./H1 - Y2T/H1H2)**2 + (1./H2 + X2T/H1H2)**2
C         
            V1 = .5 * H1 * (P11*BII11 + P21*BII21)
            V2 = .5 * H1 * (P12*BII12 + P22*BII22)
            V = .5 * H2 * (V1 + V2)
C         
            A0(ILR) = A0(ILR) + V
C         
C..       evaluate integral involving lower right and lower left
C..       basis functions.
            IF(K .EQ. 1) GO TO 60
C         
            BIJ11 = -(1./H1-Y1T/H1H2)**2-(1./H2+X1T/H1H2)*(X1T/H1H2)
            BIJ12 = -(1./H1-Y2T/H1H2)**2-(1./H2+X1T/H1H2)*(X1T/H1H2)
            BIJ21 = -(1./H1-Y1T/H1H2)**2-(1./H2+X2T/H1H2)*(X2T/H1H2)
            BIJ22 = -(1./H1-Y2T/H1H2)**2-(1./H2+X2T/H1H2)*(X2T/H1H2)
C         
            V1 = .5 * H1 * (P11*BIJ11 + P21*BIJ21)
            V2 = .5 * H1 * (P12*BIJ12 + P22*BIJ22)
            V = .5 * H2 * (V1 + V2)
C         
            A1(ILL) = A1(ILL) + V
C         
C..       evaluate integral involving upper left basis function.
 60         IF(J .EQ. N2P1 .OR. K .EQ. 1) GO TO 70
C         
            X1T = H1/2. - H1/(2.*SQRT(3.))
            X2T = H1/2. + H1/(2.*SQRT(3.))
            Y1T = -H2/2. - H2/(2.*SQRT(3.))
            Y2T = -H2/2. + H2/(2.*SQRT(3.))
C         
            BII11 = (1./H1 + Y1T/H1H2)**2 + (1./H2 - X1T/H1H2)**2
            BII12 = (1./H1 + Y2T/H1H2)**2 + (1./H2 - X1T/H1H2)**2
            BII21 = (1./H1 + Y1T/H1H2)**2 + (1./H2 - X2T/H1H2)**2
            BII22 = (1./H1 + Y2T/H1H2)**2 + (1./H2 - X2T/H1H2)**2
C         
            V1 = .5 * H1 * (P11*BII11 + P21*BII21)
            V2 = .5 * H1 * (P12*BII12 + P22*BII22)
            V = .5 * H2 * (V1 + V2)
C         
            A0(IUL) = A0(IUL) + V
C         
C..       evaluate integral involving upper left and lower left
C..       basis functions.
            IF(J .EQ. 1) GO TO 70
C         
            BIK11 = -(1./H1+Y1T/H1H2)*(Y1T/H1H2)-(1./H2-X1T/H1H2)**2
            BIK12 = -(1./H1+Y2T/H1H2)*(Y2T/H1H2)-(1./H2-X1T/H1H2)**2
            BIK21 = -(1./H1+Y1T/H1H2)*(Y1T/H1H2)-(1./H2-X2T/H1H2)**2
            BIK22 = -(1./H1+Y2T/H1H2)*(Y2T/H1H2)-(1./H2-X2T/H1H2)**2
C         
            V1 = .5 * H1 * (P11*BIK11+P21*BIK21)
            V2 = .5 * H1 * (P12*BIK12+P22*BIK22)
            V = .5 * H2 * (V1+V2)
C         
            A3(ILL) = A3(ILL)+V
C         
C..       evaluate integral involving upper left and lower right
C..       basis functions.
            IF(K .EQ. N1P1) GO TO 70
C         
            BIL11 = (1./H1+Y1T/H1H2)*(Y1T/H1H2)+(1./H2-X1T/H1H2)*
     $           (-X1T/H1H2)
            BIL12 = (1./H1+Y2T/H1H2)*(Y2T/H1H2)+(1./H2-X1T/H1H2)*
     $           (-X1T/H1H2)
            BIL21 = (1./H1+Y1T/H1H2)*(Y1T/H1H2)+(1./H2-X2T/H1H2)*
     $           (-X2T/H1H2)
            BIL22 = (1./H1+Y2T/H1H2)*(Y2T/H1H2)+(1./H2-X2T/H1H2)*
     $           (-X2T/H1H2)
C         
            V1 = .5 * H1 * (P11*BIL11 + P21*BIL21)
            V2 = .5 * H1 * (P12*BIL12 + P22*BIL22)
            V = .5 * H2 * (V1 + V2)
C         
            A2(ILR) = A2(ILR) + V
C         
C..       evaluate integral involving upper right basis function.
 70         IF(J .EQ. N2P1 .OR. K .EQ. N1P1) GO TO 100
C         
            X1T = -H1/2. - H1/(2.*SQRT(3.))
            X2T = -H1/2. + H1/(2.*SQRT(3.))
            Y1T = -H2/2. - H2/(2.*SQRT(3.))
            Y2T = -H2/2. + H2/(2.*SQRT(3.))
C         
            BII11 = (1./H1 + Y1T/H1H2)**2 + (1./H2 + X1T/H1H2)**2
            BII12 = (1./H1 + Y2T/H1H2)**2 + (1./H2 + X1T/H1H2)**2
            BII21 = (1./H1 + Y1T/H1H2)**2 + (1./H2 + X2T/H1H2)**2
            BII22 = (1./H1 + Y2T/H1H2)**2 + (1./H2 + X2T/H1H2)**2
C         
            V1 = .5 * H1 * (P11*BII11 + P21*BII21)
            V2 = .5 * H1 * (P12*BII12 + P22*BII22)
            V = .5 * H2 * (V1 + V2)
C         
            A0(IUR) = A0(IUR) + V
C         
C..       evaluate integral involving upper right and upper left
C..       basis functions.
            IF(K .EQ. 1) GO TO 80
C         
            BIJ11 = -(1./H1 + Y1T/H1H2)**2-(1./H2 + X1T/H1H2)*(X1T/H1H2)
            BIJ12 = -(1./H1 + Y2T/H1H2)**2-(1./H2 + X1T/H1H2)*(X1T/H1H2)
            BIJ21 = -(1./H1 + Y1T/H1H2)**2-(1./H2 + X2T/H1H2)*(X2T/H1H2)
            BIJ22 = -(1./H1 + Y2T/H1H2)**2-(1./H2 + X2T/H1H2)*(X2T/H1H2)
C         
            V1 = .5 * H1 * (P11*BIJ11 + P21*BIJ21)
            V2 = .5 * H1 * (P12*BIJ12 + P22*BIJ22)
            V = .5 * H2 * (V1 + V2)
C         
            A1(IUL) = A1(IUL) + V
C         
C..       evaluate integral involving upper right and lower left
C..       basis functions.
            IF(J .EQ. 1) GO TO 100
C         
            BIM11 = (1./H1+Y1T/H1H2)*(Y1T/H1H2)+(1./H2+X1T/H1H2)*
     $           (X1T/H1H2)
            BIM12 = (1./H1+Y2T/H1H2)*(Y2T/H1H2)+(1./H2+X1T/H1H2)*
     $           (X1T/H1H2)
            BIM21 = (1./H1+Y1T/H1H2)*(Y1T/H1H2)+(1./H2+X2T/H1H2)*
     $           (X2T/H1H2)
            BIM22 = (1./H1+Y2T/H1H2)*(Y2T/H1H2)+(1./H2+X2T/H1H2)*
     $           (X2T/H1H2)
C         
            V1 = .5 * H1 * (P11*BIM11 + P21*BIM21)
            V2 = .5 * H1 * (P12*BIM12 + P22*BIM22)
            V = .5 * H2 * (V1 + V2)
C         
            A4(ILL) = A4(ILL) + V
C         
C..       evaluate integral involving upper right and lower right
C..       basis functions.
 80         IF(J .EQ. 1) GO TO 100
C         
            BIK11 = -(1./H1+Y1T/H1H2)*(Y1T/H1H2)-(1./H2+X1T/H1H2)**2
            BIK12 = -(1./H1+Y2T/H1H2)*(Y2T/H1H2)-(1./H2+X1T/H1H2)**2
            BIK21 = -(1./H1+Y1T/H1H2)*(Y1T/H1H2)-(1./H2+X2T/H1H2)**2
            BIK22 = -(1./H1+Y2T/H1H2)*(Y2T/H1H2)-(1./H2+X2T/H1H2)**2
C         
            V1 = .5 * H1 * (P11*BIK11 + P21*BIK21)
            V2 = .5 * H1 * (P12*BIK12 + P22*BIK22)
            V = .5 * H2 * (V1 + V2)
C         
            A3(ILR) = A3(ILR) + V
C         
 100     CONTINUE
 101  CONTINUE
C         
C..       matrix has been generated.  set random solution and calculate
C..       right-hand side.
C$$$      dum = 0.
C$$$      iseed = ranset( 1234 )
C$$$CVD$ NOVECTOR
C$$$CVD$ NOCONCUR
C$$$      do 110 i=1,n
C$$$         soln(i) = ranf(dum)
C$$$ 110  continue
      DO 110 I = 1, N
         SOLN(I) = 1.0
 110  CONTINUE
      IF( SCALE.LT.1.0 ) SCALE = 1.01
      DO 121 I = 1, N
         A0(I) = SCALE*A0(I)
 121  CONTINUE
      F(1) = A0(1)*SOLN(1) + A1(1)*SOLN(2)
      DO 120 I=2,N-1
         F(I) = A0(I)*SOLN(I) + A1(I-1)*SOLN(I-1) + A1(I)*SOLN(I+1)
 120  CONTINUE
      F(N) = A0(N)*SOLN(N) + A1(N-1)*SOLN(N-1)
C         
      DO 170 I=1,N-N1-1
         F(I) = F(I) + A2(I)*SOLN(I+N1-1) + A3(I)*SOLN(I+N1) +
     $        A4(I)*SOLN(I+N1+1)
 170  CONTINUE
      F(N-N1) = F(N-N1) + A2(N-N1)*SOLN(N-1) + A3(N-N1)*SOLN(N)
      F(N-N1+1) = F(N-N1+1) + A2(N-N1+1)*SOLN(N)
C         
      F(N1) = F(N1) + A2(1)*SOLN(1)
      F(N1+1) = F(N1+1) + A2(2)*SOLN(2) + A3(1)*SOLN(1)
      DO 160 I=N1+2,N
         F(I) = F(I) + A2(I-N1+1)*SOLN(I-N1+1) + A3(I-N1)*SOLN(I-N1) +
     $        A4(I-N1-1)*SOLN(I-N1-1)
 160  CONTINUE
C$$$      write(3,206) (i,a0(i),a1(i),a2(i),a3(i),a4(i), i=1,n)
C$$$ 206  format(/'    I            A0(I)            A1(I)            ',
C$$$     $     'A2(I)            A3(I)            A4(I)'/
C$$$     $     (1x,i3,1x,e16.7,1x,e16.7,1x,e16.7,1x,e16.7,1x,e16.7))
C         
C*******************************************************************
C         
C         Problem is set up.  Put matrix in SLAP triad format.
C         
C*******************************************************************
C
      IF(ISYM .EQ. 0) THEN
C         Store the whole durn thing..
C         FACTOR denotes the amount of asymmetry the user desires...
         DO 210 I=1,N
            A(I) = A0(I)
            IA(I) = I
            JA(I) = I
 210     CONTINUE
         DO 220 I=1,N-1
            A(N+I) = A1(I)*FACTOR
            IA(N+I) = I
            JA(N+I) = I + 1
 220     CONTINUE
         DO 230 I=1,N-1
            A(2*N-1+I) = A1(I)
            IA(2*N-1+I) = I + 1
            JA(2*N-1+I) = I
 230     CONTINUE
         DO 240 I=1,N-N1+1
            A(3*N-2+I) = A2(I)*FACTOR
            IA(3*N-2+I) = I
            JA(3*N-2+I) = I + N1 - 1
 240     CONTINUE
         DO 250 I=1,N-N1+1
            A(4*N-N1-1+I) = A2(I)
            IA(4*N-N1-1+I) = I + N1 - 1
            JA(4*N-N1-1+I) = I
 250     CONTINUE
         DO 260 I=1,N-N1
            A(5*N-2*N1+I) = A3(I)*FACTOR
            IA(5*N-2*N1+I) = I
            JA(5*N-2*N1+I) = I + N1
 260     CONTINUE
         DO 270 I=1,N-N1
            A(6*N-3*N1+I) = A3(I)
            IA(6*N-3*N1+I) = I + N1
            JA(6*N-3*N1+I) = I
 270     CONTINUE
         DO 280 I=1,N-N1-1
            A(7*N-4*N1+I) = A4(I)*FACTOR
            IA(7*N-4*N1+I) = I
            JA(7*N-4*N1+I) = I + N1 + 1
 280     CONTINUE
         DO 290 I=1,N-N1-1
            A(8*N-5*N1-1+I) = A4(I)
            IA(8*N-5*N1-1+I) = I + N1 + 1
            JA(8*N-5*N1-1+I) = I
 290     CONTINUE
         NELT = 9*N-6*N1-2
      ELSE
C         Store the lower triangle of the Matrix...
         DO 310 I=1,N
            A(I) = A0(I)
            IA(I) = I
            JA(I) = I
 310     CONTINUE
         DO 320 I=1,N-1
            A(N+I) = A1(I)
            IA(N+I) = I + 1
            JA(N+I) = I
 320     CONTINUE
         DO 330 I=1,N-N1+1
            A(2*N-1+I) = A2(I)
            IA(2*N-1+I) = I + N1 -1
            JA(2*N-1+I) = I
 330     CONTINUE
         DO 340 I=1,N-N1
            A(3*N-N1+I) = A3(I)
            IA(3*N-N1+I) = I + N1
            JA(3*N-N1+I) = I
 340     CONTINUE
         DO 350 I=1,N-N1-1
            A(4*N-2*N1+I) = A4(I)
            IA(4*N-2*N1+I) = I + N1 + 1
            JA(4*N-2*N1+I) = I
 350     CONTINUE
         NELT = 5*N-3*N1-1
      ENDIF
      RETURN
      END
*DECK FIVEPT
      SUBROUTINE FIVEPT(NMAX, NELTMAX, IERR, NX, NY, N, NELT, ISYM,
     $     IA, JA, AZ, RHS, SOLN )
C
C         FIVEPT generates the  itpack matrices jcoef,  coef, and  rhs
C         for a 5-point   central  difference  approximation  of   the
C         PDE:
C
C         (A*u )   + (C*u )   + D*u   + E*u   + F*u  =  G
C             x x        y y       x       y
C
C         Where the coefficients A, C, D, E, F, and G are functions of
C         (x,y) (supplied by the  user)  and  the  domain is  the unit
C         square (0,1)x(0,1).    Dirichlet  boundary  conditions   are
C         imposed upon the   boundary  in the  form of    the function
C         UB(x,y) also supplied by the user.  If the solution is known
C         it should be plugged into UXY(x,y)
C
C     parameters -- 
C
C        NX, NY  Number of grid lines in the x and y dimensions (Input).
C        IERR    IERR = 0 => All went well.
C                      -1 => Number of grid points N = NX*NY too large.
C                      -2 => Number of non-zeros got too large.
C        N       Number of linear equations(Output).  N = NX*NY.
C        NELT    Number of non-zeros in the IA, JA, A arrays (Output).
C        ISYM    ISYM = 0 => All Non-zeros of the matrix must be stored.
C                            The coeficients D(X,Y) and E(X,Y) are 
C                            assumed to be non-zero in this case.
C                     = 1 => Generate a symmetric matrix and store only
C                            the lower triangle.  The coeficients D(X,Y)
C                            and E(X,Y) are assumed to be *ZERO* in this
C                            case.
C        IA, JA  Row and Column indicies of the non-zeros (Output).
C        AZ      Non-zeros (Output).
C        RHS     Vector of right-hand-side values (Output).
C        SOLN    Solution vector, if known (Output).
C
C     Specifications for parameters
C
      INTEGER  NMAX, NELTMAX, IERR, NX, NY, N, NELT, ISYM
      INTEGER  IA(NELTMAX), JA(NELTMAX)
      REAL     AZ(NELTMAX), RHS(NMAX), SOLN(NMAX)
C
      LOGICAL SYMM
C
C         Statement functions.  Define the PDE coeficients here.
C
      A(X,Y) = 1.0
      C(X,Y) = 1.0
      D(X,Y) = 4.0
      E(X,Y) = 4.0
      F(X,Y) = 1.0
      G(X,Y) = 1.0
      UB(X,Y) = 1.0
      UXY(X,Y) = 0.0
C
C         Set up the grid.  Assuming a natural ordering from 
C         left to right (X-Axis) and from down to up (Y-Axis).
C         
      N = NX*NY
      IF( N.LT.1 .OR. N.GT.NMAX ) GOTO 100
      HX = 1.0/FLOAT(NX+1)
      HY = 1.0/FLOAT(NY+1)
      HALFHX = HX/2.0 
      HALFHY = HY/2.0 
      SYMM   = ISYM .EQ. 0
C
C         Loop on equations, assuming a natural ordering from
C         left to right and from down to up.
C
      NEQ = 0
      NELT = 0
      DO 25 J = 1, NY
         YY = FLOAT(J)*HY
         DO 20 I = 1, NX
            XX = FLOAT(I)*HX
            NEQ = NEQ + 1
            SOLN(NEQ) = UXY( XX, YY)
            AE = A( XX+HALFHX, YY)
            AW = A( XX-HALFHX, YY)
            CN = C( XX, YY+HALFHY)
            CS = C( XX, YY-HALFHY)
            IF( SYMM ) THEN
               DE = 0.0
               EN = 0.0
            ELSE
               DE = D( XX+HALFHX, YY)
               EN = E( XX, YY+HALFHY)
            ENDIF
            FP = F( XX, YY)
            GP = G( XX, YY)
            CC = AE + CN + AW + CS - (HX*DE + HY*EN + HX*HY*FP)
C
C ... Center point.
C
            NELT = NELT + 1
            IF( NELT.GT.NELTMAX ) GOTO 110
            AZ(NELT)  = CC
            IA(NELT) = NEQ
            JA(NELT) = NEQ
            RHS(NEQ) = -HX*HY*GP
C
C ... East point.
C
            IF( I.NE.NX ) THEN
               IF( .NOT.SYMM ) THEN
                  NELT = NELT + 1
                  IF( NELT.GT.NELTMAX ) GOTO 110
                  AZ(NELT)  = -AE - HX*DE
                  IA(NELT) = NEQ
                  JA(NELT) = NEQ + 1
               ENDIF
            ELSE
               RHS(NEQ) = RHS(NEQ) + (AE + HX*DE)*UB( 1.0, YY) 
            ENDIF
C
C ... North point.
C
            IF( J.NE.NY) THEN
               IF( .NOT.SYMM ) THEN
                  NELT = NELT + 1
                  IF( NELT.GT.NELTMAX ) GOTO 110
                  AZ(NELT)  = -CN - HY*EN
                  IA(NELT) = NEQ
                  JA(NELT) = NEQ + NX
               ENDIF
            ELSE
               RHS(NEQ) = RHS(NEQ) + (CN + HY*EN)*UB( XX, 1.0) 
            ENDIF
C
C ... West point.
C
            IF( I.NE.1 ) THEN
               NELT = NELT + 1
               IF( NELT.GT.NELTMAX ) GOTO 110
               AZ(NELT)  = -AW
               IA(NELT) = NEQ
               JA(NELT) = NEQ - 1
            ELSE
               RHS(NEQ) = RHS(NEQ) + AW*UB(0.0,YY)
            ENDIF
C
C ... South point.
C
            IF( J.NE.1 ) THEN
               NELT = NELT + 1
               IF( NELT.GT.NELTMAX ) GOTO 110
               AZ(NELT) = -CS
               IA(NELT) = NEQ
               JA(NELT) = NEQ - NX
            ELSE
               RHS(NEQ) = RHS(NEQ) + CS*UB(XX,0.0)
            ENDIF
 20      CONTINUE
 25   CONTINUE
C
C         Normal return.
      IERR = 0
      RETURN
C
C         Error return.
 100  IERR = -1
      RETURN
 110  IERR = -2
      RETURN
      END 
      REAL FUNCTION P(X,Y)
C
C..         Coeficient function for the PDE in MATGEN1
      P = .01 + X**2 + Y**2
      RETURN
      END
*DECK VFILL
      SUBROUTINE VFILL (N,V,VAL)
C
C     vfill fills a vector, v, with a constant value, val.
C
      REAL    V(N)
      IF (N .LE. 0) RETURN
      NR=MOD(N,4)
C
C The following construct assumes a zero pass do loop.
C
      IS=1
      GOTO(1,2,3,4), NR+1
    4   IS=4
        V(1)=VAL
        V(2)=VAL
        V(3)=VAL
        GOTO 1
    3   IS=3
        V(1)=VAL
        V(2)=VAL
        GOTO 1
    2   IS=2
        V(1)=VAL
    1 DO 10 I=IS,N,4
        V(I)  =VAL
        V(I+1)=VAL
        V(I+2)=VAL
        V(I+3)=VAL
 10   CONTINUE
      RETURN
      END
      REAL FUNCTION RANF( IDUM )
C
C         This function models the Cray Fortran (CFT) function of the
C         same name for Unix F77 systems.  It is not portable.
C         It returns a random number, x, in the range: 0 < x < 1.
C****************************************************************
C         THIS ROUTINE IS NOT PORTABLE
C****************************************************************
      EXTERNAL RAND
C
      RANF = RAND( 0 )
      RETURN
      END
      REAL FUNCTION RANSET( ISEED )
C
C         This function models the Cray Fortran (CFT) function of the
C         same name for Unix F77 systems.  It is not portable.
C         It sets the random number generator seed to iseed.
C****************************************************************
C         THIS ROUTINE IS NOT PORTABLE
C****************************************************************
      EXTERNAL RAND
C
      RANSET = RAND( ISEED )
      RETURN
      END
      REAL FUNCTION SECOND(DUMMY)
C
C         This function models the Cray Fortran (CFT) function of the
C         same name for Unix F77 (4.2 Bsd) systems.  It is not portable.
C         Returns the CPU time (seconds) since begining of program.
C****************************************************************
C         THIS ROUTINE IS NOT PORTABLE
C****************************************************************
      REAL DUMMY
      REAL*4 TIME(2), ETIME
      SECOND = ETIME( TIME )
      RETURN
      END
