#include "parpre_pipeline.h"
#include "parpre_mat.h"
#include "parpre_vec.h"
#include "petscviewer.h"

static int events[5];
#define ORTHO_EVENT 0
#define COMM_EVENT 1
#define AXPY_EVENT 2
#define AYPX_EVENT 3
#define DOT_EVENT 4

#undef __FUNC__
#define __FUNC__ "ParPreVecsInit"
int ParPreVecsInit()
{
  int ierr;
  PetscFunctionBegin;
  ierr = PLogEventRegister
    (events+ORTHO_EVENT, "Vecbundle QR        ",PETSC_NULL); CHKERRQ(ierr);
  ierr = PLogEventRegister
    (events+COMM_EVENT,  "Vecbundle comm      ",PETSC_NULL); CHKERRQ(ierr);
  ierr = PLogEventRegister
    (events+AXPY_EVENT,  "Vecbundle AXPY      ",PETSC_NULL); CHKERRQ(ierr);
  ierr = PLogEventRegister
    (events+AYPX_EVENT,  "Vecbundle AYPX      ",PETSC_NULL); CHKERRQ(ierr);
  ierr = PLogEventRegister
    (events+DOT_EVENT,   "Vecbundle vecdot    ",PETSC_NULL); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsOrthogonalise_O"
int VecsOrthogonalise_O
(MPI_Comm comm,Vec *vecs,int n,MatReuse use,Mat *u,Mat u1,Mat u2,
 QRMethod meth,DotProducts dp)
{ 
  int cur,next,ierr;

  PetscFunctionBegin;
  PLogEventBegin(events[ORTHO_EVENT],0,0,0,0);
  if (use==MAT_INITIAL_MATRIX) {
    ierr = MatCreateSeqDense(MPI_COMM_SELF,n,n,PETSC_NULL,u); CHKERRQ(ierr);
  } else if (use==MAT_REUSE_MATRIX) {
    ierr = MatZeroEntries(*u); CHKERRQ(ierr);
  } else SETERRQ(1,0,"Unknown MatReuse mode");

  if (meth==GRAM_SCHMIDT) {
    Scalar *tmp;
    tmp = (Scalar *) PetscMalloc(n*sizeof(Scalar)); CHKPTRQ(tmp);
    for (cur=0; cur<n; cur++) {
      Scalar norm;
      for (next=0; next<cur; next++) {
	/* DPRequest dpr; */
	ierr = DotProductsSet(dp,vecs[next],vecs[cur]); CHKERRQ(ierr);
      }
      for (next=0; next<cur; next++) {
	Scalar prod;
	ierr = DotProductsGet(dp,&prod); CHKERRQ(ierr);
	ierr = MatSetValues
	  (*u,1,&next,1,&cur,&prod,INSERT_VALUES); CHKERRQ(ierr);
	tmp[next] = -prod;
      }
      for (next=0; next<cur; next++) {
	ierr = VecAXPY(tmp+next,vecs[next],vecs[cur]); CHKERRQ(ierr);
      }
      ierr = VecDot(vecs[cur],vecs[cur],&norm); CHKERRQ(ierr);
      norm = sqrt(norm);
      ierr = MatSetValues(*u,1,&cur,1,&cur,&norm,INSERT_VALUES); CHKERRQ(ierr);
      norm = 1./norm;
      ierr = VecScale(&norm,vecs[cur]); CHKERRQ(ierr);
    }
    PetscFree(tmp);
  } else if (meth==MOD_GRAM_SCHMIDT) {
    for (cur=0; cur<n; cur++) {
      Scalar norm;
      ierr = VecDot(vecs[cur],vecs[cur],&norm); CHKERRQ(ierr);
      norm = sqrt(norm);
      ierr = MatSetValues(*u,1,&cur,1,&cur,&norm,INSERT_VALUES); CHKERRQ(ierr);
      norm = 1./norm;
      ierr = VecScale(&norm,vecs[cur]); CHKERRQ(ierr);
      for (next=cur+1; next<n; next++) {
	Scalar prod;
	ierr = VecDot(vecs[next],vecs[cur],&prod); CHKERRQ(ierr);
	ierr = MatSetValues(*u,1,&cur,1,&next,&prod,INSERT_VALUES); CHKERRQ(ierr);
	prod = -prod;
	ierr = VecAXPY(&prod,vecs[cur],vecs[next]); CHKERRQ(ierr);
      }
    }
  } else if (meth==GRAM_SCHMIDT_TWICE) {
    int dup1=0,dup2=0;
    if ((int)u1==PETSC_NULL) {
      dup1 = 1;
      ierr = MatDuplicate(*u,MAT_DO_NOT_COPY_VALUES,&u1); CHKERRQ(ierr);
    }
    if ((int)u2==PETSC_NULL) {
      dup2 = 1;
      ierr = MatDuplicate(*u,MAT_DO_NOT_COPY_VALUES,&u2); CHKERRQ(ierr);
    }
    ierr = VecsOrthogonalise(comm,vecs,n,use,&u1,GRAM_SCHMIDT); CHKERRQ(ierr);
    ierr = VecsOrthogonalise(comm,vecs,n,use,&u2,GRAM_SCHMIDT); CHKERRQ(ierr);
    ierr = MatMatMultDense(u2,u1,*u); CHKERRQ(ierr);
    if (dup1) {ierr = MatDestroy(u1); CHKERRQ(ierr);}
    if (dup2) {ierr = MatDestroy(u2); CHKERRQ(ierr);}
  }
  ierr = MatAssemblyBegin(*u,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  ierr = MatAssemblyEnd(*u,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  PLogEventEnd(events[ORTHO_EVENT],0,0,0,0);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsOrthogonalise"
int VecsOrthogonalise
(MPI_Comm comm,Vec *vecs,int n,MatReuse use,Mat *u,QRMethod meth)
{
  int ierr;
  Mat u1,u2;
  DotProducts dp;
  PetscFunctionBegin;
  ierr = DotProductsCreate(comm,n,&dp); CHKERRQ(ierr);
  if (meth==GRAM_SCHMIDT_TWICE) {
      ierr = MatDuplicate(*u,MAT_DO_NOT_COPY_VALUES,&u1); CHKERRQ(ierr);
      ierr = MatDuplicate(*u,MAT_DO_NOT_COPY_VALUES,&u2); CHKERRQ(ierr);
  }
  ierr = VecsOrthogonalise_O
    (comm,vecs,n,use,u,(Mat)PETSC_NULL,(Mat)PETSC_NULL,meth,dp); CHKERRQ(ierr);
  if (meth==GRAM_SCHMIDT_TWICE) {
    ierr = MatDestroy(u1); CHKERRQ(ierr);
    ierr = MatDestroy(u2); CHKERRQ(ierr);
  }
  ierr = DotProductsDestroy(dp); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsCopy"
int VecsCopy(Vec *ivecs,Vec *ovecs,int n)
{
  int i,ierr;

  PetscFunctionBegin;
  for (i=0; i<n; i++) {ierr = VecCopy(ivecs[i],ovecs[i]); CHKERRQ(ierr);}
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsScale"
int VecsScale(Mat a,Vec *vecs,int n)
{
  int is,js,col,row,ierr;
  Scalar *v;

  PetscFunctionBegin;
  ierr = MatGetSize(a,&is,&js); CHKERRQ(ierr);
  if (is>n) SETERRQ(1,0,"Scaling matrix bigger than vector row");
  ierr = MatGetArray(a,&v); CHKERRQ(ierr);
  for (col=0; col<js; col++) {
    ierr = VecScale(v+col*is+col,vecs[col]); CHKERRQ(ierr);
    for (row=0; row<is; row++)
      if (row!=col) {
	ierr = VecAXPY(v+col*is+row,vecs[row],vecs[col]); CHKERRQ(ierr);
      }
  }
  ierr = MatRestoreArray(a,&v); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsAXPY"
int VecsAXPY(Mat a,Vec *x,Vec *y,int n)
{
  int is,js,col,row,ierr;
  Scalar *v;

  PetscFunctionBegin;
  PLogEventBegin(events[AXPY_EVENT],0,0,0,0);
  ierr = MatGetSize(a,&is,&js); CHKERRQ(ierr);
  if (is>n) SETERRQ(1,0,"Scaling matrix bigger than vector row");
  ierr = MatGetArray(a,&v); CHKERRQ(ierr);
  for (col=0; col<js; col++) {
    ierr = VecAXPY(v+col*is+col,x[col],y[col]); CHKERRQ(ierr);
    for (row=0; row<is; row++)
      if (row!=col) {
	ierr = VecAXPY(v+col*is+row,x[row],y[col]); CHKERRQ(ierr);
      }
  }
  ierr = MatRestoreArray(a,&v); CHKERRQ(ierr);
  PLogEventEnd(events[AXPY_EVENT],0,0,0,0);
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsAYPX"
int VecsAYPX(Mat a,Vec *x,Vec *y,int n)
{
  int is,js,col,row,ierr;
  Scalar *v;

  PetscFunctionBegin;
  PLogEventBegin(events[AYPX_EVENT],0,0,0,0);
  ierr = MatGetSize(a,&is,&js); CHKERRQ(ierr);
  if (is>n) SETERRQ(1,0,"Scaling matrix bigger than vector row");
  ierr = MatGetArray(a,&v); CHKERRQ(ierr);
  for (col=0; col<js; col++) {
    ierr = VecAYPX(v+col*is+col,x[col],y[col]); CHKERRQ(ierr);
    for (row=0; row<is; row++)
      if (row!=col) {
	ierr = VecAXPY(v+col*is+row,y[row],y[col]); CHKERRQ(ierr);
      }
  }
  ierr = MatRestoreArray(a,&v); CHKERRQ(ierr);
  PLogEventEnd(events[AYPX_EVENT],0,0,0,0);
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsMatMult"
int VecsMatMult(Mat A,Vec *ivecs,Vec *ovecs,int n)
{
  int iv,ierr;

  PetscFunctionBegin;
  for (iv=0; iv<n; iv++) {
    ierr = MatMult(A,ivecs[iv],ovecs[iv]); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsPCApply"
int VecsPCApply(PC M,Vec *ivecs,Vec *ovecs,int n)
{
  int iv,ierr;

  PetscFunctionBegin;
  for (iv=0; iv<n; iv++) {
    ierr = PCApply(M,ivecs[iv],ovecs[iv]); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsDot"
int VecsDot(MPI_Comm comm,Vec *vecs1,Vec *vecs2,int n,MatReuse use,Mat *u)
{
  DotProducts dp;
  int ierr;

  PetscFunctionBegin;
  ierr = DotProductsCreate(comm,n,&dp); CHKERRQ(ierr);
  ierr = VecsDot_O(comm,vecs1,vecs2,n,use,u,dp); CHKERRQ(ierr);
  ierr = DotProductsDestroy(dp); CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsDot_O"
int VecsDot_O(MPI_Comm comm,Vec *vecs1,Vec *vecs2,int n,MatReuse use,Mat *u,
	      DotProducts dp)
{
  int i,j,ierr;

  PetscFunctionBegin;
  PLogEventBegin(events[DOT_EVENT],0,0,0,0);
  if (use==MAT_INITIAL_MATRIX) {
    ierr = MatCreateSeqDense(MPI_COMM_SELF,n,n,PETSC_NULL,u); CHKERRQ(ierr);
  } else if (use==MAT_REUSE_MATRIX) {
    ierr = MatZeroEntries(*u); CHKERRQ(ierr);
  } else SETERRQ(1,0,"Unknown MatReuse mode");

  for (i=0; i<n; i++)
    for (j=0; j<n; j++) {
      ierr = DotProductsSet(dp,vecs1[i],vecs2[j]); CHKERRQ(ierr);
    }
  for (i=0; i<n; i++)
    for (j=0; j<n; j++) {
      Scalar v;
      ierr = DotProductsGet(dp,&v); CHKERRQ(ierr);
      ierr = MatSetValues(*u,1,&i,1,&j,&v,INSERT_VALUES); CHKERRQ(ierr);
    }
  ierr = MatAssemblyBegin(*u,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  ierr = MatAssemblyEnd(*u,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  PLogEventEnd(events[DOT_EVENT],0,0,0,0);
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "VecsSet"
int VecsSet(Scalar *x,Vec *vecs,int n)
{
  int ierr,i;
  PetscFunctionBegin;
  for (i=0; i<n; i++) {
    ierr = VecSet(x,vecs[i]); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

