#include "petscpc.h"
#include "src/sles/pc/pcimpl.h"
#include "petscsles.h"
#include "parpre_pc.h"
#include "parpre_structure.h"
#include "parpre_subdomains.h"
#include "parpre_pipeline.h"
#include "subdomains_impl.h"
#include "petscviewer.h"

#undef __FUNC__
#define __FUNC__ "PCParallelDefaultGetLocalSLES"
static int PCParallelDefaultGetLocalSLES(PC pc,SLES *sub_method)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  *sub_method = subdomains->local_method;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSetLocalSLES"
int PCParallelSetLocalSLES(PC pc,SLES local_sles)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  subdomains->local_method = local_sles;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelGetLocalSLES"
int PCParallelGetLocalSLES(PC pc,SLES *local_sles)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  if (!subdomains->getlocalmethod)
    SETERRQ(1,0,"PCParallelGetLocalSLES: no method");
  ierr = (subdomains->getlocalmethod)(pc,local_sles); CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSetGetLocalSLES"
int PCParallelSetGetLocalSLES(PC pc,int(*getlocalmethod)(PC,SLES*))
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  subdomains->getlocalmethod = getlocalmethod;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSubdomainsSetPipeline"
int PCParallelSubdomainsSetPipeline(PC pc,VecPipeline pipe)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  subdomains->main_pipe = pipe;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSubdomainsHasPipeline"
int PCParallelSubdomainsHasPipeline(PC pc,int *flag)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  *flag = (int)(subdomains->main_pipe);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSubdomainsGetPipeline"
int PCParallelSubdomainsGetPipeline(PC pc,VecPipeline *pipe)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  /*printf("pc=%d, subdomain=%d, pipe=%d\n",
    (int)pc,(int)subdomains,(int)subdomains->main_pipe);*/
  if (!subdomains->main_pipe) SETERRQ(1,0,"no pipe found");
  *pipe = subdomains->main_pipe;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSubdomainPipelineSetType"
/*
   This general routine is necessary
   (instead of getting the pipeline pointer, and setting a parameter)
   because of the Schwarz methods which have an embedded GBSSOR method;
   the pipe needs to be set in both.*/
/*@
  PCParallelSubdomainPipelineSetType -
  Set the pipeline type of a parallel preconditioner.

  Parameters: 
+ pc - the preconditioner
. pipe_type - choice of PIPELINE_NONE, PIPELINE_SEQUENTIAL,
	      PIPELINE_REDBLACK, PIPELINE_MULTICOLOUR,
	      PIPELINE_USER
- x - an object used to determine the pipeline sequence.
  Most times this can be '(void*)PETSC_NULL', but for the multicolour
  pipe you have to pass the coefficient matrix as '(void*)A'.

  @*/
int PCParallelSubdomainPipelineSetType
(PC pc,PipelineType pipe_type,PetscObject x)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  if (!PCIsParallel(pc)) 
    SETERRQ(1,0,"Subdomain pipelines only for parallel methods\n");

  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  ierr = (subdomains->pipeline_settype)(pc,pipe_type,x);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelDefaultSubdomainPipelineSetType"
int PCParallelDefaultSubdomainPipelineSetType
(PC pc,PipelineType pipe_type,PetscObject x)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  subdomains->pipe_type = pipe_type;
  subdomains->pipe_obj = x;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSetSubdomainPipelineSetType"
int PCParallelSetSubdomainPipelineSetType
(PC pc,int(*settype)(PC,PipelineType,PetscObject x))
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  if (!PCIsParallel(pc)) 
    SETERRQ(1,0,"Subdomain pipelines only for parallel methods\n");

  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  subdomains->pipeline_settype = settype;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSubdomainPipelineGetType"
/*@
  PCParallelSubdomainPipelineGetType - get the pipeline type

  Parameters:
+ pc - the preconditioner
- pipe_type - the type of the pipeline
@*/
int PCParallelSubdomainPipelineGetType(PC pc,PipelineType *pipe_type)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  if (!PCIsParallel(pc)) 
    SETERRQ(1,0,"Subdomain pipelines only for parallel methods\n");

  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  if (!subdomains->pipeline_gettype)
    SETERRQ(1,0,"No pipeline type has been declared");
  ierr = (subdomains->pipeline_gettype)(pc,pipe_type);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelDefaultSubdomainPipelineGetType"
int PCParallelDefaultSubdomainPipelineGetType(PC pc,PipelineType *pipe_type)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  *pipe_type = subdomains->pipe_type;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSetSubdomainPipelineGetType"
int PCParallelSetSubdomainPipelineGetType
(PC pc,int(*gettype)(PC,PipelineType*))
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  if (!PCIsParallel(pc)) 
    SETERRQ(1,0,"Subdomain pipelines only for parallel methods\n");

  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  subdomains->pipeline_gettype = gettype;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSubdomainPipelineGetObject"
/*@
  PCParallelSubdomainPipelineGetObject - get the object that was
  used to generate the pipeline

  Parameters:
+ pc - the preconditioner
- pipe_obj - the object
@*/
int PCParallelSubdomainPipelineGetObject(PC pc,PetscObject *pipe_obj)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  if (!PCIsParallel(pc)) 
    SETERRQ(1,0,"Subdomain pipelines only for parallel methods\n");

  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  if (!subdomains->pipeline_getobject)
    SETERRQ(1,0,"PCParallelSubdomainPipelineGetObject: no method");
  ierr = (subdomains->pipeline_getobject)(pc,pipe_obj);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelDefaultSubdomainPipelineGetObject"
int PCParallelDefaultSubdomainPipelineGetObject(PC pc,PetscObject *pipe_obj)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  *pipe_obj = subdomains->pipe_obj;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSetSubdomainPipelineGetObject"
int PCParallelSetSubdomainPipelineGetObject
(PC pc,int(*getobject)(PC,PetscObject*))
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  if (!PCIsParallel(pc)) 
    SETERRQ(1,0,"Subdomain pipelines only for parallel methods\n");

  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  subdomains->pipeline_getobject = getobject;

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSubdomainsCreate"
int PCParallelSubdomainsCreate(PC pc,int size)
{
  int ierr;
  PCParallelStruct *pc_data;

  PetscFunctionBegin;
  pc_data = (PCParallelStruct *) PetscMalloc(size);
  CHKPTRQ(pc_data);
  pc->data = (void *) pc_data;

  {
    PCParallelSubdomainStruct subdomains = (PCParallelSubdomainStruct)
      PetscMalloc(sizeof(struct _PCParallelSubdomainStruct));
    CHKPTRQ(subdomains);
    ierr = PCParallelSetSubdomains(pc,subdomains); CHKERRQ(ierr);
    subdomains->main_pipe = 0;
  }

  ierr = PCParallelSetGetLocalSLES
    (pc,&PCParallelDefaultGetLocalSLES);
  CHKERRQ(ierr);
  ierr = PCParallelSetSubdomainPipelineSetType
    (pc,&PCParallelDefaultSubdomainPipelineSetType);
  CHKERRQ(ierr);
  ierr = PCParallelSetSubdomainPipelineGetType
    (pc,&PCParallelDefaultSubdomainPipelineGetType);
  CHKERRQ(ierr);
  ierr = PCParallelSetSubdomainPipelineGetObject
    (pc,&PCParallelDefaultSubdomainPipelineGetObject);
  CHKERRQ(ierr);

  {
    SLES local_method;
    ierr = ParPreGenerateSLES(MPI_COMM_SELF,&local_method); CHKERRQ(ierr);
    ierr = PCParallelSetLocalSLES(pc,local_method);
  }

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelDestroySubdomains"
int PCParallelDestroySubdomains(PC pc)
{
  PCParallelSubdomainStruct subdomains;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetSubdomains(pc,&subdomains); CHKERRQ(ierr);
  ierr = SLESDestroy(subdomains->local_method); CHKERRQ(ierr);
  if (subdomains->main_pipe) {
    ierr = VecPipelineDestroy(subdomains->main_pipe); CHKERRQ(ierr);
  }
  PetscFree(subdomains);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelSetSubdomainSystem"
int PCParallelSetSubdomainSystem(PC pc,Mat mat, Vec vec)
{
  SLES sub_method;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetLocalSLES(pc,&sub_method); CHKERRQ(ierr);
  ierr = ParPreSetupSLES(sub_method,PETSC_NULL,mat,vec); CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelGetLocalPC"
/*@
  PCParallelGetLocalPC - get the preconditioner that is applied
  as subdomain solver

  Parameters:
+ pc - the preconditioner
- local_pc - the subdomain solver preconditioner
@*/
int PCParallelGetLocalPC(PC pc,PC *local_pc)
{
  SLES sles;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetLocalSLES(pc,&sles); CHKERRQ(ierr);
  ierr = SLESGetPC(sles,local_pc); CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCParallelLocalSolveSetFromOptions"
int PCParallelLocalSolveSetFromOptions(PC pc)
{
  SLES sles;
  char *prefix;
  int ierr;

  PetscFunctionBegin;
  ierr = PCParallelGetLocalSLES(pc,&sles); CHKERRQ(ierr);
  ierr = PCGetOptionsPrefix(pc,&prefix); CHKERRQ(ierr);
  ierr = SLESSetOptionsPrefix(sles,prefix); CHKERRQ(ierr);
  ierr = SLESAppendOptionsPrefix(sles,"sub_"); CHKERRQ(ierr);
  ierr = SLESSetFromOptions(sles); CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "PCSubdomainsView"
int PCSubdomainsView(PC pc,Viewer viewer)
{
  VecPipeline pipe;
  SLES sles;
  int ierr,flag;

  PetscFunctionBegin;
  PetscPrintf(pc->comm,">> Subdomain method\n");
  ierr = PCParallelGetLocalSLES(pc,&sles); CHKERRQ(ierr);
  ierr = SLESView(sles,viewer); CHKERRQ(ierr);
  ierr = PCParallelSubdomainsHasPipeline(pc,&flag); CHKERRQ(ierr);
  if (flag) {
    PetscPrintf(pc->comm,"-- Subdomains pipeline\n");
    ierr = PCParallelSubdomainsGetPipeline(pc,&pipe); CHKERRQ(ierr);
    ierr = VecPipelineView(pipe,viewer); CHKERRQ(ierr);
  } else PetscPrintf(pc->comm,"Subdomains have no pipeline\n");

  PetscFunctionReturn(0);
}
