#include "petsc.h"
#include "mpi.h"
#include "petscis.h"
#include "petscmat.h"
#include "src/mat/parpre_matimpl.h"
#include "src/mat/impls/aij/mpi/mpiaij.h"
#include "./ml_head.h"

/****************************************************************
 * Colouring and finding independent sets                       *
 * None of these use the multilevel data structure              *
 ****************************************************************/

#undef __FUNC__
#define __FUNC__ "FreeClr"
static int FreeClr(int nn,int *n)
{
  int ret=1,i;
 attempt:
  for (i=0; i<nn; i++)
    if (n[i]==ret) {ret++; goto attempt;}
  return ret;
}

#undef __FUNC__
#define __FUNC__ "LookAtThisRow"
static int LookAtThisRow
(int this_var, Scalar this_rand,int *idx,int nidx, int *colour_now,
 Scalar *ran,Scalar *clr,int *n,int *neigh)
{
  int j,other;

  for (j=0; j<nidx; j++) {
    other=idx[j];
    if (other==this_var) continue;
#if defined(USE_PETSC_COMPLEX)
    if (PetscReal(ran[other])>PetscReal(this_rand)) {
      if (PetscReal(clr[other])==0.) {
#else
    if (ran[other]>this_rand) {
      if (clr[other]==0.) {
#endif
	*colour_now = 0;
	return 0;
      } else {
#if defined(USE_PETSC_COMPLEX)
	neigh[(*n)++] = (int)PetscReal(clr[other]);
#else
	neigh[(*n)++] = (int)clr[other];
#endif
      }
    }
    }
  return 0;
}

#undef __FUNC__
#define __FUNC__ "LookAtUncolouredVar"
static int LookAtUncolouredVar
(int this_var,Mat A,Mat B, int *colour_now,int has_bord,
 Scalar *clr_array,Scalar *clrb_array,Scalar *ran_array,Scalar *ranb_array,
 int *neighb,int *clr)
{
  int nzA,nzB, *pcA,*pcB, nneigh=0, ierr;

  ierr = MatGetRow(A,this_var,&nzA,&pcA,PETSC_NULL); CHKERRQ(ierr);
  ierr = LookAtThisRow
    (this_var,ran_array[this_var], pcA,nzA, colour_now,ran_array,clr_array,
     &nneigh,neighb);
  CHKERRQ(ierr);
  /*  printf("Var %d, row %d, ",this_var,nzA);*/
  ierr = MatRestoreRow(A,this_var,&nzA,&pcA,0); CHKERRQ(ierr);
  
  if (*colour_now && has_bord) {
    ierr = MatGetRow(B,this_var,&nzB,&pcB,PETSC_NULL); CHKERRQ(ierr);
    if (nzB) {
      ierr = LookAtThisRow
	(-1,ran_array[this_var], pcB,nzB, colour_now,ranb_array,clrb_array,
	 &nneigh,neighb);
      CHKERRQ(ierr);
    }
    /*    printf("%d, ",nzB);*/
    ierr = MatRestoreRow(B,this_var,&nzB,&pcB,0); CHKERRQ(ierr);
  }
  *clr = FreeClr(nneigh,neighb);
  /*  printf("nn=%d, now=%d, clr=%d\n",nneigh,*colour_now,*clr);*/
  /*  if (nzB>10) MatView(B,0);*/
  return 0;
}

#undef __FUNC__
#define __FUNC__ "FindIndSet"
static int FindIndSet
(Mat mat,Vec randoms,Vec clrs,IS *indp_vars,IS *rest_vars) 
{
  Mat_MPIAIJ *aij = (Mat_MPIAIJ *) mat->data;
  Scalar *ran_array,*ranb_array=0, *clr_array,*clrb_array=0, zero=0.0;
  Vec ran_bord = 0,clr_bord = 0;
  int *neighb, rstart,rend,local_size,var,coloured, ierr;
  int has_bord = (int)aij->lvec; /* single processor case */

  if (has_bord) {
    ierr = VecDuplicate(aij->lvec,&ran_bord); CHKERRQ(ierr);
    clr_bord = aij->lvec;
  }
  {
    int rl;
    ierr = MatMaxRowLen_MPIAIJ(mat,&rl); CHKERRQ(ierr);
    neighb = (int *) PetscMalloc(rl*sizeof(int)); CHKPTRQ(neighb);
  }
  /*>>>> Loop until completely coloured <<<<*/
  if (has_bord) {
    ierr = VecScatterBegin
      (randoms,ran_bord,INSERT_VALUES,SCATTER_FORWARD,aij->Mvctx);
    CHKERRQ(ierr);
    ierr = VecScatterEnd
      (randoms,ran_bord,INSERT_VALUES,SCATTER_FORWARD,aij->Mvctx);
    CHKERRQ(ierr);
  }
  ierr = VecGetOwnershipRange(randoms,&rstart,&rend); CHKERRQ(ierr);
  local_size = rend-rstart;
  ierr = VecGetArray(randoms,&ran_array); CHKERRQ(ierr);
  if (has_bord) {
    ierr = VecGetArray(ran_bord,&ranb_array); CHKERRQ(ierr);
  }

  coloured = 0;
  ierr = VecSet(&zero,clrs); CHKERRQ(ierr);
  ierr = VecSet(&zero,clr_bord); CHKERRQ(ierr);
 pass:
  {
    int l_rem,g_rem;
    ierr = VecGetArray(clrs,&clr_array); CHKERRQ(ierr);
    if (has_bord) {
      ierr = VecGetArray(clr_bord,&clrb_array); CHKERRQ(ierr);
    }
    for (var=0; var<local_size; var++) {
#if defined(USE_PETSC_COMPLEX)
      if (!PetscReal(clr_array[var])) {
#else
      if (!clr_array[var]) {
#endif
	int colour_now = 1,clr;
	ierr = LookAtUncolouredVar
	  (var,aij->A,aij->B,&colour_now,has_bord,clr_array,clrb_array,
	   ran_array,ranb_array,neighb,&clr); CHKERRQ(ierr);
	if (colour_now) {
	  coloured++; clr_array[var] = (Scalar) clr;
	}
      }
    }
    ierr = VecRestoreArray(clrs,&clr_array); CHKERRQ(ierr);
    if (has_bord) {
      ierr = VecRestoreArray(clr_bord,&clrb_array); CHKERRQ(ierr);
    }
    l_rem = local_size-coloured;
    MPI_Allreduce(&l_rem,&g_rem,1,MPI_INT,MPI_MAX,mat->comm);
    if (!g_rem) goto finished;
    if (g_rem<0) SETERRQ(1,0,"Cannot happen: negative points to colour");
    if (has_bord) {
      ierr = VecScatterBegin
	(clrs,clr_bord,INSERT_VALUES,SCATTER_FORWARD,aij->Mvctx);
      CHKERRQ(ierr);
      ierr = VecScatterEnd
	(clrs,clr_bord,INSERT_VALUES,SCATTER_FORWARD,aij->Mvctx);
      CHKERRQ(ierr);
    }
    goto pass;
  }

 finished:
  ierr = VecRestoreArray(randoms,&ran_array); CHKERRQ(ierr);
  if (has_bord) {
    ierr = VecRestoreArray(ran_bord,&ranb_array); CHKERRQ(ierr);
    ierr = VecDestroy(ran_bord); CHKERRQ(ierr);
  }
  {
    int n_coloured=0,n_not_coloured=0;
    int *points_coloured,*points_not_coloured;
    
    for (var=0; var<local_size; var++)
#if defined(USE_PETSC_COMPLEX)
      if ((int)PetscReal(clr_array[var])==1)
#else
      if ((int)clr_array[var]==1)
#endif
	n_coloured++; else n_not_coloured++;

    points_coloured = (int *) PetscMalloc((n_coloured+1)*sizeof(int));
    CHKPTRQ(points_coloured);
    points_not_coloured = (int *) PetscMalloc((n_not_coloured+1)*sizeof(int));
    CHKPTRQ(points_not_coloured);

    n_coloured = 0; n_not_coloured = 0;
    for (var=0; var<local_size; var++)
#if defined(USE_PETSC_COMPLEX)
      if ((int)PetscReal(clr_array[var])==1)
#else
      if ((int)clr_array[var]==1)
#endif
	points_coloured[n_coloured++] = var;
      else
	points_not_coloured[n_not_coloured++] = var;

    for (var=0; var<n_coloured; var++)
	points_coloured[var] += rstart;
    ierr = ISCreateGeneral
      (mat->comm,n_coloured,points_coloured,indp_vars);
    CHKERRQ(ierr); PetscFree(points_coloured);

    for (var=0; var<n_not_coloured; var++)
	points_not_coloured[var] += rstart;
    ierr = ISCreateGeneral
      (mat->comm,n_not_coloured,points_not_coloured,rest_vars);
    CHKERRQ(ierr); PetscFree(points_not_coloured);
  }
  
  return 0;
}

#undef __FUNC__
#define __FUNC__ "FakeIndset"
static int FakeIndset(Mat mat,int isize,int jsize,int lvl,
		      IS *indp_vars,IS *rest_vars)
{
  MPI_Comm comm = mat->comm;
  int istep,jstep,rstart,rend,ierr;
  int *points_coloured,*points_not_coloured, n_coloured=0,n_not_coloured=0;

  points_coloured = (int *) PetscMalloc((isize*jsize+1)*sizeof(int));
  points_not_coloured = (int *) PetscMalloc((isize*jsize+1)*sizeof(int));
  ierr = MatGetOwnershipRange(mat,&rstart,&rend); CHKERRQ(ierr);
  for (istep=0; istep<isize; istep++)
    for (jstep=0; jstep<jsize; jstep++) {
      int num = istep*jsize+jstep;
      
      if ( (num<rend) & !(num<rstart)) {
	if (lvl-2*(lvl/2)==0) {
	  if (istep-2*(istep/2)+jstep-2*(jstep/2)!=1) {
	    points_coloured[n_coloured++] = num;
	  } else {
	    points_not_coloured[n_not_coloured++] = num;
	  }
	} else {
	  if (istep-2*(istep/2)==0) {
	    points_coloured[n_coloured++] = num;
	  } else {
	    points_not_coloured[n_not_coloured++] = num;
	  }
	}
      }
    }
  ierr = ISCreateGeneral(comm,n_coloured,points_coloured,indp_vars);
  CHKERRQ(ierr); PetscFree(points_coloured);
  ierr = ISCreateGeneral(comm,n_not_coloured,points_not_coloured,rest_vars);
  CHKERRQ(ierr); PetscFree(points_not_coloured); CHKERRQ(ierr);
  
  return 0;
}

#undef __FUNC__
#define __FUNC__ "SplitIndAndRest"
static int SplitIndAndRest(Mat mat,Vec u,Vec v,int level,
			   IS *ind,IS *rest,
			   int orth,int isize,int jsize)
{/* create random vector, derive one independent set */
  IS ind_set,rest_set;
  PetscRandom rctx;
  int ierr;
  
  if (orth) {
    ierr = FakeIndset(mat,isize,jsize,level,&ind_set,&rest_set); CHKERRQ(ierr);
  } else {
    ierr = PetscRandomCreate
      (MPI_COMM_SELF,RANDOM_DEFAULT,&rctx); CHKERRQ(ierr);
    ierr = VecSetRandom(rctx,u); CHKERRQ(ierr);
    ierr = FindIndSet(mat,u,v,&ind_set,&rest_set); CHKERRQ(ierr);
    ierr = PetscRandomDestroy(rctx); CHKERRQ(ierr);
  }

  *ind = ind_set; *rest = rest_set;

  return 0;
}

#undef __FUNC__
#define __FUNC__ "SplitOffOneLevel"
int SplitOffOneLevel
(Mat mat,Vec u,Vec v,int level,
 IS *ind,IS *rest,int *all_zero,int *one_zero,
 int orth,int isize,int jsize)
{
  MPI_Comm comm = mat->comm;
  int local_i,local_r,ierr;

  PetscFunctionBegin;
  ierr = SplitIndAndRest(mat,u,v,level,ind,rest,
			 orth,isize,jsize); CHKERRQ(ierr);
  ierr = ISGetSize(*ind,&local_i); CHKERRQ(ierr);
  ierr = ISGetSize(*rest,&local_r); CHKERRQ(ierr);

  {
    int global_rest;
    MPI_Allreduce(&local_r,&global_rest,1,MPI_INT,MPI_SUM,comm);
    *all_zero = !global_rest;
  }
  {
    int i_have_local_r,we_have_local_r;
    if (local_r) i_have_local_r = 1; else i_have_local_r = 0;
    MPI_Allreduce(&i_have_local_r,&we_have_local_r,1,
		  MPI_INT,MPI_PROD,comm);
    *one_zero = !we_have_local_r;
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__
#define __FUNC__ "SetFandCsets"
int SetFandCsets(MPI_Comm comm,int lvl,int trace,
		 IS ind,IS rest,IS *set1,IS *set2,
		 AMLCoarseGridChoice grid_choice,int all_zero)
{
  int ierr;

  PetscFunctionBegin;
  if ( (grid_choice==AMLCoarseGridDependent) || all_zero ) {
    *set1 = ind; *set2 = rest;
  } else {
    *set1 = rest; *set2 = ind;
  }
  if (trace & AMLTraceColours) {
    int s1,s2,ss1,ss2;
    ierr = ISGetSize(*set1,&s1); CHKERRQ(ierr);
    ierr = ISGetSize(*set2,&s2); CHKERRQ(ierr);
    MPI_Reduce(&s1,&ss1,1,MPI_INT,MPI_SUM,0,comm);
    MPI_Reduce(&s2,&ss2,1,MPI_INT,MPI_SUM,0,comm);
    PetscPrintf(comm,
		"Colour %d has %d points, %d points remaining\n",
		lvl,ss1,ss2);
  }
  PetscFunctionReturn(0);
}
