C#######################################################################
C PSTSWM Version 4.0 (12/1/94)                                         #
C  A message-passing benchmark code and parallel algorithm testbed     #
C  that solves the nonlinear shallow water equations using the spectral#
C  transform method.                                                   #
C Written by:                                                          #
C  Patrick Worley of Oak Ridge National Laboratory                     #
C  Ian Foster of Argonne National Laboratory                           #
C Based on the sequential code STSWM 2.0 by James Hack and Ruediger    #
C  Jakob of the National Center for Atmospheric Research.              #
C Research and development funded by the Computer Hardware, Advanced   #
C  Mathematics, and Model Physics (CHAMMP) program of the U.S.         #
C  Department of Energy.                                               # 
C                                                                      #
C Questions and comments should be directed to worley@msr.epm.ornl.gov #
C Please notify and acknowledge the authors in any research or         #
C publications utilizing PSTSWM or any part of the code.               #
C                                                                      #
C NOTICE: Neither the institutions nor the authors make any            #
C representations about the suitability of this software for any       #
C purpose. This software is provided "as is", without express or       #
C implied warranty.                                                    #
C#######################################################################
C include precision declaration definitions                            #
#include "precision.i"
C#######################################################################
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C mpi.F                                                                C
C                                                                      C
C The following routines support the use of MPI nonblocking and        C
C buffered communication routines in swap.F and sendrecv.F. Where      C
C possible, these should be in-lined for efficiency.                   C
C                                                                      C
C COMMINIT - used to initialize any data structures used by native     C
C            implementations of sendrecv and swap.                     C
C COMMEXIT - used to clean up after use of native commands in PSTSWM   C
C ADDSEND - used to add a nonblocking send request id to the list of   C
C           outstanding requests.                                      C
C RMVSEND - used to remove a nonblocking send request id from the list C
C           of outstanding requests.                                   C
C ADDRECV - used to add a nonblocking receive request id to the list   C
C           of outstanding requests.                                   C
C RMVRECV - used to remove a nonblocking receive request id from the   C
C           list of outstanding requests.                              C
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C                                                                      C
      SUBROUTINE COMMINIT(SYSMSGS, SYSVOL, COMBUF, BUFSIZ)
C
C This subroutine initializes the data structures used to support the
C nonblocking communication options in swap.F and sendrecv.F in the
C native MPI implemenations. 
C
C called by: PSTSWM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
C mpi implementation-specific parameters
#     include "mpif.h"
C pstswm-specific parameters
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C machine architecture information
#     include "machine.i"
C definition for communicator and datatype for native MPI implementation
#     include "picl.i"
C data and data structures for native MPI implementation
#     include "mpi.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C number of blocking asynchronous send requests that the system needs 
C to be able to handle 
      INTEGER SYSMSGS
C size (in bytes) of blocking asynchronous send requests that the system 
C needs to be able to handle 
      INTEGER SYSVOL
C size of user buffer space
      INTEGER BUFSIZ
C User space to use for communication buffering 
      REAL COMBUF(BUFSIZ)
C
C---- Local Variables --------------------------------------------------
C
C index variable
      INTEGER I
C error return variable
      INTEGER IERR
C
C---- Executable Statements --------------------------------------------
C
C     Provide user buffer for buffered communication
      CALL MPI_BUFFER_ATTACH(COMBUF, BUFSIZ*RBYTES, IERR)
      IF (BUFSIZ*RBYTES .LT. SYSVOL) THEN
        IF (ME .EQ. 0) THEN
          WRITE(6,1) BUFSIZ*RBYTES, SYSVOL
 1        FORMAT(/,' -- warning warning warning warning -- ',/,
     & ' USER BUFFER SIZE INSUFFICIENT FOR SYSTEM BUFFER ',/,
     & ' REQUIREMENTS. RUN MAY DEADLOCK.',/,
     & ' USER SPACE: ',I10,' BUFFER REQUIREMENT: ',I10,/,
     &     ' -- warning warning warning warning -- ',/)
        ENDIF
      ENDIF

C     Initialize data structures used to hold nonblocking communication
C     request IDs.
      FIRSTRCV = 0
      NXTRCV   = 0
      FIRSTSND = 0
      NXTSND   = 0
      DO I = 0,NPROCSX-1
       RCVIDS(I) = -1
       RCVTAGS(I) = -2
       RCVSRCS(I) = -2
       SNDIDS(I) = -1
       SNDTAGS(I) = -2
       SNDDSTS(I) = -2
      ENDDO
C
      RETURN
C
      END
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C                                                                      C
      SUBROUTINE COMMEXIT
C
C This subroutine does any clean up needed after using native commands
C commands in PSTSWM. 
C
C called by: PSTSWM
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Local Variables --------------------------------------------------
C
C Pointer to user space used for communication buffering 
      INTEGER DUMMY
C size of returned user buffer space
      INTEGER BUFSIZ
C error return variable
      INTEGER IERR
C
C---- Executable Statements --------------------------------------------
C
C     Clean-up and "return" user buffer used for buffered communication
      CALL MPI_BUFFER_DETACH(DUMMY, BUFSIZ, IERR)
C
      RETURN
C
      END
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
      SUBROUTINE ADDSEND(SNDID, SNDTAG, SNDDST)
C
C This subroutine adds a nonblocking send request id and the associated
C message tag to the list of outstanding requests.
C
C called by: SRBEGIN, SR2
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C data and data structures for native MPI implementation
#     include "mpi.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C nonblocking send request id
      INTEGER SNDID
C associated message tag value
      INTEGER SNDTAG
C associated message destination
      INTEGER SNDDST
C
C---- Executable Statements --------------------------------------------
C
C     If there is space, add to the list
      IF (SNDIDS(NXTSND) .EQ. -1) THEN
        SNDIDS(NXTSND) = SNDID
        SNDTAGS(NXTSND) = SNDTAG
        SNDDSTS(NXTSND) = SNDDST
        NXTSND = MOD(NXTSND+1,NPROCSX)
      ELSE
        WRITE (0,101)
  101   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE ADDSEND:',
     &         /,' NOT ENOUGH SPACE FOR NONBLOCKING SEND REQUEST IDS')
        STOP                                                   
      ENDIF
C
      RETURN
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      INTEGER FUNCTION RMVSEND(SNDTAG, SNDDST)
C
C This subroutine finds and returns a nonblocking send request id that
C matches the indicated message tag, removing it from the list of 
C outstanding requests in the process.
C
C called by: SREND, SR3
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C data and data structures for native MPI implementation
#     include "mpi.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C message tag value
      INTEGER SNDTAG
C message destination
      INTEGER SNDDST
C
C---- Local Variables --------------------------------------------------
C
C index variable
      INTEGER I
C
C---- Executable Statements --------------------------------------------
C
C     Find first id associated with indicated message tag.
      I = FIRSTSND
      DO WHILE ((I .NE. NXTSND) .AND. 
     &          ((SNDTAGS(I) .NE. SNDTAG) .OR. (SNDDSTS(I) .NE. SNDDST)
     &                                    .OR. (SNDIDS(I) .EQ. -1)))
        I = MOD(I+1,NPROCSX)
      ENDDO
C     If found anything, return it.
      IF (I .NE. NXTSND) THEN
        RMVSEND = SNDIDS(I)
        SNDIDS(I) = -1
C       Update "first" (oldest id) pointer.
        DO WHILE ((FIRSTSND .NE. NXTSND) .AND. 
     &            (SNDIDS(FIRSTSND) .EQ. -1))
          FIRSTSND = MOD(FIRSTSND+1,NPROCSX)
        ENDDO
      ELSE
        WRITE (0,101) SNDTAG, FIRSTSND, NXTSND
  101   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE RMVSEND:',
     &         /,' COULD NOT FIND OUTSTANDING SEND REQUEST WITH',
     &         /,' INDICATED MESSAGE TAG',
     &         /,' SNDTAG = ',I10,' FIRSTSND = ',I10,' NXTSND = ',I10)
        STOP                                                   
      ENDIF
C
      RETURN
      END
C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      SUBROUTINE ADDRECV(RCVID, RCVTAG, RCVSRC)
C
C This subroutine adds a nonblocking send request id and the associated
C message tag to the list of outstanding requests.
C
C called by: SRBEGIN, SR1
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C data and data structures for native MPI implementation
#     include "mpi.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C nonblocking send request id
      INTEGER RCVID
C associated message tag value
      INTEGER RCVTAG
C associated message source
      INTEGER RCVSRC
C
C---- Executable Statements --------------------------------------------
C
C     If there is space, add to the list
      IF (RCVIDS(NXTRCV) .EQ. -1) THEN
        RCVIDS(NXTRCV) = RCVID
        RCVTAGS(NXTRCV) = RCVTAG
        RCVSRCS(NXTRCV) = RCVSRC
        NXTRCV = MOD(NXTRCV+1,NPROCSX)
      ELSE
        WRITE (0,101)
  101   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE ADDRECV:',
     &         /,' NOT ENOUGH SPACE FOR NONBLOCKING RECEIVE REQUEST',
     &         /,' IDS')
        STOP                                                   
      ENDIF
C
      RETURN
      END
C                                                                      
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
C
      INTEGER FUNCTION RMVRECV(RCVTAG, RCVSRC)
C
C This subroutine finds and returns a nonblocking receive request id 
C that matches the indicated message tag, removing it from the list of 
C outstanding requests in the process.
C
C called by: SREND, SR2, SR3
C
C---- Implicit None ----------------------------------------------------
C
      IMPLICIT NONE
C
C---- Parameters -------------------------------------------------------
C
#     include "params.i"
C
C---- Common Blocks ----------------------------------------------------
C
C data and data structures for native MPI implementation
#     include "mpi.i"
C
C---- Arguments --------------------------------------------------------
C
C     Input 
C
C message tag value
      INTEGER RCVTAG
C message source
      INTEGER RCVSRC
C
C---- Local Variables --------------------------------------------------
C
C index variable
      INTEGER I
C
C---- Executable Statements --------------------------------------------
C
C     Find first id associated with indicated message tag.
      I = FIRSTRCV
      DO WHILE ((I .NE. NXTRCV) .AND. 
     &          ((RCVTAGS(I) .NE. RCVTAG) .OR. (RCVSRCS(I) .NE. RCVSRC)
     &                                    .OR. (RCVIDS(I) .EQ. -1)))
        I = MOD(I+1,NPROCSX)
      ENDDO
C     If found anything, return it.
      IF (I .NE. NXTRCV) THEN
        RMVRECV = RCVIDS(I)
        RCVIDS(I) = -1
C       Update "first" (oldest id) pointer.
        DO WHILE ((FIRSTRCV .NE. NXTRCV) .AND. 
     &            (RCVIDS(FIRSTRCV) .EQ. -1))
          FIRSTRCV = MOD(FIRSTRCV+1,NPROCSX)
        ENDDO
      ELSE
        WRITE (0,101) RCVTAG, FIRSTRCV, NXTRCV
  101   FORMAT(/,' PSTSWM: FATAL ERROR IN SUBROUTINE RMVRECV:',
     &         /,' COULD NOT FIND OUTSTANDING RECEIVE REQUEST WITH',
     &         /,' INDICATED MESSAGE TAG',
     &         /,' RCVTAG = ',I10,' FIRSTRCV = ',I10,' NXTRCV = ',I10)
        STOP                                                   
      ENDIF
C
      RETURN
      END
C                                                                      C
CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCC
