% Demo script for Acceleration Method
% AMdemo 
% This script reproduces the Examples 1 to 12 of the paper by using the
% acceleration method and the Simplified Topological Epsilon-Algorithms
% STEA1 and STEA2.
% It needs of the functions exinit, exhelp and Fiter.

% Authors: Claude Brezinski, Michela Redivo-Zaglia
% V1.0 - December 2016
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clearSEAW
clear variables
close all

% Choice of the example
disp(' Examples of the demo:');
disp(' 1, 2, 3, 4a, 4b, 5, 6, 7, 8a, 8b, 9, 10, 11, 12')
disp(' ');
swi = input(' Example number ','s');
swi = strtok(swi);
if isletter(swi(end))
    swin = str2double(swi(1:end-1));
else
    swin = str2double(swi);
end
 
% Check for the example chosen 
if ((swin == 4) || (swin == 8)) && isletter(swi(end)) || ...
        ((~isempty(swin)) && ismember(swin,[1,2,3,5,6,7,9,10,11,12])) 
    exhelp(swi,0);
else
    error (' AMdemo - The example chosen do not exist');
end
   
% Choice of the alpha (only for examples 1 to 5) 
if (swin > 0) && (swin <6)
    al=input(' Insert alpha ');
    if (isempty(al)) 
        error (' AMdemo - An alpha must be given for this example! ');
    end
else 
    al = 0;
end

% Initializations for the example chosen
% X is the starting element u_0
[m,Y,X,sol,param] = exinit(swi,al);  

% For Examples 8a and 8b we have to initialize the persistent value Bn
%   in the function Fiter
% For the Examples 9 and 10 we have to initialize the persistent values 
%   nterm, psum and, for Example 9, also ind
% For the Example 11 we have to initialize the persistent values nterm
if (swin == 8) || (swin == 9) || (swin == 10) || (swin == 11)
    [~,~] = Fiter(X,param,swi,0);
end

disp([' Dimension of the system m = ',num2str(m)])
% Choice of the max even column
MAXCOL = input(' Insert max even column '); 

% Check the MAXCOL value
if mod(MAXCOL,2) ~= 0
    error (' AMdemo - The max column must be an even integer');
end

% Choice of the number of iterations
disp([' The number of iterations must be greater or equal to ', ...
    num2str(MAXCOL+1),' to reach the column ', num2str(MAXCOL)])
NBC=input(' Insert the number of iterations ');

% Check the NBC value
if NBC < MAXCOL+1
    warning([' The number of terms must be greater or equal to ', ...
    num2str(MAXCOL+1),' to reach the column ', num2str(MAXCOL)]);
end

% Assign the values for the scalar epsilon-algorithm
% Tolerance for denominators 
TOL = 1e-30;
% Number of common digit asked for detecting a singularity
NDIGIT = 15; 

% Assign the tolerance for the maximun absolute value of the
% duality product
MAXD = 1e+20;

disp([' Tolerance for scalar epsilon algorithm  = ',num2str(TOL,'%6.1e')]);
disp([' Number of the common digits             = ',num2str(NDIGIT,'%d')]);
disp([' Maximum for the |duality product|       = ',num2str(MAXD,'%6.1e')]);
disp(' ');

% Initialize the variables for total number of singularities
TNS1 = 0;
TNS2 = 0;

% Define the type and colors for the graphical line STEA1
col = 'r--';
% Apply STEA1 and STEA2 to the example chosen
for swist=1:2
    stean = ['STEA', num2str(swist,'%d')];
    disp(['*** ',stean]);
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Arguments and initializations for the first call of the SEAW and STEA

    % First value <y,x_0>
    EPSINIS = trace(Y'*X);

    % Check its value
    if abs(EPSINIS) > MAXD
       warning([' AMdemo - At iteration 1 the absolute value of the ', ...
           'duality product is ' ,num2str(abs(EPSINIS),'%10.3e'),' > ', ...
           num2str(MAXD,'%15.5e')]);
    end
        
    % First call of the scalar epsilon-algorithm  
    [EPSINIS,EPSSCA,NSING] = SEAW(EPSINIS,[],MAXCOL,TOL,NDIGIT,0);
   
    % First call (initialization) of STEA algorithm with the first 
    % element x_0
    switch swist
        case 1
            [EPSINI,EPSVEC] = STEA1_3(X,[],EPSSCA,MAXCOL,TOL,0); 
        case 2
            [EPSINI,EPSVEC] = STEA2_3(X,[],EPSSCA,MAXCOL,TOL,0); 
    end
    disp(['Iteration 1 completed']);
    
    % Start of the loop   
    for i = 1:NBC-1    

        % Define the components needed for the graphical representations     
        if ~isempty(sol) % If solution is known
            nerrX(i)   = norm(sol-X);                    % |x - sol|
            nerrEPS(i) = norm(sol-EPSINI);               % |eps - sol|
        end
        
        % In output, X is the new element x_i, 
        % and nFF is |x_{i-1}-F(x_{i-1})|
        [X,nFF]=Fiter(X,param,swi);
        
        % Define another component needed for the graphical representation 
        if isempty(sol) % If solution is not known
            nerrF(i) = nFF;                               % |x - F(x)|
            [~,nerrFEPS(i)] = Fiter(EPSINI,param,swi,1);  % |eps - F(eps)|
        end  
  
        % New value of <y,x_i>
        EPSINIS = trace(Y'*X);

        % Check the value
        if abs(EPSINIS) > MAXD
           warning([' At iteration 1 the absolute value of the duality',...
               ' product is ' ,num2str(abs(EPSINIS),'%10.3e'),' > ', ...
               num2str(MAXD,'%15.5e')]);
        end
        
        % Next call of the scalar epsilon-algorithm 
        [EPSINIS,EPSSCA,NSING] = SEAW(EPSINIS,EPSSCA,MAXCOL,TOL,NDIGIT);
          
        % Next call of STEA algorithm with the next element
        switch swist
            case 1              
                [EPSINI,EPSVEC] = STEA1_3(X,EPSVEC,EPSSCA,MAXCOL,TOL); 
            case 2
                [EPSINI,EPSVEC] = STEA2_3(X,EPSVEC,EPSSCA,MAXCOL,TOL); 
        end
                     
        disp(['Iteration ', num2str(i+1,'%d'), ' completed']);
    end
    % End of the loop
    
    % Define the last components of the graphical representation 
    if ~isempty(sol) % If solution is known
        nerrX(NBC) = norm(sol-X);                       % |x - sol|
        nerrEPS(NBC) = norm(sol-EPSINI);                % |eps - sol|
    else             % If solution is not known
        [~,nerrFEPS(NBC)] = Fiter(EPSINI,param,swi,1);  % |eps - F(eps)|
        [~,nerrF(NBC)] = Fiter(X,param,swi);            % |x - F(x)|     
    end  
    
    disp(['Total number of singularities found in the eps-array = ', ...
        num2str(NSING,'%d')]);
    disp(' ');
    
    % Graphical representations
    figure(swist)
    if ~isempty(sol)
        semilogy([0:NBC-1],nerrX(1:NBC),'k-', ...
            [0:NBC-1],nerrEPS(1:NBC),col,'LineWidth',2)
        M={'||x - sol||'; ['||eps - sol|| ',lower(stean)]};
    else
        semilogy([0:NBC-1],nerrF(1:NBC),'k-', ...
            [0:NBC-1],nerrFEPS(1:NBC),col,'LineWidth',2)
        M={'||x - F(x)||'; ['||eps - F(eps)|| ',lower(stean)]};
    end
    legend(M,'LOCATION','best');
    title(['EXAMPLE ',swi,'. AM with ',stean])    
 
    % At the end of the first loop we have to do:
    if swist == 1
        TNS1 = NSING;
        % Save the vectors for figure
        if ~isempty(sol)
            nerrEPSsav = nerrEPS;
        else
            nerrFEPSsav = nerrFEPS;
        end
        % Reinitialize the example chosen
        [m,Y,X,sol,param] = exinit(swi,al);  
        clear Fiter;
        % For Examples 8a and 8b we have to reinitialize the persistent 
        % value Bn in the function Fiter
        % For the Examples 9 and 10 we have to reinitialize the persistent 
        % values nterm, psum and, for Example 9, also ind
        % For the Example 11 we have to reinitialize the persistent values 
        % nterm
        if (swin == 8) || (swin == 9) || (swin == 10) || (swin == 11)
            [~,~] = Fiter(X,param,swi,0);
        end
        % Define the type and color for the next graphical line STEA2
        col = 'b-.';
    else
        TNS2 = NSING;
    end
end

figure(3)
if ~isempty(sol)
    semilogy([0:NBC-1],nerrX(1:NBC),'k-',[0:NBC-1], ...
         nerrEPSsav(1:NBC),'r--', ...
         [0:NBC-1],nerrEPS(1:NBC),'b-.','LineWidth',2)
    legend('||x-sol||','||eps - sol|| stea1','||eps - sol|| stea2', ...
         'LOCATION','best');
else
    semilogy([0:NBC-1],nerrF(1:NBC),'k-',[0:NBC-1], ...
          nerrFEPSsav(1:NBC),'r--', ...
         [0:NBC-1],nerrFEPS(1:NBC),'b-.','LineWidth',2)
    legend('||x-F(x)||','||eps-F(eps)|| stea1','||eps-F(eps)|| stea2', ...
         'LOCATION','best');
end    
title(['EXAMPLE ',swi,'. AM with STEA1 and STEA2'])

% Final summary
disp(' ');
disp([' *** AM - Example ',swi,' ***'])
disp([' Dimension m          = ',num2str(m)]);
if (swin > 0) && (swin <6)
    disp([' alpha                = ',num2str(al,'%6.2e')]);
end 
disp([' Max even column      = ', num2str(MAXCOL,'%d')]);
disp([' Number of iterations = ', num2str(NBC,'%d')]);
disp([' Total number of isolated singularities for STEA1 = ', ...
    num2str(TNS1, '%d')]);
disp([' Total number of isolated singularities for STEA2 = ', ...
    num2str(TNS2, '%d')]);
disp(' ');
disp([' At iteration ', num2str(NBC,'%d')]);
if ~isempty(sol)
    disp([' ||x-sol||         = ',num2str(nerrX(end),'%6.2e')]);
    disp([' ||eps-sol|| stea1 = ',num2str(nerrEPSsav(end),'%6.2e')]);   
    disp([' ||eps-sol|| stea2 = ',num2str(nerrEPS(end),'%6.2e')]);   
else
    disp([' ||x-F(x)||           = ',num2str(nerrF(end),'%6.2e')]);
    disp([' ||eps-F(eps)|| stea1 = ',num2str(nerrFEPSsav(end),'%6.2e')]);   
    disp([' ||eps-F(eps)|| stea2 = ',num2str(nerrFEPS(end),'%6.2e')]);  
end 