function [EPSINI,EPSVEC,NSING]=VEAW(EPSINI,EPSVEC,MAXCOL,TOL,NDIGIT,IFLAG)
%  VEAW   Vector epsilon-algorithm with Wynn's particular rules
%                                                                               
%  [EPSINI,EPSVEC,NSING]=VEAW(EPSINI,EPSVEC,MAXCOL,TOL,NDIGIT,IFLAG)                                    
%                                                                          
%  Arguments:                                                              
%                                                                          
%  EPSINI   Input/output argument. 
%           In input, before the k-th call of the function, it must contain
%           the new term of the original sequence S_{k-1}.
%           In output, after the k-th call, it contains
%            eps_{k-1}^{(0)}             if k <= MAXCOL+1, k ODD   
%            eps_{k-2}^{(1)}             if k <  MAXCOL+1, k EVEN  
%            eps_{MAXCOL}^{(k-1-MAXCOL)} if k >  MAXCOL+1  
%           Thus, in the successive calls the function returns the sequence
%            eps_0^{(0)}, eps_1^{(0)}, eps_2^{(0)}, eps_2^{(1)} ....
%            eps_MAXCOL^{(0)}, eps_MAXCOL^{(1)}, eps_MAXCOL^{(2)}, .....
%                                                                          
%  EPSVEC   Input/output cell argument.  
%           It contains after the k-th call the last computed backward
%           diagonal of the epsilon scheme:      
%           - IF k <= MAXCOL+1 
%               eps_0^{(k-1)}, eps_1^{(k-2)}, ... eps_{k-1}^{(0)}
%           - IF k > MAXCOL+1  
%               eps_0^{(k-1)}, eps_1^{(k-2)}, ... eps_MAXCOL^{(k-1-MAXCOL)}
%                                                                          
%  MAXCOL   Input integer giving the index of the last column of the
%           epsilon scheme that the user wants to compute (lower index of
%           eps_m^{(n}). This value must be a positive even integer number
%                                                                          
%  TOL      Input real value used in tests for zero. If norm(X) < TOL, then
%           the denominator is considered to be zero and an error occurs
% 
%  NDIGIT   Input integer value. It represents the number of digits that 
%           the user allows to lose due to the cancellation error.
%           If, for any index k and for any n, the relation
%           norm( (eps_k^{(n+1)}-eps_k^{(n)})/eps_k^{(n)} ) < 10^{-NDIGIT}
%           holds, then the particular rules of Wynn shall be used to 
%           compute the value eps_{k+3}^{(n-1)}. 
%           But if, in the next call, we also have 
%           norm( (eps_k^{(n+2)}-eps_k^{(n+1)})/eps_k^{(n+1)} ) < 10^{-NDIGIT}
%           then the function stops due to a non isolated singularity    
%                                                                                
%  IFLAG    Input integer to be set to zero at the 'first' call of the
%           function for the initializations of the scheme.  
%           If this value is different from zero, or the argument is
%           missing, the function continues in expanding the previous
%           scheme. For a new application of the algorithm, the user must
%           remember to call the function with IFLAG equal to zero
%
%  NSING    Optional output integer value. It represents the total number  
%           of singularities detected in all the calls of the function
%
%  Internal working cell array:                                               
%                                                                        
%  WG     Working cell array (4 elements)                       
%                                                                         
%  Internal persistent array and cell array:                                           
%                                                                        
%  INDSAV   Integer vector in which the function saves the lower indexes of
%           the singularities. Needed for detecting, in the next call, the 
%           non isolated singularities. Its dimension do not exceed the 
%           value MAXCOL-3.
%
%  ABVEC    Cell array. In the first row the function saves the values of A
%           In the second row, the values of B. A and B are the notation
%           used by Wynn for his particular rules. The number of columns
%           do not exceed the value MAXCOL-3.
%
%  SVAL     Cell array of dimension 3 containing ........
%                                                                          
%  REMARKS:                                                                
%                                                                         
%   -  The EPSVEC cell array must not be modified by the user between
%      two consecutive calls of the function                                    
%   -  To obtain the value eps_{2*j}^{(0)} (2*J <= MAXCOL) the user must
%      call this function (2*J+1) times (including the first call)       
%     
%  AUTHORS: Claude Brezinski and Michela Redivo-Zaglia
%  V1.0 - December 2016
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
persistent INDSAV ABVEC SVAL
persistent INDM LOCINS NEWINS LISTBEG LISTEND IEC IAC IBC ANDIG NS MAXSIN IP
persistent Sfsnv Sfilv Sfsv
%
%                        ... EXECUTABLE STATEMENTS
%
%                        ... CHECK THE ARGUMENTS
if nargin < 5               % CHECK the number of input arguments
    error(' VEAW - Not enought input arguments ')
elseif nargout < 2          % CHECK the number of output arguments
    error(' VEAW - Not enought output arguments ')
elseif (mod(MAXCOL,2) ~= 0) || MAXCOL < 0 % CHECK MAXCOL 
    error(' VEAW - MAXCOL must be a positive even integer number')
end
%
%                        ... FIRST CALL OF THE FUNCTION WITH IFLAG == 0 
%                            OR TRIVIAL CASE (MAXCOL == 0)
%
if ((nargin == 6) && (IFLAG == 0)) || (MAXCOL == 0) 
    if ((nargin == 6) && (IFLAG == 0)) 
%                        ... OPEN A NEW LOG FILE FOR SEAW
        if ~isempty(Sfsnv)
            Sfsnv = Sfsnv+1;
        else
            Sfsnv = 1;              
        end
        Sfsv=['VEAW_',date,'_',num2str(Sfsnv),'.log'];
        Sfilv = fopen(Sfsv,'w');
        fprintf(Sfilv,[' === VEAW.log file === \n']);
        fprintf(Sfilv,[' === Cycle or Run ',num2str(Sfsnv),' === \n']);
    else
%                        ... OPEN THE LOG FILE FOR SEAW
        Sfilv = fopen(Sfsv,'a');    
        fprintf(Sfilv,['\n === MAXCOL = 0 ===']);
    end  
%                        ... INITIALIZATIONS
%       INDM is a counter for the calls of the function. The function
%       at each call computes the INDM+1 backward diagonal (not used
%       in the trivial case, that is when MAXCOL == 0)
    INDM   = 0;
%       MAXSIN is the maximum number of singularities that can be stored
%       The particular rules cannot be used when MAXCOL<=2
    MAXSIN = MAXCOL-3;
%
%                        ... SET THE BACKWARD DIAGONAL EQUAL TO eps_0^{(0)}
%                            OR TO eps_0^{(n})(IN THE TRIVIAL CASE) AND
%                            NSING TO ZERO 
%                           
    EPSVEC{1} = EPSINI;   
    NSING = 0;
    if MAXCOL ~= 0  
%       Initialize the persistent variables and cell array
        IP = size(EPSINI,1);
        SVAL = cell(3);
        NS = 0;
        ANDIG  = 10^(-NDIGIT);
        LOCINS = false;
        NEWINS = false;
        IEC    = false;
        IAC    = false;
        IBC    = false; 
        fprintf(Sfilv,['\n === Compute the diagonal ', ...
            num2str(INDM+1,'%d'), ' ===']);
    end
    return
%
%                        ... END OF THE FIRST CALL OF THE FUNCTION OR
%                            TRIVIAL CASE (MAXCOL == 0)
end
%                        ... NEXT CALLS OF THE FUNCTION
%
%                        ... UPDATE INDM (IT REPRESENTS THE BACKWARD  
%                            DIAGONAL TO BE COMPUTED 
INDM = INDM + 1;
fprintf(Sfilv,['\n === Compute the diagonal ', num2str(INDM+1,'%d'),' ===']);
%
%                        ... IF INDM > MAXCOL THEN THE ELEMENT 
%                            eps_MAXCOL^{(NUMBER_OF_CALLS-MAXCOL+1)}
%                            IS COMPUTED
IND = min([INDM,MAXCOL]);
%
%                        ... IF THERE IS NO LOCAL INSTABILITY PRESENT
%                            (COMING FROM THE PREVIOUS CALL)
%                            RESET THE INDEXES LISTBEG AND LISTEND AND
%                            THE VECTOR INDSAV.
%
if (~LOCINS)
    LISTBEG = 1;
    LISTEND = 0;
    INDSAV = []; 
end
%
%                        ... INITIALIZATION FOR THE AUXILIARY CELL ARRAY
%                            WE SET WG{2} EQUAL TO eps_{0}^(INDM)
WG{2} = EPSINI;
%
%                        ... STARTING LOOP FOR COMPUTING THE ELEMENTS
%                            OF THE ASCENDING DIAGONAL
%
for INDS = 0:IND - 1
%
%                        ... COMPUTE THE DENOMINATOR OF THE FORMULA
% 
    WG{1} = WG{2} - EPSVEC{INDS+1};  
    RD = norm(WG{1})^2;
    RDD = norm(EPSVEC{INDS+1})^2;  
    if (RDD < TOL) % check for the denominator
        disp(['Value of denominator ',num2str(RDD,'%15.5e'), ...
                  ' in column ', num2str(INDS+1,'%d')])
        error(['VEAW - Division by a value < TOL.', ... 
                  ' Impossible to continue.'])
    end
%
%                        ... CONTROL FOR A NEW POINT OF INSTABILITY.
%
    if (sqrt(RDD) > TOL) && (~IEC)  && (sqrt(RD/RDD) < ANDIG) && ...
             (INDS < IND-1) && (INDS < MAXCOL-2) 
        NEWINS  = true; % there is a new point of instability 
        fprintf(Sfilv,['\n New point of instability found in column number ', ...
            num2str(INDS+1,'%d')]);
        NS = NS+1; % increase the TOTAL number of singularities in the call
        LISTEND = LISTEND+1; % increase the index of INDSAV 
        if (LISTEND > MAXSIN) % check if the maximum allowed number
                              % of isolated singularities is reached
            error(['VEAW - Maximum allowed number of isolated ',...
                'singularities is reached.'])  
        end
    end
%
    if (LOCINS || NEWINS)
%
%                        ... WHEN THERE IS A LOCAL INSTABILITY PRESENT
%                            OR A NEW POINT OF INSTABILITY WE USE
%                            THE SINGULAR RULE OF THE ALGORITHM.
%
        if (NEWINS) 
%
%                        ... THERE IS A NEW POINT OF INSTABILITY ...
%
            if (LOCINS) 
%
%                        ... AND THERE IS A LOCAL INSTABILITY PRESENT.
%
%
%                        ... IF INDS COINCIDES WITH THE FIRST ELEMENT
%                            STORED IN INDSAV THEN THERE IS A NON ISOLATED
%                            SINGULARITY AND IT IS IMPOSSIBLE TO CONTINUE.
%
                if (INDS == INDSAV(LISTBEG)) 
                    error(['VEAW - NON isolated singularity.',... 
                        ' Impossible to continue.'])
                end
            end
%
%                        ... LOCAL INSTABILITY PRESENT IS TRUE.
%
            LOCINS = true;
%
%                        ... WE MUST COMPUTE A!
%
            IAC    = true;
%
%                        ... NEW POINT OF INSTABILITY TREATED!
%
            NEWINS = false;
%
%                        ... ADD THE CURRENT INDS TO THE END OF INDSAV.
%
            INDSAV(LISTEND) = INDS;
        end
%
%                        ... WHEN THE CONTROL ON A NEW POINT OF INSTABILITY
%                            IS MADE, THE CONTROL FOR THE COMPUTATION OF E
%                            IS PERFORMED.
%
        if (IEC) 
            fprintf(Sfilv,['\n PARTICULAR rule for computing E in', ...
                ' column ', num2str(INDS+1,'%d')]);
%
%                        ... WE MUST COMPUTE E AND STORE IT IN WG{1}=AUX0.
%
               % RD is .......
            RD = 1/SVAL{2}(LISTBEG)+1/TMP-1/SVAL{1}(LISTBEG)+SVAL{3}(LISTBEG);
            if (RD < TOL)    % check for the denominator
                disp(['Value of denominator ',num2str(RD,'%15.5e'), ...
                   ' in column ', num2str(INDS+1,'%d')])
                error(['VEAW - Division by a value < TOL.', ... 
                  ' Impossible to continue.'])
            end  
            WG{1} = (ABVEC{2,LISTBEG}/SVAL{2}(LISTBEG)+ WG{3}/TMP - ...
                ABVEC{1,LISTBEG}/SVAL{1}(LISTBEG)+SVAL{3}(LISTBEG)* ...
                EPSVEC{INDS})/RD;
%
%                        ... E COMPUTED!
%
            IEC     = false;
%
%                        ... INCREASE THE POINTER ON THE VECTOR INDSAV.
%
            LISTBEG = LISTBEG + 1;
%
%                        ... IF, IN THE VECTOR INDSAV, ALL THE STORED
%                            INDEXES HAVE BEEN CONSIDERED, THERE IS NO
%                            LOCAL INSTABILITY PRESENT.
%
            if (LISTBEG > LISTEND), LOCINS = false; end
%
        else
%                        ... IF WE DON'T COMPUTE E THE NORMAL RULE IS
%                            CONSIDERED.
%
            fprintf(Sfilv,['\n NORMAL rule in column ',num2str(INDS+1,'%d')]);
            RD = norm(WG{1})^2;               
            if (RD < TOL)    % check for the denominator
                disp(['Value of denominator ',num2str(RD,'%15.5e'), ...
                    ' in column ', num2str(INDS+1,'%d')])
                error(['VEAW - Division by a value < TOL in ', ... 
                  'the normal rule. Impossible to continue.'])
            end
            WG{1} = WG{1} / RD;
            if (INDS ~= 0), WG{1} = WG{1} + EPSVEC{INDS}; end 
        end
%
        if (LOCINS) 
%
%                        ... THERE IS A LOCAL INSTABILITY PRESENT ...
%
            if (IAC) 
                fprintf(Sfilv,['\n PARTICULAR rule for computing A in column ', ...
                    num2str(INDS+1,'%d')]);
%
%                        ... AND WE MUST COMPUTE AND STORE THE VALUE OF W
%
                if (INDS ~= 0) 
                    ABVEC{1,LISTEND} = EPSVEC{INDS};
                else
                    ABVEC{1,LISTEND} = zeros(IP,1);
                end
%
%                        ... COMPUTE THE VECTOR (W-C) AND ITS SQUARED NORM
%
                WG{4} = ABVEC{1,LISTEND} - WG{1};
                SVAL{1}(LISTEND) = norm(WG{4})^2;
%
%                        ... COMPUTE THE VECTOR (N-C) AND ITS SQUARED NORM
%
                WG{4} = EPSVEC{INDS+2} - WG{1};
                SVAL{2}(LISTEND) = norm(WG{4})^2;
%
%                        ... COMPUTE THE VECTOR (N-W) 
%
                WG{4} = EPSVEC{INDS+2} - ABVEC{1,LISTEND};
%
%                        ... COMPUTE A PART OF LAMBDA
%
                if (SVAL{1}(LISTEND) < TOL)  ||  (SVAL{2}(LISTEND) < TOL) 
                    disp(['Value of norms in denominators', ...
                        num2str(SVAL{1}(LISTEND), '%15.5e'), ...
                        num2str(SVAL{2}(LISTEND),'%15.5e'), ...
                        ' in column ', num2str(INDS+1,'%d')])
                    error(['VEAW - Division by a value < TOL in ', ... 
                         'the particular rule. Impossible to continue.'])
                end
                SVAL{3}(LISTEND) = norm(WG{4})^2/SVAL{1}(LISTEND)/SVAL{2}(LISTEND);
%
%                        ... END OF THE COMPUTATIONS RELATED TO A!
%
                IAC = false;
%
%                        ... WE MUST COMPUTE ALSO B!
%
                IBC = true;
%
%                        ... IF WE DON'T CONSIDER A ...
%
            elseif (IBC) 
                fprintf(Sfilv,['\n PARTICULAR rule for computing B in column ', ...
                    num2str(INDS+1,'%d')]);
%
%                        ... BUT WE MUST STORE N in B
%
                ABVEC{2,LISTEND} = EPSVEC{INDS+1};
%
%                        ... END FOR THE COMPUTATIONS RELATED TO B!
%
                IBC = false;
%
%                        ... IF WE DON'T CONSIDER A NOR B ...
%
            elseif ( (INDS ~= 0) && (INDS == (INDSAV(LISTBEG)+1)))                 
%
%                        ... AND THE CURRENT INDEX INDS IS THE NEXT
%                            INDEX OF INSTABILITY IN INDSAV WE PERFORM
%                            THE COMPUTATIONS RELATED TO D.
%
                fprintf(Sfilv,['\n PARTICULAR rule for computing D in column ', ...
                    num2str(INDS+1,'%d')]);
%
%                        ... COMPUTE THE VECTOR (S-C) AND ITS SQUARED NORM
%
                WG{4} = WG{2} - EPSVEC{INDS+1};
                TMP = norm(WG{4})^2;
%
%                        ... COMPUTE THE VECTOR (W-S) 
%
                WG{4} = ABVEC{1,LISTBEG} - WG{2};
%
%                        ... COMPUTE ANOTHER PART OF LAMBDA
%
                if (SVAL{1}(LISTBEG) < TOL)  ||  (TMP < TOL) 
                    disp(['Value of the norm in denominators', ...
                        num2str(SVAL{1}(LISTBEG), '%15.5e'), ...
                        num2str(TMP,'%15.5e'), ...
                        ' in column ', num2str(INDS+1,'%d') ]);
                    error(['VEAW - Division by a value < TOL in ', ... 
                         'the particular rule. Impossible to continue.'])
                end
                SVAL{3}(LISTBEG) = SVAL{3}(LISTBEG) + norm(WG{4})^2/ ...
                    SVAL{1}(LISTBEG)/TMP;
%
%                        ... COMPUTE THE VECTOR (S-N) 
%
                WG{4} = WG{2} - ABVEC{2,LISTBEG};
%
%                        ... END OF COMPUTATIONS OF LAMBDA
%                  
                if (abs(SVAL{2}(LISTBEG)) < TOL)  
                    error(['VEAW - Division by a value < TOL.', ... 
                         ' Impossible to continue.'])
                end
                SVAL{3}(LISTBEG) = SVAL{3}(LISTBEG) - norm(WG{4})^2/ ...
                    SVAL{2}(LISTBEG)/TMP;
%
%                        ... END OF COMPUTATIONS RELATED TO D!
% 
%
%                        ... AND WE MUST COMPUTE E!
%
                IEC = true;
            end
        end
    else
%
%                        ... WHEN THERE IS NO LOCAL INSTABILITY PRESENT
%                            NOR A NEW POINT OF INSTABILITY WE USE
%                            THE NORMAL RULE OF THE ALGORITHM.
%
        fprintf(Sfilv,['\n NORMAL rule in column ',num2str(INDS+1,'%d')]);
        RD = norm(WG{1})^2;
        if (RD < TOL)    % check for the denominator
            disp(['Value of denominator ',num2str(RD,'%15.5e'), ...
                    ' in column ', num2str(INDS+1,'%d')])
            error(['VEAW - Division by a value < TOL in ', ... 
                  'the normal rule. Impossible to continue.'])
        end
        WG{1} = WG{1} / RD;
        if (INDS ~= 0), WG{1} = WG{1} + EPSVEC{INDS}; end  
    end
%
%                        ... SHIFT THE RHOMBUS.
%
    if (INDS ~= 0), EPSVEC{INDS} = WG{3}; end  
    WG{3} = WG{2};
    WG{2} = WG{1};
%
%                        ... END LOOP FOR INDS
%
end
%
%                        ... SAVE FOR THE FINAL VALUES OF THE DIAGONAL.
%
EPSVEC{IND}   = WG{3};  
EPSVEC{IND+1} = WG{2};  
%
%                        ... SET THE RESULT VALUE.
%
if (mod(IND,2) == 0) 
    EPSINI = EPSVEC{IND+1};  
else
    EPSINI = EPSVEC{IND};
end
% Set the output parameter to the numbero of singularities
NSING = NS;
%