function [EPSINI,EPSVEC] = STEA1_1 (EPSINI,EPSVEC,EPSSCA,MAXCOL,TOL,IFLAG)
%  STEA1_1   First Simplified Topological Epsilon-Algorithm (formula 1)
%                                                                               
%  [EPSINI,EPSVEC] = STEA1_1 (EPSINI,EPSVEC,EPSSCA,MAXCOL,TOL,IFLAG)                                    
%                                                                          
%  Arguments:                                                              
%                                                                          
%  EPSINI   Input/output argument. 
%           In input, before the k-th call of the function, it must contain
%           the new term of the original sequence S_{k-1}.
%           In output, after the k-th call, it contains
%            eps_{k-1}^{(0)}             if k <= MAXCOL+1, k ODD   
%            eps_{k-2}^{(1)}             if k <  MAXCOL+1, k EVEN  
%            eps_{MAXCOL}^{(k-1-MAXCOL)} if k >  MAXCOL+1  
%           Thus, in the successive calls the function returns the sequence
%            eps_0^{(0)}, eps_0^{(1)}, eps_2^{(0)}, eps_2^{(1)} ....
%            eps_MAXCOL^{(0)}, eps_MAXCOL^{(1)}, eps_MAXCOL^{(2)}, .....
%                                                                          
%  EPSVEC   Input/output cell array argument.  
%           It contains after the k-th call the last computed backward
%           diagonal of the topological epsilon scheme:      
%           - IF k <= MAXCOL+1, k ODD
%               eps_0^{(k-1)}, eps_2^{(k-3)}, eps_4^{(k-5)} ... 
%               eps_{k-1}^{(0)}
%           - IF k <= MAXCOL+1, k EVEN
%               eps_0^{(k-1)}, eps_2^{(k-3)}, eps_4^{(k-5)} ... 
%               eps_{k-2}^{(1)}
%           - IF k > MAXCOL+1
%               eps_0^{(k-1)}, eps_2^{(k-3)}, ... eps_MAXCOL^{(k-1-MAXCOL)}
%           Before the first call its input value must be an empty array.
%                                              
%  EPSSCA   Input scalar vector.
%           In input, before the k-th call of the function, it must contain
%           the last computed backward diagonal computed by the scalar
%           epsilon algorithm:    
%           - IF k <= MAXCOL+1 
%               eps_0^{(k-1)}, eps_1^{(k-2)}, ... eps_{k-1}^{(0)}
%           - IF k > MAXCOL+1  
%               eps_0^{(k-1)}, eps_1^{(k-2)}, ... eps_MAXCOL^{(k-1-MAXCOL)}
%                                                                          
%  MAXCOL   Input integer giving the index of the last column of the
%           epsilon scheme that the user wants to compute (lower index of
%           eps_m^{(n}). This value must be a positive even integer number
%
%  TOL      Input real value used in tests for zero. If abs(X) < TOL, then
%           X is considered to be zero
%                                                                                
%  IFLAG    Input integer to be set to zero at the 'first' call of the
%           function for the initializations of the scheme.  
%           If this value is different from zero, or the argument is
%           missing, the function continues in expanding the previous
%           scheme. For a new application of the algorithm, the user must
%           remember to call the function with IFLAG equal to zero
%
%  Internal working cell array:                                               
%                                                                        
%  AUX0,AUX1 Working cell arrays (2 elements)                       
%                                                                          
%  Internal persistent arrays and cell array:                                           
%  
%  TMPSCA   This vector contains the backward diagonal of the scalar 
%           epsilon algorithm of the previous call, passed in the input
%           argument 
%                                       
%  TMPDIF   This vector contains the differences of even term of the 
%           two preceding backward diagonals of the scalar epsilon scheme
%           (needed for the even lower index rule)  
%
%  EPSDIF   Cell array containing the differences of the even lower index 
%           terms of the two preceding backward diagonals of the epsilon 
%           scheme (needed for the even lower index rule) 
%                                                                          
%  REMARKS:                                                                
%                                                                         
%   -  The EPSVEC cell array must not be modified by the user between
%      two consecutive calls of the function                                    
%   -  To obtain the value eps_{2*J}^{(0)} (2*J <= MAXCOL) the user must
%      call this function (2*J+1) times (including the first call)       

%  AUTHORS: Claude Brezinski and Michela Redivo-Zaglia
%  V1.0 - December 2016
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
persistent INDM 
persistent TMPSCA TMPDIF 
persistent EPSDIF

%
%                        ... EXECUTABLE STATEMENTS
%
%                        ... CHECK THE ARGUMENTS
if nargin < 5               % CHECK the number of input arguments
    error('STEA1_1 - Not enought input arguments ')
elseif nargout < 2          % CHECK the number of output arguments
    error('STEA1_1 - Not enought output arguments ')
elseif (mod(MAXCOL,2) ~= 0) || MAXCOL < 0 % CHECK MAXCOL 
    error('STEA1_1 - MAXCOL must be a positive even integer number')
end
%
%                        ... FIRST CALL OF THE FUNCTION WITH IFLAG == 0 
%                            OR TRIVIAL CASE (MAXCOL == 0)
if ((nargin == 6) && (IFLAG == 0)) || (MAXCOL == 0)
%                        ... INITIALIZE THE COUNTER INDM
%       INDM is a counter for the calls of the function. The function
%       at each call computes the INDM+1 backward diagonal (not used
%       in the trivial case, that is when MAXCOL == 0)
    INDM   = 0;
%
%                        ... SET THE BACKWARD DIAGONAL EQUAL TO eps_0^{(0)}
%                            OR TO eps_0^{(n}) (IN THE TRIVIAL CASE)

    EPSVEC{1} = EPSINI;   
    EPSDIF{1} = [];
    if MAXCOL ~= 0  
%                        ... INITIALIZE THE PERSISTENT ARRAYS
        TMPSCA = EPSSCA;
        TMPDIF = [];
    end
    return 
%
%                        ... END OF THE FIRST CALL OF THE FUNCTION OR
%                            TRIVIAL CASE (MAXCOL == 0)
end    
%                        ... NEXT CALLS OF THE FUNCTION
%
%                        ... UPDATE INDM (IT REPRESENTS THE BACKWARD  
%                            DIAGONAL TO BE COMPUTED) 
INDM = INDM + 1;
%
%                        ... IN THE SECOND CALL WE HAVE ONLY TO DEFINE
%                            SOME VALUES    
if (INDM == 1)
    EPSDIF{1} = EPSINI - EPSVEC{1};
    EPSVEC{1} = EPSINI; 
    TMPDIF = EPSSCA(1) - TMPSCA;
    TMPSCA = EPSSCA;
    return
end
%
%                        ... IF INDM > MAXCOL, THEN WE REDEFINE THE 
%                            NUMBER OF ELEMENTS TO BE COMPUTED
IND = fix(min([INDM,MAXCOL+1])/2);
%
%                        ... INITIALIZATION FOR THE AUXILIARY VARIABLE
%
AUX1 = EPSINI;
%
%                        ... STARTING LOOP FOR COMPUTING THE ELEMENTS
%                            OF THE ASCENDING DIAGONAL
%
for INDS = 0:IND-1
    RD = TMPDIF(INDS+1) * (EPSSCA(INDS*2+2)-TMPSCA(INDS*2+2));
    if (abs(RD) < TOL) % check for the denominator
        disp(['Value of denominator ',num2str(RD,'%15.5e'), ...
             ' in column ', num2str(INDS+1,'%d')])
        error(['STEA1_1 - Division by a value < TOL.', ... 
             ' Impossible to continue.'])
    end
    AUX0 = EPSVEC{INDS+1} + EPSDIF{INDS+1}/RD ; 
%  
%                        ... SHIFT THE VALUES
%
    TMPDIF(INDS+1) = EPSSCA(INDS*2+1)-TMPSCA(INDS*2+1);
    EPSDIF{INDS+1} = AUX1 - EPSVEC{INDS+1};
    EPSVEC{INDS+1} = AUX1; 
    AUX1 = AUX0;  
%                        ... END LOOP FOR INDS
%
end
%  
%                        ... SAVE THE PERSISTENT VALUES FOR THE NEXT CALL
%
if ((mod(INDM-1,2) == 0) && (IND ~= MAXCOL/2))
    TMPDIF(IND+1) = EPSSCA(IND*2+1)-TMPSCA(IND*2+1);
    EPSDIF{IND+1} = AUX1 - EPSVEC{IND+1};
end
TMPSCA = EPSSCA;  
%
%                        ... SET THE RESULT VALUES
%
EPSVEC{IND+1} = AUX0;  
EPSINI = EPSVEC{IND+1};   