% Script for Restarted Method (sequence of El-Sayed, Al-Dbiban method)
% RMmain_EA
% This script shows how to use the RESTARTED method by using the
% simplified topological epsilon-algorithms. 
% The example is the nonlinear matrix equation (see Section 4.4 of 
% the paper) 
%  X + A' inv(X) A = I
% We use the Method of S.M. El-Sayed and A.M. Al-Dbiban (2005)
% The function Fiter_EA is needed
 
% Authors: Claude Brezinski, Michela Redivo-Zaglia
% V1.0 - December 2016
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clearSEAW
clear variables

%<<< %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initializations for SEAW and STEA
% Define the algorithm STEA we want to use (1 or 2)
STEAn = 2;
% Define the formula we want to use (1,2,3 or 4)
STEAv = 3;
% Choice of the maximum even column in the epsilon-array
MAXCOL = 6;
% Choice of number of cycles
NCY = 7;
% Set the number of terms of the original sequence
NBC = MAXCOL+1;
% Assign the tolerance for denominators 
TOL = 1e-20;
% Number of common digit asked for detecting a singularity
NDIGIT = 15;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% >>>

% Check the algorithm asked
if ((STEAn~=1) && (STEAn~=2)) || (STEAv<1) || (STEAv>4)
    error('Such STEA or such formula do not exists')
end
% Define the function handle
STEAs = ['STEA',num2str(STEAn),'_',num2str(STEAv)];
STEAf = str2func(STEAs);
% Check the MAXCOL and NBC values
if mod(MAXCOL,2) ~= 0
    error (' The max column must be an even integer');
end
% Preallocate the vectors for the graphical representation
nX = zeros(NCY*NBC,1); nXit = nX; nS = zeros(NCY+1,1);

% Assign a tolerance for the maximun absolute value of the
% duality product
MAXD = 1e+20;

%<<< %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initializations for the example and method chosen
% A is a 3x3 matrix
A = [0.37,0.13,0.12; -0.30,0.34,0.12; 0.11,-0.17,0.29];
% m is the dimension 
m = size(A,1);
% Define a handle function for computing the Frobenious norm of the
% residual
nres = @(Y) (norm(Y + A'*(Y\A) - eye(size(A)),'fro'));
% X is the starting matrix X_0
X = eye(m);
% Y is the starting value for the auxiliary matrix of the iterative method
Y = X;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% >>>

% Computations

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initialize the variable for the total number of singularities
TNS = 0;

% Start the outer loop 
for i = 1:NCY
    % Compute the Frobenious norms of the residual matrix u_0
    nXit(MAXCOL*(i-1)+i) = nres(X);
    % Frobenious norm of the first extrapolated term = initial matrix
    nS(i) = nres(X);
    % Compute <y,u_0>= trace(u_0)
    EPSINIS = trace(X);
    % Check its value
    if abs(EPSINIS) > MAXD
        warning([' At outer iteration ',num2str(i,'%d'), ...
           ', the absolute value of the duality product is '...
           ,num2str(abs(EPSINIS),'%10.3e'),' > ', num2str(MAXD,'%15.5e')])
    end
    % First call of SEAW
    [EPSINIS,EPSSCA,NSING]=SEAW(EPSINIS,[],MAXCOL,TOL,NDIGIT,0);
    % First call of STEA
    [EPSINI,EPSVEC] = STEAf(X,[],EPSSCA,MAXCOL,TOL,0);
    disp(['Inner iteration 1 of cycle ',num2str(i,'%d'),' completed']);
%   Start of the inner loop 
    for n = 1:NBC-1   
        % In output, X is the new element of the sequence u_n 
        [X,Y] = Fiter_EA(X,A,Y); 
        % Compute the norm of the residual of u_{n}
        nXit(NBC*(i-1)+n+1) = nres(X);
        % Compute <y,u_n> = trace(u_n)
        EPSINIS = trace(X);
        if (abs(EPSINIS) > MAXD) % check for the denominator
            disp(['Value of the dual product ',num2str(abs(EPSINIS), ...
                '%15.5e'), ' at iteration ', num2str(n+1,'%d')])
            warning(['The norm of the dual product is > ', ...
                num2str(MAXD,'%15.5e')])
        end
        % Next calls of SEAW and STEA
        [EPSINIS,EPSSCA,NSING]=SEAW(EPSINIS,EPSSCA,MAXCOL,TOL,NDIGIT);
        [EPSINI,EPSVEC] = STEAf (X,EPSVEC,EPSSCA,MAXCOL,TOL);
        disp(['Inner iteration ', num2str(n+1,'%d'), ' of cycle ', ...
            num2str(i,'%d'),' completed']);
    end
    % End of the inner loop
    
    % Set the restarting u_0=eps_{2k}^(0) element for the next outer loop
    X = EPSINI;
    % Define the corresponding auxiliary matrix for the next outer loop
    Y = inv(EPSINI);
    
    disp(['* Outer iteration ', num2str(i,'%d'), ' completed']);
    disp(['* Total number of singularities found in the eps-array = ', ...
          num2str(NSING,'%d')]);
    disp(' '); 
    % Update the total number of singularities
    TNS = TNS+NSING;
end
% End of the outer loop

% Define the last extrapolated component for the graphical representation 
nS(NCY+1) = nres(EPSINI);
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Graphical representations

% Compute the sequence produced by the original method
% X is the starting matrix X_0
X = eye(m); 
% Y is the starting value for the auxiliary matrix of the iterative method
Y = X;
% Compute the Frobenious norms of the residual matrix
nX(1) = nres(X);
for j=1:NCY*NBC-1
    % In output, X is the new element X_j   
    [X,Y] = Fiter_EA(X,A,Y);
    % Compute the norm of the residual (basic method)
    nX(j+1) = nres(X);
end

figure(21)
semilogy([0:NBC*NCY-1],nX,'k-',[0:NBC*NCY-1],nXit,'r--', ...
            [0:NBC:NBC*NCY],nS,'b*','LineWidth',2);
legend('||f(X_{ori})||_F EA Method', '||f(X))||_F EA Method rest.', ...
    ['||f(eps)||_F ',STEAs], 'LOCATION','best');
title(['An example - EA Method + Restarted Method with ',STEAs])    
    
% Final summary
disp(' ');
disp(['An example - Restarted Method with ',STEAs]);
disp('Basic method of S.M. El-Sayed and A.M. Al-Dbiban');
disp(' ');
disp([' Dimension m                       = ', num2str(m(1))]);
disp([' Number of terms for each cycle    = ', num2str(NBC,'%d')]);
disp([' Max even column                   = ', num2str(MAXCOL,'%d')]);
disp([' Tolerance for the algorithms      = ', num2str(TOL,'%6.1e')]);
disp([' Number of the common digits       = ', num2str(NDIGIT,'%d')]);
disp([' Number of cycles                  = ', num2str(NCY,'%d')]);
disp([' Maximum for the |duality product| = ', num2str(MAXD,'%6.1e')]);
disp([' Maximum number of possible isolated singularities for each cycle = ', ...
    num2str(MAXCOL-3,'%d')]);
disp([' Total number of isolated singularities found in all the cycles   = ', ...
    num2str(TNS,'%d')]);
disp(' ');
disp([' At the last iteration we obtain']);
disp([' ||f(X_{ori})||_F     = ',num2str(nX(end),'%6.2e')]);
disp([' ||f(eps)||_F         = ',num2str(nS(end),'%6.2e')]);
