% Demo script for Restarted Method
% RMdemo
% This script reproduces the Examples 1 to 8 of the paper by using the
% restarted method and the Simplified Topological Epsilon-Algorithms
% STEA1 and STEA2.
% It needs of the functions exinit, exhelp and Fiter.

% Authors: Claude Brezinski, Michela Redivo-Zaglia
% V1.0 - December 2016
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clearSEAW
clear variables
close all

% Choice of the example
disp(' Examples of the demo:');
disp(' 1, 2, 3, 4a, 4b, 5, 6, 7, 8a, 8b')
disp(' ');
swi=input(' Example number ','s');
swi=strtok(swi);
if isletter(swi(end))
    swin = str2double(swi(1:end-1));
else
    swin = str2double(swi);
end

% Check for the example chosen 
if ((swin == 4) || (swin == 8)) && isletter(swi(end)) || ...
        ((~isempty(swin)) && ismember(swin,[1,2,3,5,6,7])) 
    exhelp(swi,1);
else
    error (' RMdemo - The example chosen do not exist or cannot be used');
end

% Choice of the alpha (only for examples 1 to 5) 
if (swin > 0) && (swin <6)
    al=input(' Insert alpha ');
    if (isempty(al)) 
        error (' RMdemo - An alpha must be given for this example! ');
    end
else 
    al = 0;
end

% Initializations for the example chosen for 
% computing the sequence u_{i+1}=F(u_i), i=0,1,...,
% without using the Restarted method.
% X is the starting element u_0.
[m,~,X,sol,param] = exinit(swi,al);  

% For Examples 8a and 8b we have to initialize the persistent value Bn
% in the function Fiter
if swin == 8
    [~,~] = Fiter(X,param,swi,0);
end

disp([' Dimension of the system m = ',num2str(m)])

% Choice of the max even column
% For the Generalized Steffensen method this value must be equal to
% 2*m, with m dimension of the system
MAXCOL = input(' Insert max even column (= 2*m for GSM) '); 

% Check the MAXCOL value 
if mod(MAXCOL,2) ~= 0
    error (' RMdemo - The max column must be an even integer');
end
% Set the number of terms of the original sequence
NBC = MAXCOL+1;
% Choice of the number of outer loops
NCY=input(' Insert the number of outer loops ');

% Assign the values for the scalar epsilon-algorithm
% Tolerance for denominators 
TOL = 1e-30;
% Number of common digits asked for detecting a singularity
NDIGIT = 15; 
% Assign the tolerance for the maximun absolute value of the
% duality product
MAXD = 1e+20;

disp([' Tolerance for scalar epsilon algorithm  = ',num2str(TOL,'%6.1e')]);
disp([' Number of the common digits             = ',num2str(NDIGIT,'%d')]);
disp([' Maximum for the |duality product|       = ',num2str(MAXD,'%6.1e')]);
disp(' ');

tot = NBC*NCY;

for i=1:tot
    % Define a component needed for the graphical representations
    if ~isempty(sol) % If solution is known
        nerrITX(i) = norm(sol-X);            % |x - sol|
    end
    % Compute the original u_i sequence and its norm
    [X,nFF] = Fiter(X,param,swi);
    % Define a component needed for the graphical representations 
    if isempty(sol) % If solution is not known
        nerrFX(i) = nFF;                     % |x - F(x)|
    end
end

% Reinitialize the example chosen for the 
% Restarted method.
[m,Y,X,sol,param] = exinit(swi,al);  

% For Examples 8a and 8b we have to initialize the persistent value Bn
% in the function Fiter
if swin == 8
    [~,~] = Fiter(X,param,swi,0);
end

% Initialize the variables for total number of singularities
TNS1 = 0;
TNS2 = 0;

% Apply STEA1 and STEA2 to the example chosen
for swist=1:2
    stean = ['STEA', num2str(swist,'%d')];
    disp(['*** ',stean]);

    % initialize the counter of the NBC*NCY elements
    ii=1;
    
    % Start the outer loop of the Restarted method
    for i=1:NCY
        
        % First value <y,u_0>
        EPSINIS = trace(Y'*X);
        
        % Check its value
        if abs(EPSINIS) > MAXD
            warning([' RMdemo - At outer iteration ',num2str(i,'%d'), ...
               ', the absolute value of the duality product is '...
               ,num2str(abs(EPSINIS),'%10.3e'),' > ', ...
               num2str(MAXD,'%15.5e')]);
        end

        % First call of the scalar epsilon-algorithm  
        [EPSINIS,EPSSCA,NSING] = SEAW(EPSINIS,[],MAXCOL,TOL,NDIGIT,0);
   
        % First call (initialization) of STEA algorithm 
        switch swist
            case 1
                [EPSINI,EPSVEC] = STEA1_3(X,[],EPSSCA,MAXCOL,TOL,0); 
            case 2
                [EPSINI,EPSVEC] = STEA2_3(X,[],EPSSCA,MAXCOL,TOL,0); 
        end
        disp(['Inner iteration 1 completed']);
        
        % Initialize the components needed for the graphical representations     
        if ~isempty(sol) % If solution is known
            nerrEPS(i) = norm(sol-EPSINI);               % |eps - sol|
        else             % If solution is not known
            [~,nerrFEPS(i)] = Fiter(EPSINI,param,swi,1); % |eps-F(eps)|
        end
            
        % Start of the inner loop of the Restarted method
        for n = 1:NBC-1  
            
            % Define the components needed for the graphical representations     
            if ~isempty(sol)                      % If solution is known
                nerrIT(ii)  = norm(sol-X);                  % |u - sol|
            end
            
            % In output, X is the new element u_i,
            % and nFF is |u_{i-1} - F(u_{i-1})|
            [X,nFF] = Fiter(X,param,swi); 
            
            % Define another component for the graphical representation             
            if isempty(sol)                   % If solution is not known
                nerrF(ii) = nFF;                            % |u - F(u)|
            end
            
            % New value of <y,u_i>
            EPSINIS = trace(Y'*X);
            
            % Check the value
            if abs(EPSINIS) > MAXD
                warning([' RMdemo - At inner iteration ',num2str(n,'%d'), ...
                    ', outer iteration ',num2str(i,'%d'), ...
                    ', the absolute value of the duality product is ',...
                    num2str(abs(EPSINIS),'%10.3e'),' > ', ...
                    num2str(MAXD,'%15.5e')]);
            end

            % Next call of the scalar epsilon-algorithm 
            [EPSINIS,EPSSCA,NSING]= SEAW(EPSINIS,EPSSCA,MAXCOL,TOL,NDIGIT);
 
            % Next element u_i
            EPSINI = X; 
            
            % Next call of STEA algorithm 
            switch swist
                case 1              
                  [EPSINI,EPSVEC]=STEA1_3(EPSINI,EPSVEC,EPSSCA,MAXCOL,TOL); 
                case 2
                  [EPSINI,EPSVEC]=STEA2_3(EPSINI,EPSVEC,EPSSCA,MAXCOL,TOL); 
            end      
 
            % increase the counter of the elements for graphical
            % representation
            ii = ii+1;
            
            disp(['Inner iteration ', num2str(n+1,'%d'), ' completed']);
        end
        % End of the inner loop of the Restarted method
      
        % Define the last components of the graphical representation 
        if ~isempty(sol) % If solution is known
            nerrIT(ii)  = norm(sol-X);                       % |u - sol|
            nerrEPS(i+1) = norm(sol-EPSINI);                 % |eps - sol|
        else             % If solution is not known
            [~,nerrF(ii)] = Fiter(X,param,swi,1);            % |u - F(u)|     
            [~,nerrFEPS(i+1)] = Fiter(EPSINI,param,swi,1);   % |eps-F(eps)|
        end  
        
        % Set the restarting u_0=eps_{2k}^(0) element for the next outer loop
        X = EPSINI;

        % increase the counter of the elements for graphical representation
        ii=ii+1;
        
        disp(['* Outer iteration ', num2str(i,'%d'), ' completed']);
        disp(['* Total number of singularities found in the eps-array = ', ...
              num2str(NSING,'%d')]);
        disp(' ');
        switch swist
            case 1
                TNS1 = TNS1+NSING; 
            case 2
                TNS2 = TNS2+NSING; 
        end
    end 
    % End of the outer loop of the Restarted method

    % Graphical representation of the results

    figure(swist*10)
    if ~isempty(sol)
        semilogy([0:tot-1],nerrITX,'k-',[0:tot-1],nerrIT,'b--',...
            [0:NBC:tot],nerrEPS,'r*','LineWidth',2)
        M={'||x_{ori} - sol||'; ['||x-sol|| ',lower(stean)] ; ...
            ['||eps - sol|| ',lower(stean)]};
    else
        semilogy([0:tot-1],nerrFX,'k-',[0:tot-1],nerrF,'b--',...
            [0:NBC:tot],nerrFEPS,'r*','LineWidth',2)
        M={'||x_{ori} - F(x_{ori})||'; ['||x-F(x)|| ',lower(stean)] ; ...
            ['||eps - F(eps)|| ',lower(stean)]};
    end
    legend(M,'LOCATION','best');
    if MAXCOL == 2*m
        title(['EXAMPLE ',swi,'. GSM with ',stean])  
    else  
        title(['EXAMPLE ',swi,'. RM with ',stean])  
    end
       
    % At the end of the first outer loop we have to do:
    if swist == 1
        % Save the vectors for figure
        if ~isempty(sol)
            nerrITsav = nerrIT;
            nerrEPSsav = nerrEPS;
        else
            nerrFsav = nerrF;
            nerrFEPSsav = nerrFEPS;
        end
        % Reinitialize the example chosen
        [m,Y,X,sol,param] = exinit(swi,al);  
        
        % For Examples 8a and 8b we have to initialize the persistent value
        % Bn in the function Fiter
        if swin == 8
            [~,~] = Fiter(X,param,swi,0);
        end
    end
end

% Other graphical representations

figure(30)
if ~isempty(sol)
        semilogy([0:tot-1],nerrITX,'k-',[0:tot-1],nerrITsav,'r--', ...
            [0:NBC:tot],nerrEPSsav,'r*', [0:tot-1],nerrIT,'b-.', ...
            [0:NBC:tot],nerrEPS,'b*', 'LineWidth',2)
        legend('||x_{ori}-sol||','||x-sol|| stea1', '||eps - sol|| stea1', ...
            '||x-sol|| stea2', '||eps - sol|| stea2 ', 'LOCATION','best');
else
        semilogy([0:tot-1],nerrFX,'k-',[0:tot-1],nerrFsav,'r--', ...
            [0:NBC:tot],nerrFEPSsav,'r*', [0:tot-1],nerrF,'b-.', ...
            [0:NBC:tot],nerrFEPS,'b*', 'LineWidth',2)
        legend('||x_{ori}-F(x_{ori})||','||x-F(x)|| stea1', ...
            '||eps - F(eps)|| stea1', '||x-F(x)|| stea2', ...
            '||eps - F(eps)|| stea2', 'LOCATION','best');
end    
if MAXCOL == 2*m
    title(['EXAMPLE ',swi,'. GSM with STEA1 and STEA2'])
else
    title(['EXAMPLE ',swi,'. RM with STEA1 and STEA2'])
end


figure(40)
if ~isempty(sol)
        semilogy([0:NCY],nerrEPSsav,'r--*',[0:NCY], ...
            nerrEPS,'b-.*', 'LineWidth',2)
        legend('||eps-sol|| stea1','||eps-sol|| stea2','LOCATION','best');
else
        semilogy([0:NCY],nerrFEPSsav,'r--*',[0:NCY], ...
            nerrFEPS,'b-.*', 'LineWidth',2)
        legend('||eps-F(eps)|| stea1','||eps-F(eps)|| stea2','LOCATION', ...
            'best');
end 
xlabel('cycles')
if MAXCOL == 2*m
    title(['EXAMPLE ',swi,'. GSM with STEA1 and STEA2'])
else
    title(['EXAMPLE ',swi,'. RM with STEA1 and STEA2'])
end

% Final summary
disp(' ');
if MAXCOL == 2*m
    disp([' *** GSM - Example ',swi,' ***'])
else
    disp([' *** RM - Example ',swi,' ***'])
end
disp([' Dimension m                   = ',num2str(m)])
if (swin > 0) && (swin <6)
     disp([' alpha                         = ',num2str(al,'%6.2e')]);
end 
disp([' Max even column               = ',num2str(MAXCOL,'%d')]);
disp(' Number of terms for each ');
disp([' inner iteration               = ',num2str(NBC,'%d')]);
disp([' Number of cycles              = ',num2str(NCY,'%d')]);
disp([' Total number of isolated singularities for STEA1 (all cycles) = ', ...
    num2str(TNS1, '%d')]);
disp([' Total number of isolated singularities for STEA2 (all cycles) = ', ...
    num2str(TNS2, '%d')]);
disp(' ');
disp(' At last iteration of the Restarted Method');
if ~isempty(sol)
    disp([' ||x_{ori}-sol||   = ',num2str(nerrITX(end),'%6.2e')]);
    disp([' ||eps-sol|| stea1 = ',num2str(nerrEPSsav(end),'%6.2e')]);   
    disp([' ||eps-sol|| stea2 = ',num2str(nerrEPS(end),'%6.2e')]);   
else
    disp([' ||x_{ori}-F(x_{ori})|| = ',num2str(nerrFX(end),'%6.2e')]);
    disp([' ||eps-F(eps)|| stea1   = ',num2str(nerrFEPSsav(end),'%6.2e')]);   
    disp([' ||eps-F(eps)|| stea2   = ',num2str(nerrFEPS(end),'%6.2e')]);  
end 