% Script for Acceleration Method (sequence of El-Sayed, Al-Dbiban method)
% AMmain_EA
% This script shows how to use the ACCELERATION method by using the
% simplified topological epsilon-algorithms. 
% The example is the nonlinear matrix equation (see Section 4.4 of 
% the paper) 
%  X + A' inv(X) A = I
% We use the Method of S.M. El-Sayed and A.M. Al-Dbiban (2005)
% The function Fiter_EA is needed

% Authors: Claude Brezinski, Michela Redivo-Zaglia
% V1.0 - December 2016
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clearSEAW
clear variables

%<<< %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initializations for SEAW and STEA
% Define the algorithm STEA we want to use (1 or 2)
STEAn = 2;
% Define the formula we want to use (1,2,3 or 4)
STEAv = 3;
% Choice of the maximum even column in the epsilon-array
MAXCOL = 6;
% Choice of the number of terms of the original sequence
NBC = 51;
% Assign the tolerance for denominators 
TOL = 1e-20;
% Number of common digit asked for detecting a singularity
NDIGIT = 15;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% >>>

% Check the algorithm asked
if ((STEAn~=1) && (STEAn~=2)) || (STEAv<1) || (STEAv>4)
    error('Such STEA or such formula do not exists')
end
% Define the function handle
STEAs = ['STEA',num2str(STEAn),'_',num2str(STEAv)];
STEAf = str2func(STEAs);
% Check the MAXCOL and NBC values
if mod(MAXCOL,2) ~= 0
    error (' The max column must be an even integer');
end
if NBC < MAXCOL+1
    warning([' The number of terms must be greater or equal to ', ...
    num2str(MAXCOL+1),' to reach the column ', num2str(MAXCOL)]);
end

% Preallocate the vectors for the graphical representation
nX = zeros(NBC,1); nS = nX;

% Assign a tolerance for the maximun absolute value of the
% duality product
MAXD = 1e+20;

%<<< %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initializations for the example and method chosen
% A is a 3x3 matrix
A = [0.37,0.13,0.12; -0.30,0.34,0.12; 0.11,-0.17,0.29];
% m is the dimension 
m = size(A,1);
% Define a handle function for computing the Frobenious norm of the
% residual
nres = @(Y) (norm(Y + A'*(Y\A) - eye(size(A)),'fro'));
% X is the starting matrix X_0
X = eye(m);
% Y is the starting value for the auxiliary matrix of the iterative method
Y = X;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% >>>

% Computations

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% First call of the SEAW and STEA
% X_0 is the starting matrix X
% Compute the Frobenious norms of the residual matrix
% Original method
nX(1) = nres(X);
% Frobenious norm of the first extrapolated term
nS(1) = nres(X);
% Compute <y,X_0>= trace(X_0)
EPSINIS = trace(X);
if abs(EPSINIS) > MAXD
   warning([' At iteration 1 the absolute value of the duality product is '...
     ,num2str(abs(EPSINIS),'%10.3e'),' > ', num2str(MAXD,'%15.5e')]);
end
% First call of SEAW
[EPSINIS,EPSSCA,NSING] = SEAW(EPSINIS,[],MAXCOL,TOL,NDIGIT,0);
% First call of STEA
[EPSINI,EPSVEC] = STEAf (X,[],EPSSCA,MAXCOL,TOL,0);
disp(['Iteration 1 completed']);

% Start of the loop   
for n = 1:NBC-1   
    % In output, X is the new element X_n   
    [X,Y] = Fiter_EA(X,A,Y);
    % Compute the norm of the residual (basic method)
    nX(n+1) = nres(X);
    % Compute <y,X_n> = trace(X_n)
    EPSINIS = trace(X);
    % Check its value
    if abs(EPSINIS) > MAXD
        warning([' At iteration ',num2str(n+1,'%d'), ...
           ', the absolute value of the duality product is '...
           ,num2str(abs(EPSINIS),'%10.3e'),' > ', num2str(MAXD,'%15.5e')]);
    end
    % next call of SEAW and STEA
    [EPSINIS,EPSSCA,NSING] = SEAW(EPSINIS,EPSSCA,MAXCOL,TOL,NDIGIT);
    [EPSINI,EPSVEC] = STEAf (X,EPSVEC,EPSSCA,MAXCOL,TOL);
    % Compute the norm of the residual (new extrapolated term)
    nS(n+1) = nres(EPSINI);
    disp(['Iteration ', num2str(n+1,'%d'), ' completed']);
end
% End of the loop

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Graphical representations

figure(11)
semilogy([0:NBC-1],nX(1:NBC),'k-',[0:NBC-1], ...
          nS(1:NBC),'r--','LineWidth',2)
legend('||f(X)||_F EA Method',['||f(eps)||_F ',STEAs], ...
         'LOCATION','best');
title(['An example - EA Method + Acceleration Method with ',STEAs])    
    
% Final summary
disp(' ');
disp(['An example - Acceleration Method with ',STEAs]);
disp('Basic method of S.M. El-Sayed and A.M. Al-Dbiban');
disp(' ');
disp([' Dimension m                       = ', num2str(m(1))]);
disp([' Number of terms of the sequence   = ', num2str(NBC,'%d')]);
disp([' Max even column                   = ', num2str(MAXCOL,'%d')]);
disp([' Tolerance for the algorithms      = ',num2str(TOL,'%6.1e')]);
disp([' Number of the common digits       = ',num2str(NDIGIT,'%d')]);
disp([' Maximum for the |duality product| = ',num2str(MAXD,'%6.1e')]);
disp([' Maximum number of possible isolated singularities = ', ...
    num2str(MAXCOL-3,'%d')]);
disp([' Total number of isolated singularities found in the eps-array = ', ...
    num2str(NSING,'%d')]);
disp(' ');
disp([' At iteration ', num2str(NBC,'%d'),' we obtain']);
disp([' ||f(X)||_F         = ',num2str(nX(end),'%6.2e')]);
disp([' ||f(eps)||_F       = ',num2str(nS(end),'%6.2e')]);

