function [m,Y,X,sol,param] = exinit(swi,al)  
% EXINIT Initializes the Examples 1 to 12 of the paper 
%
% [m,Y,X,sol,param] = exinit(swi,al)
%
% Inputs:
%   swi     Example number (string)
%   al      Optional (default alpha = 1). It is the value of alpha 
%           needed for Examples 1 to 5.
%
% Outputs:
%   m       Row dimension of the array
%   Y       Additional array (for duality product)
%   X       First term of the sequence
%   sol     True solution if exists, or empty value, if no solution
%   param   Cell array defining additional parameters needed for Fiter
%           function. For examples 1 to 5, param{1} contains the alpha
%           value. For the other ones, it contains other data needed.

% Authors: Claude Brezinski, Michela Redivo-Zaglia
% V1.0 - December 2016
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Define the default alpha, if not given as input argument
if nargin < 2
    al = 1;
end

% Select the example asked
switch swi
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 1    Solution of a system of nonlinear equations (K.M. Brown)
%              Figure 1 of the paper
%
    case '1'
        % Dimension
        m = 5; 
        
        Y = ones(m,1);
        X = ones(m,1)/2;
        sol = ones(m,1);

        % Set the alpha
        param{1} = al;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 2     Solution of a system of nonlinear equations. 
%               Figure 2 of the paper
%
    case '2'
        % Dimensions m=5, m=10, m=100, m=1000
        m = 10; 
        
        Y = ones(m,1);
        X = 0.5*ones(m,1)/m;
        sol = zeros(m,1);

        % Set the alpha
        param{1} = al;
        
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 3     Solution of a system of nonlinear equations. 
%               Figure 3 of the paper
%
    case '3'
        % Dimension (it is not possible to modify the dimension)
        m = 3; 
        
        Y = ones(m,1);
        X = zeros(m,1);
        sol = [-1 1 0]';
        
        % Set the alpha
        param{1} = al;
        
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 4a    Solution of a system of nonlinear equations. This system 
%               has at least two solutions. This is the first one.
%               (Figure 4 of the paper)
%
    case '4a'
        % Dimension (it is not possible to modify the dimension)
        m = 2; 
        
        Y = ones(m,1);
        % For solution [1,1]
        X = 1.3*ones(m,1); 
        sol = [1 1]';
        
        % Set the alpha
        param{1} = al;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 4b    Solution of a system of nonlinear equations. This system 
%               has at least two solutions. This is the second one. 
%               (Figure 5 of the paper)
%
    case '4b'
        % Dimension (it is not possible to modify the dimension)
        m = 2; 
        
        Y = ones(m,1);     
        % For solution [-2,-2]
        X = -ones(m,1);
        sol = [-2 -2]';

        % Set the alpha
        param{1} = al;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 5     Solution of a system of nonlinear equations (Grau Sanchez 
%               et al)
%               Figure 6 of the paper
%
    case '5'
        % Dimension (it is not possible to modify the dimension)
        m = 7; 
        
        Y = ones(m,1);
        X = ones(m,1)/10;
        sol = 0.14427495072088622350*ones(m,1);

        % Set the alpha
        param{1} = al;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 6     Solution of a system of linear equations with several
%               right hand sides AX = B.
%               Figure 7 of the paper
%
    case '6'
        % Dimension of A
        m = 5;
        % Number of right hand sides
        s = 2; 
        
        Y = ones(m,s);
        X = zeros(m,s);

        % Define the solutions (first s colums of matrix pei)
        sol = eye(m,s)+ones(m,s);
        
        % Set the parameters needed
        param{1} = gallery('parter',m)/3;  %%% A
        param{2} = param{1}*sol;           %%% B

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 7     Solution of a matrix equation (C.-H. Guo)
%               Figure 8 of the paper
%
    case '7'
        % Dimension
        m = 5; 
        
        Y = eye(m,m);
        
        % Set the parameters needed
        param{1} = gallery('prolate',m);        % S
        r = 1.1*max(abs(eig(param{1})));        % r > \rho(S)
        param{1} = param{1}/r;                  % A = S/r
        param{2} = eye(m,m)+param{1}'*param{1}; % Q = I + A^* A
        X = param{2};                           % X_0 = Q
        sol = eye(m,m);                         % X_+
        
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 8a    Solution of a matrix equation (Z.-Z. Bai). 
%               The solution is known when the A_i = p^i A_0, A_0 given.
%               Otherwise the solution is unknown.
%               In this example the solution is set unknown, but the data
%               are the same as Example 8b.
%               (Figure 9 of the paper)
%
    case '8a'
        % Dimension (it is not possible to modify the dimension with the
        % following choice of A_0. It is possible if user modify A_0 with
        % right dimension)
        m = 5; 
        
        Y = eye(m,m);
        
        % Define the matrix B_0
        param{1} = eye(m,m);                             
        % Define the value of p (it can be changed, but 0<p<1 )
        param{3}=0.47;  
        
        % Define the matrix A_0
        param{2}=[0.05 0.1 0.2 0.3 0.1; 0.2 0.05 0.1 0.1 0.3; ...
            0.1 0.2 0.3 0.05 0.1; 0.1 0.05 0.2 0.1 0.3; ...
            0.3 0.1 0.1 0.2 0.05]; 
        param{2}=4/3*(1-param{3})*param{2}; 
        
        X = param{1}*param{2};
        sol=[];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 8b    Solution of a matrix equation (Z.-Z. Bai). 
%               The solution is known when the A_i = p^i A_0, A_0 given.
%               Otherwise the solution is unknown.
%               In this example the solution is known.
%
    case '8b'
        % Dimension (it is not possible to modify the dimension)
        m = 5; 
        
        Y = eye(m,m);
        
        % Define the matrix B_0
        param{1} = eye(m,m);                             
        % Define the value of p (it can be changed, but 0<p<1 )
        param{3}=0.47;  
        
        % Define the matrix A_0
        param{2}=[0.05 0.1 0.2 0.3 0.1; 0.2 0.05 0.1 0.1 0.3; ...
            0.1 0.2 0.3 0.05 0.1; 0.1 0.05 0.2 0.1 0.3; ...
            0.3 0.1 0.1 0.2 0.05]; 
        param{2}=4/3*(1-param{3})*param{2}; 
        
        X = param{1}*param{2};
        
        % sol = (I-sqrtm(I-4pA_0))/(2p)
        sol=(eye(m,m)-sqrtm(eye(m,m)-4*param{3}*param{2}))/(2*param{3});
        
        % The following expressions theoretically are zero matrices
        % sol*(eye(n,n)-param{3}*sol)-param{2}
        % param{2}*inv(eye(n,n)-param{3}*sol)-sol


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 9      Computation of the exponential function exp(A t)
%                Only for Acceleration method.
%                Figure 10 of the paper
%
    case '9'
        % Dimension
        m = 100; 
        
        Y = eye(m,m);
        
        % Define A t
        D = diag(diag(gallery('frank',m)));
        U = gallery('orthog',m);
        t = -0.099;  
        param{1}=U*D*U'*t; 

        X=eye(m,m); 
        sol=U*diag(diag(exp(D*t)))*U';
 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 10     Computation of the inverse of (I - A t)
%                Only for Acceleration method.
%                Figure 11 of the paper
%
    case '10'
        % Dimension
        m = 50; 
        
        Y = eye(m,m);
        
        % Define A t
        D = diag(diag(gallery('frank',m)));
        U = gallery('orthog',m);
        t = -0.0099;
        param{1}=U*D*U'*t; 

        X=eye(m,m);
        sol=inv(eye(m,m)-param{1});

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 11     Computation of the square root of A
%                Only for Acceleration method.
%                Figure 12 of the paper
%
    case '11'
        % Dimension
        m = 50; 
        
        Y = eye(m,m);
        
        % Define A 
        param{1} = gallery('moler',m); 

        X = param{1};        % X_0 = A
        sol=sqrtm(param{1});

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% EXEMPLE 12     Binomial iteration for the square root of (I-C)
%                Only for Acceleration method.
%                Figure 13 of the paper
%
    case '12'
        % Dimension
        m = 500; 
        
        Y = eye(m,m);
        
        % Define C 
        param{1} = gallery('moler',m)./110000; 

        X = zeros(m,m);        % X_0 = 0
        sol=eye(m,m) - sqrtm(eye(m,m)-param{1});
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    otherwise
        error(' exinit - The example chosen do not exist')
end